/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* statistics_cb.c -- GUI functions for the statistics window */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#if STDC_HEADERS
#include <stdlib.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#endif

#include <message_windows.h>
#include <sql_common.h>
#include <sql_stat.h>
#include <schedwi_time.h>
#include <statistics_cb.h>


/*
 * Initialize the statistics
 *
 * Return:
 *       0 --> No error
 *   other --> SQL error (a message has been displayed to the user)
 */
static int
dialog_statistics_init (GtkWidget *dialog_statistics, const gchar *id,
			const gchar *name)
{
	char *s;
	char **row;
	GtkWidget *label;
	guint64 total, num_run;

	/* Dialog name */
	if (name != NULL && name[0] != '\0') {
		s = g_strdup_printf (_("Statistics for %s"), name);
		gtk_window_set_title (GTK_WINDOW (dialog_statistics), s);
		g_free (s);
	}

	/* Retrieve and fill the fields in the dialog */
	if (sql_stat_get (id, &row,
			(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
			_("Database error")) != 0)
	{
		return -1;
	}

	/* Successful runs */
	label = lookup_widget (dialog_statistics, "label_statistics_success");
	gtk_label_set_text (GTK_LABEL (label), (row != NULL) ? row[0] : "0");

	/* Failed runs */
	label = lookup_widget (dialog_statistics, "label_statistics_failed");
	gtk_label_set_text (GTK_LABEL (label), (row != NULL) ? row[1] : "0");

	/* Average duration */
	label = lookup_widget (	dialog_statistics,
				"label_statistics_avg_duration");
	if (row == NULL) {
		gtk_label_set_text (GTK_LABEL (label), "0");
	}
	else {
		total = g_ascii_strtoull (row[2], NULL, 0);
		num_run = g_ascii_strtoull (row[0], NULL, 0);
		if (num_run > 0) {
			s = schedwi_time_duration_to_str (total / num_run);
		}
		else {
			s = NULL;
		}
		if (s == NULL) {
			gtk_label_set_text (GTK_LABEL (label), _("Unknown"));
		}
		else {
			gtk_label_set_text (GTK_LABEL (label), s);
			free (s);
		}
	}

	/* Last run duration */
	label = lookup_widget (	dialog_statistics,
				"label_statistics_last_duration");
	if (row == NULL) {
		gtk_label_set_text (GTK_LABEL (label), "0");
	}
	else {
		total = g_ascii_strtoull (row[3], NULL, 0);
		s = schedwi_time_duration_to_str (total);
		if (s == NULL) {
			gtk_label_set_text (GTK_LABEL (label), _("Unknown"));
		}
		else {
			gtk_label_set_text (GTK_LABEL (label), s);
			free (s);
		}
	}

	sql_free_row (row);
	return 0;
}


/*
 * Create a statistics dialog window
 *
 * Return:
 *   The new GtkWidget (`show'ed by this function) or
 *   NULL in case of error (an error message has been displayed for the user)
 */
GtkWidget *
new_dialog_statistics (const gchar *id, const gchar *name)
{
	GtkWidget *widget;

	widget = create_dialog_statistics ();
	if (dialog_statistics_init (widget, id, name) != 0) {
		gtk_widget_destroy (widget);
		return NULL;
	}
	else {
		gtk_widget_show (widget);
		return widget;
	}
}

/*------------------------======= End Of File =======------------------------*/
