/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* sql_common.c -- Useful MYSQL functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <pthread.h>

#include <mysql/mysql.h>
#include <lib_functions.h>
#include <conf.h>
#include <sql_common.h>


static MYSQL *sql = NULL;
static int nb_handlers = 0;


/*
 * Mutex to protect the SQL request (around mysql_query()
 * and mysql_store_result()
 */
static pthread_mutex_t lock = PTHREAD_MUTEX_INITIALIZER;


/*
 * Establish a database connection.
 *
 * Return:
 *   0 --> No error.  The static sql handler is opened
 *  -1 --> Error.  If err_msg is not NULL, it is set with the error message
 *         (it must be freed by the caller)
 */
static int
init_db_connection (char **err_msg)
{
	const char *host, *user, *passwd, *db, *unix_socket;
	char *msg;
	long int port;

	/* Get the connection parameter from the configuration file */
	host = user = passwd = db = unix_socket = NULL;
	port = 0;
	conf_get_param_string ("DB_HOST", &host);
	conf_get_param_string ("DB_USER", &user);
	conf_get_param_string ("DB_PASSWD", &passwd);
	conf_get_param_string ("DB_DATABASE", &db);
	conf_get_param_int ("DB_PORT", &port);
	conf_get_param_string ("DB_UNIXSOCKET", &unix_socket);

	sql = mysql_init (NULL);
	if (sql == NULL) {
		if (err_msg != NULL) {
			msg = _("Memory allocation error");
			*err_msg = (char *) malloc (schedwi_strlen (msg) + 1);
			if (*err_msg != NULL) {
				strcpy (*err_msg, msg);
			}
		}
		return -1;
	}

	mysql_options (sql, MYSQL_READ_DEFAULT_GROUP, PACKAGE);

	if (mysql_real_connect (sql, host, user, passwd, db,
				(unsigned int)port, unix_socket, 0) == NULL)
	{
		compose_error_message (	_("Failed to connect to database"),
					err_msg);
		mysql_close (sql);
		sql = NULL;
		return -1;
	}
	return 0;
}


/*
 * Free the database connection
 */
void
destroy_db_connection ()
{
	if (sql != NULL) {
		mysql_close (sql);
		sql = NULL;
		nb_handlers = 0;
	}
}


/*
 * Get a MYSQL handler
 *
 * Return:
 *    The handler or
 *    NULL in case of error.  If err_msg is not NULL, it is set with the error
 *    message (it must be freed by the caller)
 */
MYSQL *
begin_mysql (char **err_msg)
{
	if (sql == NULL || nb_handlers <= 0) {
		if (init_db_connection (err_msg) != 0) {
			return NULL;
		}
		nb_handlers = 1;
	}
	else {
		nb_handlers++;
	}
	return sql;
}


/*
 * Free a MYSQL handler
 */
void
end_mysql ()
{
	nb_handlers--;
	if (sql != NULL && nb_handlers <= 0) {
		destroy_db_connection ();
	}
}


/*
 * Built an SQL error message
 *
 * Return:
 *   0 --> No error (if not NULL, msg_out contains the error message
 *         to be freed by the caller)
 *  -1 --> Memory allocation error
 */
int
compose_error_message (const char *msg, char **msg_out)
{
	const char *msg_sql;
	char *separator;
	unsigned int msg_len;

	if (msg_out == NULL) {
		return 0;
	}

	msg_sql = mysql_error (sql);

	if (msg == NULL) {
		msg_len = 0;
	}
	else {
		msg_len = schedwi_strlen (msg);
	}

	separator = _(": ");
	*msg_out = (char *) malloc (	  msg_len
					+ schedwi_strlen (separator)
					+ schedwi_strlen (msg_sql)
					+ 1);
	if (*msg_out == NULL) {
		return -1;
	}

	(*msg_out)[0] = '\0';
	if (msg != NULL) {
		strcpy (*msg_out, msg);
	}
	if (msg != NULL && msg_sql[0] != '\0') {
		strcat (*msg_out, separator);
	}
	strcat (*msg_out, msg_sql);
	return 0;
}


/*
 * Escape the provided string (if the string is NULL, an empty string is
 * returned - ie. the first char is 0 and the string still need to be
 * freed by the caller)
 *
 * Return:
 *   The escaped string (to be freed by the caller) or
 *   NULL in case of error
 */
char *
sql_escape (MYSQL *sql, const char *s)
{
	char *res;
	unsigned int len;

#if HAVE_ASSERT_H
	assert (sql != NULL);
#endif

	len = (s != NULL) ? schedwi_strlen (s) : 0;

	res = (char *) malloc (len * 2 + 1);
	if (res == NULL) {
		return NULL;
	}
	if (s == NULL) {
		res[0] = '\0';
	}
	else {
		mysql_real_escape_string(sql, res, s, len);
	}
	return res;
}


/*
 * Escape the provided binary string (if the string is NULL, an empty string is
 * returned - ie. the first char is 0 and the string still need to be
 * freed by the caller)
 *
 * Return:
 *   The escaped string (to be freed by the caller) or
 *   NULL in case of error
 */
static char *
sql_escape_bin (MYSQL *sql, const char *s, unsigned long int len)
{
	char *res;

#if HAVE_ASSERT_H
	assert (sql != NULL);
#endif

	res = (char *) malloc (len * 2 + 1);
	if (res == NULL) {
		return NULL;
	}
	if (s == NULL) {
		res[0] = '\0';
	}
	else {
		mysql_real_escape_string(sql, res, s, len);
	}
	return res;
}


#define MAX_NUM_ARGS 10

struct sql_args {
	sql_type type;
	union {
		long int value_int;
		char *value_string;
		char value_bool;
	};
};
typedef struct sql_args sql_args_t;


/*
 * Free the content of a sql_args array
 */
static void
free_args (sql_args_t *args, int len)
{
	if (args != NULL) {
		while (--len >= 0) {
			if (args[len].type == SQL_STRING) {
				if (args[len].value_string != NULL) {
					free (args[len].value_string);
				}
			}
		}
	}
}


/*
 * Run a database request
 *
 * buf (and buf_len) is used as a working buffer.  If buf is NULL, an internal
 * buffer is used. If *buf is NULL, sql_request() will allocate a buffer for
 * internal use (*buf must be freed by the caller) and *buf (and *buf_len) will
 * be set with this buffer.  Alternatively, before calling sql_request() *buf
 * can contain a pointer to a malloc()-allocated buffer *buf_len bytes in size.
 * If the buffer is not large enough, it will be resized, updating *buf and
 * *buf_len to reflect the buffer address and size.
 *
 * The variable argument list should contain a sql_type followed by the value.
 * For the sql_type SQL_BIN, the value must be followed by the lenght of the
 * string value.
 * The list is terminated by SQL_END.  For examples:
 *
 *     sql_request (	sql, NULL, NULL, &err,
 *   			"DELETE FROM hosts WHERE id=%ld AND hostname=\"%s\"",
 *   			SQL_INT, 12,
 *   			SQL_STRING, "foo.company.org",
 *   			SQL_END);
 *
 *     img_data = (char *)gdk_pixdata_serialize (pixdata, &len_img_data);
 *     sql_request (	sql, NULL, NULL, &err,
 *			"INSERT INTO imgs (id,data) VALUES (%ld,\"%s\"),
 *			SQL_INT, 144,
 *			SQL_BIN, img_data, len_img_data,
 *                      SQL_END);
 *     g_free (img_data);
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message)
 * other --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 *           The returned value is the one returned by mysql_errno()
 */
static unsigned int
sql_request (	MYSQL *sql, char **buf, unsigned int *buf_len, char **err_msg,
		const char *fmt, va_list ap)
{
	sql_args_t args[MAX_NUM_ARGS];
	sql_type t;
	int i, ret;
	unsigned int sql_ret;
	char c;
	char *s;
	unsigned int req_len;
	char *req;
	char req_to_free;
	unsigned long int bin_len;

#if HAVE_ASSERT_H
	assert (sql != NULL && fmt != NULL);
#endif

	req_len = 0;
	for (	i = 0;
		i < MAX_NUM_ARGS && (t = va_arg (ap, sql_type)) != SQL_END;
		i++)
	{
		args[i].type = t;
		switch (t) {
			case SQL_INT:
				args[i].value_int = va_arg (ap, long int);
				req_len += 22;
				break;

			case SQL_BOOL:
				c = va_arg (ap, int);
				args[i].value_bool = (c == 0 || c == '0') ?0:1; 
				req_len += 1;
				break;

			case SQL_STRING:
				/* Escape the provided string */
				s = sql_escape (sql, va_arg (ap, char *));
				if (s == NULL) {
					free_args (args, i);
					if (err_msg != NULL) {
						s =_("Memory allocation error");
						*err_msg = (char *) malloc (
							schedwi_strlen (s) + 1);
						if (*err_msg != NULL) {
							strcpy (*err_msg, s);
						}
					}
					return 1;
				}
				args[i].value_string = s;
				req_len += schedwi_strlen (s);
				break;

			case SQL_STRING_NON_ESCAPE:
				s = va_arg (ap, char *);
				args[i].value_string = s;
				req_len += schedwi_strlen (s);
				break;

			case SQL_BIN:
				/* Escape the binary string */
				s = va_arg (ap, char *);
				bin_len = va_arg (ap, unsigned long int);
				s = sql_escape_bin (sql, s, bin_len);
				if (s == NULL) {
					free_args (args, i);
					if (err_msg != NULL) {
						s =_("Memory allocation error");
						*err_msg = (char *) malloc (
							schedwi_strlen (s) + 1);
						if (*err_msg != NULL) {
							strcpy (*err_msg, s);
						}
					}
					return 1;
				}
				args[i].value_string = s;
				req_len += schedwi_strlen (s);
				break;

			default:
				break;
		}
	}

	/* Fill the end of the args array with NULL strings */
	while (i < MAX_NUM_ARGS) {
		args[i].type = SQL_STRING;
		args[i].value_string = NULL;
		i++;
	}

	/* Allocate memory for the SQL request */
	req_len += schedwi_strlen (fmt) + 1;
	if (buf == NULL || buf_len == NULL) {
		/* Create a new buffer */
		req = (char *) malloc (req_len);
		if (req == NULL) {
			free_args (args, MAX_NUM_ARGS);
			if (err_msg != NULL) {
				s = _("Memory allocation error");
				*err_msg = (char *) malloc (schedwi_strlen (s)
								+ 1);
				if (*err_msg != NULL) {
					strcpy (*err_msg, s);
				}
			}
			return 1;
		}
		req_to_free = 1;
	}
	else {
		/* Try to use the provided buffer */
		if (*buf_len < req_len || *buf == NULL) {
			if (*buf != NULL) {
				free (*buf);
			}
			*buf = (char *) malloc (req_len);
			if (*buf == NULL) {
				free_args (args, MAX_NUM_ARGS);
				*buf_len = 0;
				if (err_msg != NULL) {
					s = _("Memory allocation error");
					*err_msg = (char *) malloc (
							schedwi_strlen (s) + 1);
					if (*err_msg != NULL) {
						strcpy (*err_msg, s);
					}
				}
				return 1;
			}
			*buf_len = req_len;
		}
		req = *buf;
		req_to_free = 0;
	}

	/* Build the request */
	ret = snprintf (req, req_len, fmt,
		(   args[0].type == SQL_STRING || args[0].type == SQL_BIN
		 || args[0].type == SQL_STRING_NON_ESCAPE)
			? args[0].value_string
			: ((args[0].type == SQL_INT) 	? args[0].value_int
							: args[0].value_bool)
#if MAX_NUM_ARGS > 1
		,(  args[1].type == SQL_STRING || args[1].type == SQL_BIN
		 || args[1].type == SQL_STRING_NON_ESCAPE)
			? args[1].value_string
			: ((args[1].type == SQL_INT)	? args[1].value_int
							: args[1].value_bool)
#if MAX_NUM_ARGS > 2
		,(  args[2].type == SQL_STRING || args[2].type == SQL_BIN
		 || args[2].type == SQL_STRING_NON_ESCAPE)
			? args[2].value_string
			: ((args[2].type == SQL_INT)	? args[2].value_int
							: args[2].value_bool)
#if MAX_NUM_ARGS > 3
		,(  args[3].type == SQL_STRING || args[3].type == SQL_BIN
		 || args[3].type == SQL_STRING_NON_ESCAPE)
			? args[3].value_string
			: ((args[3].type == SQL_INT)	? args[3].value_int
							: args[3].value_bool)
#if MAX_NUM_ARGS > 4
		,(  args[4].type == SQL_STRING || args[4].type == SQL_BIN
		 || args[4].type == SQL_STRING_NON_ESCAPE)
			? args[4].value_string
			: ((args[4].type == SQL_INT)	? args[4].value_int
							: args[4].value_bool)
#if MAX_NUM_ARGS > 5
		,(  args[5].type == SQL_STRING || args[5].type == SQL_BIN
		 || args[5].type == SQL_STRING_NON_ESCAPE)
			? args[5].value_string
			: ((args[5].type == SQL_INT)	? args[5].value_int
							: args[5].value_bool)
#if MAX_NUM_ARGS > 6
		,(  args[6].type == SQL_STRING || args[6].type == SQL_BIN
		 || args[6].type == SQL_STRING_NON_ESCAPE)
			? args[6].value_string
			: ((args[6].type == SQL_INT)	? args[6].value_int
							: args[6].value_bool)
#if MAX_NUM_ARGS > 7
		,(  args[7].type == SQL_STRING || args[7].type == SQL_BIN
		 || args[7].type == SQL_STRING_NON_ESCAPE)
			? args[7].value_string
			: ((args[7].type == SQL_INT)	? args[7].value_int
							: args[7].value_bool)
#if MAX_NUM_ARGS > 8
		,(  args[8].type == SQL_STRING || args[8].type == SQL_BIN
		 || args[8].type == SQL_STRING_NON_ESCAPE)
			? args[8].value_string
			: ((args[8].type == SQL_INT)	? args[8].value_int
							: args[8].value_bool)
#if MAX_NUM_ARGS > 9
		,(  args[9].type == SQL_STRING || args[9].type == SQL_BIN
		 || args[9].type == SQL_STRING_NON_ESCAPE)
			? args[9].value_string
			: ((args[9].type == SQL_INT)	? args[9].value_int
							: args[9].value_bool)
#if MAX_NUM_ARGS > 10
		,(  args[10].type == SQL_STRING || args[10].type == SQL_BIN
		 || args[10].type == SQL_STRING_NON_ESCAPE)
			? args[10].value_string
			: ((args[10].type == SQL_INT)	? args[10].value_int
							: args[10].value_bool)
#if MAX_NUM_ARGS > 11
		,(  args[11].type == SQL_STRING || args[11].type == SQL_BIN
		 || args[11].type == SQL_STRING_NON_ESCAPE)
			? args[11].value_string
			: ((args[11].type == SQL_INT)	? args[11].value_int
							: args[11].value_bool)
#if MAX_NUM_ARGS > 12
		,(  args[12].type == SQL_STRING || args[12].type == SQL_BIN
		 || args[12].type == SQL_STRING_NON_ESCAPE)
			? args[12].value_string
			: ((args[12].type == SQL_INT)	? args[12].value_int
							: args[12].value_bool)
#if MAX_NUM_ARGS > 13
		,(  args[13].type == SQL_STRING || args[13].type == SQL_BIN
		 || args[13].type == SQL_STRING_NON_ESCAPE)
			? args[13].value_string
			: ((args[13].type == SQL_INT)	? args[13].value_int
							: args[13].value_bool)
#if MAX_NUM_ARGS > 14
		,(  args[14].type == SQL_STRING || args[14].type == SQL_BIN
		 || args[14].type == SQL_STRING_NON_ESCAPE)
			? args[14].value_string
			: ((args[14].type == SQL_INT)	? args[14].value_int
							: args[14].value_bool)
#if MAX_NUM_ARGS > 15
		,(  args[15].type == SQL_STRING || args[15].type == SQL_BIN
		 || args[15].type == SQL_STRING_NON_ESCAPE)
			? args[15].value_string
			: ((args[15].type == SQL_INT)	? args[15].value_int
							: args[15].value_bool)
#if MAX_NUM_ARGS > 16
		,(  args[16].type == SQL_STRING || args[16].type == SQL_BIN
		 || args[16].type == SQL_STRING_NON_ESCAPE)
			? args[16].value_string
			: ((args[16].type == SQL_INT)	? args[16].value_int
							: args[16].value_bool)
#if MAX_NUM_ARGS > 17
		,(  args[17].type == SQL_STRING || args[17].type == SQL_BIN
		 || args[17].type == SQL_STRING_NON_ESCAPE)
			? args[17].value_string
			: ((args[17].type == SQL_INT)	? args[17].value_int
							: args[17].value_bool)
#if MAX_NUM_ARGS > 18
		,(  args[18].type == SQL_STRING || args[18].type == SQL_BIN
		 || args[18].type == SQL_STRING_NON_ESCAPE)
			? args[18].value_string
			: ((args[18].type == SQL_INT)	? args[18].value_int
							: args[18].value_bool)
#if MAX_NUM_ARGS > 19
		,(  args[19].type == SQL_STRING || args[19].type == SQL_BIN
		 || args[19].type == SQL_STRING_NON_ESCAPE)
			? args[19].value_string
			: ((args[19].type == SQL_INT)	? args[19].value_int
							: args[19].value_bool)
#if MAX_NUM_ARGS > 20
		,(  args[20].type == SQL_STRING || args[20].type == SQL_BIN
		 || args[20].type == SQL_STRING_NON_ESCAPE)
			? args[20].value_string
			: ((args[20].type == SQL_INT)	? args[20].value_int
							: args[20].value_bool)
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
	);

#if HAVE_ASSERT_H
	assert (ret < req_len && ret >=0);
#endif

	free_args (args, MAX_NUM_ARGS);

	/* Run the request */
	if (mysql_query (sql, req) != 0) {
		if (req_to_free != 0) {
			free (req);
		}
		sql_ret = mysql_errno (sql);
		compose_error_message (_("Database error"), err_msg);
		return sql_ret;
	}

	if (req_to_free != 0) {
		free (req);
	}
	return 0;
}


/*
 * Run a non-select database request (insert, delete, update, ...)
 *
 * buf (and buf_len) is used as a working buffer.  If buf is NULL, an internal
 * buffer is used. If *buf is NULL, sql_non_select() will allocate a buffer for
 * internal use (*buf must be freed by the caller) and *buf (and *buf_len) will
 * be set with this buffer.  Alternatively,  before calling sql_non_select()
 * *buf can contain a pointer to a malloc()-allocated buffer *buf_len bytes in
 * size.  If the buffer is not large enough, it will be resized, updating *buf
 * and *buf_len to reflect the buffer address and size.
 *
 * The variable argument list should contain a sql_type followed by the value.
 * The list is terminated by SQL_END.  For example:
 *
 *     sql_non_select (	NULL, NULL, &err, NULL, NULL,
 *   			"DELETE FROM hosts WHERE id=%ld AND hostname=\"%s\"",
 *   			SQL_INT, 12,
 *   			SQL_STRING, "foo.company.org",
 *   			SQL_END);
 *
 * Return:
 *     0 --> No error (if id is not NULL it contains the value of the last ID
 *           inserted or updated in the database.  This is only relevant for
 *           INSERT and UPDATE requests)
 *     1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message)
 *     2 --> Database connection error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message)
 * other --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 *           The returned value is the one returned by mysql_errno()
 */
unsigned int
sql_non_select (	char **buf, unsigned int *buf_len, char **err_msg,
			unsigned long int *id,
			unsigned long long int *affected_rows,
			const char *fmt, ...)
{
	va_list ap;
	MYSQL *sql;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (fmt != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (err_msg);
	if (sql == NULL) {
		return 2;
	}

	/* Run the request */
	pthread_mutex_lock (&lock);
	va_start (ap, fmt);
	ret = sql_request (sql, buf, buf_len, err_msg, fmt, ap);
	va_end (ap);

	if (ret != 0) {
		pthread_mutex_unlock (&lock);
		end_mysql ();
		return ret;
	}

	/* Get the number of affected rows */
	if (affected_rows != NULL) {
		*affected_rows = (unsigned long long int)
					mysql_affected_rows (sql);
	}

	/* Get the last id inserted in the database */
	if (id != NULL) {
		*id = (unsigned long int) mysql_insert_id (sql);
	}
	pthread_mutex_unlock (&lock);
	
	end_mysql ();
	return 0;
}


/*
 * Free a result row
 */
void
sql_free_row (char **row)
{
	int i;

	if (row != NULL) {
		for (i = 0; row[i] != NULL; i++) {
			free (row[i]);
		}
		free (row);
	}
}


/*
 * Run a select database request
 *
 * buf (and buf_len) is used as a working buffer.  If buf is NULL, an internal
 * buffer is used. If *buf is NULL, sql_select() will allocate a buffer for
 * internal use (*buf must be freed by the caller) and *buf (and *buf_len) will
 * be set with this buffer.  Alternatively,  before calling sql_select() *buf
 * can contain a pointer to a malloc()-allocated buffer *buf_len bytes in size.
 * If the buffer is not large enough, it will be resized, updating *buf and
 * *buf_len to reflect the buffer address and size.
 *
 * The variable argument list should contain a sql_type followed by the value.
 * For the sql_type SQL_BIN, the value must be followed by the lenght of the
 * string value.
 * The list is terminated by SQL_END.  For example:
 *
 *         sql_select (	NULL, NULL, &err, NULL, &list_row, NULL,
 *   			"SELECT FROM hosts WHERE id=%ld AND hostname=\"%s\"",
 *   			SQL_INT, 12,
 *   			SQL_STRING, "foo.company.org",
 *   			SQL_END);
 *
 * Return:
 *     0 --> No error.  rows contains the list of rows and must be freed by the
 *           caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *           If not NULL, fields_len contains the list of fields len and must
 *           be freed by the caller by
 *              lwc_delLL (fields_len, (void (*)(const void *)) free);
 *     1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message)
 *     2 --> Database connection error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message)
 * other --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 *           The returned value is the one returned by mysql_errno()
 */
unsigned int
sql_select (	char **buf, unsigned int *buf_len, char **err_msg,
		unsigned int *num_columns, lwc_LL **rows, lwc_LL **fields_len,
		const char *fmt, ...)
{
	va_list ap;
	MYSQL *sql;
	lwc_LL *list, *list_len;
	char *s;
	MYSQL_RES *result;
        MYSQL_ROW row;
	unsigned int cols, i, ret;
	char **row_array;
	unsigned long int *lengths, *len_array;

#if HAVE_ASSERT_H
	assert (rows != NULL && fmt != NULL);
#endif

	/* Allocate memory for the linked list */
	list = lwc_newLL ();
	if (list == NULL) {
		if (err_msg != NULL) {
			s = _("Memory allocation error");
			*err_msg = (char *) malloc (schedwi_strlen (s) + 1);
			if (*err_msg != NULL) {
				strcpy (*err_msg, s);
			}
		}
		return 1;
	}

	/* Allocate memory for the field lenghts linked list */
	if (fields_len != NULL) {
		list_len = lwc_newLL ();
		if (list_len == NULL) {
			lwc_delLL (list, NULL);
			if (err_msg != NULL) {
				s = _("Memory allocation error");
				*err_msg = (char *) malloc (schedwi_strlen (s)
								+ 1);
				if (*err_msg != NULL) {
					strcpy (*err_msg, s);
				}
			}
			return 1;
		}
	}
	else {
		list_len = NULL;
	}
		

	/* Connect to the database */
	sql = begin_mysql (err_msg);
	if (sql == NULL) {
		lwc_delLL (list, NULL);
		lwc_delLL (list_len, NULL);
		return 2;
	}

	/* Run the request */
	pthread_mutex_lock (&lock);
	va_start (ap, fmt);
	ret = sql_request (sql, buf, buf_len, err_msg, fmt, ap);
	va_end (ap);

	if (ret != 0) {
		pthread_mutex_unlock (&lock);
		end_mysql ();
		lwc_delLL (list, NULL);
		lwc_delLL (list_len, NULL);
		return ret;
	}

	/* Get the result */
	result = mysql_store_result (sql);
	pthread_mutex_unlock (&lock);
	if (result == NULL) {
		ret = mysql_errno(sql);
		compose_error_message (_("SELECT"), err_msg);
		end_mysql ();
		lwc_delLL (list, NULL);
		lwc_delLL (list_len, NULL);
		return ret;
	}

	/* Copy each returned row */
	cols = mysql_num_fields (result);
	while ((row = mysql_fetch_row (result)) != NULL) {

		/* Allocate a new row array */
		row_array = (char **) malloc (sizeof (char *) * (cols + 1));
		if (row_array == NULL) {
			mysql_free_result(result);
			end_mysql ();
			lwc_delLL (list, (void (*)(const void *))sql_free_row);
			lwc_delLL (list_len, (void (*)(const void *))free);
			if (err_msg != NULL) {
				s = _("Memory allocation error");
				*err_msg = (char *) malloc (schedwi_strlen (s)
								+ 1);
				if (*err_msg != NULL) {
					strcpy (*err_msg, s);
				}
			}
			return 1;
		}
		row_array[cols] = NULL;

		/* Retrieve the length of the fields in the row */
		lengths = mysql_fetch_lengths (result);

		/* Allocate a new row length array */
		if (fields_len != NULL) {
			len_array = (unsigned long int *) malloc (
					sizeof (unsigned long int) * cols);
			if (len_array == NULL) {
				mysql_free_result(result);
				end_mysql ();
				lwc_delLL (list, (void (*)(const void *))
								sql_free_row);
				lwc_delLL (list_len, (void (*)(const void *))
								free);
				if (err_msg != NULL) {
					s = _("Memory allocation error");
					*err_msg = (char *) malloc (
							schedwi_strlen (s) + 1);
					if (*err_msg != NULL) {
						strcpy (*err_msg, s);
					}
				}
				return 1;
			}
			schedwi_memcpy (len_array, lengths,
					sizeof (unsigned long int) * cols);
			if (lwc_addEndLL (list_len, len_array) != 0) {
				mysql_free_result(result);
				end_mysql ();
				lwc_delLL (list, (void (*)(const void *))
								sql_free_row);
				lwc_delLL (list_len, (void (*)(const void *))
								free);
				free (len_array);
				if (err_msg != NULL) {
					s = _("Memory allocation error");
					*err_msg = (char *) malloc (
							schedwi_strlen (s) + 1);
					if (*err_msg != NULL) {
						strcpy (*err_msg, s);
					}
				}
				return 1;
			}
		}

		/* Copy the returned row */
		for (i = 0; i < cols; i++) {
			row_array[i] = (char *) malloc (lengths[i] + 1);
			if (row_array[i] == NULL) {
				mysql_free_result(result);
				end_mysql ();
				lwc_delLL (list, (void (*)(const void *))
								sql_free_row);
				lwc_delLL (list_len, (void (*)(const void *))
								free);
				sql_free_row (row_array);
				if (err_msg != NULL) {
					s = _("Memory allocation error");
					*err_msg = (char *) malloc (
							schedwi_strlen (s) + 1);
					if (*err_msg != NULL) {
						strcpy (*err_msg, s);
					}
				}
				return 1;
			}
			schedwi_memcpy (row_array[i], row[i], lengths[i]);
			row_array[i][lengths[i]] = '\0';
		}

		/* Add the new row to the linked list */
		if (lwc_addEndLL (list, row_array) != 0) {
			mysql_free_result(result);
			end_mysql ();
			lwc_delLL (list, (void (*)(const void *))sql_free_row);
			lwc_delLL (list_len, (void (*)(const void *))free);
			sql_free_row (row_array);
			if (err_msg != NULL) {
				s = _("Memory allocation error");
				*err_msg = (char *) malloc (schedwi_strlen (s)
								+ 1);
				if (*err_msg != NULL) {
					strcpy (*err_msg, s);
				}
			}
			return 1;
		}
	}

	mysql_free_result(result);
	end_mysql ();
	if (num_columns != NULL) {
		*num_columns = cols;
	}
	*rows = list;
	if (fields_len != NULL) {
		*fields_len = list_len;
	}
	return 0;
}


/*
 * Function to call at the begining of a new thread
 */
void
sql_thread_init ()
{
	mysql_thread_init ();
}


/*
 * Function to call before exiting from a thread
 */
void
sql_thread_end ()
{
	mysql_thread_end ();
}

/*-----------------============== End Of File ==============-----------------*/
