/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* pref_cb.c -- GUI functions for the preferences window */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#include <math.h>

#include <schedwi_gconf.h>
#include <database_test_cb.h>
#include <pref_cb.h>


/*
 * Internal structure used to store the initial database parameters to
 * restore them if the user changed them and asks to restore them
 */
struct pref_backup_database_params {
	gchar *host, *user, *database, *socket_file;
	gint port;
};
typedef struct pref_backup_database_params pref_backup_database_params_t;


/*
 * Create and return a new pref_backup_database_params_t object.  The
 * returned object must be freed by destroy_pref_backup_database_params()
 */
static pref_backup_database_params_t *
new_pref_backup_database_params ()
{
	pref_backup_database_params_t *ptr;

	ptr = g_new0 (pref_backup_database_params_t, 1);
	ptr->host = gconf_client_get_string (schedwi_gconf,
					GCONF_PATH "/database/host",
					NULL);
	ptr->user = gconf_client_get_string (schedwi_gconf,
					GCONF_PATH "/database/username",
					NULL);
	ptr->database = gconf_client_get_string (schedwi_gconf,
					GCONF_PATH "/database/database",
					NULL);
	ptr->port = gconf_client_get_int (schedwi_gconf,
					GCONF_PATH "/database/port",
					NULL);
	ptr->socket_file = gconf_client_get_string (schedwi_gconf,
					GCONF_PATH "/database/socket",
					NULL);
	return ptr;
}


/*
 * Destroy the provided pref_backup_database_params_t object
 */
static void
destroy_pref_backup_database_params (pref_backup_database_params_t *ptr)
{
	if (ptr != NULL) {
		g_free (ptr->host);
		g_free (ptr->user);
		g_free (ptr->database);
		g_free (ptr->socket_file);
		g_free (ptr);
	}
}


/*
 * Destroy the provided object.  This function is used
 * by g_object_set_data_full()
 */
static void
destroy_notify_pref_backup_database_params (gpointer data)
{
	destroy_pref_backup_database_params (
					(pref_backup_database_params_t *)data);
}


/*
 * Callbacks
 */
void
pref_grid_ratio_clicked (GtkButton *button)
{
	GtkWidget *w;
	GdkPixbuf *p;
	gint active;

	active = GPOINTER_TO_INT (g_object_get_data (	G_OBJECT (button),
							"active"));

	if (active == 0) {
		p = create_pixbuf ("gschedwi/vchain-broken.png");
		g_object_set_data (	G_OBJECT (button), "active", 
					GINT_TO_POINTER (1));
	}
	else {
		p = create_pixbuf ("gschedwi/vchain.png");
		g_object_set_data (	G_OBJECT (button), "active",
					GINT_TO_POINTER (0));

		w = lookup_widget (	GTK_WIDGET (button),
					"spinbutton_pref_grid_width");
		g_object_set_data (G_OBJECT (button), "width",
				GINT_TO_POINTER (
					gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w))));
		w = lookup_widget (	GTK_WIDGET (button),
					"spinbutton_pref_grid_height");
		g_object_set_data (G_OBJECT (button), "height",
				GINT_TO_POINTER (
					gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w))));
	}
	w = lookup_widget (GTK_WIDGET (button), "image_pref_grid_ratio");
	gtk_image_set_from_pixbuf (GTK_IMAGE (w), p);
	g_object_unref (p);
}

void
pref_grid_width_changed_cb (GtkSpinButton *spin)
{
	GtkWidget *w;
	gint value, active, width, height;
	gdouble ratio, min, max;

	value = gtk_spin_button_get_value_as_int (spin);

	w = lookup_widget (GTK_WIDGET (spin), "button_pref_grid_ratio");
	active = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (w), "active"));
	if (active == 0) {
		width = GPOINTER_TO_INT (g_object_get_data (	G_OBJECT (w),
								"width"));
		height = GPOINTER_TO_INT (g_object_get_data (	G_OBJECT (w),
								"height"));
		if (width != 0 && height != 0) {
			gtk_spin_button_get_range (spin, &min, &max);

			ratio = (gdouble)width / (gdouble)height;
			height = round (value / ratio);

			if (height > max) {
				gtk_spin_button_set_value (spin, value - 1);
				return;
			}
			if (height < min) {
				gtk_spin_button_set_value (spin, value + 1);
				return;
			}

			w = lookup_widget (	GTK_WIDGET (spin),
						"spinbutton_pref_grid_height");
			if (height != gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w)))
			{
				gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
								height);
			}
		}
	}

	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/grid/width",
				NULL) == TRUE)
	{
		gconf_client_set_int (schedwi_gconf,
				GCONF_PATH "/ui/grid/width",
				value,
				NULL);
	}
}

void
pref_grid_height_changed_cb (GtkSpinButton *spin)
{
	GtkWidget *w;
	gint value, active, width, height;
	gdouble ratio, min, max;

	value = gtk_spin_button_get_value_as_int (spin);

	w = lookup_widget (GTK_WIDGET (spin), "button_pref_grid_ratio");
	active = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (w), "active"));
	if (active == 0) {
		width = GPOINTER_TO_INT (g_object_get_data (	G_OBJECT (w),
								"width"));
		height = GPOINTER_TO_INT (g_object_get_data (	G_OBJECT (w),
								"height"));
		if (width != 0 && height != 0) {
			gtk_spin_button_get_range (spin, &min, &max);

			ratio = (gdouble)width / (gdouble)height;
			width = round (value * ratio); 

			if (width > max) {
				gtk_spin_button_set_value (spin, value - 1);
				return;
			}
			if (width < min) {
				gtk_spin_button_set_value (spin, value + 1);
				return;
			}

			w = lookup_widget (	GTK_WIDGET (spin),
						"spinbutton_pref_grid_width");
			if (width != gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w)))
			{
				gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
								width);
			}
		}
	}

	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/grid/height",
				NULL) == TRUE)
	{
		gconf_client_set_int (schedwi_gconf,
				GCONF_PATH "/ui/grid/height",
				value,
				NULL);
	}
}

void
pref_grid_line_style_changed_cb (GtkComboBox *combo_box)
{
	gchar *s;

	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/grid/line_style",
				NULL) == TRUE)
	{
		s = gtk_combo_box_get_active_text (combo_box);
		if (s != NULL) {
			if (g_ascii_strcasecmp (s, _("Solid")) == 0) {
				gconf_client_set_string (
					schedwi_gconf,
					GCONF_PATH "/ui/grid/line_style",
					"SOLID",
					NULL);
			}
			else {
			if (g_ascii_strcasecmp (s, _("Dashed")) == 0) {
				gconf_client_set_string (
					schedwi_gconf,
					GCONF_PATH "/ui/grid/line_style",
					"DASH",
					NULL);
			}
			else {
				gconf_client_set_string (
					schedwi_gconf,
					GCONF_PATH "/ui/grid/line_style",
					"CROSS",
					NULL);
			}}
			g_free (s);
		}
	} 
}

void
pref_grid_color_set_cb (GtkColorButton *color_button)
{
	GdkColor color;
	gchar *s;

	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/grid/color",
				NULL) == TRUE)
	{
		gtk_color_button_get_color (color_button, &color);
		s = g_strdup_printf ("#%2.2x%2.2x%2.2x",
					color.red / 257,
					color.green / 257,
					color.blue / 257);
		gconf_client_set_string (schedwi_gconf,
				GCONF_PATH "/ui/grid/color",
				s,
				NULL);
		g_free (s);
	}
}

void
pref_calendar_months_by_row_changed_cb (GtkSpinButton *spin)
{
	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/months_by_row",
				NULL) == TRUE)
	{
		gconf_client_set_int (schedwi_gconf,
				GCONF_PATH "/ui/calendar/months_by_row",
				gtk_spin_button_get_value_as_int (spin),
				NULL);
	}
}

void
pref_calendar_first_day_of_week_changed_cb (GtkComboBox *combo_box)
{
	gchar *s;

	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
				NULL) == TRUE)
	{
		s = gtk_combo_box_get_active_text (combo_box);
		if (s != NULL) {
			if (g_ascii_strcasecmp (s, _("Sunday")) == 0) {
				gconf_client_set_string (
					schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
					"Sunday",
					NULL);
			}
			else {
			if (g_ascii_strcasecmp (s, _("Monday")) == 0) {
				gconf_client_set_string (
					schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
					"Monday",
					NULL);
			}
			else {
			if (g_ascii_strcasecmp (s, _("Tuesday")) == 0) {
				gconf_client_set_string (
					schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
					"Tuesday",
					NULL);
			}
			else {
			if (g_ascii_strcasecmp (s, _("Wednesday")) == 0) {
				gconf_client_set_string (
					schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
					"Wednesday",
					NULL);
			}
			else {
			if (g_ascii_strcasecmp (s, _("Thursday")) == 0) {
				gconf_client_set_string (
					schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
					"Thursday",
					NULL);
			}
			else {
			if (g_ascii_strcasecmp (s, _("Friday")) == 0) {
				gconf_client_set_string (
					schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
					"Friday",
					NULL);
			}
			else {
				gconf_client_set_string (
					schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
					"Saturday",
					NULL);
			}}}}}}
			g_free (s);
		}
	} 
}

void
pref_calendar_use_system_font_toggled_cb (GtkToggleButton *toggle_button)
{
	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/use_system_font",
				NULL) == TRUE)
	{
		gconf_client_set_bool (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/use_system_font",
				gtk_toggle_button_get_active (toggle_button),
				NULL);
	}

}

void
pref_calendar_font_set_cb (GtkFontButton *font_button)
{
	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/font",
				NULL) == TRUE)
	{
		gconf_client_set_string (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/font",
				gtk_font_button_get_font_name (font_button),
				NULL);
	}
}

void
pref_database_host_changed_cb (GtkEntry *entry)
{
	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/database/host",
				NULL) == TRUE)
	{
		gconf_client_set_string (
				schedwi_gconf,
				GCONF_PATH "/database/host",
				gtk_entry_get_text (entry),
				NULL);
	}
}

void
pref_database_user_changed_cb (GtkEntry *entry)
{
	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/database/username",
				NULL) == TRUE)
	{
		gconf_client_set_string (
				schedwi_gconf,
				GCONF_PATH "/database/username",
				gtk_entry_get_text (entry),
				NULL);
	}
}

void
pref_database_port_changed_cb (GtkSpinButton *spin)
{
	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/database/port",
				NULL) == TRUE)
	{
		gconf_client_set_int (schedwi_gconf,
				GCONF_PATH "/database/port",
				gtk_spin_button_get_value_as_int (spin),
				NULL);
	}
}

void
pref_database_database_changed_cb (GtkEntry *entry)
{
	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/database/database",
				NULL) == TRUE)
	{
		gconf_client_set_string (
				schedwi_gconf,
				GCONF_PATH "/database/database",
				gtk_entry_get_text (entry),
				NULL);
	}
}

void
pref_database_socket_changed_cb (GtkEditable *editable)
{
	gchar *s;

	if (	   schedwi_gconf != NULL
		&& gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/database/socket",
				NULL) == TRUE)
	{
		s = gtk_editable_get_chars (editable, 0, -1);
		if (s != NULL) {
			gconf_client_set_string (
				schedwi_gconf,
				GCONF_PATH "/database/socket",
				s,
				NULL);
			g_free (s);
		}
	}
}


/*
 * Callback for the `Test' database button
 */
void
pref_database_test_clicked_cb (GtkButton *button)
{
	new_dialog_database_test ();
}


/*
 * Restore the database parameters
 */
void
pref_database_restore_activated_cb (GtkButton *button)
{
	GtkWidget *widget, *w;
	pref_backup_database_params_t *ptr;

	/* Retrieve the previously stored backup object */
	widget = lookup_widget (GTK_WIDGET (button), "dialog_preferences");
	ptr = (pref_backup_database_params_t *)g_object_get_data (
							G_OBJECT (widget),
							"database_backup");
	if (ptr != NULL && schedwi_gconf != NULL) {
		if (gconf_client_key_is_writable (schedwi_gconf,
						GCONF_PATH "/database/host",
						NULL) == TRUE)
		{
			w = lookup_widget (	widget,
						"entry_pref_database_server");
			gtk_entry_set_text (GTK_ENTRY (w), ptr->host);
			gconf_client_set_string (schedwi_gconf,
					GCONF_PATH "/database/host",
					ptr->host,
					NULL);
		}

		if (gconf_client_key_is_writable (schedwi_gconf,
					GCONF_PATH "/database/username",
					NULL) == TRUE)
		{
			w = lookup_widget (widget,
					"entry_pref_database_username");
			gtk_entry_set_text (GTK_ENTRY (w), ptr->user);
			gconf_client_set_string (schedwi_gconf,
					GCONF_PATH "/database/username",
					ptr->user,
					NULL);
		}

		if (gconf_client_key_is_writable (schedwi_gconf,
					GCONF_PATH "/database/database",
					NULL) == TRUE)
		{
			w = lookup_widget (widget, "entry_pref_database_name");
			gtk_entry_set_text (GTK_ENTRY (w), ptr->database);
			gconf_client_set_string (schedwi_gconf,
					GCONF_PATH "/database/database",
					ptr->database,
					NULL);
		}

		if (gconf_client_key_is_writable (schedwi_gconf,
					GCONF_PATH "/database/port",
					NULL) == TRUE)
		{
			w = lookup_widget (widget,
					"spinbutton_pref_database_port");
			gtk_spin_button_set_value (	GTK_SPIN_BUTTON (w),
							ptr->port);
			gconf_client_set_int (	schedwi_gconf,
						GCONF_PATH "/database/port",
						ptr->port,
						NULL);
			
		}

		if (gconf_client_key_is_writable (schedwi_gconf,
					GCONF_PATH "/database/socket",
					NULL) == TRUE)
		{
			w = lookup_widget (	widget,
						"fileentry_database_socket");
			gnome_file_entry_set_filename (GNOME_FILE_ENTRY (w),
						ptr->socket_file);
			gconf_client_set_string (schedwi_gconf,
						GCONF_PATH "/database/socket",
						ptr->socket_file,
						NULL);
		}
	}
}


/*
 * Initialize the new Preference window
 */
static gboolean
init_pref (GtkWidget *widget)
{
	GtkWidget *w;
	gint v;
	gchar *s;
	gboolean b;
	pref_backup_database_params_t *ptr;

	/*
	 * Calendar
	 */

	w = lookup_widget (widget, "spinbutton_pref_cal_num_month");
	v = gconf_client_get_int (schedwi_gconf,
				GCONF_PATH "/ui/calendar/months_by_row",
				NULL);
	if (v != 0) {
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (w), v);
	}
	gtk_widget_set_sensitive (w,
			gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/months_by_row",
				NULL));

	w = lookup_widget (widget, "radiobutton_pref_cal_font_theme");
	b =  gconf_client_get_bool (schedwi_gconf,
				GCONF_PATH "/ui/calendar/use_system_font",
				NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), b);
	gtk_widget_set_sensitive (w,
			gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/use_system_font",
				NULL));
	w = lookup_widget (widget, "radiobutton_pref_cal_font");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), !b);
	gtk_widget_set_sensitive (w,
			gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/use_system_font",
				NULL));

	
	w = lookup_widget (widget, "fontbutton_pref_cal_font");
	s = gconf_client_get_string (	schedwi_gconf,
					GCONF_PATH "/ui/calendar/font",
					NULL);
	if (s != NULL) {
		gtk_font_button_set_font_name (	GTK_FONT_BUTTON (w),
						s);
		g_free (s);
	}
	gtk_widget_set_sensitive (w,
			gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/font",
				NULL));

	w = lookup_widget (widget, "combobox_pref_cal_first_day");
	s = gconf_client_get_string (schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
				NULL);
	if (s != NULL) {
		if (g_ascii_strncasecmp (s, "Monday", 3) == 0) {
			gtk_combo_box_set_active (GTK_COMBO_BOX (w), 0);
		}
		else {
		if (g_ascii_strncasecmp (s, "Sunday", 3) == 0) {
			gtk_combo_box_set_active (GTK_COMBO_BOX (w), 6);
		}
		else {
		if (g_ascii_strncasecmp (s, "Tuesday", 3) == 0) {
			gtk_combo_box_set_active (GTK_COMBO_BOX (w), 1);
		}
		else {
		if (g_ascii_strncasecmp (s, "Wednesday", 3) == 0) {
			gtk_combo_box_set_active (GTK_COMBO_BOX (w), 2);
		}
		else {
		if (g_ascii_strncasecmp (s, "Thursday", 3) == 0) {
			gtk_combo_box_set_active (GTK_COMBO_BOX (w), 3);
		}
		else {
		if (g_ascii_strncasecmp (s, "Friday", 3) == 0) {
			gtk_combo_box_set_active (GTK_COMBO_BOX (w), 4);
		}
		else {
			gtk_combo_box_set_active (GTK_COMBO_BOX (w), 5);
		}}}}}}
		g_free (s);
	}
	else {
		gtk_combo_box_set_active (GTK_COMBO_BOX (w), 0);
	}
	gtk_widget_set_sensitive (w,
			gconf_client_key_is_writable (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
				NULL));

	/*
	 * Database
	 */

	/* Retrieve the parameters from gconf */
	ptr = new_pref_backup_database_params ();

	w = lookup_widget (widget, "entry_pref_database_server");
	if (ptr->host != NULL) {
		gtk_entry_set_text (GTK_ENTRY (w), ptr->host);
	}
	gtk_widget_set_sensitive (w, gconf_client_key_is_writable (
						schedwi_gconf,
						GCONF_PATH "/database/host",
						NULL));

	w = lookup_widget (widget, "entry_pref_database_username");
	if (ptr->user != NULL) {
		gtk_entry_set_text (GTK_ENTRY (w), ptr->user);
	}
	gtk_widget_set_sensitive (w, gconf_client_key_is_writable (
					schedwi_gconf,
					GCONF_PATH "/database/username",
					NULL));

	w = lookup_widget (widget, "entry_pref_database_name");
	if (ptr->database != NULL) {
		gtk_entry_set_text (GTK_ENTRY (w), ptr->database);
	}
	gtk_widget_set_sensitive (w, gconf_client_key_is_writable (
					schedwi_gconf,
					GCONF_PATH "/database/database",
					NULL));

	w = lookup_widget (widget, "spinbutton_pref_database_port");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w), ptr->port);
	gtk_widget_set_sensitive (w, gconf_client_key_is_writable (
						schedwi_gconf,
						GCONF_PATH "/database/port",
						NULL));

	w = lookup_widget (widget, "fileentry_database_socket");
	if (ptr->socket_file != NULL) {
		gnome_file_entry_set_filename (	GNOME_FILE_ENTRY (w),
						ptr->socket_file);
	}
	gtk_widget_set_sensitive (w, gconf_client_key_is_writable (
						schedwi_gconf,
						GCONF_PATH "/database/socket",
						NULL));

	/* Store the current database parameters for a later restoration */ 
	g_object_set_data_full (G_OBJECT (widget), "database_backup", ptr,
				destroy_notify_pref_backup_database_params);
	return TRUE;
}


/*
 * Create a new Preference window
 */
GtkWidget *
new_dialog_pref (void (*destroy_cb)(gpointer), gpointer user_data)
{
	GtkWidget *widget;

	widget = create_dialog_preferences ();
	if (init_pref (widget) == FALSE) {
		gtk_widget_destroy (widget);
		return NULL;
	}
	else {
		g_object_set_data (G_OBJECT (widget), "destroy_cb",
						destroy_cb);
		g_object_set_data (G_OBJECT (widget), "destroy_cb_data",
						user_data);
		gtk_widget_show (widget);
		return widget;
	}
}


/*
 * delete-event callback (when the user closes the window)
 */
gboolean
pref_delete_event (GtkWidget *widget)
{
	GtkWidget *window;
	void (*destroy_cb)(gpointer);
	gpointer user_data;

	/*
	 * Call the destroy_cb function provided when the
	 * new_dialog_pref() function was called
	 */
	window = lookup_widget (widget, "dialog_preferences");
	destroy_cb = (void (*)(gpointer))g_object_get_data (
							G_OBJECT(window),
							"destroy_cb");
	user_data = g_object_get_data (G_OBJECT (window), "destroy_cb_data");
	if (destroy_cb != NULL) {
		destroy_cb (user_data);
	}
	else {
		/* Destroy the window */
		return FALSE;
	}
	/*
	 * Return TRUE to stop other handlers from being invoked for
	 * the event. Otherwise, the window will be destroyed
	 */
	return TRUE;

}


/*
 * CLOSE button callback
 */
void
pref_close_clicked (GtkButton *button)
{
	GtkWidget *window;
	void (*destroy_cb)(gpointer);
	gpointer user_data;

	/*
	 * Call the destroy_cb function provided when the
	 * new_dialog_pref() function was called
	 */
	window = lookup_widget (GTK_WIDGET (button), "dialog_preferences");
	destroy_cb = (void (*)(gpointer))g_object_get_data (
							G_OBJECT(window),
							"destroy_cb");
	user_data = g_object_get_data (G_OBJECT (window), "destroy_cb_data");
	if (destroy_cb != NULL) {
		destroy_cb (user_data);
	}
	else {
		/* Destroy the window */
		gtk_widget_destroy (window);
	}
}

/*------------------------======= End Of File =======------------------------*/
