/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*
 * job_tooltips.c -- Tooltips for the job and jobset canvas items
 *
 * These functions provide tooltips for canvas items (as the GTK tooltips
 * don't work with such objects).  This code has been inspired by the
 * gtktooltips.c file from GTK.
 */

#include <schedwi.h>

#include "support.h"
#include "interface.h"

#include <job_tooltips.h>


#define DEFAULT_DELAY 500		/* Default delay in ms */


/*
 * Draw a border around the tooltip window
 */
static gboolean
job_tooltips_paint_window (job_tooltips_ptr tooltips)
{
	GtkRequisition req;
 
	gtk_widget_size_request (tooltips->tip_window, &req);
	gtk_paint_flat_box (	tooltips->tip_window->style,
				tooltips->tip_window->window,
				GTK_STATE_NORMAL, GTK_SHADOW_OUT, 
				NULL, GTK_WIDGET(tooltips->tip_window),
				"tooltip",
				0, 0, req.width, req.height);
	return FALSE;
}


/*
 * Create the tooltip window and initialize the provided
 * job_tooltips_ptr object
 */
static void
job_tooltips_build (job_tooltips_ptr tooltips)
{
	if (tooltips != NULL && tooltips->tip_window == NULL) {
		tooltips->tip_window = create_window_job_tooltips ();
		gtk_widget_set_name (tooltips->tip_window, "gtk-tooltips");
		gtk_widget_set_app_paintable (tooltips->tip_window, TRUE);
		gtk_widget_show_all (lookup_widget (	tooltips->tip_window,
							"table_job_tooltips"));
		tooltips->tip_label_title = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_title");

		tooltips->tip_label_0 = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_label_0");
		tooltips->tip_value_0 = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_value_0");

		tooltips->tip_label_1 = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_label_1");
		tooltips->tip_value_1 = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_value_1");

		tooltips->tip_label_2 = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_label_2");
		tooltips->tip_value_2 = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_value_2");

		tooltips->tip_label_3 = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_label_3");
		tooltips->tip_value_3 = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_value_3");

		tooltips->tip_label_4 = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_label_4");
		tooltips->tip_value_4 = lookup_widget (
						tooltips->tip_window,
						"label_job_tooltips_value_4");

		g_signal_connect_swapped (tooltips->tip_window,
				"expose_event",
				G_CALLBACK (job_tooltips_paint_window), 
				tooltips);
	}
}


/*
 * Hide the tooltip window
 */
static void
job_tooltips_hide (job_tooltips_ptr tooltips)
{
	if (tooltips != NULL && tooltips->tip_window != NULL) {
		gtk_widget_hide (tooltips->tip_window);
	}
}


/*
 * Show the tooltip window (at the correct position)
 */
static void
job_tooltips_show (job_tooltips_ptr tooltips)
{
	GdkScreen *screen;
	gint x, y, h, w, px, py, screen_width, screen_height;
	GtkRequisition requisition;


	if (	   tooltips != NULL
		&& tooltips->tip_window != NULL
		&& tooltips->widget != NULL)
	{

		/*
		 * Display the tooltip on the same screen as
		 * the provided widget
		 */
		screen = gtk_widget_get_screen (tooltips->widget);
		gtk_window_set_screen (	GTK_WINDOW (tooltips->tip_window),
					screen);

		/* Move the tooltip to be visible */
		screen_width = gdk_screen_get_width (screen);
		screen_height = gdk_screen_get_height (screen);

		gtk_widget_size_request (tooltips->tip_window, &requisition);
		w = requisition.width;
		h = requisition.height;

		gdk_display_get_pointer (gdk_screen_get_display (screen),
					NULL, &px, &py, NULL);

		if (px + w > screen_width) {
			x = screen_width - w;
		}
		else {
			if (px < w / 2) {
				x = 0;
			}
			else {
				x = px - w / 2;
			}
		}

		if (py + 10 + h > screen_height) {
			y = py - h - 10;
		}
		else {
			y = py + 10;
		}

		gtk_window_move (GTK_WINDOW (tooltips->tip_window), x, y);
		gtk_widget_show (tooltips->tip_window);
	}
}


/*
 * Timeout callback to show the tooltip window
 */
static gboolean
job_tooltips_show_cb (gpointer data)
{
	job_tooltips_ptr tooltips = (job_tooltips_ptr) data;

	if (tooltips != NULL) {
		job_tooltips_show (tooltips);
		tooltips->timer_tag = -1;
	}
	return FALSE;
}


/*
 * Show the tooltip window (with a delay) with the provided title.
 * `widget' is the widget for which a tooltip must be shown.
 */
void
job_tooltips_start (	job_tooltips_ptr tooltips, GtkWidget *widget,
			const char *title,
			const char *label_0, const char *value_0,
			const char *label_1, const char *value_1,
			const char *label_2, const char *value_2,
			const char *label_3, const char *value_3,
			const char *label_4, const char *value_4)
{
	gchar *s;

	if (tooltips == NULL) {
		return;
	}

	if (tooltips->tip_window == NULL) {
		job_tooltips_build (tooltips);
	}

	tooltips->widget = widget;

	if (title != NULL) {
		s = g_strdup_printf ("<b>%s</b>", title);
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_label_title),
					s);
		g_free (s);
		gtk_widget_show (tooltips->tip_label_title);
	}
	else {
		gtk_widget_hide (tooltips->tip_label_title);
	}

	if (label_0 != NULL) {
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_label_0),
					label_0);
		gtk_widget_show (tooltips->tip_label_0);
	}
	else {
		gtk_widget_hide (tooltips->tip_label_0);
	}

	if (value_0 != NULL) {
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_value_0),
					value_0);
		gtk_widget_show (tooltips->tip_value_0);
	}
	else {
		gtk_widget_hide (tooltips->tip_value_0);
	}

	if (label_1 != NULL) {
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_label_1),
					label_1);
		gtk_widget_show (tooltips->tip_label_1);
	}
	else {
		gtk_widget_hide (tooltips->tip_label_1);
	}

	if (value_1 != NULL) {
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_value_1),
					value_1);
		gtk_widget_show (tooltips->tip_value_1);
	}
	else {
		gtk_widget_hide (tooltips->tip_value_1);
	}

	if (label_2 != NULL) {
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_label_2),
					label_2);
		gtk_widget_show (tooltips->tip_label_2);
	}
	else {
		gtk_widget_hide (tooltips->tip_label_2);
	}

	if (value_2 != NULL) {
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_value_2),
					value_2);
		gtk_widget_show (tooltips->tip_value_2);
	}
	else {
		gtk_widget_hide (tooltips->tip_value_2);
	}

	if (label_3 != NULL) {
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_label_3),
					label_3);
		gtk_widget_show (tooltips->tip_label_3);
	}
	else {
		gtk_widget_hide (tooltips->tip_label_3);
	}

	if (value_3 != NULL) {
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_value_3),
					value_3);
		gtk_widget_show (tooltips->tip_value_3);
	}
	else {
		gtk_widget_hide (tooltips->tip_value_3);
	}

	if (label_4 != NULL) {
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_label_4),
					label_4);
		gtk_widget_show (tooltips->tip_label_4);
	}
	else {
		gtk_widget_hide (tooltips->tip_label_4);
	}

	if (value_4 != NULL) {
		gtk_label_set_markup (	GTK_LABEL (tooltips->tip_value_4),
					value_4);
		gtk_widget_show (tooltips->tip_value_4);
	}
	else {
		gtk_widget_hide (tooltips->tip_value_4);
	}

	tooltips->timer_tag = g_timeout_add (	DEFAULT_DELAY, 
						job_tooltips_show_cb,
						tooltips);
}


/*
 * Hide the tooltip window (if it had time to popup)
 */
void
job_tooltips_stop (job_tooltips_ptr tooltips)
{
	if (tooltips != NULL) {
		if (tooltips->timer_tag >= 0) {
			g_source_remove (tooltips->timer_tag);
		}
		tooltips->timer_tag = -1;
		job_tooltips_hide (tooltips);
	}
}


/*
 * Destroy the provided job_tooltips_ptr object
 */
void
job_tooltips_destroy (job_tooltips_ptr tooltips)
{
	if (tooltips != NULL) {
		if (tooltips->timer_tag >= 0) {
			g_source_remove (tooltips->timer_tag);
		}
		if (tooltips->tip_window != NULL) {
			gtk_widget_destroy (tooltips->tip_window);
		}
		g_free (tooltips);
	}
}


/*
 * Create and return a new job_tooltips_ptr object.  This object must be
 * destroyed by the caller by job_tooltips_destroy()
 */
job_tooltips_ptr
job_tooltips_new ()
{
	job_tooltips_ptr tooltips;

	tooltips = g_new0 (job_tooltips, 1);
	tooltips->timer_tag = -1;
	return tooltips;
}

/*-----------------============== End Of File ==============-----------------*/
