/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* job_cb.c -- GUI functions for the job/jobset window */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#include "support.h"
#include "interface.h"

#include <schedwi_g_utils.h>
#include <cache_pixbuf.h>
#include <cursor.h>
#include <message_windows.h>
#include <sql_children_job.h>
#include <sql_job_hierarchy.h>
#include <sql_hosts.h>
#include <sql_env.h>
#include <env_widget.h>
#include <sql_constraint_file.h>
#include <calendar_list_cb.h>
#include <main_cb.h>
#include <workload.h>
#include <job_cb.h>


struct _job_cb_params {
	gdouble x, y;
	gchar *parent_id;
	gchar *id;
	gchar type;
	gboolean new_job;
};
typedef struct _job_cb_params job_cb_params;
typedef struct _job_cb_params *job_cb_params_ptr;


/*
 * Create and return a new job_cb_params object.
 *
 * Return:
 *   The new object to be freed by the caller by job_cb_params_destroy()
 */
static job_cb_params_ptr
job_cb_params_new (	gchar type, gboolean new_job, gdouble x, gdouble y,
			const gchar *id, const gchar *parent_id)
{
	job_cb_params_ptr ptr;

	ptr = g_new(job_cb_params, 1);
	ptr->parent_id = g_strdup (parent_id);
	ptr->id = (id != NULL) ? g_strdup (id) : NULL;
	ptr->type = type;
	ptr->new_job = new_job;
	ptr->x = x;
	ptr->y = y;
	return ptr;
}


/*
 * Destroy the provided job_cb_params object
 */
static void
job_cb_params_destroy (gpointer obj)
{
	job_cb_params_ptr ptr = (job_cb_params_ptr) obj;
	g_free (ptr->id);
	g_free (ptr->parent_id);
	g_free (ptr);
}


/*
 * Free a GString
 */
static void
string_free (gpointer string)
{
	g_string_free ((GString *)string, TRUE);
}


/*
 * Callback when the selection changes in the argument list
 * The buttons are made sensitive depending of the selected item
 */
static void
arguments_tree_selection_changed_cb (	GtkTreeSelection *selection,
					gpointer data)
{
	GtkWidget *widget = data;
	GtkWidget *remove, *up, *down, *top, *bottom;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreePath *path;

	remove = lookup_widget (widget, "button_job_arg_del");
	up = lookup_widget (widget, "button_job_arg_up");
	down = lookup_widget (widget, "button_job_arg_down");
	top = lookup_widget (widget, "button_job_arg_top");
	bottom = lookup_widget (widget, "button_job_arg_bottom");

	if (gtk_tree_selection_get_selected (selection, &model, &iter) == FALSE)
	{
		gtk_widget_set_sensitive (remove, FALSE);
		gtk_widget_set_sensitive (up, FALSE);
		gtk_widget_set_sensitive (down, FALSE);
		gtk_widget_set_sensitive (top, FALSE);
		gtk_widget_set_sensitive (bottom, FALSE);
		return;
	}

	/* Is there an item before the selected one? */
	gtk_widget_set_sensitive (remove, TRUE);
	path =  gtk_tree_model_get_path (model, &iter);
	if (gtk_tree_path_prev (path) == TRUE) {
		gtk_widget_set_sensitive (up, TRUE);
		gtk_widget_set_sensitive (top, TRUE);
	}
	else {
		gtk_widget_set_sensitive (up, FALSE);
		gtk_widget_set_sensitive (top, FALSE);
	}
	gtk_tree_path_free (path);

	/* Is there an item bellow the selected one? */
	if (gtk_tree_model_iter_next (model, &iter) == TRUE) {
		gtk_widget_set_sensitive (down, TRUE);
		gtk_widget_set_sensitive (bottom, TRUE);
	}
	else {
		gtk_widget_set_sensitive (down, FALSE);
		gtk_widget_set_sensitive (bottom, FALSE);
	}
}


/*
 * SQL callback functions to fill the dialog fields
 */
static void
job_set_id (const char *value, void *data)
{
	g_object_set_data_full (G_OBJECT (data), "job_id",
				(gpointer) g_string_new (value), string_free);
}

static void
set_label (GtkWidget *label, const char *str)
{
	gchar *s;

	s = g_strdup_printf ("<b>%s</b>", str);
	gtk_label_set_markup (GTK_LABEL (label), s);
	g_free (s);
}

static void
job_set_name (const char *value, void *data)
{
	GtkWidget *label;

	label = lookup_widget (GTK_WIDGET (data), "label_job_name");
	set_label (label, value);
}

static void
job_set_enable (const char *value, void *data)
{
	GtkWidget *label;

	label = lookup_widget (GTK_WIDGET (data), "label_job_enabled");
	set_label (label, (value == NULL || value[0] == '0')	? _("No")
								: _("Yes"));
}

static void
job_set_description (const char *value, void *data)
{
	GtkWidget *text;
	GtkTextBuffer *buffer;

	text = lookup_widget (GTK_WIDGET (data), "textview_job_description");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
	gtk_text_buffer_set_text (buffer, value, -1);
}

static int
job_set_hostname_search_value (	void *data, const char *host_id,
				const char *host_name, const char *port)
{
	GPtrArray *a = (GPtrArray *)data;
	GtkWidget *label;
	gchar *s;
	char *id;

	label = (GtkWidget *) g_ptr_array_index (a, 0);
	id = (char *) g_ptr_array_index (a, 1);

	if (id != NULL && host_id != NULL && strcmp (id, host_id) == 0) {
		s = g_strdup_printf (_("%s (port %s)"), host_name, port);
		set_label (label, s);
		g_free (s);
	}
	return 0;
}

static gboolean
job_set_hostname (char **values, char **values_parent, GtkWidget *dialog_job)
{
	GtkWidget *label;
	GPtrArray *a;

	label = lookup_widget (dialog_job, "label_job_hostname");
	a = g_ptr_array_sized_new (2);
	g_ptr_array_add (a, label);
	g_ptr_array_add (a, (values != NULL) ? values[0]: values_parent[0]);

	/* Retrieve the host list and fill the combobox */
	if (sql_hosts_list (	job_set_hostname_search_value,
				a,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		g_ptr_array_free (a, TRUE);
		return FALSE;
	}
	g_ptr_array_free (a, TRUE);

	return TRUE;
}


/*
 * Initialize the GtkLabel widget
 */
static gboolean
job_set_entry (	char **values, char **values_parent, GtkWidget *dialog_job,
		const gchar *widget_name)
{
	GtkWidget *label;
	char *s;

	g_assert (widget_name != NULL);

	label = lookup_widget (dialog_job, widget_name);

	/* Fill the label */
	s = (values != NULL) 	? values[0]
				: ((values_parent != NULL)
							? values_parent[0]
							: NULL);
	if (s != NULL) {
		set_label (label, s);
	}

	return TRUE;
}


/*
 * Initialize the GtkLabel widget (a limit in minutes)
 */
static gboolean
job_set_entry_lim (	char **values, char **values_parent,
			GtkWidget *dialog_job, const gchar *widget_name)
{
	GtkWidget *label;
	char *s;
	gchar *text;

	g_assert (widget_name != NULL);

	label = lookup_widget (dialog_job, widget_name);

	/* Fill the label */
	s = (values != NULL) 	? values[0]
				: ((values_parent != NULL)
							? values_parent[0]
							: NULL);
	if (s == NULL || s[0] == '\0' || (s[0] == '0' && s[1] == '\0')) {
		set_label (label, _("No limit"));
	}
	else {
		text = g_strdup_printf (_("%s minutes"), s);
		set_label (label, text);
		g_free (text);
	}

	return TRUE;
}


/*
 * Initialize the GtkLabel widget (a check box)
 */
static gboolean
job_set_check (	char **values, char **values_parent,
		GtkWidget *dialog_job, const gchar *widget_name)
{
	GtkWidget *label;

	g_assert (widget_name != NULL);

	label = lookup_widget (dialog_job, widget_name);

	if (values != NULL) {
		set_label (label, (values[0][0] == '0') ? _("No") : _("Yes"));
	}
	else {
		if (values_parent != NULL) {
			set_label (	label,
					(values_parent[0][0] == '0') ? _("No")
								: _("Yes"));
		}
		else {
			set_label (label, _("No"));
		}
	}

	return TRUE;
}


static gboolean
job_set_start_time (	char **values, char **values_parent,
			GtkWidget *dialog_job)
{
	GtkWidget *label;
	gint v, hour, min;
	gchar *s;

	label = lookup_widget (dialog_job, "label_job_start_time");

	if (values != NULL) {
		v = atoi (values[0]);
		hour = v / 100;
		min  = v % 100;
	}
	else {
		if (values_parent != NULL) {
			v = atoi (values_parent[0]);
			hour = v / 100;
			min  = v % 100;
		}
		else {
			hour = min = 0;
		}
	}

	/*
	 * If it's earlier than noon, add AM to avoid confusions (except for
	 * 00:00)
	 */
	if (hour < 12 && (hour != 0 || min != 0)) {
		s = g_strdup_printf (_("%2.2d:%2.2d AM"), hour, min);
	}
	else {
		s = g_strdup_printf (_("%2.2d:%2.2d"), hour, min);
	}
	set_label (label, s);
	g_free (s);

	return TRUE;
}

static gboolean
job_set_calendar (	char **values, char **values_parent,
			GtkWidget *dialog_job, int workload_date)
{
	GtkWidget *label;
	gchar *path;
	GtkTreeStore *model;

	label = lookup_widget (dialog_job, "label_job_calendar");

	/* Build the calendar model */
	if (calendar_list_build_model (&model, workload_date) != 0) {
		return FALSE;
	}

	/* Retrieve the calendar path */
	if (values != NULL) {
		path = calendar_list_get_path (model, values[0]);
	}
	else {
		if (values_parent != NULL) {
			path = calendar_list_get_path (model, values_parent[0]);
		}
		else {
			path = g_strdup (_("Every Day"));
		}

	}
	set_label (label, path);
	g_free (path);
	g_object_unref (model);

	return TRUE;
}


static void
job_set_arguments_fill (GtkWidget *view, lwc_LL *values)
{
	GtkListStore *store;
	GtkTreeIter iter;
	char **row;


	/* Create and fill the GtkListStore object */
	store = gtk_list_store_new (1, G_TYPE_STRING);

	lwc_rewindLL (values);
	while ((row = (char **)lwc_nextLL (values)) != NULL) {
		if (atoi (row[0]) != -1) {
			gtk_list_store_append (store, &iter);
			gtk_list_store_set (store, &iter, 0, row[1], -1);
		}
	}

	/* Associate the model with the view */
	gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (store));
	g_object_unref (store);
}


static gboolean
job_set_arguments (	lwc_LL *values, lwc_LL *values_parent,
			GtkWidget *dialog_job)
{
	GtkWidget *view;
	GtkCellRenderer *renderer;
	GtkTreeSelection *select;

	view = lookup_widget (dialog_job, "treeview_job_arg");

	/* Fill the list view */
	job_set_arguments_fill (view, (values == NULL)? values_parent: values);

	/* Build the list view */
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (	GTK_TREE_VIEW (view),
							-1,
							_("Command arguments"),
							renderer,
							"text", 0,
							NULL);

	/* Selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_NONE);

	return TRUE;
}


static gboolean
job_set_environments (lwc_LL *values, GtkWidget *dialog_job)
{
	GtkWidget *view, *label;
	env_widget_ptr ptr;

	view = lookup_widget (dialog_job, "treeview_job_env");

	/* Build the environment widget */
	ptr = env_widget_new (view, NULL, NULL, NULL, NULL, NULL, NULL);

	/* Fill the environment list */
	env_widget_build_rows (ptr, values);

	/* Set the history label */
	label = lookup_widget (dialog_job, "label_job_parent_env");
	set_label (label, values != NULL ? _("No") : _("Yes"));
	return TRUE;
}


static gboolean
job_set_constraint_files (lwc_LL *values, GtkWidget *dialog_job)
{
	GtkWidget *view;
	GtkListStore *store;
	GtkTreeIter iter;
	char **row;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSelection *select;
	gchar *s;

	/* Build and fill the model for the constraint file */
	store = gtk_list_store_new (	3,
					G_TYPE_STRING,	/* Host Name */	
					G_TYPE_STRING,  /* File name */
					G_TYPE_BOOLEAN);/* `Must exist' flag */

	while ((row = (char **)lwc_delStartLL (values)) != NULL) {
		/*
		 * row[0] --> job Id
		 * row[1] --> host Id
		 * row[2] --> host name
		 * row[3] --> host port
		 * row[4] --> File name
		 * row[5] --> `Must exist' flag
		 */
		s = g_strdup_printf (_("%s (port %s)"), row[2], row[3]);
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (	store, &iter,
					0, s,
					1, row[4],
					2, (row[5][0] == '0')? FALSE: TRUE,
					-1);
		g_free (s);
		sql_free_row (row);
	}

	view = lookup_widget (dialog_job, "treeview_job_files");

	/* Associate the model with the view */
	gtk_tree_view_set_model (	GTK_TREE_VIEW (view),
					GTK_TREE_MODEL (store));
	g_object_unref (store);

	/*
	 * Add the columns and cells
	 */

	/* Host */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Host"),
							renderer,
							"text", 0,
							NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (view), column);

	/* File name */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("File name"),
							renderer,
							"text", 1,
							NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (view), column);

	/* `Must exist' flag */
	renderer = gtk_cell_renderer_toggle_new ();
	gtk_cell_renderer_toggle_set_radio (
				GTK_CELL_RENDERER_TOGGLE (renderer), FALSE);
	column = gtk_tree_view_column_new_with_attributes (_("Must exist"),
							renderer,
							"active", 2,
							NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (view), column);

	/* Selection */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_NONE);

	return TRUE;
}


static void
set_icon (GtkWidget *image, char **values, unsigned long int *values_len)
{
	GdkPixbuf *p;
	gchar *err_msg = NULL;

	if (add_pixbuf_to_cache (	values[0], values_len[0],
					&p, NULL, NULL, NULL, NULL,
					&err_msg) != TRUE)
	{
		warning_window (_("Could not load the icon"), err_msg);
		g_free (err_msg);
		p = NULL;
	}
	else {
		gtk_image_set_from_pixbuf (GTK_IMAGE (image), p);
		g_object_ref (p);
	}
	/* Store the image in the "image" property */
	g_object_set_data_full (G_OBJECT (image), "image", p, g_object_unref);
}


/*
 * Initialize the window
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error popup has been displayed for the user)
 */
static int
dialog_job_init (	GtkWidget *dialog_job,
			int workload_date,
			const gchar *id, const gchar *name, gchar type,
			gdouble x, gdouble y, const gchar *parent_id)
{
	gchar *s;
	job_cb_params_ptr ptr;
	sql_job_hierarchy_ptr hierarchy_ptr;
	char **values, **values_parent;
	lwc_LL *lvalues, *lvalues_parent;
	GtkWidget *w;

	/* Store the default basic parameters in the dialog widget */
	ptr = job_cb_params_new (type,
				(id == NULL || name == NULL)? TRUE : FALSE,
				x, y, id, parent_id);
	g_object_set_data_full (G_OBJECT (dialog_job), "job_parameters",
				(gpointer) ptr, job_cb_params_destroy);

	/* Dialog name */
	if (ptr->new_job == FALSE) {
		s = g_strdup_printf ((type == 0)? _("Jobset %s"): _("Job %s"),
					name);
	}
	else {
		s = g_strdup ((type == 0)? _("New jobset"): _("New job"));
	}
	gtk_window_set_title (GTK_WINDOW (dialog_job), s);
	g_free (s);

	/* Adapt the tab labels if it is a jobset */
	if (type == 0) {
		w = lookup_widget (dialog_job, "label_job_tab_command");
		gtk_label_set_text (GTK_LABEL (w), _("Default command"));
	}

	/* Retrieve the main parameters from the database */
	if (ptr->new_job == FALSE && sql_children_job_get_main (
				workload_date, id,
				job_set_id,
				NULL,
				job_set_name,
				NULL,
				job_set_enable,
				job_set_description,
				NULL, NULL, NULL, NULL,
				dialog_job,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return -1;
	}

	/* Build the hierarchy object */
	hierarchy_ptr = sql_job_hierarchy_new (
				workload_date,
				(ptr->new_job == FALSE)? id: parent_id,
				(ptr->new_job == FALSE)? 0: 1,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	if (hierarchy_ptr == NULL) {
		return -1;
	}
	
	/* Host name */
	if (sql_job_hierarchy_get (	hierarchy_ptr,
					"job_host_s", "host_id",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_hostname (values, values_parent, dialog_job) == FALSE) {
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* Command arguments */
	if (sql_job_hierarchy_get_multi (	hierarchy_ptr,
						"job_arguments_s",
						"position,argument",
						"position",
						&lvalues, NULL,
						&lvalues_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_arguments (lvalues, lvalues_parent, dialog_job) == FALSE) {
		sql_job_hierarchy_destroy (hierarchy_ptr);
		lwc_delLL (lvalues, (void (*)(const void *))sql_free_row);
		lwc_delLL (	lvalues_parent,
				 (void (*)(const void *))sql_free_row);
		return -1;
	}
	lwc_delLL (lvalues, (void (*)(const void *))sql_free_row);
	lwc_delLL (lvalues_parent, (void (*)(const void *))sql_free_row);

	/* Command */
	if (sql_job_hierarchy_get (	hierarchy_ptr,
					"job_command_s", "command",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_entry (	values, values_parent, dialog_job,
				"label_job_command") == FALSE)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* User name */
	if (sql_job_hierarchy_get (	hierarchy_ptr,
					"job_username_s", "username",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_entry (	values, values_parent, dialog_job,
				"label_job_username") == FALSE)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* Output file name */
	if (sql_job_hierarchy_get (	hierarchy_ptr,
					"job_file_out_s", "file_out",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_entry (	values, values_parent, dialog_job,
				"label_job_stdout") == FALSE)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* Error file name */
	if (sql_job_hierarchy_get (	hierarchy_ptr,
					"job_file_err_s", "file_err",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_entry (	values, values_parent, dialog_job,
				"label_job_stderr") == FALSE)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* Return code */
	if (sql_job_hierarchy_get (	hierarchy_ptr,
					"job_success_return_code_s",
					"success_ret",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_entry (	values, values_parent, dialog_job,
				"label_job_retcode") == FALSE)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* Start limit */
	if (sql_job_hierarchy_get (	hierarchy_ptr,
					"job_start_limit_s",
					"start_limit",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_entry_lim (	values, values_parent, dialog_job,
				"label_job_start_limit") == FALSE)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* Max duration */
	if (sql_job_hierarchy_get (	hierarchy_ptr,
					"job_max_duration_s",
					"max_duration",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_entry_lim (	values, values_parent, dialog_job,
				"label_job_max_duration") == FALSE)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* Load user environment */
	if (sql_job_hierarchy_get (	hierarchy_ptr,
					"job_loadenv_s",
					"loadenv",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_check (	values, values_parent, dialog_job,
				"label_job_load_user_env") == FALSE)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* Start time */
	if (sql_job_hierarchy_get_by_id (hierarchy_ptr,
					"job_main_s",
					"start_time",
					"-1",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_start_time (values, values_parent, dialog_job) == FALSE) {
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* Calendar */
	if (sql_job_hierarchy_get_by_id (hierarchy_ptr,
					"job_main_s",
					"cal_id",
					"0",
					&values, NULL,
					&values_parent, NULL) != 0)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_calendar (	values, values_parent,
				dialog_job, workload_date) == FALSE)
	{
		sql_job_hierarchy_destroy (hierarchy_ptr);
		sql_free_row (values);
		sql_free_row (values_parent);
		return -1;
	}
	sql_free_row (values);
	sql_free_row (values_parent);

	/* Environments */
	if (sql_job_hierarchy_env_list (hierarchy_ptr, &lvalues, NULL) != 0) {
		sql_job_hierarchy_destroy (hierarchy_ptr);
		return -1;
	}
	if (job_set_environments (lvalues, dialog_job) == FALSE) {
		sql_job_hierarchy_destroy (hierarchy_ptr);
		lwc_delLL (lvalues, (void (*)(const void *))sql_free_row);
		return -1;
	}
	lwc_delLL (lvalues, (void (*)(const void *))sql_free_row);

	/* File constraints */
	if (ptr->new_job == FALSE) {
		if (sql_constraint_file_get_full (workload_date, &lvalues, id,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
		{
			sql_job_hierarchy_destroy (hierarchy_ptr);
			return -1;
		}
	}
	else {
		lvalues = NULL;
	}
	if (job_set_constraint_files (lvalues, dialog_job) == FALSE) {
		sql_job_hierarchy_destroy (hierarchy_ptr);
		lwc_delLL (lvalues, (void (*)(const void *))sql_free_row);
		return -1;
	}
	lwc_delLL (lvalues, (void (*)(const void *)) sql_free_row);

	sql_job_hierarchy_destroy (hierarchy_ptr);

	/* Root jobset */
	if (id != NULL && id[0] == '1' && id[1] == '\0') {
		w = lookup_widget (	dialog_job,
					"label_job_parent_env_label");
		gtk_widget_destroy (w);
		w = lookup_widget (	dialog_job,
					"label_job_parent_env");
		gtk_widget_destroy (w);
	}

	return 0;
}


/*
 * Create a new job dialog
 *
 * Return:
 *   The GtkWidget of the new window (which has been `show'ed by this function)
 *   or NULL in case of error
 */
GtkWidget *
new_dialog_job (int workload_date, const gchar *id, const gchar *name,
		gchar type, gdouble x, gdouble y, const gchar *parent_id,
		void (*reload)(void *, const gchar *),
		void *reload_data)
{
	GtkWidget *dialog_job;


	cursor_busy (NULL);

	dialog_job = create_dialog_job ();
	if (dialog_job_init (	dialog_job, workload_date,
				id, name, type, x, y, parent_id) != 0)
	{
		gtk_widget_destroy (dialog_job);
		cursor_normal (NULL);
		return NULL;
	}
	else {
		/* Store the callback function and its parameter */
		g_object_set_data (	G_OBJECT (dialog_job),
					"reload_callback", (gpointer)reload);
		g_object_set_data (	G_OBJECT (dialog_job),
					"reload_data", (gpointer)reload_data);
		gtk_widget_show (dialog_job);
		cursor_normal (NULL);
		return dialog_job;
	}
}


/*
 * Callback for the host toggle button to use parent jobset parameter
 */
void
job_cb_h_host_toggled (GtkToggleButton *togglebutton)
{
	GtkWidget *combo;
	gboolean active;
	char **values;
	GPtrArray *a;
	guint i;


	combo = lookup_widget (GTK_WIDGET (togglebutton), "combobox_job_host");

	active = gtk_toggle_button_get_active (togglebutton);
	if (active == TRUE) {
		values = (char **) g_object_get_data (	G_OBJECT (combo),
							"values_parent");
	}
	else {
		values = (char **) g_object_get_data (	G_OBJECT (combo),
							"values");
	}

	if (values != NULL) {
		/* Retrieve the array of the combo box items */
		a = (GPtrArray *) g_object_get_data (	G_OBJECT (combo),
							"id_array");
		/* Select the host */
		for (i = 0; i < a->len; i++) {
			if (strcmp (	values[0],
					(char *)g_ptr_array_index (a, i)) == 0)
			{
				gtk_combo_box_set_active (
						GTK_COMBO_BOX (combo), i);
				break;
			}
		}
	}

	gtk_widget_set_sensitive (combo, (active == TRUE)? FALSE: TRUE);
}


/*
 * Callback for the start time toggle button to use parent jobset parameter
 */
void
job_cb_h_time_toggled (GtkToggleButton *togglebutton)
{
	GtkWidget *spin_hour, *spin_min;
	gboolean active;
	char **values;
	gint v, hour, min;


	spin_hour = lookup_widget (	GTK_WIDGET (togglebutton),
					"spinbutton_job_hour");
	spin_min = lookup_widget (	GTK_WIDGET (togglebutton),
					"spinbutton_job_min");

	active = gtk_toggle_button_get_active (togglebutton);
	if (active == TRUE) {
		values = (char **) g_object_get_data (	G_OBJECT (spin_hour),
							"values_parent");
	}
	else {
		values = (char **) g_object_get_data (	G_OBJECT (spin_hour),
							"values");
	}

	if (values != NULL) {
		v = atoi (values[0]);
		hour = v / 100;
		min  = v % 100;
		gtk_spin_button_set_value (	GTK_SPIN_BUTTON (spin_hour),
						(gdouble)hour);
		gtk_spin_button_set_value (	GTK_SPIN_BUTTON (spin_min),
						(gdouble)min);
	}
	else {
		if (active == TRUE) {
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_hour),
							0.0);
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_min),
							0.0);
		}
	}

	gtk_widget_set_sensitive (spin_hour, (active == TRUE)? FALSE: TRUE);
	gtk_widget_set_sensitive (spin_min, (active == TRUE)? FALSE: TRUE);
}


/*
 * Callback for the calendar toggle button to use parent jobset parameter
 */
void
job_cb_h_calendar_toggled (GtkToggleButton *togglebutton)
{
	GtkWidget *label, *button;
	gchar *path, *id;
	gboolean active;

	label = lookup_widget (	GTK_WIDGET (togglebutton),
				"label_job_calendar_path");
	button = lookup_widget (GTK_WIDGET (togglebutton),
				"button_job_calendar");

	active = gtk_toggle_button_get_active (togglebutton);
	if (active == TRUE) {
		path = (gchar *) g_object_get_data (	G_OBJECT (label),
							"values_parent_path");
		id = (gchar *) g_object_get_data (	G_OBJECT (label),
							"values_parent_id");
	}
	else {
		path = (gchar *) g_object_get_data (	G_OBJECT (label),
							"values_path");
		id = (gchar *) g_object_get_data (	G_OBJECT (label),
							"values_id");
	}

	/* Store in the label widget the current displayed calendar */ 
	if (id != NULL) {
		g_object_set_data_full (G_OBJECT (label), "current_id",
					(gpointer)g_strdup (id),
					(GDestroyNotify) g_free);
	}
	
	if (path != NULL) {
		gtk_label_set_text (GTK_LABEL (label), path);
	}

	gtk_widget_set_sensitive (button, (active == TRUE)? FALSE: TRUE);
}


/*
 * Callback for the toggle button associated with the provided entry
 */
void
job_cb_h_entry_toggled (GtkToggleButton *togglebutton,
			const gchar *widget_name)
{
	GtkWidget *entry;
	gboolean active;
	char  **values;

	g_assert (widget_name != NULL);

	entry = lookup_widget (GTK_WIDGET (togglebutton), widget_name);

	active = gtk_toggle_button_get_active (togglebutton);
	if (active == TRUE) {
		values = (char **) g_object_get_data (	G_OBJECT (entry),
							"values_parent");
	}
	else {
		values = (char **) g_object_get_data (	G_OBJECT (entry),
							"values");
	}

	if (values != NULL) {
		gtk_entry_set_text (GTK_ENTRY (entry), values[0]);
	}
	else {
		if (active == TRUE) {
			gtk_entry_set_text (GTK_ENTRY (entry), "");
		}
	}

	gtk_widget_set_sensitive (entry, (active == TRUE)? FALSE: TRUE);
}


/*
 * Callback for the toggle button associated with the GtkSpinButton
 */
void
job_cb_h_spin_toggled (GtkToggleButton *togglebutton, const gchar *widget_name)
{
	GtkWidget *spin;
	gboolean active;
	char  **values;
	gdouble v;

	g_assert (widget_name != NULL);

	spin = lookup_widget (GTK_WIDGET (togglebutton), widget_name);

	active = gtk_toggle_button_get_active (togglebutton);
	if (active == TRUE) {
		values = (char **) g_object_get_data (	G_OBJECT (spin),
							"values_parent");
	}
	else {
		values = (char **) g_object_get_data (	G_OBJECT (spin),
							"values");
	}

	if (values != NULL) {
		v = (gdouble) atoi (values[0]);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), v);
	}
	else {
		if (active == TRUE) {
			gtk_spin_button_set_value (	GTK_SPIN_BUTTON (spin),
							0.0);
		}
	}

	gtk_widget_set_sensitive (spin, (active == TRUE)? FALSE: TRUE);
}

/*
 * Callback for the icon toggle button to use the jobset parameter
 */
void
job_cb_h_icon_toggled (	GtkToggleButton *togglebutton,
			const gchar *image_name,
			const gchar *button_save,
			const gchar *button_upload)
{
	GtkWidget *image, *bt_save, *bt_upload;
	gboolean active;
	char  **values;
	unsigned long int *values_len;

	image = lookup_widget (GTK_WIDGET (togglebutton), image_name);
	bt_save = lookup_widget (GTK_WIDGET (togglebutton), button_save);
	bt_upload = lookup_widget (GTK_WIDGET (togglebutton), button_upload);

	active = gtk_toggle_button_get_active (togglebutton);
	if (active == TRUE) {
		values = (char **) g_object_get_data (	G_OBJECT (image),
							"values_parent");
		values_len = (unsigned long int *) g_object_get_data (
							G_OBJECT (image),
							"values_parent_len");
	}
	else {
		values = (char **) g_object_get_data (	G_OBJECT (image),
							"values");
		values_len = (unsigned long int *) g_object_get_data (
							G_OBJECT (image),
							"values_len");
	}

	if (values != NULL) {
		set_icon (image, values, values_len);
	}

	if (active == TRUE) {
		gtk_widget_set_sensitive (bt_save, FALSE);
		gtk_widget_set_sensitive (bt_upload, FALSE);
	}
	else {
		gtk_widget_set_sensitive (bt_save, TRUE);
		gtk_widget_set_sensitive (bt_upload, TRUE);
	}
}


/*
 * Callback for the Save button for icons
 */
void
job_cb_icon_save (GtkButton *button, const gchar *image_name)
{
	GtkWidget *dialog_job, *dialog, *image;
	gchar *s, *format;
	GError *err = NULL;
	GdkPixbuf *p;
	static gchar *filename = NULL;

	dialog_job = lookup_widget (GTK_WIDGET (button), "dialog_job");
	image = lookup_widget (GTK_WIDGET (button), image_name);

	/* Save file chooser dialog */
	dialog = gtk_file_chooser_dialog_new (
					"Save Icon",
					GTK_WINDOW (dialog_job),
					GTK_FILE_CHOOSER_ACTION_SAVE,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
					NULL);
/*TODO Gtk 2.8
 * gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);
 */

	if (filename != NULL) {
		gtk_file_chooser_set_filename (	GTK_FILE_CHOOSER (dialog),
						filename);
	}

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {

		/* Retrieve the image pixbuf */
		p = (GdkPixbuf *) g_object_get_data (	G_OBJECT (image),
							"image");

		/* Get the output file name */
		if (filename != NULL) {
			g_free (filename);
		}
		filename = gtk_file_chooser_get_filename (
						GTK_FILE_CHOOSER (dialog));

		/* Does the file already exist */
		if (g_file_test (filename, G_FILE_TEST_EXISTS) == TRUE) {

			s = g_strdup_printf (
					_("A file named '%s' already exists."),
					filename);
			if (question_replace_window (dialog, s,
		_("Do you want to replace it with the image you are saving?"))
					== FALSE)
			{
				gtk_widget_destroy (dialog);
				g_free (s);
				return;
			}
			g_free (s);
		}

		/* Get the output format from the file extension */
		if (	   g_strrstr (filename, ".jpeg") != NULL
			|| g_strrstr (filename, ".jpg") != NULL)
		{
			format = "jpeg";
		}
		else {
		if (g_strrstr (filename, ".ico") != NULL) {
			format = "ico";
		}
		else {
		if (g_strrstr (filename, ".bmp") != NULL) {
			format = "bmp";
		}
		else {
			format = "png";
		}}}

		/* Write the image */
		if (gdk_pixbuf_save (p, filename, format, &err, NULL) != TRUE)
		{
			gtk_widget_destroy (dialog);
			error_window (	_("Failed to save the icon"),
					(err != NULL) ? err->message : NULL);
			if (err != NULL) {
				g_error_free (err);
			}
			return;
		}
		if (err != NULL) {
			g_error_free (err);
		}
	}
	gtk_widget_destroy (dialog);
}


/*
 * Update the image preview in file chooser dialog
 */
static void
update_preview_cb (GtkFileChooser *file_chooser, gpointer data)
{
	GtkWidget *preview;
	char *filename;
	GdkPixbuf *pixbuf;
	gboolean have_preview;

	preview = GTK_WIDGET (data);
	filename = gtk_file_chooser_get_preview_filename (file_chooser);

	if (filename != NULL) {
		pixbuf = gdk_pixbuf_new_from_file_at_size (filename, 128, 128,
								NULL);
		have_preview = (pixbuf != NULL);
		g_free (filename);
	}
	else {
		have_preview = FALSE;
		pixbuf = NULL;
	}

	gtk_image_set_from_pixbuf (GTK_IMAGE (preview), pixbuf);
	if (pixbuf != NULL) {
		g_object_unref (pixbuf);
	}

	gtk_file_chooser_set_preview_widget_active (	file_chooser,
							have_preview);
}


/*
 * Callback for the Upload icon button
 */
void
job_cb_icon_upload (GtkButton *button, const gchar *image_name)
{
	GtkWidget *dialog_job, *dialog, *preview, *image;
	static gchar *filename = NULL;
	GError *err = NULL;
	GdkPixbuf *p;

	dialog_job = lookup_widget (GTK_WIDGET (button), "dialog_job");
	image = lookup_widget (GTK_WIDGET (button), image_name);

	/* Load file chooser dialog */
	dialog = gtk_file_chooser_dialog_new (
					"Load Icon",
					GTK_WINDOW (dialog_job),
					GTK_FILE_CHOOSER_ACTION_OPEN,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					NULL);
	preview = gtk_image_new ();
	gtk_file_chooser_set_preview_widget (	GTK_FILE_CHOOSER (dialog),
						preview);
	g_signal_connect (	dialog, "update-preview",
				G_CALLBACK (update_preview_cb), preview);

	if (filename != NULL) {
		gtk_file_chooser_set_filename (	GTK_FILE_CHOOSER (dialog),
						filename);
	}

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {

		/* Get the input file name */
		if (filename != NULL) {
			g_free (filename);
		}
		filename = gtk_file_chooser_get_filename (
						GTK_FILE_CHOOSER (dialog));

		/* Load the image pixbuf */
		p =  gdk_pixbuf_new_from_file (filename, &err);
		if (p == NULL) {
			gtk_widget_destroy (dialog);
			error_window (	_("Failed to load the icon"),
					(err != NULL) ? err->message : NULL);
			if (err != NULL) {
				g_error_free (err);
			}
			return;
		}

		if (err != NULL) {
			g_error_free (err);
		}

		/* Store the image in the "image" property */
		g_object_set_data_full (G_OBJECT (image), "image", p,
					g_object_unref);
		gtk_image_set_from_pixbuf (GTK_IMAGE (image), p);
	}
	gtk_widget_destroy (dialog);
}


/*
 * Callback for the load user environment toggle button to use
 * the jobset parameter
 */
void
job_cb_h_loadenv_toggled (GtkToggleButton *togglebutton)
{
	GtkWidget *check;
	gboolean active, v;
	char  **values;


	check = lookup_widget (	GTK_WIDGET (togglebutton),
				"checkbutton_job_env_loaduser");

	active = gtk_toggle_button_get_active (togglebutton);
	if (active == TRUE) {
		values = (char **) g_object_get_data (	G_OBJECT (check),
							"values_parent");
	}
	else {
		values = (char **) g_object_get_data (	G_OBJECT (check),
							"values");
	}

	if (values != NULL) {
		v = (values[0][0] == '0') ? FALSE : TRUE;
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), v);
	}
	else {
		if (active == TRUE) {
			gtk_toggle_button_set_active (
					GTK_TOGGLE_BUTTON (check), FALSE);
		}
	}

	gtk_widget_set_sensitive (check, (active == TRUE)? FALSE: TRUE);
}


/*
 * Callback for the command arguments toggle button to use parent
 * jobset parameter
 */
void
job_cb_h_arg_toggled (GtkToggleButton *togglebutton)
{
	GtkWidget *view, *w;
	gboolean active;
	lwc_LL *values;


	view = lookup_widget (GTK_WIDGET (togglebutton), "treeview_job_arg");

	active = gtk_toggle_button_get_active (togglebutton);
	if (active == TRUE) {
		values = (lwc_LL *) g_object_get_data (	G_OBJECT (view),
							"values_parent");
	}
	else {
		values = (lwc_LL *) g_object_get_data (	G_OBJECT (view),
							"values");
	}

	if (values != NULL) {
		job_set_arguments_fill (view, values);
	}

	gtk_widget_set_sensitive (view, (active == TRUE)? FALSE: TRUE);
	w = lookup_widget (GTK_WIDGET (togglebutton), "button_job_arg_new");
	gtk_widget_set_sensitive (w, (active == TRUE)? FALSE: TRUE);

	if (active == TRUE) {
		w = lookup_widget (	GTK_WIDGET (togglebutton),
					"button_job_arg_del");
		gtk_widget_set_sensitive (w, FALSE);
		w = lookup_widget (	GTK_WIDGET (togglebutton),
					"button_job_arg_up");
		gtk_widget_set_sensitive (w, FALSE);
		w = lookup_widget (	GTK_WIDGET (togglebutton),
					"button_job_arg_down");
		gtk_widget_set_sensitive (w, FALSE);
		w = lookup_widget (	GTK_WIDGET (togglebutton),
					"button_job_arg_top");
		gtk_widget_set_sensitive (w, FALSE);
		w = lookup_widget (	GTK_WIDGET (togglebutton),
					"button_job_arg_bottom");
		gtk_widget_set_sensitive (w, FALSE);
	}
}


/*
 * Callback for the `Move up' button for arguments
 */
void
job_cb_arg_up_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, prev;
	GtkTreePath *path;

	view = lookup_widget (GTK_WIDGET (button), "treeview_job_arg");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		path = gtk_tree_model_get_path (model, &iter);
		gtk_tree_path_prev (path);
		gtk_tree_model_get_iter (model, &prev, path);
		gtk_list_store_swap (GTK_LIST_STORE (model), &iter, &prev);
		gtk_tree_path_free (path);
		arguments_tree_selection_changed_cb (select, (gpointer)button);
	}
}


/*
 * Callback for the `Move down' button for arguments
 */
void
job_cb_arg_down_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, next;

	view = lookup_widget (GTK_WIDGET (button), "treeview_job_arg");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		next = iter;
		gtk_tree_model_iter_next (model, &next);
		gtk_list_store_swap (GTK_LIST_STORE (model), &iter, &next);
		arguments_tree_selection_changed_cb (select, (gpointer)button);
	}
}


/*
 * Callback for the `Move to top' button for arguments
 */
void
job_cb_arg_top_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter;

	view = lookup_widget (GTK_WIDGET (button), "treeview_job_arg");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		gtk_list_store_move_after (	GTK_LIST_STORE (model),
						&iter, NULL);
		arguments_tree_selection_changed_cb (select, (gpointer)button);
	}

}


/*
 * Callback for the `Move to bottom' button for arguments
 */
void
job_cb_arg_bottom_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter;

	view = lookup_widget (GTK_WIDGET (button), "treeview_job_arg");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		gtk_list_store_move_before (	GTK_LIST_STORE (model),
						&iter, NULL);
		arguments_tree_selection_changed_cb (select, (gpointer)button);
	}
}


/*
 * Callback for the `Remove' button for arguments
 */
void
job_cb_arg_del_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, next;

	view = lookup_widget (GTK_WIDGET (button), "treeview_job_arg");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		/* Move the selection to the next argument */
		next = iter;
		if (gtk_tree_model_iter_next (model, &next) == TRUE) {
			gtk_tree_selection_select_iter (select, &next);
		}
		/* Remove the argument */
		gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
		/* Refresh the selection */
		arguments_tree_selection_changed_cb (select, (gpointer)button);
	}
}


/*
 * Callback for the `Add' button for arguments
 */
void
job_cb_arg_add_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, new_iter;
	GtkTreePath *path;
	GtkTreeViewColumn *column;

	view = lookup_widget (GTK_WIDGET (button), "treeview_job_arg");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));

	/* If an argument is selected, add the new one just after it */
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE) {
		gtk_list_store_insert_after (	GTK_LIST_STORE (model),
						&new_iter,
						&iter);
		arguments_tree_selection_changed_cb (select, (gpointer)button);
	}
	else {
		/* Otherwise, add it to the end */
		model = gtk_tree_view_get_model (GTK_TREE_VIEW (view));
		gtk_list_store_append (GTK_LIST_STORE (model), &new_iter);
	}

	/* Edit the argument */
	path = gtk_tree_model_get_path (model, &new_iter);
	column = gtk_tree_view_get_column (GTK_TREE_VIEW (view), 0);
	gtk_tree_view_set_cursor (	GTK_TREE_VIEW (view),
					path,
					column,
					TRUE);
	gtk_tree_path_free (path);
}


/*
 * Callback for the `Remove' button for constraint files
 */
void
job_cb_constraint_file_del_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter, next;

	view = lookup_widget (GTK_WIDGET (button), "treeview_job_files");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE)
	{
		/* Move the selection to the next constraint file */
		next = iter;
		if (gtk_tree_model_iter_next (model, &next) == TRUE) {
			gtk_tree_selection_select_iter (select, &next);
		}
		/* Remove the environment */
		gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
	}
}


/*
 * Callback for the `Add' button for constraint files
 */
void
job_cb_constraint_file_add_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *select;
	GtkTreeModel *model, *host_store;
	GtkTreeIter iter, new_iter;
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	gchar *s;

	view = lookup_widget (GTK_WIDGET (button), "treeview_job_files");
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));

	/* If a file is selected, add the new one just after it */
	if (gtk_tree_selection_get_selected (select, &model, &iter) == TRUE) {
		gtk_list_store_insert_after (	GTK_LIST_STORE (model),
						&new_iter,
						&iter);
	}
	else {
		/* Otherwise, add it to the end */
		model = gtk_tree_view_get_model (GTK_TREE_VIEW (view));
		gtk_list_store_append (GTK_LIST_STORE (model), &new_iter);
	}

	/* Set the associated host name to the first item of the model */
	host_store = (GtkTreeModel *)g_object_get_data (G_OBJECT (view),
							"host_model");
	if (	   host_store != NULL
		&& gtk_tree_model_get_iter_first (host_store, &iter) == TRUE)
	{
		gtk_tree_model_get (host_store, &iter, 1, &s, -1);
		gtk_list_store_set (	GTK_LIST_STORE (model), &new_iter,
					0, s,
					-1);
		g_free (s);
	}

	/* Edit the argument */
	path = gtk_tree_model_get_path (model, &new_iter);
	column = gtk_tree_view_get_column (GTK_TREE_VIEW (view), 1);
	gtk_tree_view_set_cursor (	GTK_TREE_VIEW (view),
					path,
					column,
					TRUE);
	gtk_tree_path_free (path);
}


/*
 * Callback for the environment toggle button to use parent
 * jobset parameter
 */
void
job_cb_h_env_toggled (GtkToggleButton *togglebutton)
{
	GtkWidget *view, *w;
	gboolean active;
	lwc_LL *values;
	env_widget_ptr ptr;


	view = lookup_widget (GTK_WIDGET (togglebutton), "treeview_job_env");

	active = gtk_toggle_button_get_active (togglebutton);
	if (active == TRUE) {
		values = NULL;
	}
	else {
		values = (lwc_LL *) g_object_get_data (	G_OBJECT (view),
							"values");
	}

	ptr = env_widget_get_from_view (view);
	env_widget_build_rows (ptr, values);

	w = lookup_widget (GTK_WIDGET (togglebutton), "frame_job_env");
	gtk_widget_set_sensitive (w, (active == TRUE)? FALSE: TRUE);
}


/*
 * Return the status of a toggle button given its name
 */
static gboolean
job_cb_toggle_button_get_active (	GtkWidget *dialog_job,
					const gchar *widget_h_name)
{
	GtkWidget *check_h;

	check_h = lookup_widget (dialog_job, widget_h_name);
	return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check_h));
}


/*
 * Update in the database the value of the provided entry field
 * 
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error (an error message has been displayed
 */
static gboolean
job_cb_save_entry (	GtkWidget *dialog_job, 
			job_cb_params_ptr ptr,
			const gchar *id,
			const gchar *widget_name, const gchar *widget_h_name,
			const char *table_name, const char *col_name)
{
	GtkWidget *entry;
	gboolean  active;
	const gchar *value;

	active = job_cb_toggle_button_get_active (dialog_job, widget_h_name);
	if (active == TRUE) {
		if (sql_job_hierarchy_del (id, table_name,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
		{
			return FALSE;
		}
		return TRUE;
	}

	entry = lookup_widget (dialog_job, widget_name);
	value = gtk_entry_get_text (GTK_ENTRY (entry));
	if (sql_job_hierarchy_add (id, table_name, col_name, value,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return FALSE;
	}
	return TRUE;
}


/*
 * Update in the database the value of the provided spinbutton field
 * 
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error (an error message has been displayed
 */
static gboolean
job_cb_save_spin (	GtkWidget *dialog_job, 
			job_cb_params_ptr ptr,
			const gchar *id,
			const gchar *widget_name, const gchar *widget_h_name,
			const char *table_name, const char *col_name)
{
	GtkWidget *spin;
	gboolean  active;
	gint v;
	gchar *value;

	active = job_cb_toggle_button_get_active (dialog_job, widget_h_name);
	if (active == TRUE) {
		if (sql_job_hierarchy_del (id, table_name,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
		{
			return FALSE;
		}
		return TRUE;
	}

	spin = lookup_widget (dialog_job, widget_name);
	v = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin));
	value = schedwi_ulltostr (v);
	if (sql_job_hierarchy_add (id, table_name, col_name, value,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		g_free (value);
		return FALSE;
	}
	g_free (value);
	return TRUE;
}


/*
 * Update in the database the value of the provided icon
 *
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error (an error message has been displayed
 */
static gboolean
job_cb_save_icon (	GtkWidget *dialog_job, 
			job_cb_params_ptr ptr,
			const gchar *id,
			const gchar *widget_name, const gchar *widget_h_name,
			const char *table_name, const char *col_name)
{
	GtkWidget *image;
	gboolean  active;
	GdkPixbuf *p;
	gpointer ptr_icon;
	GdkPixdata pixdata;
	guint8 *stream;
	guint stream_len;

	active = job_cb_toggle_button_get_active (dialog_job, widget_h_name);
	if (active == TRUE) {
		if (sql_job_hierarchy_del (id, table_name,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
		{
			return FALSE;
		}
		return TRUE;
	}

	/* Retrieve the image pixbuf */
	image = lookup_widget (dialog_job, widget_name);
	p = (GdkPixbuf *) g_object_get_data (G_OBJECT (image), "image");

	/* Save the icon */
	if (p != NULL) {
		/* Serialize the pixbuf */
		ptr_icon = gdk_pixdata_from_pixbuf (&pixdata, p, TRUE);
		stream = gdk_pixdata_serialize (&pixdata, &stream_len);
		g_free (ptr_icon);

		if (sql_job_hierarchy_add_icon (id, table_name, col_name,
				(char *)stream, (unsigned long int)stream_len,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
		{
			g_free (stream);
			return FALSE;
		}
		g_free (stream);
	}

	return TRUE;
}


/*
 * Update in the database the value of the `Load user environment' toggle
 * 
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error (an error message has been displayed
 */
static gboolean
job_cb_save_load_user_env (	GtkWidget *dialog_job, 
				job_cb_params_ptr ptr, const gchar *id)
{
	GtkWidget *toggle;
	gboolean  active;

	active = job_cb_toggle_button_get_active (dialog_job,
						"checkbutton_job_h_loadenv");
	if (active == TRUE) {
		if (sql_job_hierarchy_del (id, "job_loadenv",
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
		{
			return FALSE;
		}
		return TRUE;
	}

	toggle = lookup_widget (dialog_job, "checkbutton_job_env_loaduser");
	active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggle));
	if (sql_job_hierarchy_add (id, "job_loadenv", "loadenv",
				(active == TRUE)? "1": "0",
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return FALSE;
	}
	return TRUE;
}


/*
 * Update in the database the environment list
 * 
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error (an error message has been displayed
 */
static gboolean
job_cb_save_env (GtkWidget *dialog_job, job_cb_params_ptr ptr, const gchar *id)
{
	GtkWidget *view;
	gboolean  active;
	env_widget_ptr ptr_env;


	/*
	 * First, delete all the previous environments
	 */
	if (sql_job_hierarchy_del (id, "job_environment",
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return FALSE;
	}

	active = job_cb_toggle_button_get_active (dialog_job,
					"checkbutton_job_env_loadparent");
	if (active == TRUE) {
		return TRUE;
	}

	/*
	 * Save the environment list to the database
	 */
	view = lookup_widget (dialog_job, "treeview_job_env");
	ptr_env = env_widget_get_from_view (view);
	if (env_widget_save_list (	ptr_env, id,
					sql_job_hierarchy_add_env) != 0)
	{
		return FALSE;
	}
	return TRUE;
}


/*
 * Add the argument to the provided list
 *
 * Return:
 *   FALSE --> No error
 *    TRUE --> Memory allocation error (this return value stops the
 *             gtk_tree_model_foreach() calling function)
 */
static gboolean
job_cb_add_arguments_to_list (	GtkTreeModel *model,
				GtkTreePath *path,
				GtkTreeIter *iter,
				gpointer data)
{
	lwc_LL *current_lst = data;
	gchar *arg;

	gtk_tree_model_get (model, iter, 0, &arg, -1);
	if (lwc_addEndLL (current_lst, arg) != 0) {
		g_free (arg);
		lwc_emptyLL (current_lst, (void (*)(const void *))g_free);
		error_window (_("Memory allocation error"), NULL);
		return TRUE;
	}
	return FALSE;
}


/*
 * Update in the database the arguments of the command
 * 
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error (an error message has been displayed
 */
static gboolean
job_cb_save_arguments (	GtkWidget *dialog_job, 
			job_cb_params_ptr ptr, const gchar *id)
{
	GtkWidget *view;
	gboolean  active;
	GtkTreeModel *model;
	lwc_LL *list;

	/* Allocate the list of the arguments to save */
	list = lwc_newLL ();
	if (list == NULL) {
		error_window (_("Memory allocation error"), NULL);
		return FALSE;
	}

	/*
	 * First, delete all the previous arguments
	 */
	if (sql_job_hierarchy_del (id, "job_arguments",
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		lwc_delLL (list, NULL);
		return FALSE;
	}

	active = job_cb_toggle_button_get_active (dialog_job,
						"checkbutton_job_h_arg");
	if (active == TRUE) {
		lwc_delLL (list, NULL);
		return TRUE;
	}

	/*
	 * Add the arguments in the database
	 */
	view = lookup_widget (dialog_job, "treeview_job_arg");
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (view));

	/* Build the list */
	gtk_tree_model_foreach (model, job_cb_add_arguments_to_list, list);

	/* Insert the list in the database */
	if (sql_job_hierarchy_add_array (id, "job_arguments", "argument", list,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		lwc_delLL (list, (void (*)(const void *)) g_free);
		return FALSE;
	}

	lwc_delLL (list, (void (*)(const void *)) g_free);
	return TRUE;
}


/*
 * Update in the database the host name associated with the job/jobset
 * 
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error (an error message has been displayed
 */
static gboolean
job_cb_save_hostname (	GtkWidget *dialog_job, 
			job_cb_params_ptr ptr, const gchar *id)
{
	GtkWidget *combo;
	gboolean  active;
	GPtrArray *a;
	gint selected;

	combo = lookup_widget (dialog_job, "combobox_job_host");
	selected = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
	active = job_cb_toggle_button_get_active (dialog_job,
						"checkbutton_job_h_host");
	if (active == TRUE || selected < 0) {
		if (sql_job_hierarchy_del (id, "job_host",
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
		{
			return FALSE;
		}
		return TRUE;
	}

	/* Retrieve the array of the combo box host name IDs */
	a = (GPtrArray *) g_object_get_data (G_OBJECT (combo), "id_array");
	if (sql_job_hierarchy_add (id, "job_host", "host_id",
				(char *)g_ptr_array_index (a, selected),
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return FALSE;
	}
	return TRUE;
}


/*
 * Free the provided string array
 */
static void
job_cb_free_constraint (gchar **item)
{
	if (item != NULL) {
		g_free (item[0]);
		g_free (item[1]);
		g_free (item[2]);
		g_free (item);
	}
}


/*
 * Search for the provided host name in the model to get the associated host ID
 *
 * Return:
 *   FALSE --> Not found yet
 *    TRUE --> Found
 */
static gboolean
job_cb_search_host (	GtkTreeModel *model,
			GtkTreePath *path,
			GtkTreeIter *iter,
			gpointer data)
{
	gchar **host_inout = (gchar **)data;
	gchar *host, *host_id;

	gtk_tree_model_get (	model, iter,
				0, &host_id,
				1, &host,
				-1);
	if (host != NULL && strcmp (host, *host_inout) == 0) {
		*host_inout = host_id;
		g_free (host);
		return TRUE;
	}

	g_free (host);
	g_free (host_id);
	return FALSE;
}


/*
 * Add the provided constraint file to the provided list
 *
 * Return:
 *   FALSE --> No error
 *    TRUE --> Memory allocation error (this return value stops the
 *             gtk_tree_model_foreach() calling function)
 */
static gboolean
job_cb_add_constraint_files_to_list (	GtkTreeModel *model,
					GtkTreePath *path,
					GtkTreeIter *iter,
					gpointer data)
{
	GPtrArray *a = (GPtrArray *)data;
	lwc_LL *current_lst;
	gchar **constraint_file;
	gboolean exist;
	gchar *host, *host_id;
	GtkTreeModel *host_model;

	host_model = (GtkTreeModel *)g_ptr_array_index (a, 0);
	current_lst = (lwc_LL *)g_ptr_array_index (a, 1);

	constraint_file = g_new0 (gchar *, 3);
	gtk_tree_model_get (	model, iter,
				0, &host,
				1, &(constraint_file[1]),
				2, &exist,
				-1);
	constraint_file[2] = g_strdup ((exist == TRUE)? "1": "0");

	/* Search the host name in the host model to get the host ID */
	host_id = host;
	gtk_tree_model_foreach (host_model, job_cb_search_host, &host_id);
	g_free (host);
	constraint_file[0] = host_id;

	if (lwc_addEndLL (current_lst, constraint_file) != 0) {
		job_cb_free_constraint (constraint_file);
		lwc_emptyLL (current_lst,
			(void (*)(const void *))job_cb_free_constraint);
		error_window (_("Memory allocation error"), NULL);
		return TRUE;
	}
	return FALSE;
}


/*
 * Update in the database the constraint files associated with the job/jobset
 * 
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error (an error message has been displayed
 */
static gboolean
job_cb_save_constraint_files (	GtkWidget *dialog_job, 
				job_cb_params_ptr ptr, const gchar *id)
{
	GtkWidget *view;
	GtkTreeModel *model, *host_model;
	lwc_LL *list;
	GPtrArray *a;

	/* Allocate the list of files to save */
	list = lwc_newLL ();
	if (list == NULL) {
		error_window (_("Memory allocation error"), NULL);
		return FALSE;
	}

	/*
	 * First, delete all the previous constraint files
	 */
	if (sql_job_hierarchy_del (id, "constraint_file",
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		lwc_delLL (list, NULL);
		return FALSE;
	}

	/*
	 * Add the constraint files in the database
	 */
	view = lookup_widget (dialog_job, "treeview_job_files");
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (view));
	host_model = (GtkTreeModel *)g_object_get_data (G_OBJECT (view),
							"host_model");

	a = g_ptr_array_sized_new (2);
	g_ptr_array_add (a, host_model);
	g_ptr_array_add (a, list);

	/* Build the list */
	gtk_tree_model_foreach (model, job_cb_add_constraint_files_to_list, a);
	g_ptr_array_free (a, TRUE);

	/* Insert the list in the database */
	if (sql_job_hierarchy_add_constraint_files (id, list,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		lwc_delLL (list, 
			(void (*)(const void *)) job_cb_free_constraint);
		return FALSE;
	}

	lwc_delLL (list, (void (*)(const void *)) job_cb_free_constraint);
	return TRUE;
}


/*
 * Retrieve the main job/jobset parameters.
 * description, cal_id and start_time must be freed by the caller by g_free()
 */
static void
job_cb_get_main_parameters (	GtkWidget *dialog_job,
				const gchar **name,
				char *enabled,
				gchar **description,
				gchar **cal_id,
				gchar **start_time)
{
	GtkWidget *w;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	gint start_time_int;
	gchar *id;

	/* Name */
	w = lookup_widget (dialog_job, "entry_job_name");
	*name = gtk_entry_get_text (GTK_ENTRY (w));

	/* Enabled */
	w = lookup_widget (dialog_job, "checkbutton_job_enabled");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)) == TRUE) {
		*enabled = 1; 
	}
	else {
		*enabled = 0;
	}

	/* Description */
	w = lookup_widget (dialog_job, "textview_job_description");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (w));
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	*description = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

	/* Calendar */
	w = lookup_widget (dialog_job, "checkbutton_job_h_calendar");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)) == TRUE) {
		*cal_id = g_strdup ("0");
	}
	else {
		w = lookup_widget (dialog_job, "label_job_calendar_path");
		id = (gchar *) g_object_get_data (G_OBJECT (w), "current_id");
		*cal_id = g_strdup ((id != NULL) ? id : "0");
	}

	/* Start time */
	w = lookup_widget (dialog_job, "checkbutton_job_h_time");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)) == TRUE) {
		*start_time = g_strdup ("-1");
	}
	else {
		w = lookup_widget (dialog_job, "spinbutton_job_hour");
		start_time_int = 100 * gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w));
		w = lookup_widget (dialog_job, "spinbutton_job_min");
		start_time_int += gtk_spin_button_get_value_as_int (
							GTK_SPIN_BUTTON (w));
		*start_time = schedwi_ulltostr (start_time_int);
	}
}


/*
 * Update the parameters for the job/jobset in the database
 *
 * Return:
 *   TRUE --> No error
 *  FALSE --> Error (an error message has been displayed)
 */
static gboolean
job_cb_update (GtkWidget *dialog_job, job_cb_params_ptr ptr)
{
	const  gchar *name;
	gchar *description, *cal_id, *start_time;
	char enabled;

	/* Update the main parameters */
	job_cb_get_main_parameters (	dialog_job, &name, &enabled,
					&description, &cal_id, &start_time);
	if (sql_job_hierarchy_update_job (ptr->id, name, enabled, description,
				cal_id, start_time,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		g_free (description);
		g_free (cal_id);
		g_free (start_time);
		return FALSE;
	}
	g_free (description);
	g_free (cal_id);
	g_free (start_time);

	/* Update the other parameters */
	if (	   job_cb_save_entry (	dialog_job, ptr, ptr->id,
					"entry_job_command",
					"checkbutton_job_h_command",
					"job_command", "command") == FALSE
		|| job_cb_save_entry (	dialog_job, ptr, ptr->id,
					"entry_job_username",
					"checkbutton_job_h_username",
					"job_username", "username") == FALSE
		|| job_cb_save_entry (	dialog_job, ptr, ptr->id,
					"entry_job_stdout",
					"checkbutton_job_h_stdout",
					"job_file_out", "file_out") == FALSE
		|| job_cb_save_entry (	dialog_job, ptr, ptr->id,
					"entry_job_stderr",
					"checkbutton_job_h_stderr",
					"job_file_err", "file_err") == FALSE
		|| job_cb_save_spin (	dialog_job, ptr, ptr->id,
					"spinbutton_job_ret",
					"checkbutton_job_h_ret",
					"job_success_return_code",
					"success_ret") == FALSE
		|| job_cb_save_spin (	dialog_job, ptr, ptr->id,
					"spinbutton_job_start_limit",
					"checkbutton_job_h_start_limit",
					"job_start_limit",
					"start_limit") == FALSE
		|| job_cb_save_spin (	dialog_job, ptr, ptr->id,
					"spinbutton_job_max_duration",
					"checkbutton_job_h_max_duration",
					"job_max_duration",
					"max_duration") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, ptr->id,
					"image_job_default",
					"checkbutton_job_icon_h_default",
					"job_icon_default", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, ptr->id,
					"image_job_waiting",
					"checkbutton_job_icon_h_waiting",
					"job_icon_waiting", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, ptr->id,
					"image_job_running",
					"checkbutton_job_icon_h_running",
					"job_icon_running", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, ptr->id,
					"image_job_completed",
					"checkbutton_job_icon_h_completed",
					"job_icon_completed", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, ptr->id,
					"image_job_failed",
					"checkbutton_job_icon_h_failed",
					"job_icon_failed", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, ptr->id,
					"image_jobset_default",
					"checkbutton_jobset_icon_h_default",
					"jobset_icon_default", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, ptr->id,
					"image_jobset_waiting",
					"checkbutton_jobset_icon_h_waiting",
					"jobset_icon_waiting", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, ptr->id,
					"image_jobset_running",
					"checkbutton_jobset_icon_h_running",
					"jobset_icon_running", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, ptr->id,
					"image_jobset_completed",
					"checkbutton_jobset_icon_h_completed",
					"jobset_icon_completed",
					"icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, ptr->id,
					"image_jobset_failed",
					"checkbutton_jobset_icon_h_failed",
					"jobset_icon_failed", "icon") == FALSE
		|| job_cb_save_load_user_env (	dialog_job,
						ptr, ptr->id) == FALSE
		|| job_cb_save_arguments (dialog_job, ptr, ptr->id) == FALSE
		|| job_cb_save_env (dialog_job, ptr, ptr->id) == FALSE
		|| job_cb_save_hostname (dialog_job, ptr, ptr->id) == FALSE
		|| job_cb_save_constraint_files (	dialog_job,
							ptr, ptr->id) == FALSE)
	{
		return FALSE;
	}
	return TRUE;
}


/*
 * Save the parameters for the new job/jobset in the database
 *
 * Return:
 *   TRUE --> No error
 *  FALSE --> Error (an error message has been displayed)
 */
static gboolean
job_cb_save (GtkWidget *dialog_job, job_cb_params_ptr ptr)
{
	const gchar *name;
	gchar *description, *cal_id, *start_time, *id;
	char enabled;
	unsigned long int added_id;

	/* Save the main parameters */
	job_cb_get_main_parameters (	dialog_job, &name, &enabled,
					&description, &cal_id, &start_time);
	if (sql_job_hierarchy_save_job (
				ptr->parent_id, name, ptr->type,
				enabled, description,
				(long int)ptr->x, (long int)ptr->y,
				cal_id, start_time, &added_id,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		g_free (description);
		g_free (cal_id);
		g_free (start_time);
		return FALSE;
	}
	g_free (description);
	g_free (cal_id);
	g_free (start_time);

	id = schedwi_ulltostr (added_id);

	/* Update the other parameters */
	if (	   job_cb_save_entry (	dialog_job, ptr, id,
					"entry_job_command",
					"checkbutton_job_h_command",
					"job_command", "command") == FALSE
		|| job_cb_save_entry (	dialog_job, ptr, id,
					"entry_job_username",
					"checkbutton_job_h_username",
					"job_username", "username") == FALSE
		|| job_cb_save_entry (	dialog_job, ptr, id,
					"entry_job_stdout",
					"checkbutton_job_h_stdout",
					"job_file_out", "file_out") == FALSE
		|| job_cb_save_entry (	dialog_job, ptr, id,
					"entry_job_stderr",
					"checkbutton_job_h_stderr",
					"job_file_err", "file_err") == FALSE
		|| job_cb_save_spin (	dialog_job, ptr, id,
					"spinbutton_job_ret",
					"checkbutton_job_h_ret",
					"job_success_return_code",
					"success_ret") == FALSE
		|| job_cb_save_spin (	dialog_job, ptr, id,
					"spinbutton_job_start_limit",
					"checkbutton_job_h_start_limit",
					"job_start_limit",
					"start_limit") == FALSE
		|| job_cb_save_spin (	dialog_job, ptr, id,
					"spinbutton_job_max_duration",
					"checkbutton_job_h_max_duration",
					"job_max_duration",
					"max_duration") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, id,
					"image_job_default",
					"checkbutton_job_icon_h_default",
					"job_icon_default", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, id,
					"image_job_waiting",
					"checkbutton_job_icon_h_waiting",
					"job_icon_waiting", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, id,
					"image_job_running",
					"checkbutton_job_icon_h_running",
					"job_icon_running", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, id,
					"image_job_completed",
					"checkbutton_job_icon_h_completed",
					"job_icon_completed", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, id,
					"image_job_failed",
					"checkbutton_job_icon_h_failed",
					"job_icon_failed", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, id,
					"image_jobset_default",
					"checkbutton_jobset_icon_h_default",
					"jobset_icon_default", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, id,
					"image_jobset_waiting",
					"checkbutton_jobset_icon_h_waiting",
					"jobset_icon_waiting", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, id,
					"image_jobset_running",
					"checkbutton_jobset_icon_h_running",
					"jobset_icon_running", "icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, id,
					"image_jobset_completed",
					"checkbutton_jobset_icon_h_completed",
					"jobset_icon_completed",
					"icon") == FALSE
		|| job_cb_save_icon (	dialog_job, ptr, id,
					"image_jobset_failed",
					"checkbutton_jobset_icon_h_failed",
					"jobset_icon_failed", "icon") == FALSE
		|| job_cb_save_load_user_env (dialog_job, ptr, id) == FALSE
		|| job_cb_save_arguments (dialog_job, ptr, id) == FALSE
		|| job_cb_save_env (dialog_job, ptr, id) == FALSE
		|| job_cb_save_hostname (dialog_job, ptr, id) == FALSE
		|| job_cb_save_constraint_files (dialog_job, ptr, id) == FALSE)
	{
		sql_job_hierarchy_del_job (id, NULL, NULL);
		sql_job_hierarchy_del (id, "job_command", NULL, NULL);
		sql_job_hierarchy_del (id, "job_username", NULL, NULL);
		sql_job_hierarchy_del (id, "job_file_out", NULL, NULL);
		sql_job_hierarchy_del (id, "job_file_err", NULL, NULL);
		sql_job_hierarchy_del (id, "job_success_return_code",
					NULL, NULL);
		sql_job_hierarchy_del (id, "job_start_limit", NULL, NULL);
		sql_job_hierarchy_del (id, "job_max_duration", NULL, NULL);
		sql_job_hierarchy_del (id, "job_icon_default", NULL, NULL);
		sql_job_hierarchy_del (id, "job_icon_waiting", NULL, NULL);
		sql_job_hierarchy_del (id, "job_icon_running", NULL, NULL);
		sql_job_hierarchy_del (id, "job_icon_completed", NULL, NULL);
		sql_job_hierarchy_del (id, "job_icon_failed", NULL, NULL);
		sql_job_hierarchy_del (id, "jobset_icon_default", NULL, NULL);
		sql_job_hierarchy_del (id, "jobset_icon_waiting", NULL, NULL);
		sql_job_hierarchy_del (id, "jobset_icon_running", NULL, NULL);
		sql_job_hierarchy_del (id, "jobset_icon_completed",
					NULL, NULL);
		sql_job_hierarchy_del (id, "jobset_icon_failed", NULL, NULL);
		sql_job_hierarchy_del (id, "job_loadenv", NULL, NULL);
		sql_job_hierarchy_del (id, "job_arguments", NULL, NULL);
		sql_job_hierarchy_del (id, "job_environment", NULL, NULL);
		sql_job_hierarchy_del (id, "job_host", NULL, NULL);
		sql_job_hierarchy_del (id, "constraint_file", NULL, NULL);
		g_free (id);
		return FALSE;
	}
	g_free (id);
	return TRUE;
}


/*
 * Check that the provided constraint file is not empty
 *
 * Return:
 *   FALSE --> No error (constraint file defined)
 *    TRUE --> Error (constraint file empty).  This return value stops the
 *             gtk_tree_model_foreach() calling function
 */
static gboolean
job_cb_check_constraint_files (	GtkTreeModel *model,
				GtkTreePath *path,
				GtkTreeIter *iter,
				gpointer data)
{
	GPtrArray *a = (GPtrArray *)data;
	gboolean *error_flag;
	gchar *constraint_file;
	GtkTreeSelection *select;

	select = (GtkTreeSelection *)g_ptr_array_index (a, 0);
	error_flag = (gboolean *)g_ptr_array_index (a, 1);

	gtk_tree_model_get (model, iter, 1, &constraint_file, -1);
	if (constraint_file == NULL || constraint_file[0] == '\0') {
		*error_flag = TRUE;
		g_free (constraint_file);
		gtk_tree_selection_select_iter (select, iter);
		return TRUE;
	}
	g_free (constraint_file);
	return FALSE;
}


/*
 * Check all the required fields
 *
 * Return:
 *   TRUE --> No error
 *  FALSE --> Error (an error message has been displayed)
 */
static gboolean
job_cb_check_error (GtkWidget *dialog_job)
{
	GtkWidget *w, *error_widget = NULL;
	GtkTreeModel *model;
	GtkTreeSelection *select;
	GString *err_msg;
	gint err;
	const gchar *s;
	gboolean error_flag;
	GPtrArray *a;

	err_msg = g_string_new ("");
	err = 0;

	/* Job/jobset name */
	w = lookup_widget (dialog_job, "entry_job_name");
	s = gtk_entry_get_text (GTK_ENTRY (w));
	if (s == NULL || s[0] == '\0') {
		err_msg = g_string_append (err_msg,
			_("The Name field is required.\n"));
		err++;
		error_widget = w;
	}

	/* User name */
	w = lookup_widget (dialog_job, "checkbutton_job_h_username");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)) == FALSE) {
		w = lookup_widget (dialog_job, "entry_job_username");
		s = gtk_entry_get_text (GTK_ENTRY (w));
		if (s == NULL || s[0] == '\0') {
			err_msg = g_string_append (err_msg,
				_("The User Name field is required.\n"));
			err++;
			error_widget = w;
		}
	}

	/* Constraint files */
	w = lookup_widget (dialog_job, "treeview_job_files");
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (w));
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (w));
	error_flag = FALSE;
	a = g_ptr_array_sized_new (2);
	g_ptr_array_add (a, select);
	g_ptr_array_add (a, &error_flag);
	gtk_tree_model_foreach (model, job_cb_check_constraint_files, a);
	g_ptr_array_free (a, TRUE);
	if (error_flag == TRUE) {
		err_msg = g_string_append (err_msg,
			_("A constraint file name is empty.\n"));
		err++;
		error_widget = w;
	}

	switch (err) {
		case 0:
			g_string_free (err_msg, TRUE);
			return TRUE;

		case 1:
			error_window (_("A required information is missing"),
					err_msg->str);
			g_string_free (err_msg, TRUE);
			break;

		default:
			error_window (_("Required informations are missing"),
					err_msg->str);
			g_string_free (err_msg, TRUE);
			break;
	}

	gtk_widget_grab_focus (error_widget);

	return FALSE;
}


/*
 * Check that the provided argument is not empty
 *
 * Return:
 *   FALSE --> No error (argument not empty)
 *    TRUE --> Error (argument empty).  This return value stops the
 *             gtk_tree_model_foreach() calling function
 */
static gboolean
job_cb_check_arguments (GtkTreeModel *model,
			GtkTreePath *path,
			GtkTreeIter *iter,
			gpointer data)
{
	GPtrArray *a = (GPtrArray *)data;
	gboolean *error_flag;
	gchar *argument;
	GtkTreeSelection *select;

	select = (GtkTreeSelection *)g_ptr_array_index (a, 0);
	error_flag = (gboolean *)g_ptr_array_index (a, 1);

	gtk_tree_model_get (model, iter, 0, &argument, -1);
	if (argument == NULL || argument[0] == '\0') {
		*error_flag = TRUE;
		g_free (argument);
		gtk_tree_selection_select_iter (select, iter);
		return TRUE;
	}
	g_free (argument);
	return FALSE;
}


/*
 * Check that there is not an empty argument
 *
 * Return:
 *   TRUE --> No error
 *  FALSE --> Warning (an error message has been displayed)
 */
static gboolean
job_cb_check_warning (GtkWidget *dialog_job)
{
	GtkWidget *w;
	GtkTreeModel *model;
	GtkTreeSelection *select;
	gboolean error_flag;
	GPtrArray *a;

	w = lookup_widget (dialog_job, "checkbutton_job_h_arg");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)) == TRUE) {
		return TRUE;
	}

	w = lookup_widget (dialog_job, "treeview_job_arg");
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (w));
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (w));
	error_flag = FALSE;
	a = g_ptr_array_sized_new (2);
	g_ptr_array_add (a, select);
	g_ptr_array_add (a, &error_flag);
	gtk_tree_model_foreach (model, job_cb_check_arguments, a);
	g_ptr_array_free (a, TRUE);
	if (error_flag == FALSE) {
		return TRUE;
	}

	if (question_window (dialog_job,
			_("A command argument is empty"),
			_("Would you like to save it anyway?")) == TRUE)
	{
		return TRUE;
	}

	gtk_widget_grab_focus (w);
	return FALSE;
}


/*
 * Callback for the OK button
 */
void
job_cb_ok_clicked (GtkButton *button)
{
	GtkWidget *dialog_job, *w;
	job_cb_params_ptr ptr;
	gchar *name;
	void (*reload)(void *, const gchar *);
	void *reload_data;

	cursor_busy (GTK_WIDGET (button));
	dialog_job = lookup_widget (GTK_WIDGET (button), "dialog_job");

	if (job_cb_check_error (dialog_job) == FALSE) {
		cursor_normal (GTK_WIDGET (button));
		return;
	}

	if (job_cb_check_warning (dialog_job) == FALSE) {
		cursor_normal (GTK_WIDGET (button));
		return;
	}

	ptr = (job_cb_params_ptr)g_object_get_data (	G_OBJECT (dialog_job),
							"job_parameters");

	if (ptr->new_job == FALSE) {
		if (job_cb_update (dialog_job, ptr) != TRUE) {
			return;
		}
	}
	else {
		if (job_cb_save (dialog_job, ptr) != TRUE) {
			return;
		}
	}

	/* Call the callback function to draw/redraw the job/jobset */
	reload = g_object_get_data (G_OBJECT (dialog_job), "reload_callback");
	reload_data = g_object_get_data (G_OBJECT (dialog_job), "reload_data");
	w = lookup_widget (dialog_job, "entry_job_name");
	name = g_strdup (gtk_entry_get_text (GTK_ENTRY (w)));
	gtk_widget_destroy (dialog_job);
	if (reload != NULL) {
		reload (reload_data, name);
	}
	g_free (name);
	cursor_normal (GTK_WIDGET (button));
}


/*
 * Callback to refresh the canvas
 */
static void
jobset_refresh (void *user_data, const gchar *foo)
{
	jobset_list_refresh (application_main);
}


/*
 * Callback for the `Default job parameters' menu item
 */
void
job_cb_on_default_parameters_activated (GtkMenuItem *menuitem)
				
{
	int workload_date;

	workload_date = (int) workload_get_workload (application_main);
	new_dialog_job (workload_date, "1", "Root", 0, 0.0, 0.0, "0",
			jobset_refresh , NULL);
}

/*-----------------============== End Of File ==============-----------------*/
