/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* gcalendar_list.c -- Management functions for the calendars */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#endif

#include <sql_calendar.h>
#include <lwc_linkedlist.h>
#include <utils.h>
#include <message_windows.h>
#include <gcalendar_list.h>


/*
 * Retrieve and build the calendar array. list and nmemb (number of elements
 * in the list) are set. The last element of the list has an id set to 0.
 *
 * Return:
 *   0 --> No error.  list is set and must be freed by the caller (free()).
 *         Calendars with syntax error are ignored (but a message is displayed)
 *  -1 --> Memory allocation error.  An error message has been displayed.
 *  -2 --> Database error.  An error message has been displayed.
 */
static int
get_calendar_list (	calendar_t_ptr *list, unsigned int *nmemb,
			schedwi_date workload_date)
{
	lwc_LL *cals;
	char **row;
	gchar *s;
	unsigned int ret, i;
	calendar_t_ptr ptr;
	cal_errcode_t ret1, ret2, ret3;
	unsigned short int year;

	g_assert (list != NULL);

	/* Retrieve all the calendars */
	ret = sql_cal_only_list (schedwi_date_to_int (workload_date), &cals,
				(void (*)(void *, const char*, unsigned int))
					error_window_ignore_errno,
					_("Database error"));
	if (ret != 0) {
		return (ret == 1) ? -1 : -2;
	}

	/* Allocate the array */
	ptr = g_new (calendar_t, lwc_getNumNode (cals) + 1);

	/* Copy the list in the array */
	year = workload_date.year;
	i = 0;
	while ((row = (char **) lwc_delStartLL (cals)) != NULL) {
		/*
		 * row[0] --> ID
		 * row[1] --> Name
		 * row[2] --> Formula
		 */
		ret1 = str2cal (&(ptr[i].calendar[0]), row[2], year - 1, NULL);
		ret2 = str2cal (&(ptr[i].calendar[1]), row[2], year, NULL);
		ret3 = str2cal (&(ptr[i].calendar[2]), row[2], year + 1, NULL);
		switch ((ret1 == CAL_NOERROR) ? 
				((ret2 == CAL_NOERROR) ? ret3 : ret2) : ret1)
		{
			case CAL_NOERROR:
				ptr[i++].id = strtoull (row[0], NULL, 0);
				break;

			case CAL_MALLOC:
				error_window (	_("Memory allocation error"),
						NULL);
				break;

			case CAL_EMPTYFIELD:
				s = g_strdup_printf (
			_("calendar `%s' (ID: %s): a required field is empty"),
					row[1], row[0]);
				error_window (	_("Calendar syntax error"),
						NULL);
				g_free (s);
				break;

			case CAL_BADMONTHNAME:
				s = g_strdup_printf (
			_("calendar `%s' (ID: %s): invalid month name"),
					row[1], row[0]);
				error_window (	_("Calendar syntax error"),
						NULL);
				g_free (s);
				break;

			case CAL_BADDAYNAME:
				s = g_strdup_printf (
			_("calendar `%s' (ID: %s): invalid day name"),
					row[1], row[0]);
				error_window (	_("Calendar syntax error"),
						NULL);
				g_free (s);
				break;

			case CAL_MONTHOOR:
				s = g_strdup_printf (
	_("calendar `%s' (ID: %s): month number must be between 1 and 12"),
					row[1], row[0]);
				error_window (	_("Calendar syntax error"),
						NULL);
				g_free (s);
				break;

			case CAL_DAYOOR:
				s = g_strdup_printf (
	_("calendar `%s' (ID: %s): day number must be between 1 and 31"),
					row[1], row[0]);
				error_window (	_("Calendar syntax error"),
						NULL);
				g_free (s);
				break;

			case CAL_BADOPTION:
				s = g_strdup_printf (
			_("calendar `%s' (ID: %s): invalid option"),
					row[1], row[0]);
				error_window (	_("Calendar syntax error"),
						NULL);
				g_free (s);
				break;

			default:
				s = g_strdup_printf (
					_("calendar `%s' (ID: %s)"),
					row[1], row[0]);
				error_window (	_("Calendar syntax error"),
						NULL);
				g_free (s);
				break;
		}
		sql_free_row (row);
	}
	lwc_delLL (cals, NULL);
	ptr[i].id = 0;

	*list = ptr;
	if (nmemb != NULL) {
		*nmemb = i;
	}
	return 0;
}


/*
 * Create a new calendar_list_t object and fill it with the compiled calendars
 * for the provided year, the previous one and the next one.
 *
 * Return:
 *   The new object (to be freed by destroy_calendar_list()) or
 *   NULL in case of error (an error message has been displayed)
 */
calendar_list_t_ptr
new_calendar_list (schedwi_date workload_date)
{
	calendar_list_t_ptr ptr;

	ptr = g_new (calendar_list_t, 1);
	if (get_calendar_list (	&(ptr->list), &(ptr->ncals),
				workload_date) != 0)
	{
		g_free (ptr);
		return NULL;
	}
	ptr->year = workload_date.year;
	return ptr;
}


/*
 * Free a calendar_list_t object
 */
void
destroy_calendar_list (calendar_list_t_ptr ptr)
{
	if (ptr != NULL) {
		g_free (ptr->list);
		g_free (ptr);
	}
}


/*
 * Retrieve all the active calendars for the provided day (day/month/year).
 * The returned string is a list of the calendar IDs (separated by `,')
 * str is set and must be freed by the caller (g_free()).
 */
void
get_calendar_list_for_the_day ( calendar_list_t_ptr ptr,
				schedwi_date date, char **str)
{
	char *s;
	unsigned int i, j, n, year_idx;

	g_assert (   ptr != NULL && str != NULL
		&& (	   ptr->year - 1 == date.year
			|| ptr->year == date.year
			|| ptr->year + 1 == date.year));

	/* Allocate the string */
	s = (char *)g_malloc (ptr->ncals * 25 + 1);

	/*
	 * In the calendar list, copy the ID of the calendars
	 * matching the provided day/month/year
	 */
	year_idx = 1 + date.year - ptr->year;
	for (i = j = 0; i < ptr->ncals; i++) {
		if (calmatch (&(ptr->list[i].calendar[year_idx]),
			date.day, date.month, date.year) == CAL_NOERROR)
		{
			n = copy_ulltostr (ptr->list[i].id, s + j);
			s[j + n] = ',';
			j += n + 1;
		}
	}

	/* Remove the last `,' */
	if (j == 0) {
		s[0] = '\0';
	}
	else {
		s[j - 1] = '\0';
	}

	*str = s;
}

/*-----------------============== End Of File ==============-----------------*/
