#
##
##  This file is part of pyFormex 2.1  (Sat Oct 24 14:52:22 CEST 2020)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: http://pyformex.org
##  Project page:  http://savannah.nongnu.org/projects/pyformex/
##  Copyright 2004-2020 (C) Benedict Verhegghe (benedict.verhegghe@ugent.be)
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
"""pyFormex GUI module initialization.

This module is intended to form a single access point to the Python
wrappers for the QT libraries, which form the base of the pyFormex GUI.
By using a single access point, we can better deal with API changes
in the wrappers.

All pyFormex modules accessing QT libraries should do this by importing
from this module.

This module also detects the underlying windowing system.
Currently, the pyFormex GUI is only guaranteed on X11.
For other systems, a warning will be printed that some things may not work.
"""

import pyformex as pf
from pyformex import Path
from pyformex import utils

BINDINGS = ('pyside2', 'pyside', 'pyqt5', 'pyqt4')
bindings = pf.cfg['gui/bindings'].lower()

if bindings in BINDINGS:
    utils.Module.require(bindings)

else:
    # Try using any existing bindings
    bindings = ''
    for b in BINDINGS:
        if utils.Module.has(b):
            bindings = b
            break

if bindings == 'pyside2':
    from PySide2 import QtCore, QtGui, QtOpenGL, QtWidgets
    from PySide2.QtCore import Signal, Slot

elif bindings == 'pyside':
    from PySide import QtCore, QtGui, QtOpenGL
    import PySide.QtGui as QtWidgets
    from PySide.QtCore import Signal, Slot

elif bindings == 'pyqt4':
    import sip
    try:
        sip.setapi('QDate', 2)
        sip.setapi('QDateTime', 2)
        sip.setapi('QString', 2)
        sip.setapi('QTextStream', 2)
        sip.setapi('QTime', 2)
        sip.setapi('QUrl', 2)
        sip.setapi('QVariant', 2)
    except ValueError as e:
        raise RuntimeError('Could not set PyQt4 API version (%s)' % e)

    from PyQt4 import QtCore, QtGui, QtOpenGL
    import PyQt4.QtGui as QtWidgets
    from PyQt4.QtCore import pyqtSignal as Signal
    from PyQt4.QtCore import pyqtSlot as Slot

elif bindings == 'pyqt5':   # UNTESTED!
    import sip
    try:
        sip.setapi('QDate', 2)
        sip.setapi('QDateTime', 2)
        sip.setapi('QString', 2)
        sip.setapi('QTextStream', 2)
        sip.setapi('QTime', 2)
        sip.setapi('QUrl', 2)
        sip.setapi('QVariant', 2)
    except ValueError as e:
        raise RuntimeError('Could not set PyQt5 API version (%s)' % e)

    from PyQt5 import QtCore, QtGui, QtOpenGL
    import PyQt5.QtGui as QtWidgets
    from PyQt5.QtCore import pyqtSignal as Signal
    from PyQt5.QtCore import pyqtSlot as Slot

else:
    bindings = ''

if not bindings:
    raise ValueError("\n"+"*"*40+"\nThe pyFormex GUI requires Python bindings for the Qt libraries.\n These are available as PySide2, PySide, PyQt5 or PyQt4.\nI could not find neither of them, so I can not continue.\nInstall one of these bindings for Python3 (including its OpenGL component)\nand then retry."+"\n"+"*"*40)


if bindings == 'pyside2':
    def QByteArrayToStr(self):
        """Convert a QByteArray to a Python3 str"""
        return str(self)[2:-1]
elif bindings == 'pyside':
    def QByteArrayToStr(self):
        """Convert a QByteArray to a Python3 str"""
        return str(self)
else:
    raise RuntimeError("Fatal error: currently only pyside2/pyside bindings are supported, not pyqt4/pyqt5")

QtCore.QByteArray.toStr = QByteArrayToStr


# Some Qt wrappers

class QPixmap(QtGui.QPixmap):
    def __init__(self, *args):
        if isinstance(args[0], Path):
            args = (str(args[0]),) + args[1:]
        QtGui.QPixmap.__init__(self, *args)


class QImage(QtGui.QImage):
    def __init__(self, *args):
        if isinstance(args[0], Path):
            args = (str(args[0]),) + args[1:]
        QtGui.QImage.__init__(self, *args)




# Load X11 color names

def loadX11Colors(filename):
    """Load the X11 colors"""
    x11colors = {}
    with open(filename) as f:
        for line in f:
            s = line.strip().split()
            name = ''.join(s[3:]).lower()
            try:
                x11colors[name] = tuple(int(i) for i in s[:3])
            except Exception:
                pass

    return x11colors

pf.X11colors = loadX11Colors("/etc/X11/rgb.txt")
pf.X11 = len(pf.X11colors) > 0


# End
