/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: security.c,v $
-- Revision 1.21  2008/09/13 15:09:27  jsedwards
-- Fix Bug #2103915 - pass phrase now obscured in "show" mode.
--
-- Revision 1.20  2008/09/07 14:42:46  jsedwards
-- Fixed so pass phrase from environment variable works correctly.
--
-- Revision 1.19  2008/09/07 12:38:40  jsedwards
-- Made display_help function, changed Escape to not exit, and simplified the
-- backspace code.
--
-- Revision 1.18  2008/09/07 12:28:01  jsedwards
-- Added 'last' and 'none' display modes and 'quit' or 'exit' to exit the
-- program.
--
-- Revision 1.17  2008/09/07 11:22:24  jsedwards
-- Changed to have 'show' and 'hide' modes for pass phrase.
--
-- Revision 1.16  2008/09/07 03:18:55  jsedwards
-- First attempt at reading pass phrase in non-canonical mode.
--
-- Revision 1.15  2008/09/06 15:34:24  jsedwards
-- Changed to use TEST_ENVIRONMENT_VARIABLE.
--
-- Revision 1.14  2008/09/06 13:36:50  jsedwards
-- Change pass phrase from TEST_PASS_PHRASE define to environment variable.
--
-- Revision 1.13  2008/03/12 04:06:44  jsedwards
-- Changed to use the GNU MD5 context and functions instead of the RSA context
-- and functions.
--
-- Revision 1.12  2007/09/15 14:31:58  jsedwards
-- Added conditional code so that if TEST_PASS_PHRASE is defined, it will use
-- it instead of asking for pass phrase from user.
--
-- Revision 1.11  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.10  2007/01/16 14:12:02  jsedwards
-- Changed to use MD5 and SHA1 to generate key from short pass phrases.
--
-- Revision 1.9  2007/01/16 13:37:22  jsedwards
-- Eliminated get_short_key_from_password and made get_key_from_password call
-- convert_short_pass_phrase_to_reference if the password entered was too
-- short by itself.
--
-- Revision 1.8  2007/01/16 02:27:04  jsedwards
-- Added new routine for short pass phrases.
--
-- Revision 1.7  2006/12/01 14:37:23  jsedwards
-- Fix the year in the copyright.
--
-- Revision 1.6  2006/11/11 12:01:06  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.5  2006/10/26 01:51:29  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.4.2.4  2006/09/09 13:03:33  jsedwards
-- Moved "create_root_object" routine from security.c to objectify.c so that
-- the security module didn't have references to so many other parts of the
-- system.
--
-- Revision 1.4.2.3  2006/09/06 13:15:25  jsedwards
-- Removed nwos_seed_sequence function and instead pass pointers to the values
-- in the call to next_sequence function.
--
-- Revision 1.4.2.2  2006/09/02 15:04:30  jsedwards
-- Add reference list class reference to root object because now it has to
-- read the reference list to get it's size and we can't easily read it
-- without the read routine verifying it's class.
--
-- Revision 1.4.2.1  2006/09/02 01:12:25  jsedwards
-- Add root object reference in call to fill_in_common_header so it can put
-- it into the header.
--
-- Revision 1.4  2005/12/30 03:16:12  jsedwards
-- Changed "Password" prompt to "Pass phrase".
--
-- Revision 1.3  2005/12/28 13:03:24  jsedwards
-- Added sequence generator (from Fine random_number_generator).  Changed the
-- get reference from password routine to get a variable length key from the
-- password.  It's actually more like a pass phrase now because it requires
-- the input to be long enough to get enough bits to fill up the key.
--
-- Revision 1.2  2005/12/27 19:45:27  jsedwards
-- Added routine to create a root object.
--
-- Revision 1.1  2005/12/27 18:02:42  jsedwards
-- Initial version converts a pass phrase into an object reference (ObjRef).
--
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"

#include "gnu/md5.h"
#include "gnu/sha1.h"

#define ESC 0x1b

#define MAX_NUMBER_SIZE 512    /* number can be up to 4096 bits */
#define SHORT_NUMBER_SIZE 36

#define MULTIPLIER 3141592621U
#define ADDEND     101U


uint32 nwos_next_sequence(uint32* linear, uint32* serial)
{
    uint32 feedback;

    *linear = (*linear * MULTIPLIER) + ADDEND;

    /* feedback_mask: BIT 32 is 0x80004000 */
    /* exclusive or of bits 31 and 14 */
    feedback = ((*serial >> 31) ^ (*serial >> 14)) & 1;

    *serial = (*serial << 1) | feedback;

    return *linear ^ *serial;
}


void convert_pass_phrase_to_reference(char* pass_phrase, uint8 key[], size_t key_size)
{
    int i;
    int j;
    uint8 number[MAX_NUMBER_SIZE];
    uint8 mult;
    uint32 acc;
    char *p;

    mult = '~' - ' ' + 2;

#if 0
    printf("mult: %02x\n", mult);
#endif

    memset(&number, 0, MAX_NUMBER_SIZE);

    for (p = pass_phrase; *p != '\0'; p++)
    {
	acc = *p;

	if (' ' <= acc && acc <= '~')
	{
	    acc = acc - ' ' + 1;
	}
	else
	{
	    acc = 0;
	}

	/* multiply and add big endian ([0] is most significant byte) */
	for (i = MAX_NUMBER_SIZE; i > 0; i--)
	{
	    acc = (number[i-1] * mult) + acc;
	    number[i-1] = acc;
	    acc = acc >> 8;
	}
#if 0
	for (i = 0; i < MAX_NUMBER_SIZE; i++) if (number[i] != 0) break;

	while (i < MAX_NUMBER_SIZE)
	  {
	    printf("%02x", number[i++]);
	  }
	printf("\n");
#endif
    }

    /* exclusive or each reference size chunk into the result */

    memset(key, 0, key_size);

    j = key_size - 1;
    for (i = MAX_NUMBER_SIZE; i > 0; i--)
    {
	key[j] = key[j] ^ number[i-1];
	j--;
	if (j < 0) j = key_size - 1;
    }
}


void convert_short_pass_phrase_to_reference(char* pass_phrase, uint8 key[], size_t key_size)
{
#ifndef PUBLIC_MODE
    int i;
    int j;
    uint8 number[SHORT_NUMBER_SIZE];
    uint8 mult;
    uint32 acc;
    char *p;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    uint8 md5_digest[16];
    uint8 sha1_digest[20];

    mult = '~' - ' ' + 2;

#if 0
    printf("mult: %02x\n", mult);
#endif

    memset(&number, 0, SHORT_NUMBER_SIZE);

    for (p = pass_phrase; *p != '\0'; p++)
    {
	acc = *p;

	if (' ' <= acc && acc <= '~')
	{
	    acc = acc - ' ' + 1;
	}
	else
	{
	    acc = 0;
	}

	/* multiply and add big endian ([0] is most significant byte) */
	for (i = SHORT_NUMBER_SIZE; i > 0; i--)
	{
	    acc = (number[i-1] * mult) + acc;
	    number[i-1] = acc;
	    acc = acc >> 8;
	}
#if 0
	for (i = 0; i < SHORT_NUMBER_SIZE; i++) if (number[i] != 0) break;

	while (i < SHORT_NUMBER_SIZE)
	  {
	    printf("%02x", number[i++]);
	  }
	printf("\n");
#endif
    }

    for (i = 0; i < SHORT_NUMBER_SIZE; i++) if (number[i] != 0) break;

    md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
    md5_process_bytes(&number[i], SHORT_NUMBER_SIZE-i, &md5_context);    /* include this data in the md5 checksum */
    md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */

    sha1_init_ctx(&sha1_context);
    sha1_process_bytes(&number[i], SHORT_NUMBER_SIZE-i, &sha1_context);    /* include this data in the sha1 checksum */
    sha1_finish_ctx(&sha1_context, sha1_digest);

    /* exclusive or each reference size chunk into the result */

    memset(key, 0, key_size);

    j = 0;
    for (i = 0; i < 20; i++)
    {
	key[j] = key[j] ^ sha1_digest[i];
	j++;
	if (j == key_size) j = 0;
    }
    for (i = 0; i < 16; i++)
    {
	key[j] = key[j] ^ md5_digest[i];
	j++;
	if (j == key_size) j = 0;
    }
#endif
}


static void display_help()
{
    fputc('\n', stderr);
    fprintf(stderr, "Valid characters are A-Z, a-z, 0-9, @#$%%& *~^ +-= ;:,.!? /|\\ `'\" _ ()[]{}<>.\n");
    fputc('\n', stderr);
    fprintf(stderr, "Enter 'show' to display the characters typed, 'hide' to display asterisks,\n");
    fprintf(stderr, "'last' to display the last character typed, 'none' for no display at all,\n");
    fprintf(stderr, "and 'quit', 'exit', or Control-C to exit back to the shell without running.\n");
    fputc('\n', stderr);
}


void nwos_get_key_from_password(uint8 key[], size_t key_size)
{
    bool ok = false;
    char buffer[MAX_NUMBER_SIZE];
    char *test_phrase;
    int chars_needed;
    int num_chars = 0;
    int chars_read;
    int i;
    struct termios oldtio, newtio;
    static enum { None, Hide, Last, Show } display_mode = Hide;


    chars_needed = (key_size * 80) / 65; /* we get 6.5 bits for each character entered, we need 8 for each output byte */

    while (chars_needed * 65 < key_size * 80) chars_needed++;   /* adjust up for truncation errors */

    test_phrase = getenv(TEST_ENVIRONMENT_VARIABLE);

    if (test_phrase != NULL)
    {
	strncpy(buffer, test_phrase, sizeof(buffer));  /* copy environment variable value to buffer */
	num_chars = strlen(test_phrase);
    }
    else
    {
	tcgetattr(STDIN_FILENO, &oldtio); /* save current port settings */

	memcpy(&newtio, &oldtio, sizeof(newtio));

	newtio.c_lflag &= ~(ICANON | ECHO | ISIG);   /* by default turn off canonical, echo, and signals */

	while (!ok)
	{
	    fprintf(stderr, "Pass phrase: ");
	    fflush(stderr);

	    tcflush(STDIN_FILENO, TCIFLUSH);
	    tcsetattr(STDIN_FILENO, TCSANOW, &newtio);

	    num_chars = 0;
	    chars_read = 0;

	    while (chars_read == 0 && num_chars < sizeof(buffer))
	    {
		chars_read = read(STDIN_FILENO, buffer + num_chars, sizeof(buffer) - num_chars);

		if (chars_read == 0) break;

		while (chars_read > 0)
		{
		    if (buffer[num_chars] == newtio.c_cc[VERASE])  /* backspace */
		    {
			if (num_chars > 0)
			{
			    num_chars--;

			    if (display_mode != None)
			    {
				fputc('\b', stderr);
				fputc(' ', stderr);
				fputc('\b', stderr);
			    }

			    if (display_mode == Last && num_chars > 0)
			    {
				fputc('\b', stderr);
				fputc(buffer[num_chars - 1], stderr);
			    }
			}
		    }
		    else if (buffer[num_chars] == newtio.c_cc[VKILL])  /* erase line */
		    {
			num_chars = 0;
		    }
		    else if (' ' <= buffer[num_chars] && buffer[num_chars] <= '~')
		    {
			switch (display_mode)
			{
			  case None:
			    /* don't display anything */
			    break;

			  case Hide:
			    fputc('*', stderr);
			    break;

			  case Last:
			    if (num_chars > 0)
			    {
				fputc('\b', stderr);
				fputc('*', stderr);
			    }
			    fputc(buffer[num_chars], stderr);
			    break;

			  case Show:
			    fputc(buffer[num_chars], stderr);
			    break;
			}
			num_chars++;
		    }
		    else   /* anything else */
		    {
			break;   /* we are outta here */
		    }

		    chars_read--;
		}

		if (buffer[num_chars] == '\n')
		{
		    if (display_mode == Show)
		    {
			for (i = 0; i < num_chars; i++) fputc('\b', stderr);
			for (i = 0; i < num_chars; i++) fputc('*', stderr);
		    }
		    else if (display_mode == Last)
		    {
			fputc('\b', stderr);
			fputc('*', stderr);
		    }
		    fputc('\n', stderr);
		    chars_read--;
		    break;
		}
	    }

	    tcsetattr(STDIN_FILENO, TCSANOW, &oldtio);    /* restore to original settings */

	    if (chars_read > 0)
	    {
		if (buffer[num_chars] == newtio.c_cc[VINTR] || buffer[num_chars] == newtio.c_cc[VQUIT])
		{
		    fprintf(stderr, "\n");
		    exit(1);
		}
		else
		{
		    fprintf(stderr, "\nUnknown input while reading pass phrase: ");
		    for (i = 0; i < chars_read; i++)
		    {
			fprintf(stderr, "%02x ", buffer[num_chars]);
			num_chars++;
		    }
		    fprintf(stderr, "\n");

		    display_help();
		}
	    }
	    else if (num_chars == sizeof(buffer))    /* pass phrase is too long */
	    {
		fprintf(stderr, "password was too long, must be less than %d characters\n", MAX_NUMBER_SIZE);
	    }
	    else if (num_chars < 10)            /* pass phrase is too short */
	    {
		buffer[num_chars] = '\0';

		if (strcasecmp(buffer, "show") == 0)
		{
		    display_mode = Show;
		}
		else if (strcasecmp(buffer, "hide") == 0)
		{
		    display_mode = Hide;
		}
		else if (strcasecmp(buffer, "last") == 0)
		{
		    display_mode = Last;
		}
		else if (strcasecmp(buffer, "none") == 0)
		{
		    display_mode = None;
		}
		else if (strcasecmp(buffer, "quit") == 0 || strcasecmp(buffer, "exit") == 0)
		{
		    exit(1);
		}
		else if (strcasecmp(buffer, "help") == 0)
		{
		    display_help();
		}
		else
		{
		    fprintf(stderr, "password was too short, must be at least 10 characters for\n"
			    "short password or %d characters for higher security password\n", chars_needed);
		}
	    }
	    else
	    {
		buffer[num_chars] = '\0';   /* eliminate the newline character */

		ok = true;   /* we should be good to go */
	    }
	}
    }

    assert(num_chars >= 10);
    assert(strlen(buffer) == num_chars);

    if (num_chars < chars_needed)
    {
	convert_short_pass_phrase_to_reference(buffer, key, key_size);
    }
    else
    {
	convert_pass_phrase_to_reference(buffer, key, key_size);
    }
}




#if 0
int main(int argc, char* argv[])
{
  ObjRef key;
  int i;

  key = convert_pass_phrase_to_reference(argv[1]);

  for (i = 0; i < sizeof(ObjRef); i++) printf("%02x", key.id[i]);
  printf("\n");

  return 0;
}
#endif

#if 0
int main(int argc, char* argv[])
{
  uint8 big_key[28];
  int i;

  nwos_get_key_from_password(big_key, sizeof(big_key));

  for (i = 0; i < sizeof(big_key); i++)
  {
      printf("%02x", big_key[i]);
  }

  printf("\n");

  return 0;
}
#endif

