/*
--             This file is part of the New World OS project
--                 Copyright (C) 2005-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: class_definition.c,v $
-- Revision 1.29  2008/08/31 21:53:51  jsedwards
-- Added an assert around calls to nwos_read_variable_sized_object_from_disk
-- and nwos_read_object_from_disk because now they return false when they fail
-- instead of asserting themselves.
--
-- Revision 1.28  2008/07/19 13:07:03  jsedwards
-- Added asserts to verify that after an updated class is cloned that it can
-- be accessed succesfully.  NO changes to the code operation.
--
-- Revision 1.27  2008/07/17 12:07:18  jsedwards
-- Changed nwos_find_or_create_private_class_definition to return
-- ObjCreateResult instead of void.
--
-- Revision 1.26  2008/03/15 16:41:00  jsedwards
-- Changed 'nwos_find_private_class_definition' to call new
-- 'nwos_get_object_class_without_update function' instead of
-- 'nwos_get_object_class' to keep from getting in a recursive mess.
--
-- Revision 1.25  2007/07/28 01:20:09  jsedwards
-- Changes to deal with classes that have been updated.
--
-- Revision 1.24  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.23  2007/06/21 22:34:15  jsedwards
-- Added function to determine the size of an object automatically from it's
-- class.
--
-- Revision 1.22  2007/04/12 02:43:27  jsedwards
-- Add #ifdef public_mode around find or create public feature definition.
--
-- Revision 1.21  2007/03/23 14:04:29  jsedwards
-- Finished routine to create feature definition, not tested yet!
--
-- Revision 1.20  2007/03/23 13:22:06  jsedwards
-- Added find_public_feature_definition and started adding create public
-- feature.
--
-- Revision 1.19  2007/01/04 17:30:28  jsedwards
-- Changed to always get a completely random id for the private class clone
-- and print clone message to log instead of stdout.
--
-- Revision 1.18  2007/01/03 14:26:08  jsedwards
-- Changed the name of create_private_class_definition to
-- find_or_create_private_class_definition.
--
-- Revision 1.17  2007/01/03 14:20:39  jsedwards
-- Changed create_class_definition to create_private_class_definition which
-- clones the public class.
--
-- Revision 1.16  2007/01/02 11:23:32  jsedwards
-- Removed assert so we can clone the class_definition_class itself.
--
-- Revision 1.15  2006/12/21 13:15:41  jsedwards
-- Ifdef out find_class_definition function, at least for now.
--
-- Revision 1.14  2006/12/20 12:37:26  jsedwards
-- Split class_definition_class_ref into public and private variants.  Added
-- find public and find private class definiton functions.
--
-- Revision 1.13  2006/12/15 11:28:05  jsedwards
-- Change find routine to change the global variable when it steps down one
-- level.  Can only handle two areas (NOT three).
--
-- Revision 1.12  2006/12/14 13:49:59  jsedwards
-- Modified find_class_definition function to search through levels of class
-- definitions (encrypted, private, public).
--
-- Revision 1.11  2006/12/14 13:20:35  jsedwards
-- Added clone_class function.
--
-- Revision 1.10  2006/12/14 12:26:12  jsedwards
-- Change to clear kludge instead of class_ref.
--
-- Revision 1.9  2006/12/08 17:12:57  jsedwards
-- Moved 'read_class_definition' function from objectify.c and fixed it to
-- read the new variable length class_definition.  HOWEVER it is kludged now
-- to only NOT read the feature definitions into the class object because
-- none of the places it is called is set up to deal with variable sized
-- class objects.  Needs to be fixed!!
--
-- Revision 1.8  2006/12/05 04:17:08  jsedwards
-- Changed find_class to deal with the new variable sized class objects.
--
-- Revision 1.7  2006/12/01 14:37:23  jsedwards
-- Fix the year in the copyright.
--
-- Revision 1.6  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.5  2006/11/29 18:29:03  jsedwards
-- Fixed bug where memory following class_ref was overwritten if pointers were
-- not the same size as an ObjRef.
--
-- Revision 1.4  2006/11/11 14:19:07  jsedwards
-- Moved find_class_defintion function from objectify.c.
--
-- Revision 1.3  2006/11/11 12:01:01  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.2  2006/10/26 01:51:23  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.2  2006/10/25 12:22:27  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.1.2.1  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.1  2005/12/31 17:54:54  jsedwards
-- Extracted "create_class_definition" routine from "big_bang.c" and created
-- this file.
--
*/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"


static size_t get_class_object_size(void* class_obj)
{
    return sizeof(C_struct_Class_Definition) + (((C_struct_Class_Definition*)class_obj)->count * sizeof(ObjRef));
}


/*********************************************************************************/
/* This routine calculates the size of an object, including variable sized ones. */
/*********************************************************************************/

#define MAX_BUILT_IN_TYPES 4

static size_t feature_size(ObjRef* class_ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    char name[32];
    int i;

    static struct
    {
	ObjRef class;
	size_t size;
    } dictionary[MAX_BUILT_IN_TYPES];

    /* first see if we have figured this one out before */
    for (i = 0; i < MAX_BUILT_IN_TYPES; i++)
    {
	if (is_void_reference(&dictionary[i].class)) break;
	if (is_same_object(class_ref, &dictionary[i].class)) break;
    }

    assert(i < MAX_BUILT_IN_TYPES);

    if (is_void_reference(&dictionary[i].class))
    {
	copy_reference(&dictionary[i].class, class_ref);

	assert(nwos_read_variable_sized_object_from_disk(class_ref, kludge, sizeof(kludge), &get_class_object_size));
	nwos_name_to_string(&class_def_obj_ptr->name, name, sizeof(name));

	if (strcmp(name, "Object Reference") == 0)
	{
	    dictionary[i].size = sizeof(ObjRef);
	}
	else if (strcmp(name, "Byte") == 0)
	{
	    dictionary[i].size = sizeof(uint8);
	}
	else if (strcmp(name, "Character") == 0)
	{
	    dictionary[i].size = sizeof(char);
	}
	else if (strcmp(name, "Time Stamp") == 0)
	{
	    dictionary[i].size = sizeof(TimeStamp);
	}
	else
	{
	    bool Known_Built_in_Type = false;
	    assert(Known_Built_in_Type);
	}

	assert(MAX_BUILT_IN_TYPES == 4);  /* this must match how many there are */
    }

    return dictionary[i].size;
}


size_t nwos_get_object_size(void* object)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    C_struct_Feature_Definition feature_obj;
    char label[32] = "";
    EveryObject* header = (EveryObject*) object;
    size_t result = sizeof(EveryObject);
    size_t size;
    size_t prev_size = 0;
    uint8 prev_count = 0;
    int i;
    int item_count;
    static bool no_more_calls;

    if (no_more_calls) return 0;

    no_more_calls = true;

    assert(nwos_read_variable_sized_object_from_disk(&header->common.class_definition, kludge, sizeof(kludge), &get_class_object_size));

    for (i = 0; i < class_def_obj_ptr->count; i++)
    {
	assert(nwos_read_object_from_disk(&class_def_obj_ptr->feature[i], &feature_obj, sizeof(feature_obj)));

	size = feature_size(&feature_obj.class);

	/* deal with variable sized object if previous was named count */
	if (feature_obj.count == 0 && strcmp(label, "count"))
	{
	    if (prev_count == 1)   /* uint8 count */
	    {
		item_count = *((uint8*)object + result - 1);
	    }
	    else
	    {
		assert(prev_count == 4);
		item_count = nwos_decode_variable_sized_count((uint8*)object + result - 4);
	    }
	}
	else
	{
	    item_count = feature_obj.count;
	}

	result += size * item_count;

	/* deal with alignment pain in the ass */
	if (prev_size == 1 && (size == 4 || feature_obj.count == 0 || i+1 == class_def_obj_ptr->count))
	{
	    result += 3 - (prev_count - 1) % 4;
	}

	nwos_name_to_string(&feature_obj.label, label, sizeof(label));
#if 0
	printf("label: %s  size: %zd  prev_size: %zd  count: %u  item_count: %d  result: %zd\n",
	       label, size, prev_size, feature_obj.count, item_count, result);
#endif
	prev_size = size;
	prev_count = feature_obj.count;
    }

    no_more_calls = false;

    return result;
}





/*****************************************************************************/
/* This routine clones an updated public class definition for a class that   */
/* has been cloned.                                                          */
/*****************************************************************************/

void nwos_clone_updated_class_definition(ObjRef* old_private_ref, ObjRef* clone_ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    char msg[128];
    ObjRef public_ref;


    nwos_generate_new_completely_random_id(clone_ref);

    assert(nwos_read_variable_sized_object_from_disk(old_private_ref, kludge, sizeof(kludge), &get_class_object_size));

    copy_reference(&class_def_obj_ptr->header.object.next_version, clone_ref);

    nwos_crc32_calculate((uint8*) &class_def_obj_ptr->header.object, sizeof(ObjectHeader), class_def_obj_ptr->header.common.header_chksum);

    nwos_overwrite_object_to_disk(old_private_ref, class_def_obj_ptr, get_class_object_size(class_def_obj_ptr));

    /* get the old public object */
    assert(nwos_read_variable_sized_object_from_disk(&class_def_obj_ptr->header.object.clone_of, kludge, sizeof(kludge), &get_class_object_size));

    assert(!is_void_reference(&class_def_obj_ptr->header.object.next_version));

    while (!is_void_reference(&class_def_obj_ptr->header.object.next_version))
    {
	copy_reference(&public_ref, &class_def_obj_ptr->header.object.next_version);
	assert(nwos_read_variable_sized_object_from_disk(&public_ref, kludge, sizeof(kludge), &get_class_object_size));
    }

    snprintf(msg, sizeof(msg), "clone_class_definition(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
	   public_ref.id[0], public_ref.id[1], public_ref.id[2], public_ref.id[3],
	   clone_ref->id[0], clone_ref->id[1], clone_ref->id[2], clone_ref->id[3]);
    nwos_log(msg);

    copy_reference(&class_def_obj_ptr->header.common.id, clone_ref);

    nwos_get_time_stamp(class_def_obj_ptr->header.common.creation_time);

    copy_reference(&class_def_obj_ptr->header.common.class_definition, &nwos_private_class_definition_class_ref);

    copy_reference(&class_def_obj_ptr->header.object.clone_of, &public_ref);
    copy_reference(&class_def_obj_ptr->header.object.prev_version, old_private_ref);

    nwos_create_reference_list(clone_ref, &class_def_obj_ptr->header.object.references);

    nwos_crc32_calculate((uint8*) &class_def_obj_ptr->header.object, sizeof(ObjectHeader), class_def_obj_ptr->header.common.header_chksum);

    nwos_write_object_to_disk(clone_ref, class_def_obj_ptr, get_class_object_size(class_def_obj_ptr));

    nwos_add_to_references(clone_ref, &nwos_private_class_definition_class_ref);
}



/*********************************************************************************/
/* This routine finds or creates a private class definition with the given name. */
/*********************************************************************************/

ObjCreateResult nwos_find_or_create_private_class_definition(char* name_of_class, ObjRef* ref)
{
    ObjRef public_ref;
    ObjRef old_private_ref;
    ObjRef new_private_ref;
    EveryObject header;
    ObjCreateResult result = FOUND_EXISTING;

    if (nwos_find_private_class_definition(name_of_class, ref))   /* see if public is updated */
    {
	nwos_read_object_headers_from_disk(ref, &header);

	nwos_read_object_headers_from_disk(&header.object.clone_of, &header);

	if (!is_void_reference(&header.object.next_version))  /* the public class has been updated */
	{
	    copy_reference(&old_private_ref, ref);

	    nwos_clone_updated_class_definition(&old_private_ref, ref);
	    assert(!is_same_object(&old_private_ref, ref));

	    assert(nwos_find_private_class_definition(name_of_class, &new_private_ref));
	    assert(is_same_object(&new_private_ref, ref));

	    nwos_read_object_headers_from_disk(ref, &header);

	    nwos_read_object_headers_from_disk(&header.object.clone_of, &header);

	    assert(is_void_reference(&header.object.next_version));

	    result = CREATED_NEW;
	}
    }
    else
    {
	/* if there isn't a private one there certainly had better be a public one */
	assert(nwos_find_public_class_definition(name_of_class, &public_ref));

	nwos_clone_class_definition(&public_ref, ref);

	result = CREATED_NEW;
    }

    return result;
}


/*****************************************************************************/
/* This routine clones a public class definition.                            */
/*****************************************************************************/

void nwos_clone_class_definition(ObjRef* class_to_clone_ref, ObjRef* clone_ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    char msg[128];

    memset(kludge, 0, sizeof(kludge));

    assert(nwos_read_variable_sized_object_from_disk(class_to_clone_ref, kludge, sizeof(kludge), &get_class_object_size));

    nwos_generate_new_completely_random_id(clone_ref);

    snprintf(msg, sizeof(msg), "clone_class_definition(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
	   class_to_clone_ref->id[0], class_to_clone_ref->id[1], class_to_clone_ref->id[2], class_to_clone_ref->id[3],
	   clone_ref->id[0], clone_ref->id[1], clone_ref->id[2], clone_ref->id[3]);
    nwos_log(msg);

    copy_reference(&class_def_obj_ptr->header.common.id, clone_ref);

    nwos_get_time_stamp(class_def_obj_ptr->header.common.creation_time);

    copy_reference(&class_def_obj_ptr->header.common.class_definition, &nwos_private_class_definition_class_ref);

    copy_reference(&class_def_obj_ptr->header.object.clone_of, class_to_clone_ref);

    nwos_create_reference_list(clone_ref, &class_def_obj_ptr->header.object.references);

    nwos_crc32_calculate((uint8*) &class_def_obj_ptr->header.object, sizeof(ObjectHeader), class_def_obj_ptr->header.common.header_chksum);

    nwos_write_object_to_disk(clone_ref, class_def_obj_ptr, get_class_object_size(class_def_obj_ptr));

    nwos_add_to_references(clone_ref, &nwos_private_class_definition_class_ref);
}



/*****************************************************************************/
/* This routine finds the class definition with the given name.              */
/*****************************************************************************/

bool nwos_find_public_class_definition(char* name, ObjRef* class_ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    ReferenceList* ref_list;
    int num_classes;
    ObjRef object_class;
    char buffer[128];
    int i;
    bool result = false;

    memset(kludge, 0, sizeof(kludge));

    assert(nwos_read_variable_sized_object_from_disk(&nwos_public_class_definition_class_ref, kludge, sizeof(kludge), &get_class_object_size));

    ref_list = nwos_malloc_reference_list(&class_def_obj_ptr->header.object.references);

    num_classes = ref_list->common_header.num_refs;

    /* printf("num_classes (in find_class_definition): %d\n", num_classes); */

    for (i = 0; i < num_classes; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &nwos_public_class_definition_class_ref))   /* it is a class definition object */
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_class_object_size));

	    if (is_void_reference(&class_def_obj_ptr->header.object.next_version))
	    {
		nwos_name_to_string(&class_def_obj_ptr->name, buffer, sizeof(buffer));  /* get it's name */

		if (strcasecmp(name, buffer) == 0)
		{
		    memcpy(class_ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return result;
}


bool nwos_find_private_class_definition(char* name, ObjRef* class_ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    ReferenceList* ref_list;
    int num_classes;
    ObjRef object_class;
    char buffer[128];
    int i;
    bool result = false;

    memset(kludge, 0, sizeof(kludge));

    assert(nwos_read_variable_sized_object_from_disk(&nwos_private_class_definition_class_ref, kludge, sizeof(kludge), &get_class_object_size));

    ref_list = nwos_malloc_reference_list(&class_def_obj_ptr->header.object.references);

    num_classes = ref_list->common_header.num_refs;

    /* printf("num_classes (in find_class_definition): %d\n", num_classes); */

    for (i = 0; i < num_classes; i++)
    {
	/* find out what kind of object it is without update to keep from getting into a recursive mess */
	nwos_get_object_class_without_update(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &nwos_private_class_definition_class_ref))   /* it is a class definition object */
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_class_object_size));

	    if (is_void_reference(&class_def_obj_ptr->header.object.next_version))
	    {
		nwos_name_to_string(&class_def_obj_ptr->name, buffer, sizeof(buffer));   /* get it's name */

		if (strcasecmp(name, buffer) == 0)
		{
		    memcpy(class_ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return result;
}


#if 0
bool nwos_find_class_definition(char* name, ObjRef* class_ref)
{
    bool result = true;

    if (is_void_reference(&nwos_private_class_definition_class_ref) || !nwos_find_private_class_definition(name, class_ref))
    {
	result = nwos_find_public_class_definition(name, class_ref);
    }

    return result;
}
#endif


void nwos_read_class_definition(ObjRef* ref, C_struct_Class_Definition* class_def)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    assert(!is_void_reference(ref));

    assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), get_class_object_size));

    /* Holy shit Batman this is a mess! */
    /* memcpy(class_def, kludge, get_class_object_size(kludge)); */

    memcpy(class_def, kludge, sizeof(C_struct_Class_Definition));  /* for now just copy the first part of the class without features*/
}


static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);

    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}


bool nwos_find_public_feature_definition(ObjRef* type_ref, char* label, int count, ObjRef* ref)
{
    ObjRef label_ref;
    ObjRef feature_class_ref;
    ObjRef object_class;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* ptr_name_obj = (C_struct_Name*)kludge;
    C_struct_Feature_Definition feature_obj;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;

    if (nwos_find_public_name(label, &label_ref))
    {
	printf("Found public name: %s\n", label);

	assert(nwos_find_public_class_definition("FEATURE DEFINITION", &feature_class_ref));

	assert(nwos_read_variable_sized_object_from_disk(&label_ref, kludge, sizeof(kludge), &get_name_object_size));
	ref_list = nwos_malloc_reference_list(&ptr_name_obj->header.object.references);

	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&feature_class_ref, &object_class))
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &feature_obj, sizeof(feature_obj)));

		assert(is_same_object(&feature_obj.label, &label_ref));

		if (is_same_object(&feature_obj.class, type_ref) && feature_obj.count == count)
		{
		    copy_reference(ref, &ref_list->references[i]);
		    result = true;
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


#ifdef PUBLIC_MODE
void nwos_find_or_create_public_feature_definition(ObjRef* type_ref, char* label, int count, ObjRef* ref)
{
    C_struct_Feature_Definition feature_obj;
    ObjRef feature_class_ref;

    if (!nwos_find_public_feature_definition(type_ref, label, count, ref))
    {
	assert(nwos_find_public_class_definition("FEATURE DEFINITION", &feature_class_ref));

	nwos_generate_new_id(ref);

	memset(&feature_obj, 0, sizeof(feature_obj));

	nwos_fill_in_common_header(&feature_obj.header.common, ref, &feature_class_ref);

	copy_reference(&feature_obj.class, type_ref);
	nwos_create_name(label, &feature_obj.label);
	feature_obj.count = count;

	nwos_create_reference_list(ref, &feature_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &feature_obj.header.object, sizeof(ObjectHeader), feature_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &feature_obj.inherit, sizeof(feature_obj) - sizeof(EveryObject), feature_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &feature_obj, sizeof(feature_obj));

        nwos_add_to_references(ref, &feature_class_ref);
        nwos_add_to_references(ref, &feature_obj.label);
    }
}
#endif
