/*
 *  This file is part of librpc.
 *  (c) Copyright 2013 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  librpc is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  librpc is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with librpc.  If not, see <http://www.gnu.org/licenses/>.
 */
using Gee;

namespace FirstPass
{

    class Property
    {
        public string propname;
        public string classname;
    }

    class Argument
    {
        public string argname;
        public string classname;
    }

    class Method
    {
        public string returntype;
        public string name;
        public Argument[] args;
        public bool pass_caller = false;
        public string[] errors;
    }

    string? namespace_guess;
    string[] serializable_types;
    HashMap<string, ArrayList<string>> errordefs;
    string[] roots;
    string[] remotes;
    HashMap<string, ArrayList<Property>> properties;
    HashMap<string, ArrayList<Method>> methods;
    HashMap<string, ArrayList<Method>> methods_rmt;

    public void firstpass(string if_name, string of_name)
    {
        string[] lines = read_file(if_name);

        namespace_guess = scan_for_namespace(lines);
        serializable_types = scan_for_serializables(lines);
        errordefs = scan_for_errordomains(lines);

        roots = scan_for_root_classes(lines);
        remotes = scan_for_remote_classes(lines);
        string[] to_scan = {};
        foreach (string s in roots) to_scan += s;
        foreach (string s in remotes) to_scan += s;
        properties = new HashMap<string, ArrayList<Property>>();
        methods = new HashMap<string, ArrayList<Method>>();
        methods_rmt = new HashMap<string, ArrayList<Method>>();

        while (to_scan.length > 0)
        {
            string[] scan_again = {};
            foreach (string rmtclass in to_scan)
            {
                properties[rmtclass] = new ArrayList<Property>();
                methods[rmtclass] = new ArrayList<Method>();
                methods_rmt[rmtclass] = new ArrayList<Method>();
                string[] classlines;
                if (rmtclass in roots)
                    classlines = extract_class(lines, @"I$(rmtclass)RootDispatcher");
                else
                    classlines = extract_class(lines, @"I$(rmtclass)");
                Property[] ar_prop = scan_for_properties(classlines);
                foreach (Property prop in ar_prop)
                {
                    properties[rmtclass].add(prop);
                    if (! (prop.classname in remotes))
                    {
                        remotes += prop.classname;
                        scan_again += prop.classname;
                    }
                }
                Method[] ar_meth = scan_for_methods(classlines);
                foreach (Method meth in ar_meth)
                {
                    methods[rmtclass].add(meth);
                }
                Method[] ar_meth_rmt = scan_for_methods_rmt(classlines);
                foreach (Method meth in ar_meth_rmt)
                {
                    methods_rmt[rmtclass].add(meth);
                }
            }
            to_scan = scan_again;
        }

        string output = "";
        output += "errors:\n";
        foreach (string e in errordefs.keys)
        {
            output += @" $(e)\n";
            foreach (string ec in errordefs[e])
            {
                output += @"  $(ec)\n";
            }
        }

        output += "serializables:\n";
        foreach (string s in serializable_types)
        {
            output += @" $(s)\n";
        }

        output += "dispatchers:\n";
        foreach (string r in roots)
        {
            output +=         @" $(r)\n";
            output +=          "  properties:\n";
            foreach (Property p in properties[r])
            {
                output +=     @"   $(p.classname) $(p.propname)\n";
            }
            output +=          "  methods_rmt:\n";
            foreach (Method m in methods_rmt[r])
            {
                output +=     @"   $(m.returntype) $(m.name)\n";
                output +=      "    arguments:\n";
                foreach (Argument a in m.args)
                {
                    output += @"     $(a.classname) $(a.argname)\n";
                }
            }
            output +=          "  methods:\n";
            foreach (Method m in methods[r])
            {
                output +=     @"   $(m.returntype) $(m.name)\n";
                output +=      "    arguments:\n";
                foreach (Argument a in m.args)
                {
                    output += @"     $(a.classname) $(a.argname)\n";
                }
                if (m.pass_caller)
                    output +=  "    pass_caller\n";
                output +=      "    throws:\n";
                foreach (string err in m.errors)
                {
                    output += @"     $(err)\n";
                }
            }
        }

        output += "remoteclasses:\n";
        foreach (string r in remotes) if (!(r in roots))
        {
            output +=         @" $(r)\n";
            output +=          "  properties:\n";
            foreach (Property p in properties[r])
            {
                output +=     @"   $(p.classname) $(p.propname)\n";
            }
            output +=          "  methods_rmt:\n";
            foreach (Method m in methods_rmt[r])
            {
                output +=     @"   $(m.returntype) $(m.name)\n";
                output +=      "    arguments:\n";
                foreach (Argument a in m.args)
                {
                    output += @"     $(a.classname) $(a.argname)\n";
                }
            }
            output +=          "  methods:\n";
            foreach (Method m in methods[r])
            {
                output +=     @"   $(m.returntype) $(m.name)\n";
                output +=      "    arguments:\n";
                foreach (Argument a in m.args)
                {
                    output += @"     $(a.classname) $(a.argname)\n";
                }
                if (m.pass_caller)
                    output +=  "    pass_caller\n";
                output +=      "    throws:\n";
                foreach (string err in m.errors)
                {
                    output += @"     $(err)\n";
                }
            }
        }

        if (namespace_guess != null) output += @"namespace: $(namespace_guess)\n";

        write_file(of_name, output);
    }

    string[] scan_for_root_classes(string[] lines)
    {
        string[] ret = {};
        foreach (string line in lines)
        {
            Regex r = new Regex("""\bpublic\b.+\binterface\b.+\bI.*RootDispatcher\b""");
            if (r.match(line))
            {
                MatchInfo m2;
                Regex r2 = new Regex("""\bI.*RootDispatcher\b""");
                if (r2.match(line, 0, out m2))
                {
                    string? iname = m2.fetch(0);
                    string classname = iname.substring(1, iname.length - 15);
                    ret += classname;
                }
            }
        }
        return ret;
    }

    string[] scan_for_remote_classes(string[] lines)
    {
        string[] ret = {};
        foreach (string line in lines)
        {
            Regex r = new Regex("""\bpublic\b.+\binterface\b.+\bI[a-zA-Z0-9_]+\b""");
            if (r.match(line))
            {
                MatchInfo m2;
                Regex r2 = new Regex("""\bI[a-zA-Z0-9_]+\b""");
                if (r2.match(line, 0, out m2))
                {
                    string? iname = m2.fetch(0);
                    string classname = iname.substring(1);
                    if (classname.length > 14 && 
                        classname.substring(classname.length-14) == "RootDispatcher")
                        continue;
                    ret += classname;
                }
            }
        }
        return ret;
    }

    string[] separate_parentheses(string[] lines, string oppar="{", string clpar="}")
    {
        // split { and } in single lines
        string[] lines2 = {};
        foreach (string line in lines)
        {
            while (true)
            {
                int splita = line.index_of(oppar);
                int splitb = line.index_of(clpar);
                if (splita != splitb)
                {
                    // at least one is found
                    int split = splita;
                    if (splita == -1) split = splitb;
                    else if (splitb == -1) split = splita;
                    else if (splita > splitb) split = splitb;
                    string p1 = line.substring(0, split);
                    string p2 = line.substring(split, 1);
                    string p3 = line.substring(split+1);
                    lines2 += p1;
                    lines2 += p2;
                    line = p3;
                }
                else break;
            }
            lines2 += line;
        }
        return lines2;
    }

    string[] extract_class(string[] lines, string classname)
    {
        string[] ret = {};
        bool started = false;
        bool closing = false;
        int openpar = 0;
        foreach (string line in lines)
        {
            if (! started)
            {
                Regex r = new Regex("""\bpublic\b.+\binterface\b.+\b""" + classname + """\b""");
                if (r.match(line))
                    started = true;
            }
            if (started)
            {
                ret += line;
                MatchInfo m2;
                Regex r2 = new Regex("""\{""");
                if (r2.match(line, 0, out m2))
                {
                    openpar++;
                    while (m2.next()) openpar++;
                }
                MatchInfo m3;
                Regex r3 = new Regex("""\}""");
                if (r3.match(line, 0, out m3))
                {
                    closing = true;
                    openpar--;
                    while (m3.next()) openpar--;
                }
            }
            // it has to have met at least a closing par
            // and as many opening as closing.
            if (closing && (openpar == 0)) break;
        }
        return ret;
    }

    Property[] scan_for_properties(string[] lines)
    {
        string[] lines2 = separate_parentheses(lines);
        lines = lines2;

        Property[] ret = {};
        int openpar = 0;
        foreach (string line in lines)
        {
            // remember counting of open par
            // and evaluate only lines in level 1
            MatchInfo mpar2;
            Regex rpar2 = new Regex("""\{""");
            if (rpar2.match(line, 0, out mpar2))
            {
                openpar++;
                while (mpar2.next()) openpar++;
            }
            MatchInfo mpar3;
            Regex rpar3 = new Regex("""\}""");
            if (rpar3.match(line, 0, out mpar3))
            {
                openpar--;
                while (mpar3.next()) openpar--;
            }
            if (openpar != 1) continue;

            Regex r = new Regex("""\(""");
            if (!r.match(line))
            {
                MatchInfo m2;
                Regex r2 = new Regex("""^\s*public\b""");
                if (r2.match(line, 0, out m2))
                {
                    string? pubmod = m2.fetch(0);
                    string secondpart = line.substring(pubmod.length);
                    MatchInfo m3;
                    Regex r3 = new Regex("""[a-zA-Z0-9_]+""");
                    if (r3.match(secondpart, 0, out m3))
                    {
                        string classname = m3.fetch(0);
                        if (classname.substring(0,1) == "I")
                        {
                            classname = classname.substring(1);
                            if (m3.next())
                            {
                                Property p = new Property();
                                p.classname = classname;
                                p.propname = m3.fetch(0);
                                ret += p;
                            }
                        }
                    }
                }
            }
        }
        return ret;
    }

    string? scan_for_namespace(string[] lines)
    {
        string ret = "untitled";
        foreach (string line in lines)
        {
            MatchInfo m;
            Regex r = new Regex("""^namespace\s+[a-zA-Z0-9_.]+""");
            if (r.match(line, 0, out m))
            {
                ret = m.fetch(0).substring(10);
                MatchInfo m2;
                Regex r2 = new Regex("""[a-zA-Z0-9_.]+""");
                if (r2.match(ret, 0, out m2))
                {
                    ret = m2.fetch(0);
                    return ret;
                }
            }
        }
        return null;
    }

    string[] scan_for_serializables(string[] lines)
    {
        string[] ret = {};
        foreach (string line in lines)
        {
            Regex r = new Regex("""\bpublic\b.+\bclass\b.+\bISerializable\b""");
            if (r.match(line))
            {
                MatchInfo m2;
                Regex r2 = new Regex("""[a-zA-Z0-9_]+""");
                if (r2.match(line, 0, out m2))
                {
                    while (true)
                    {
                        string keyword = m2.fetch(0);
                        if (keyword == "class") break;
                        m2.next();
                    }
                    m2.next();
                    string classname = m2.fetch(0);
                    ret += classname;
                }
            }
        }
        return ret;
    }

    HashMap<string, ArrayList<string>> scan_for_errordomains(string[] lines)
    {
        HashMap<string, ArrayList<string>> ret = new HashMap<string, ArrayList<string>>();
        foreach (string line in lines)
        {
            Regex r = new Regex("""\bpublic\b.+\berrordomain\b""");
            if (r.match(line))
            {
                MatchInfo m2;
                Regex r2 = new Regex("""[a-zA-Z0-9_]+""");
                if (r2.match(line, 0, out m2))
                {
                    while (true)
                    {
                        string keyword = m2.fetch(0);
                        if (keyword == "errordomain") break;
                        m2.next();
                    }
                    m2.next();
                    string errordm = m2.fetch(0);
                    ret[errordm] = new ArrayList<string>();
                }
            }
        }
        foreach (string errordm in ret.keys)
        {
            string[] lines2 = extract_errordomain(lines, errordm);
            string[] codes = scan_for_errorcodes(lines2);
            foreach (string code in codes) ret[errordm].add(code);
        }
        return ret;
    }

    string[] extract_errordomain(string[] lines, string errordm)
    {
        string[] ret = {};
        bool started = false;
        bool closing = false;
        int openpar = 0;
        foreach (string line in lines)
        {
            if (! started)
            {
                Regex r = new Regex("""\bpublic\b.+\berrordomain\b.+""" + errordm);
                if (r.match(line))
                    started = true;
            }
            if (started)
            {
                ret += line;
                MatchInfo m2;
                Regex r2 = new Regex("""\{""");
                if (r2.match(line, 0, out m2))
                {
                    openpar++;
                    while (m2.next()) openpar++;
                }
                MatchInfo m3;
                Regex r3 = new Regex("""\}""");
                if (r3.match(line, 0, out m3))
                {
                    closing = true;
                    openpar--;
                    while (m3.next()) openpar--;
                }
            }
            // it has to have met at least a closing par
            // and as many opening as closing.
            if (closing && (openpar == 0)) break;
        }
        return ret;
    }

    string[] scan_for_errorcodes(string[] lines)
    {
        string[] lines2 = separate_parentheses(lines);
        lines = lines2;

        string[] ret = {};
        int openpar = 0;
        foreach (string line in lines)
        {
            // remember counting of open par
            // and evaluate only lines in level 1
            MatchInfo mpar2;
            Regex rpar2 = new Regex("""\{""");
            if (rpar2.match(line, 0, out mpar2))
            {
                openpar++;
                while (mpar2.next()) openpar++;
            }
            MatchInfo mpar3;
            Regex rpar3 = new Regex("""\}""");
            if (rpar3.match(line, 0, out mpar3))
            {
                openpar--;
                while (mpar3.next()) openpar--;
            }
            if (openpar != 1) continue;

            MatchInfo m3;
            Regex r3 = new Regex("""[a-zA-Z0-9_]+""");
            if (r3.match(line, 0, out m3))
            {
                ret += m3.fetch(0);
                while (m3.next())
                    ret += m3.fetch(0);
            }
        }
        return ret;
    }

    Method[] scan_for_methods(string[] lines)
    {
        Method[] ret = {};

        string[] method_lines = {};
        string concat = "";
        foreach (string line in lines)
        {
            if (concat != "")
            {
                if (";" in line)
                {
                    method_lines += concat + " " + line;
                    concat = "";
                }
                else
                {
                    concat += " " + line;
                }
            }
            else
            {
                Regex r_exclude = new Regex("""_getter\(\)""");
                if (! r_exclude.match(line))
                {
                    Regex r = new Regex("""^\s*public\s+abstract\s+""");
                    MatchInfo m;
                    if (r.match(line, 0, out m))
                    {
                        string toremove = m.fetch(0);
                        string start = line.substring(toremove.length);
                        if (";" in start)
                        {
                            method_lines += start;
                        }
                        else
                        {
                            concat = start;
                        }
                    }
                }
            }
        }

        foreach (string method_line in method_lines)
        {
            int start_pos;
            int end_pos;
            Method met = new Method();
            met.args = {};
            met.errors = {};

            // if it is a method that returns a remote class ignore it
            Regex r_meth_rmt = new Regex("""\).*\bthrows\b.*\bRPCError\b.*;""");
            if (! r_meth_rmt.match(method_line)) continue;

            MatchInfo m2;
            // A classname can contain letters numbers and underscore.
            // It can be prefixed with a namespace.
            // It can contain generics (supports only Gee.List<xxxx>)
            // It can be nullable.
            Regex r2 = new Regex("""[a-zA-Z0-9_.<>]+\??""");
            if (! r2.match(method_line, 0, out m2))
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  where is the return type?\n");
                continue;
            }
            met.returntype = m2.fetch(0);
            checktype(met.returntype);
            // stdout.printf(@"$(met.returntype)\n");
            if (! m2.next())
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  where is the name?\n");
                continue;
            }
            met.name = m2.fetch(0);
            // stdout.printf(@"$(met.name)\n");
            if (! m2.fetch_pos(0, out start_pos, out end_pos))
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  where is the name?\n");
                continue;
            }
            // stdout.printf(@"start $(start_pos) end $(end_pos)\n");
            string remaining_line = method_line.substring(end_pos);
            // stdout.printf(@"remaining $(remaining_line)\n");
            Regex rcheck = new Regex("""^\s*\(.*\)""");
            MatchInfo margs;
            if (! rcheck.match(remaining_line, 0, out margs))
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  where are the parens?\n");
                continue;
            }
            if (! margs.fetch_pos(0, out start_pos, out end_pos))
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  where are the parens?\n");
                continue;
            }
            // stdout.printf(@"args: start $(start_pos) end $(end_pos)\n");
            string args_line = remaining_line.substring(start_pos, end_pos-start_pos);
            // stdout.printf(@"args $(args_line)\n");
            args_line = args_line.substring(args_line.index_of("(")+1);
            // stdout.printf(@"args $(args_line)\n");
            args_line = args_line.substring(0, args_line.index_of(")"));
            // stdout.printf(@"args $(args_line)\n");
            string[] args = args_line.split(",");
            for (int i = 0; i < args.length; i++)
            {
                string arg = args[i];
                if (i == args.length-1)
                {
                    // only last arg may be "CallerInfo? _rpc_caller=null"
                    Regex rcaller = new Regex("""^\s*CallerInfo\?\s+_rpc_caller\s*=\s*null\s*$""");
                    if (rcaller.match(arg))
                    {
                        met.pass_caller = true;
                        continue;
                    }
                }
                Argument arg1 = new Argument();
                MatchInfo m3;
                // A classname can contain letters numbers and underscore.
                // It can be prefixed with a namespace.
                // It can contain generics (supports only Gee.List<xxxx>)
                // It can be nullable.
                Regex r3 = new Regex("""[a-zA-Z0-9_.<>]+\??""");
                if (! r3.match(arg, 0, out m3))
                {
                    stdout.printf(@"Cannot process argument \"$(arg)\" for \"$(method_line)\"\n");
                    continue;
                }
                arg1.classname = m3.fetch(0);
                checktype(arg1.classname);
                if (! m3.next())
                {
                    stdout.printf(@"Cannot process argument \"$(arg)\" for \"$(method_line)\"\n");
                    continue;
                }
                arg1.argname = m3.fetch(0);
                met.args += arg1;
            }
            MatchInfo merrors;
            Regex rerrors = new Regex("""\).*\bthrows\b.*\bRPCError\b.*;""");
            if (! rerrors.match(method_line, 0, out merrors))
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  missing RPCError.\n");
                continue;
            }
            string error_line = merrors.fetch(0);
            error_line = error_line.substring(error_line.index_of("throws") + 6);
            error_line = error_line.substring(0, error_line.index_of(";"));
            // stdout.printf(@"errors $(error_line)\n");
            MatchInfo merr;
            Regex rerr = new Regex("""[a-zA-Z0-9_]+""");
            if (! rerr.match(error_line, 0, out merr))
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  throws what?\n");
                continue;
            }
            while (true)
            {
                string strerr = merr.fetch(0);
                if (strerr != "RPCError") met.errors += strerr;
                if (! merr.next()) break;
            }
            // stdout.printf(@"Method $(met)\n");
            ret += met;
        }
        return ret;
    }

    Method[] scan_for_methods_rmt(string[] lines)
    {
        Method[] ret = {};

        string[] method_lines = {};
        string concat = "";
        foreach (string line in lines)
        {
            if (concat != "")
            {
                if (";" in line)
                {
                    method_lines += concat + " " + line;
                    concat = "";
                }
                else
                {
                    concat += " " + line;
                }
            }
            else
            {
                Regex r_exclude = new Regex("""_getter\(\)""");
                if (! r_exclude.match(line))
                {
                    Regex r = new Regex("""^\s*public\s+abstract\s+""");
                    MatchInfo m;
                    if (r.match(line, 0, out m))
                    {
                        string toremove = m.fetch(0);
                        string start = line.substring(toremove.length);
                        if (";" in start)
                        {
                            method_lines += start;
                        }
                        else
                        {
                            concat = start;
                        }
                    }
                }
            }
        }

        foreach (string method_line in method_lines)
        {
            int start_pos;
            int end_pos;
            Method met = new Method();
            met.args = {};
            met.errors = {};

            // if it is a method that returns a serializable class ignore it
            Regex r_meth_ser = new Regex("""\).*\bthrows\b.*\bRPCError\b.*;""");
            if (r_meth_ser.match(method_line)) continue;

            MatchInfo m2;
            // A classname can contain letters numbers and underscore.
            // It can be prefixed with a namespace.
            Regex r2 = new Regex("""[a-zA-Z0-9_.]+""");
            if (! r2.match(method_line, 0, out m2))
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  where is the return type?\n");
                continue;
            }
            met.returntype = m2.fetch(0);
            if (met.returntype.substring(0, 1) != "I")
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  return type (remote class) should start with I\n");
                continue;
            }
            met.returntype = met.returntype.substring(1);
            checkreturntype_rmt(met.returntype);
            // stdout.printf(@"$(met.returntype)\n");
            if (! m2.next())
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  where is the name?\n");
                continue;
            }
            met.name = m2.fetch(0);
            // stdout.printf(@"$(met.name)\n");
            if (! m2.fetch_pos(0, out start_pos, out end_pos))
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  where is the name?\n");
                continue;
            }
            // stdout.printf(@"start $(start_pos) end $(end_pos)\n");
            string remaining_line = method_line.substring(end_pos);
            // stdout.printf(@"remaining $(remaining_line)\n");
            Regex rcheck = new Regex("""^\s*\(.*\)""");
            MatchInfo margs;
            if (! rcheck.match(remaining_line, 0, out margs))
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  where are the parens?\n");
                continue;
            }
            if (! margs.fetch_pos(0, out start_pos, out end_pos))
            {
                stdout.printf(@"Cannot process \"$(method_line)\"\n");
                stdout.printf( "  where are the parens?\n");
                continue;
            }
            // stdout.printf(@"args: start $(start_pos) end $(end_pos)\n");
            string args_line = remaining_line.substring(start_pos, end_pos-start_pos);
            // stdout.printf(@"args $(args_line)\n");
            args_line = args_line.substring(args_line.index_of("(")+1);
            // stdout.printf(@"args $(args_line)\n");
            args_line = args_line.substring(0, args_line.index_of(")"));
            // stdout.printf(@"args $(args_line)\n");
            string[] args = args_line.split(",");
            for (int i = 0; i < args.length; i++)
            {
                string arg = args[i];
                Argument arg1 = new Argument();
                MatchInfo m3;
                // A classname can contain letters numbers and underscore.
                Regex r3 = new Regex("""[a-zA-Z0-9_]+""");
                if (! r3.match(arg, 0, out m3))
                {
                    stdout.printf(@"Cannot process argument \"$(arg)\" for \"$(method_line)\"\n");
                    continue;
                }
                arg1.classname = m3.fetch(0);
                checkargtype_rmt(arg1.classname);
                if (! m3.next())
                {
                    stdout.printf(@"Cannot process argument \"$(arg)\" for \"$(method_line)\"\n");
                    continue;
                }
                arg1.argname = m3.fetch(0);
                met.args += arg1;
            }
            // stdout.printf(@"Method $(met)\n");
            ret += met;
        }
        return ret;
    }

    void checkreturntype_rmt(string t)
    {
        if (t in remotes) return;
        abort(@"$t not a valid return type for a method that should return a remote class");
    }

    void checkargtype_rmt(string t)
    {
        string[] basic = {"int"};
        if (t in basic) return;
        abort(@"$t not a valid argument type for a method that should return a remote class");
    }

    void checktype(string t)
    {
        string? msg = checktype_msg(t);
        if (msg != null) stdout.printf(@"Warning: firstpass: $(msg)\n");
    }

    string? checktype_msg(owned string t)
    {
        string[] basic = {"string", "bool", "int", "void",
                          "string?", "bool?", "int?", "ISerializable", "RemoteCall"};
        if (t in basic) return null;
        if (t in serializable_types) return null;
        if (t == "") abort("type = ''");
        if (t == "?") abort("type = '?'");
        if (t.substring(t.length-1) == "?") t = t.substring(0, t.length-1);
        if (t in serializable_types) return null;
        if (t.length > 9 && t.substring(0, 9) == "Gee.List<") t = t.substring(9, t.length-10);
        if (t in basic) return null;
        if (t in serializable_types) return null;
        return @"Did you forget to include $(t) in serializables?";
    }

    void abort(string msg)
    {
        stdout.printf(@"Aborting: firstpass: $(msg)\n");
        Process.exit (1);
    }

    string[] read_file(string path)
    {
        string[] ret = new string[0];
        if (FileUtils.test(path, FileTest.EXISTS))
        {
            try
            {
                string contents;
                assert(FileUtils.get_contents(path, out contents));
                ret = contents.split("\n");
            }
            catch (FileError e) {error("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));}
        }
        return ret;
    }

    void write_file(string path, string contents)
    {
        try
        {
            assert(FileUtils.set_contents(path, contents));
        }
        catch (FileError e) {error("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));}
    }

}

