/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;

namespace zcd
{
    /** Packs a message of variable size together with the indication of its size
      */
    uchar[] data_pack(uchar[] data)
    {
        uint data_sz = data.length;
        Variant data_hdr = new Variant.uint32(data_sz);
        uint data_hdr_sz = (uint)data_hdr.get_size();
        uchar []ser = null;
        ser = new uchar[data_hdr_sz + data_sz];
        for (int i = 0; i < data_sz; i++) ser[data_hdr_sz + i] = data[i];
            if (!endianness_network) data_hdr = data_hdr.byteswap();
        data_hdr.store(ser);
        return ser;
    }

    /** Reads from a stream a message of variable size (use with tcp streams)
      */
    uchar[] data_unpack_from_stream(IConnectedStreamSocket socket)
    {
        try
        {
            int i;
            uchar[] readBuffer = new uchar[0];
            Variant data_hdr_tmp = new Variant.uint32(0);
            uint data_hdr_sz = (uint)data_hdr_tmp.get_size();
            while (true)
            {
                uchar[] rawPacket = socket.recv((int)(data_hdr_sz - readBuffer.length));
                uchar[] tempBuffer = new uchar[readBuffer.length + rawPacket.length];
                for (i = 0; i < readBuffer.length; i++) tempBuffer[i] = readBuffer[i];
                for (i = 0; i < rawPacket.length; i++) tempBuffer[readBuffer.length + i] = rawPacket[i];
                readBuffer = tempBuffer;
                if (readBuffer.length == data_hdr_sz)
                {
                    Variant data_hdr = Variant.new_from_data<uchar[]>(VariantType.UINT32, readBuffer, false);
                    if (!endianness_network) data_hdr = data_hdr.byteswap();
                    uint data_sz = (uint)data_hdr;
                    log_debug(@"zcd: data_unpack_from_stream: start reading $(data_sz) bytes from stream.");
                    // TODO Do I have enough memory?
                    readBuffer = new uchar[0];
                    while (readBuffer.length != data_sz)
                    {
                        rawPacket = socket.recv((int)(data_sz - readBuffer.length));
                        tempBuffer = new uchar[readBuffer.length + rawPacket.length];
                        for (i = 0; i < readBuffer.length; i++) tempBuffer[i] = readBuffer[i];
                        for (i = 0; i < rawPacket.length; i++) tempBuffer[readBuffer.length + i] = rawPacket[i];
                        readBuffer = tempBuffer;
                    }
                    return readBuffer;
                }
            }
        }
        catch (Error e)
        {
            return new uchar[0];
        }
    }

    /** An instance of this class is used when we want to send a message via TCP.
      */
    public class TCPClient : Object, FakeRmt
    {
        public string dest_addr {get; private set;}
        public uint16 dest_port;
        private string my_addr;
        private bool wait_response;
        private bool connected;
        public bool calling {get; private set;}
        public bool retry_connect {get; set; default=true;}
        private IConnectedStreamSocket? socket;
        public TCPClient(string dest_addr, uint16? dest_port=null, string? my_addr=null, bool wait_response=true)
        {
            if (dest_port == null) dest_port = (uint16)269;
            this.dest_addr = dest_addr;
            this.dest_port = dest_port;
            this.my_addr = my_addr;
            this.wait_response = wait_response;
            socket = null;
            connected = false;
            calling = false;
        }

        public ISerializable rmt(RemoteCall data) throws RPCError
        {
            zcd.log_debug(@"$(this.get_type().name()): sending $data");
            // TODO locking for thread safety
            if (calling)
            {
                // A TCPClient instance cannot handle more than one rpc call at a time.
                // When a new RPC is needed and another one is 'calling' with this
                // instance of TCPClient, then another instance should be created.
                // Anyway, this control will tolerate it and emit a WARNING.
                zcd.log_warn("TCPClient: a remote call through TCP is delayed...");
                while (calling) Thread.usleep(1000);
                zcd.log_warn("TCPClient: a remote call through TCP was delayed. Now takes place.");
            }
            try
            {
                // Now other threads cannot make a RPC call
                // until this one has finished
                calling = true;

                TCPRequest message = new TCPRequest(wait_response, data);
                rpc_send(message.serialize());
                if (wait_response)
                    return rpc_receive();
                return new SerializableNone();
            }
            finally
            {
                calling = false;
            }
        }

        public void rpc_send(uchar[] serdata) throws RPCError
        {
            // 30 seconds max to try connecting
            Timer timeout = new Timer();
            int interval = 5;
            while (!connected)
            {
                zcd.log_debug("TCPClient: trying to connect...");
                connect();
                zcd.log_debug("TCPClient: connect returns " + (connected?"True":"False"));
                if (!connected)
                {
                    if (!retry_connect || timeout.elapsed() > 30)
                    {
                        throw new RPCError.NETWORK_ERROR(
                                @"Failed connecting to (\"$(dest_addr)\", $(dest_port))");
                    }
                    zcd.log_debug(@"wait $(interval) before trying again to connect a TCPClient...");
                    // wait <n> ms.
                    Thread.usleep(interval * 1000);
                    interval *= 2;
                    if (interval > 10000) interval = 10000;
                }
            }
            zcd.log_debug("TCPClient: sending message...");
            try
            {
                socket.send(data_pack(serdata));
            }
            catch (Error e)
            {
                connected = false;
                throw new RPCError.NETWORK_ERROR(e.message);
            }
            zcd.log_debug("TCPClient: message sent.");
        }

        public ISerializable rpc_receive() throws RPCError
        {
            if (connected)
            {
                zcd.log_debug("TCPClient: receiving response...");
                uchar[] recv_encoded_data = data_unpack_from_stream(socket);
                zcd.log_debug("TCPClient: got response.");

                if (recv_encoded_data.length == 0)
                {
                    zcd.log_debug("TCPClient: throwing 'Connection closed before reply'.");
                    connected = false;
                    throw new RPCError.NETWORK_ERROR(
                                "Connection closed before reply");
                }

                zcd.log_debug("TCPClient: deserializing.");
                ISerializable ret;
                try {
                    ret = ISerializable.deserialize(recv_encoded_data);
                }
                catch (SerializerError e) {
                    throw new RPCError.SERIALIZER_ERROR(
                                "Error deserializing response");
                }
                return ret;
            }
            else throw new RPCError.NETWORK_ERROR(
                                "Connection closed before reply");
        }

        public new void connect()
        {
            try
            {
                ClientStreamSocket x = new ClientStreamSocket(my_addr);
                socket = x.socket_connect(dest_addr, dest_port);
                connected = true;
            }
            catch (Error e)
            {
                zcd.log_debug("TCPClient: socket connect error: %s".printf(e.message));
            }
        }

        public void close() throws RPCError
        {
            if (connected)
            {
                if (socket != null)
                {
                    try
                    {
                        socket.close();
                    }
                    catch (Error e)
                    {
                        throw new RPCError.NETWORK_ERROR(e.message);
                    }
                    socket = null;
                }
                connected = false;
            }
        }

        ~TCPClient()
        {
            zcd.log_debug(@"TCPClient: destructor: \"$(dest_addr)\"");
            if (connected) close();
        }
    }
}

