/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Tasklets;

namespace zcd
{
    /** Packs a message of variable size together with the indication of its size
      */
    uchar[] data_pack(uchar[] data)
    {
        uint data_sz = data.length;
        Variant data_hdr = new Variant.uint32(data_sz);
        uint data_hdr_sz = (uint)data_hdr.get_size();
        uchar []ser = null;
        ser = new uchar[data_hdr_sz + data_sz];
        for (int i = 0; i < data_sz; i++) ser[data_hdr_sz + i] = data[i];
            if (!endianness_network) data_hdr = data_hdr.byteswap();
        data_hdr.store(ser);
        return ser;
    }

    /** Reads from a stream a message of variable size (use with tcp streams)
      */
    uchar[] data_unpack_from_stream(IConnectedStreamSocket socket)
    {
        try
        {
            int i;
            uchar[] readBuffer = new uchar[0];
            Variant data_hdr_tmp = new Variant.uint32(0);
            uint data_hdr_sz = (uint)data_hdr_tmp.get_size();
            while (true)
            {
                uchar[] rawPacket = socket.recv((int)(data_hdr_sz - readBuffer.length));
                uchar[] tempBuffer = new uchar[readBuffer.length + rawPacket.length];
                for (i = 0; i < readBuffer.length; i++) tempBuffer[i] = readBuffer[i];
                for (i = 0; i < rawPacket.length; i++) tempBuffer[readBuffer.length + i] = rawPacket[i];
                readBuffer = tempBuffer;
                if (readBuffer.length == data_hdr_sz)
                {
                    Variant data_hdr = Variant.new_from_data<uchar[]>(VariantType.UINT32, readBuffer, false);
                    if (!endianness_network) data_hdr = data_hdr.byteswap();
                    uint data_sz = (uint)data_hdr;
                    log_debug(@"zcd: data_unpack_from_stream: start reading $(data_sz) bytes from stream.");
                    // TODO Do I have enough memory?
                    readBuffer = new uchar[0];
                    while (readBuffer.length != data_sz)
                    {
                        rawPacket = socket.recv((int)(data_sz - readBuffer.length));
                        tempBuffer = new uchar[readBuffer.length + rawPacket.length];
                        for (i = 0; i < readBuffer.length; i++) tempBuffer[i] = readBuffer[i];
                        for (i = 0; i < rawPacket.length; i++) tempBuffer[readBuffer.length + i] = rawPacket[i];
                        readBuffer = tempBuffer;
                    }
                    return readBuffer;
                }
            }
        }
        catch (Error e)
        {
            return new uchar[0];
        }
    }

    /** Reads from a packet a message of variable size (use with udp packets)
      */
    uchar[] data_unpack_from_buffer(uchar[] buffer)
    {
        int i;
        Variant data_hdr_tmp = new Variant.uint32(0);
        uint data_hdr_sz = (uint)data_hdr_tmp.get_size();
        if (buffer.length < data_hdr_sz) return new uchar[0];
        uchar[] readBuffer = new uchar[data_hdr_sz];
        for (i = 0; i < data_hdr_sz; i++) readBuffer[i] = buffer[i];
        Variant data_hdr = Variant.new_from_data<uchar[]>(VariantType.UINT32, readBuffer, false);
        if (!endianness_network) data_hdr = data_hdr.byteswap();
        uint data_sz = (uint)data_hdr;
        if (buffer.length != data_sz + data_hdr_sz) return new uchar[0];
        readBuffer = new uchar[data_sz];
        for (i = 0; i < data_sz; i++) readBuffer[i] = buffer[data_hdr_sz + i];
        return readBuffer;
    }

    /** Informations about an host that sent us a message.
      */
    public class CallerInfo : Object
    {
        public string? caller_ip;
        public string? my_ip;
        public string? dev;
        public CallerInfo(string? caller_ip, string? my_ip, string? dev)
        {
            this.caller_ip = caller_ip;
            this.my_ip = my_ip;
            this.dev = dev;
        }
    }

    /** This callback is used to interpret a request. It is called this way:
      *   rpcdispatcher, data, response = self.callback(caller, data_request)
      * where the arguments are:
      *   caller: a CallerInfo instance.
      *   data_request: what has been read from socket as a request.
      * and the results are:
      *   rpcdispatcher: if not None, we must do the call to the remotable method:
      *      rpcdispatcher.marshalled_dispatch(caller, data)
      *   response: if rpcdispatcher is None and response is not None then send response.
      *             if rpcdispatcher is None and response is None then do nothing.
      */
    [CCode (has_target = false)]
    public delegate void TCPCallbackDelegate (CallerInfo caller,
                                              TCPRequest tcprequest,
                                              out RPCDispatcher? rpcdispatcher,
                                              out uchar[] data,
                                              out uchar[] response);
    struct struct_helper_TCPServer_handle_connection
    {
        public TCPServer self;
        public IConnectedStreamSocket sock;
    }
    struct struct_helper_TCPServer_listen
    {
        public TCPServer self;
    }

    /** An instance of this class is created for the whole application in order
      * to listen to TCP requests.
      */
    public class TCPServer : Object
    {
        private uint16 port;
        private TCPCallbackDelegate? callback;
        private Tasklet? listen_handle = null;
        private ServerStreamSocket? listen_s = null;
        public TCPServer(TCPCallbackDelegate? callback, uint16 port=269)
        {
            this.port = port;
            this.callback = callback;
            listen_handle = null;
        }

        /** This method is executed in a separated tasklet. It handles a connection. We do not know
          *  a priori if we'll receive 0, 1 or more requests in this connection.
          */
        protected virtual void impl_handle_connection(IConnectedStreamSocket sock) throws Error
        {
            Tasklet.declare_self("TCPServer.handle_connection");
            assert(callback != null);
            try
            {
                zcd.log_debug(@"TCPServer: handling a connection from $(sock.peer_address) to my address $(sock.my_address)");
                CallerInfo caller = new CallerInfo(sock.peer_address, sock.my_address, null);
                while (true)
                {
                    RPCDispatcher? rpcdispatcher;
                    uchar[] data;
                    uchar[] response;
                    uchar[] data_request = data_unpack_from_stream(sock);
                    if (data_request.length == 0) break;
                    TCPRequest tcprequest = (TCPRequest)ISerializable.deserialize(data_request);
                    zcd.log_debug(@"TCPServer: from $(sock.peer_address): got $tcprequest");
                    callback(caller, tcprequest, out rpcdispatcher,
                             out data, out response);
                    if (rpcdispatcher != null)
                    {
                        if (tcprequest.wait_response)
                        {
                            zcd.log_debug(@"TCPServer: from $(sock.peer_address): executing...");
                            response = rpcdispatcher.marshalled_dispatch(caller, data);
                        }
                        else
                        {
                            zcd.log_debug(@"TCPServer: from $(sock.peer_address): spawning execution.");
                            // Call rpcdispatcher.marshalled_dispatch(data) in a new tasklet
                            struct_helper_RPCDispatcher_marshalled_dispatch arg = struct_helper_RPCDispatcher_marshalled_dispatch();
                            arg.self = rpcdispatcher;
                            arg.caller = caller;
                            arg.data = data;
                            Tasklet.spawn((FunctionDelegate)helper_marshalled_dispatch, &arg);
                            response = null;
                        }
                    }
                    if (response != null && response.length > 0)
                    {
                        zcd.log_debug(@"TCPServer: from $(sock.peer_address): sending response...");
                        sock.send(data_pack(response));
                        zcd.log_debug(@"TCPServer: from $(sock.peer_address): response sent.");
                    }
                }
                zcd.log_debug(@"TCPServer: terminating connection from $(sock.peer_address) to my address $(sock.my_address)");
            }
            catch (Error e)
            {
                zcd.log_warn(@"TCPServer: from $(sock.peer_address): An error occurred during connection handling");
            }
            finally
            {
                try {
                    sock.close();
                } catch (Error e) {}
            }
        }

        private static void * helper_handle_connection(void *v) throws Error
        {
            struct_helper_TCPServer_handle_connection *tuple_p = (struct_helper_TCPServer_handle_connection *)v;
            // The caller function has to add a reference to the ref-counted instances
            TCPServer self_save = tuple_p->self;
            IConnectedStreamSocket sock_save = tuple_p->sock;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_handle_connection(sock_save);
            // void method, return null
            return null;
        }

        public void handle_connection(IConnectedStreamSocket sock)
        {
            struct_helper_TCPServer_handle_connection arg = struct_helper_TCPServer_handle_connection();
            arg.self = this;
            arg.sock = sock;
            Tasklet.spawn((FunctionDelegate)helper_handle_connection, &arg);
        }

        private void impl_listen() throws Error
        {
            Tasklet.declare_self("TCPServer.listen");
            // TODO since listen is a microfunc and a fatal error should be reported to
            //   the application (e.g. to respawn listen), then we should add a signal to
            //   the class.
            try
            {
                if (listen_handle != null) return;
                listen_handle = Tasklet.self();
                listen_s = new ServerStreamSocket(port);
                Tasklets.Timer timer = null;
                int num_errors = 0;
                int excessive = 50;
                long period_msec = 2000;
                while (true)
                {
                    try
                    {
                        handle_connection(listen_s.accept());
                    }
                    catch (Error e)
                    {
                        zcd.log_warn(@"TCPServer: An error occurred during socket.accept(): $(e.message)");
                        // check for an excessive number of errors in a short period
                        if (timer == null || timer.is_expired())
                        {
                            timer = new Tasklets.Timer(period_msec);
                            num_errors = 0;
                        }
                        num_errors++;
                        if (num_errors > excessive) throw e;
                    }
                }
            }
            catch (Error e)
            {
                zcd.log_error(@"TCPServer: An error occurred during listening for connections: $(e.message)");
                if (listen_s != null)
                {
                    try
                    {
                        listen_s.close();
                    }
                    catch (Error e) {}
                    listen_s = null;
                }
                // TODO signal that the TCPServer is not listening any more
            }
        }

        private static void * helper_listen(void *v) throws Error
        {
            struct_helper_TCPServer_listen *tuple_p = (struct_helper_TCPServer_listen *)v;
            // The caller function has to add a reference to the ref-counted instances
            TCPServer self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_listen();
            // void method, return null
            return null;
        }

        public void listen()
        {
            struct_helper_TCPServer_listen arg = struct_helper_TCPServer_listen();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_listen, &arg);
        }

        public void stop()
        {
            zcd.log_debug("TCPServer: stop");
            if (listen_handle != null)
            {
                listen_handle.abort();
                listen_handle = null;
            }
            if (listen_s != null)
            {
                try {
                    listen_s.close();
                } catch (Error e) {}
                listen_s = null;
           }
        }

        public bool is_listening()
        {
            return listen_handle != null;
        }

        ~TCPServer()
        {
            zcd.log_debug("TCPServer: destructor");
            stop();
        }
    }

    /** An instance of this class is used when we want to send a message via TCP.
      */
    public class TCPClient : Object, FakeRmt
    {
        public string dest_addr {get; private set;}
        public uint16 dest_port;
        private string my_addr;
        private bool wait_response;
        private bool connected;
        public bool calling {get; private set;}
        public bool retry_connect {get; set; default=true;}
        private IConnectedStreamSocket? socket;
        public TCPClient(string dest_addr, uint16? dest_port=null, string? my_addr=null, bool wait_response=true)
        {
            if (dest_port == null) dest_port = (uint16)269;
            this.dest_addr = dest_addr;
            this.dest_port = dest_port;
            this.my_addr = my_addr;
            this.wait_response = wait_response;
            socket = null;
            connected = false;
            calling = false;
        }

        public ISerializable rmt(RemoteCall data) throws RPCError
        {
            zcd.log_debug(@"$(this.get_type().name()): sending $data");
            if (calling)
            {
                // A TCPClient instance cannot handle more than one rpc call at a time.
                // When a new RPC is needed and another one is 'calling' with this
                // instance of TCPClient, then another instance should be created.
                // Anyway, this control will tolerate it and emit a WARNING.
                zcd.log_warn("TCPClient: a remote call through TCP is delayed...");
                while (calling) Tasklet.nap(0, 1000);
                zcd.log_warn("TCPClient: a remote call through TCP was delayed. Now takes place.");
            }
            try
            {
                // Now other tasklets cannot make a RPC call
                // until this one has finished
                calling = true;

                TCPRequest message = new TCPRequest(wait_response, data);
                rpc_send(message.serialize());
                if (wait_response)
                    return rpc_receive();
                return new SerializableNone();
            }
            finally
            {
                calling = false;
            }
        }

        public void rpc_send(uchar[] serdata) throws RPCError
        {
            // 30 seconds max to try connecting
            Tasklets.Timer timeout = new Tasklets.Timer(30000);
            int interval = 5;
            while (!connected)
            {
                zcd.log_debug("TCPClient: trying to connect...");
                connect();
                zcd.log_debug("TCPClient: connect returns " + (connected?"True":"False"));
                if (!connected)
                {
                    if (!retry_connect || timeout.is_expired())
                    {
                        throw new RPCError.NETWORK_ERROR(
                                "Failed connecting to (\"%s\", %d)"
                                .printf(dest_addr, dest_port));
                    }
                    zcd.log_debug("wait %d before trying again to connect a TCPClient..."
                                 .printf(interval));
                    // wait <n> ms.
                    ms_wait(interval);
                    interval *= 2;
                    if (interval > 10000) interval = 10000;
                }
            }
            zcd.log_debug("TCPClient: sending message...");
            try
            {
                socket.send(data_pack(serdata));
            }
            catch (Error e)
            {
                connected = false;
                throw new RPCError.NETWORK_ERROR(e.message);
            }
            zcd.log_debug("TCPClient: message sent.");
        }

        public ISerializable rpc_receive() throws RPCError
        {
            if (connected)
            {
                zcd.log_debug("TCPClient: receiving response...");
                uchar[] recv_encoded_data = data_unpack_from_stream(socket);
                zcd.log_debug("TCPClient: got response.");

                if (recv_encoded_data.length == 0)
                {
                    zcd.log_debug("TCPClient: throwing 'Connection closed before reply'.");
                    connected = false;
                    throw new RPCError.NETWORK_ERROR(
                                "Connection closed before reply");
                }

                zcd.log_debug("TCPClient: deserializing.");
                ISerializable ret;
                try {
                    ret = ISerializable.deserialize(recv_encoded_data);
                }
                catch (SerializerError e) {
                    throw new RPCError.SERIALIZER_ERROR(
                                "Error deserializing response");
                }
                return ret;
            }
            else throw new RPCError.NETWORK_ERROR(
                                "Connection closed before reply");
        }

        public new void connect()
        {
            try
            {
                ClientStreamSocket x = new ClientStreamSocket(my_addr);
                socket = x.socket_connect(dest_addr, dest_port);
                connected = true;
            }
            catch (Error e)
            {
                zcd.log_debug("TCPClient: socket connect error: %s".printf(e.message));
            }
        }

        public void close() throws RPCError
        {
            if (connected)
            {
                if (socket != null)
                {
                    try
                    {
                        socket.close();
                    }
                    catch (Error e)
                    {
                        throw new RPCError.NETWORK_ERROR(e.message);
                    }
                    socket = null;
                }
                connected = false;
            }
        }

        ~TCPClient()
        {
            zcd.log_debug(@"TCPClient: destructor: \"$(dest_addr)\"");
            if (connected) close();
        }
    }

    Object _KEEP_ALIVE = null;
    Object get_KEEP_ALIVE()
    {
        if (_KEEP_ALIVE == null) _KEEP_ALIVE = new Object();
        return _KEEP_ALIVE;
    }

    [CCode (has_target = false)]
    public delegate void BroadcastCallbackDelegate (CallerInfo caller,
                                                    UDPPayload payload,
                                                    out Gee.List<RPCDispatcher> rpcdispatchers,
                                                    out uchar[] data);
    [CCode (has_target = false)]
    public delegate void UnicastCallbackDelegate (CallerInfo caller,
                                                  UDPPayload payload,
                                                  out RPCDispatcher? rpcdispatcher,
                                                  out uchar[] data,
                                                  out Gee.List<string> devs_response);
    struct struct_helper_UDPServer_send_keepalive_forever_start_micro
    {
        public UDPServer self;
        public Gee.List<string> devs;
        public int request_id;
        public int interval;
    }
    struct struct_helper_UDPServer_listen
    {
        public UDPServer self;
    }
    struct struct_helper_UDPServer_handle_request
    {
        public UDPServer self;
        public uchar[] message;
        public string rmt_ip;
        public uint16 rmt_port;
    }

    /** An instance of this class is created for each managed NIC in order
      * to listen to UDP requests.
      */
    public class UDPServer : Object
    {
        private UnicastCallbackDelegate unicast_callback;
        private BroadcastCallbackDelegate broadcast_callback;
        private string dev;
        private uint16 port;
        private uint16 peerport;
        private Tasklet? listen_handle = null;
        private ServerDatagramSocket listen_s = null;
        private ArrayList<int> keeping_alive_list;

        /** These callbacks are used to interpret a unicast/broadcast request.
          * They are called this way:
          *             rpcdispatchers, data = this.broadcast_callback(CallerInfo, UDPPayload)
          *             rpcdispatcher, data, devs_response = this.unicast_callback(CallerInfo, UDPPayload)
          * where the results are:
          *   rpcdispatchers: a list of rpcdispatcher (might be empty)
          *   rpcdispatcher:  a dispatcher (or null)
          *   data:           to be passed to marshalled_dispatch
          *   devs_response:  if null we use a new tasklet and do not send response to the client.
          *                   else, we'll send the response through those devs.
          */
        public UDPServer(UnicastCallbackDelegate unicast_callback,
                         BroadcastCallbackDelegate broadcast_callback,
                         string dev, uint16? port=null, uint16? peerport=null)
        {
            if (port == null) port = (uint16)269;
            this.unicast_callback = unicast_callback;
            this.broadcast_callback = broadcast_callback;
            // These callbacks are used to interpret a unicast/broadcast request.
            this.dev = dev;
            this.port = port;
            if (peerport == null) this.peerport = port;
            else this.peerport = peerport;
            listen_handle = null;
            keeping_alive_list = new ArrayList<int>();
        }

        /** This callback is provided for a server that does want to ignore
          * unicast requests.
          */
        public static void ignore_unicast(CallerInfo caller,
                                          UDPPayload payload,
                                          out RPCDispatcher? rpcdispatcher,
                                          out uchar[] data,
                                          out Gee.List<string> devs_response)
        {
            rpcdispatcher = null;
            data = null;
            devs_response = null;
        }

        /** This callback is provided for a server that does want to ignore
          * broadcast requests.
          */
        public static void ignore_broadcast(CallerInfo caller,
                                            UDPPayload payload,
                                            out Gee.List<RPCDispatcher> rpcdispatchers,
                                            out uchar[] data)
        {
            rpcdispatchers = new ArrayList<RPCDispatcher>();
            data = null;
        }

        public void send_keepalive_forever_start(Gee.List<string> devs, int request_id, int interval=800)
        {
            // Starts sending a keepalive each interval
            keeping_alive_list.add(request_id);
            zcd.log_debug(@"keepalive_forever $(request_id)");
            send_keepalive_forever_start_micro(devs, request_id, interval);
        }

        private void impl_send_keepalive_forever_start_micro(Gee.List<string> devs, int request_id, int interval) throws Error
        {
            Tasklet.declare_self("UDPServer.send_keepalive_forever");
            try
            {
                // Sends a keepalive each interval
                UDPResponseClient resp_client = new UDPResponseClient(devs, request_id, peerport);
                while (true)
                {
                    ms_wait(interval);
                    zcd.log_debug(@"keepalive_forever $(request_id): checking keeping_alive_list");
                    if (keeping_alive_list.contains(request_id))
                    {
                        zcd.log_debug(@"keepalive_forever $(request_id): sending keepalive");
                        resp_client.send_keepalive();
                    }
                    else
                    {
                        zcd.log_debug(@"keepalive_forever $(request_id): stopped.");
                        break;
                    }
                }
            }
            catch (Error e)
            {
                zcd.log_warn("keepalive_forever %d: error %s".printf(request_id, e.message));
            }
        }

        private static void * helper_send_keepalive_forever_start_micro(void *v) throws Error
        {
            struct_helper_UDPServer_send_keepalive_forever_start_micro *tuple_p = (struct_helper_UDPServer_send_keepalive_forever_start_micro *)v;
            // The caller function has to add a reference to the ref-counted instances
            UDPServer self_save = tuple_p->self;
            Gee.List<string> devs_save = tuple_p->devs;
            // The caller function has to copy the value of byvalue parameters
            int request_id_save = tuple_p->request_id;
            int interval_save = tuple_p->interval;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_send_keepalive_forever_start_micro(devs_save, request_id_save, interval_save);
            // void method, return null
            return null;
        }

        public void send_keepalive_forever_start_micro(Gee.List<string> devs, int request_id, int interval)
        {
            struct_helper_UDPServer_send_keepalive_forever_start_micro arg = struct_helper_UDPServer_send_keepalive_forever_start_micro();
            arg.self = this;
            arg.devs = devs; // arrays are reference counted.
            arg.request_id = request_id;
            arg.interval = interval;
            Tasklet.spawn((FunctionDelegate)helper_send_keepalive_forever_start_micro, &arg);
        }

        public void send_keepalive_forever_stop(int request_id)
        {
            if (keeping_alive_list.contains(request_id))
            {
                keeping_alive_list.remove(request_id);
            }
        }

        /** This function implement a simple Rpc UDP server
          */
        private void impl_listen() throws Error
        {
            Tasklet.declare_self("UDPServer.listen");
            try
            {
                /*
                    *WARNING*
                    If the message to be received is greater than the buffer
                    size, it will be lost!
                    Use UDP RPC only for small calls, i.e. if the arguments passed to the
                    remote function are small when packed.
                    *WARNING*
                */
                if (listen_handle != null) return;
                listen_handle = Tasklet.self();
                listen_s = new ServerDatagramSocket(port, null, dev);
                while (true)
                {
                    try
                    {
                        string rmt_ip;
                        uint16 rmt_port;
                        uchar[] message = listen_s.recvfrom(8192, out rmt_ip, out rmt_port);
                        handle_request(message, rmt_ip, rmt_port);
                    }
                    catch (Error e)
                    {
                        zcd.log_warn("UDPServer.listen: temporary error %s".printf(e.message));
                        Tasklet.nap(0, 20000);
                    }
                }
            }
            catch (Error e)
            {
                zcd.log_error("UDPServer.listen: fatal error %s".printf(e.message));
            }
        }

        private static void * helper_listen(void *v) throws Error
        {
            struct_helper_UDPServer_listen *tuple_p = (struct_helper_UDPServer_listen *)v;
            // The caller function has to add a reference to the ref-counted instances
            UDPServer self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_listen();
            // void method, return null
            return null;
        }

        public void listen()
        {
            // Before spawning a new tasklet, do all the things that we want to ensure that are
            // completed when we start to communicate on the network. That is, ensure that the
            // NIC is ready.
#if linux
            int ret;
            string msg_doing = @"trying to disable rp_filter on NIC $(dev)";
            string command = @"sysctl -w net.ipv4.conf.$(dev).rp_filter=0";
            log_debug(@"UDPServer: $(msg_doing)");
            try {
                CommandResult com_ret = Tasklet.exec_command(command);
                if (com_ret.exit_status != 0) log_warn(@"UDPServer: '$(command)' returned $(com_ret.exit_status) instead of 0.");
                else log_info(@"Disabled rp_filter on NIC $(dev)");
            } catch (Error e) {
                log_error(@"UDPServer: Error while $(msg_doing): $(e.domain).$(e.code) '$(e.message)'");
            }

            msg_doing = "trying to disable rp_filter on every NIC";
            command = "sysctl -w net.ipv4.conf.all.rp_filter=0";
            log_debug(@"UDPServer: $(msg_doing)");
            try {
                CommandResult com_ret = Tasklet.exec_command(command);
                if (com_ret.exit_status != 0) log_warn(@"UDPServer: '$(command)' returned $(com_ret.exit_status) instead of 0.");
            } catch (Error e) {
                log_error(@"UDPServer: Error while $(msg_doing): $(e.domain).$(e.code) '$(e.message)'");
            }

            msg_doing = @"trying to set up NIC $(dev)";
            command = @"ip l set $(dev) up";
            log_debug(@"UDPServer: $(msg_doing)");
            try {
                ret = Tasklet.system(command);
                if (ret != 0) log_warn(@"UDPServer: '$(command)' returned $(ret) instead of 0.");
                else log_info(@"Set up NIC $(dev)");
            } catch (Error e) {
                log_error(@"UDPServer: Error while $(msg_doing): $(e.domain).$(e.code) '$(e.message)'");
            }
#elif win32
            error("Your platform is not supported yet");
#else
            error("Your platform is not supported yet");
#endif

            struct_helper_UDPServer_listen arg = struct_helper_UDPServer_listen();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_listen, &arg);

            // After having spawn the new tasklet, before returning to the caller wait a bit, to
            // ensure that the socket is listening.
            Tasklet.nap(0, 10000);
        }

        public void stop()
        {
            if (listen_handle != null)
            {
                listen_handle.abort();
                listen_handle = null;
                listen_s = null;
            }
        }

        public bool is_listening()
        {
            return listen_handle != null;
        }

        void hex_dump(uint8[] buf)
        {
            /* dumps buf to stdout. Looks like:
             * [0000] 75 6E 6B 6E 6F 77 6E 20
             *                  30 FF 00 00 00 00 39 00 unknown 0.....9.
             * (in a single line of course)
             */

            uchar *p = buf;
            string addrstr = "";
            string hexstr = "";
            string charstr = "";

            for (int n = 1; n <= buf.length; n++)
            {
                if (n % 16 == 1)
                {
                    addrstr = "%.4x".printf((uint)p-(uint)buf);
                }

                uchar c = *p;
                if (c == '\0' || "( ):-_qwertyuiopasdfghjklzxcvbnmQWERTYUIOPASDFGHJKLZXCVBNM1234567890".index_of_char(c) < 0)
                    c = '.';

                // store hex str (for left side)
                hexstr += "%02X ".printf(*p);
                charstr += "%c".printf(c);

                if (n % 16 == 0)
                {
                    // line completed
                    stdout.printf("[%4.4s]   %-50.50s  %s\n", addrstr, hexstr, charstr);
                    hexstr = "";
                    charstr = "";
                }
                else if (n % 16 == 8)
                {
                    // half line: add whitespaces
                    hexstr += "  ";
                    charstr += " ";
                }

                p++; // next byte
            }

            if (hexstr.length > 0)
            {
                // print rest of buffer if not empty
                stdout.printf("[%4.4s]   %-50.50s  %s\n", addrstr, hexstr, charstr);
            }
        }

        private void impl_handle_request(uchar[] message, string rmt_ip, uint16 rmt_port) throws Error
        {
            Tasklet.declare_self("UDPServer.handle_request");
            CallerInfo caller = new CallerInfo(rmt_ip, rmt_port.to_string(), dev);
            try
            {
                int lenpacket = message.length;
                if (lenpacket > 4000) zcd.log_warn("CAUTION! Handling UDP packet of %d bytes.".printf(lenpacket));
                else if (lenpacket > 3000) zcd.log_debug("WARNING!!! Handling UDP packet of %d bytes.".printf(lenpacket));
                else if (lenpacket > 1024) zcd.log_debug("Handling UDP packet of %d bytes.".printf(lenpacket));
                //if (Ntk.Settings.TESTING) {stdout.printf("\nUDPServer: message follows:\n"); hex_dump((uint8[])message);}
                uchar[] message_got = data_unpack_from_buffer(message);
                //if (Ntk.Settings.TESTING) {stdout.printf("\nUDPServer: unpacked follows:\n"); hex_dump((uint8[])message_got);}
                if (! ISerializable.is_instance_of(typeof(UDPMessage), message_got))
                    throw new RPCError.MALFORMED_PACKET("Not a UDPMessage");
                UDPMessage udpmessage = (UDPMessage)ISerializable.deserialize(message_got);
                zcd.log_debug(@"UDPServer: got $udpmessage");
                if (udpmessage.is_request_broadcast())
                {
                    int request_id = udpmessage.request_id;
                    if ((! is_in_pending_requests(request_id)) && (! is_in_executed_requests(request_id)))
                    {
                        ExecutedRequest executed_request = new ExecutedRequest();
                        add_to_executed_requests(request_id, executed_request);
                        if (! udpmessage.data.get_type().is_a(typeof(UDPPayload)))
                            throw new RPCError.MALFORMED_PACKET("Not a UDPPayload");
                        Gee.List<RPCDispatcher> rpcdispatchers;
                        uchar[] data;
                        broadcast_callback(caller, (UDPPayload)udpmessage.data,
                                                    out rpcdispatchers,
                                                    out data);
                        foreach (RPCDispatcher rpcdispatcher in rpcdispatchers)
                        {
                            // Call rpcdispatcher.marshalled_dispatch(data) in a new tasklet
                            struct_helper_RPCDispatcher_marshalled_dispatch arg = struct_helper_RPCDispatcher_marshalled_dispatch();
                            arg.self = rpcdispatcher;
                            arg.caller = caller;
                            arg.data = data;
                            Tasklet.spawn((FunctionDelegate)helper_marshalled_dispatch, &arg);
                        }
                    }
                }
                else if (udpmessage.is_request_unicast())
                {
                    int request_id = udpmessage.request_id;
                    if (! is_in_executed_requests(request_id))
                    {
                        ExecutedRequest executed_request = new ExecutedRequest();
                        add_to_executed_requests(request_id, executed_request);
                        if (! udpmessage.data.get_type().is_a(typeof(UDPPayload)))
                            throw new RPCError.MALFORMED_PACKET("Not a UDPPayload");
                        RPCDispatcher? rpcdispatcher;
                        uchar[] data;
                        Gee.List<string> devs_response;
                        unicast_callback(caller, (UDPPayload)udpmessage.data,
                                                    out rpcdispatcher,
                                                    out data,
                                                    out devs_response);
                        if (rpcdispatcher != null)
                        {
                            if (devs_response != null && udpmessage.wait_response)
                            {
                                send_keepalive_forever_start(devs_response, request_id);
                                uchar[] response = rpcdispatcher.marshalled_dispatch(caller, data);
                                send_keepalive_forever_stop(request_id);
                                UDPResponseClient resp_client = new UDPResponseClient(devs_response, request_id, peerport);
                                resp_client.send_response(ISerializable.deserialize(response));
                            }
                            else
                            {
                                rpcdispatcher.marshalled_dispatch(caller, data);
                            }
                        }
                    }
                }
                else if (udpmessage.is_response())
                {
                    int request_id = udpmessage.request_id;
                    PendingRequest pending_request = retrieve_pending_request(request_id);
                    if (pending_request != null)
                    {
                        Channel channel = pending_request.channel;
                        channel.send(udpmessage.data);
                    }
                }
                else if (udpmessage.is_keepalive())
                {
                    int request_id = udpmessage.request_id;
                    PendingRequest pending_request = retrieve_pending_request(request_id);
                    if (pending_request != null)
                    {
                        Channel channel = pending_request.channel;
                        channel.send(get_KEEP_ALIVE());
                    }
                }
            }
            catch (Error e)
            {
                zcd.log_warn(@"An error occurred during request handling. $(e.message)");
            }
        }

        private static void * helper_handle_request(void *v) throws Error
        {
            struct_helper_UDPServer_handle_request *tuple_p = (struct_helper_UDPServer_handle_request *)v;
            // The caller function has to add a reference to the ref-counted instances
            UDPServer self_save = tuple_p->self;
            uchar[] message_save = tuple_p->message;
            // The caller function has to copy the value of byvalue parameters
            string rmt_ip_save = tuple_p->rmt_ip;
            uint16 rmt_port_save = tuple_p->rmt_port;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_handle_request(message_save, rmt_ip_save, rmt_port_save);
            // void method, return null
            return null;
        }

        public void handle_request(uchar[] message, string rmt_ip, uint16 rmt_port)
        {
            struct_helper_UDPServer_handle_request arg = struct_helper_UDPServer_handle_request();
            arg.self = this;
            arg.message = message; // arrays are reference counted.
            arg.rmt_ip = rmt_ip;
            arg.rmt_port = rmt_port;
            Tasklet.spawn((FunctionDelegate)helper_handle_request, &arg);
        }
    }

    /** An instance of a derived of this class is used when we want to send a message via UDP.
      */
    public abstract class UDPRequestClient : Object, FakeRmt
    {
        private BroadcastClientDatagramSocket[] sockets = null;
        public uint16 dest_port;
        private bool wait_response;
        public UDPRequestClient(uint16? dest_port=null, bool wait_response=true)
        {
            if (dest_port == null) dest_port = (uint16)269;
            this.dest_port = dest_port;
            this.wait_response = wait_response;
        }

        public ISerializable rmt(RemoteCall data) throws RPCError
        {
            zcd.log_debug(@"$(this.get_type().name()): sending $data");
            try
            {
                if (sockets == null)
                    initialize();
                UDPMessage udpmessage;
                if (is_unicast())
                {
                    udpmessage = UDPMessage.make_request_unicast((ISerializable)get_udp_payload(data), wait_response);
                }
                else
                {
                    udpmessage = UDPMessage.make_request_broadcast((ISerializable)get_udp_payload(data));
                    wait_response = false;
                }
                int request_id = udpmessage.request_id;
                PendingRequest pending_request = new PendingRequest();
                add_to_pending_requests(request_id, pending_request);

                zcd.log_debug(@"UDPRequestClient: sending it wrapped in $udpmessage");
                rpc_send(udpmessage.serialize());
                if (wait_response)
                {
                    pending_request.channel = new Channel();
                    ISerializable ret;
                    while(true)
                    {
                        Value v_ret;
                        try
                        {
                            v_ret = pending_request.channel.recv_with_timeout(3000);
                        }
                        catch (ChannelError e)
                        {
                            assert(e is ChannelError.TIMEOUT_EXPIRED);
                            throw new RPCError.GENERIC("UDP call: remote host not responding.");
                        }
                        Object o_ret = (Object)v_ret;
                        // Handling keepalives
                        if (o_ret == get_KEEP_ALIVE())
                        {
                            zcd.log_debug("UDPRequestClient got refreshed by a keepalive.");
                        }
                        else
                        {
                            ret = (ISerializable)o_ret;
                            break;
                        }
                    }
                    return ret;
                }
                return new SerializableNone();
            }
            catch (Error e)
            {
                zcd.log_debug("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));
                throw new RPCError.GENERIC(e.message);
            }
        }


        public void initialize() throws RPCError
        {
            try
            {
                string[] devs = get_devs();
                sockets = new BroadcastClientDatagramSocket[devs.length];
                for (int i = 0; i < devs.length; i++)
                {
                    string dev = devs[i];
                    sockets[i] = new BroadcastClientDatagramSocket(dev, dest_port);
                }
            }
            catch (Error e)
            {
                zcd.log_debug("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));
                throw new RPCError.GENERIC(e.message);
            }
        }

        public abstract string[] get_devs();
        public abstract UDPPayload get_udp_payload(RemoteCall data);
        public abstract bool is_unicast();

        public void rpc_send(uchar[] serdata) throws RPCError
        {
            try
            {
                foreach (BroadcastClientDatagramSocket soc in sockets)
                {
                    soc.send(data_pack(serdata));
                }
            }
            catch (Error e)
            {
                zcd.log_debug("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));
                throw new RPCError.GENERIC(e.message);
            }
        }

        ~UDPRequestClient()
        {
            if (sockets != null)
            {
                foreach (BroadcastClientDatagramSocket soc in sockets)
                {
                    soc.close();
                }
            }
        }
    }

    public class UDPResponseClient : Object
    {
        private int request_id;
        private BroadcastClientDatagramSocket[] sockets;

        public UDPResponseClient(Gee.List<string> devs, int request_id, uint16 port=269) throws RPCError
        {
            try
            {
                this.request_id = request_id;
                sockets = new BroadcastClientDatagramSocket[devs.size];
                int i = 0;
                foreach (string dev in devs)
                {
                    sockets[i] = new BroadcastClientDatagramSocket(dev, port);
                    i++;
                }
            }
            catch (Error e)
            {
                zcd.log_debug("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));
                throw new RPCError.GENERIC(e.message);
            }
        }

        public void send_keepalive() throws RPCError
        {
            try
            {
                UDPMessage udpmessage = UDPMessage.make_keep_alive(request_id);
                foreach (BroadcastClientDatagramSocket soc in sockets)
                {
                    soc.send(data_pack(udpmessage.serialize()));
                }
            }
            catch (Error e)
            {
                throw new RPCError.GENERIC(e.message);
            }
        }

        public void send_response(ISerializable response) throws RPCError
        {
            try
            {
                UDPMessage udpmessage = UDPMessage.make_response(request_id, response);
                foreach (BroadcastClientDatagramSocket soc in sockets)
                {
                    soc.send(data_pack(udpmessage.serialize()));
                }
            }
            catch (Error e)
            {
                zcd.log_debug("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));
                throw new RPCError.GENERIC(e.message);
            }
        }

        ~UDPResponseClient()
        {
            foreach (BroadcastClientDatagramSocket soc in sockets)
            {
                soc.close();
            }
        }
    }

    public class PendingRequest : Object
    {
        public Channel channel;
    }

    ArrayList<int> pending_requests_ids = null;
    HashMap<int, PendingRequest> pending_requests = null;

    void add_to_pending_requests(int request_id, PendingRequest pending_request) throws RPCError
    {
        if (pending_requests_ids == null) pending_requests_ids = new ArrayList<int>();
        if (pending_requests == null) pending_requests = new HashMap<int, PendingRequest>();
        purge_old_pending_requests();
        if (is_in_pending_requests(request_id))
            throw new RPCError.GENERIC("add_to_pending_requests: key duplicated.");
        pending_requests_ids.add(request_id);
        pending_requests[request_id] = pending_request;
    }

    void purge_old_pending_requests()
    {
        if (pending_requests_ids == null) return;
        // keep the number of pending_requests less than a maximum
        while (pending_requests_ids.size > 50)
        {
            int id_to_remove = pending_requests_ids.remove_at(0);
            pending_requests.unset(id_to_remove);
        }
    }

    bool is_in_pending_requests(int request_id)
    {
        if (pending_requests_ids == null) return false;
        return pending_requests.has_key(request_id);
    }

    PendingRequest? retrieve_pending_request(int request_id)
    {
        if (pending_requests_ids == null) return null;
        if (pending_requests.has_key(request_id))
            return pending_requests[request_id];
        return null;
    }

    public class ExecutedRequest : Object
    {
    }

    ArrayList<int> executed_requests_ids = null;
    HashMap<int, ExecutedRequest> executed_requests = null;

    void add_to_executed_requests(int request_id, ExecutedRequest executed_request) throws RPCError
    {
        if (executed_requests_ids == null) executed_requests_ids = new ArrayList<int>();
        if (executed_requests == null) executed_requests = new HashMap<int, ExecutedRequest>();
        purge_old_executed_requests();
        if (is_in_executed_requests(request_id))
            throw new RPCError.GENERIC("add_to_executed_requests: key duplicated.");
        executed_requests_ids.add(request_id);
        executed_requests[request_id] = executed_request;
    }

    void purge_old_executed_requests()
    {
        if (executed_requests_ids == null) return;
        // keep the number of executed_requests less than a maximum
        while (executed_requests_ids.size > 50)
        {
            int id_to_remove = executed_requests_ids.remove_at(0);
            executed_requests.unset(id_to_remove);
        }
    }

    bool is_in_executed_requests(int request_id)
    {
        if (executed_requests_ids == null) return false;
        return executed_requests.has_key(request_id);
    }

    ExecutedRequest? retrieve_executed_request(int request_id)
    {
        if (executed_requests_ids == null) return null;
        if (executed_requests.has_key(request_id))
            return executed_requests[request_id];
        return null;
    }
}

