/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011-2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Netsukuku;
using Tasklets;
using zcd;

namespace Netsukuku
{
#if log_tasklet
    private string tasklet_id()
    {
        return @"[$(Tasklet.self().id)] ";
    }
#else
    private string tasklet_id()
    {
        return "";
    }
#endif
    internal void log_debug(string msg) {Posix.syslog(Posix.LOG_DEBUG, tasklet_id()+msg);}
    internal void log_info(string msg) {Posix.syslog(Posix.LOG_INFO, tasklet_id()+msg);}
    internal void log_notice(string msg) {Posix.syslog(Posix.LOG_NOTICE, tasklet_id()+msg);}
    internal void log_warn(string msg) {Posix.syslog(Posix.LOG_WARNING, tasklet_id()+msg);}
    internal void log_error(string msg) {Posix.syslog(Posix.LOG_ERR, tasklet_id()+msg);}
    internal void log_critical(string msg) {Posix.syslog(Posix.LOG_CRIT, tasklet_id()+msg);}

    public ArrayList<int> valid_ids(int levels, int gsize, int lvl, PartialNIP partial_nip)
    {
        ArrayList<int> ret = new ArrayList<int>();
        for (int i = 0; i < gsize; i++) ret.add(i);
        return ret;
    }

    public class FakeAddrMgr : Object
    {
        public AddressManagerFakeRmt get_broadcast_client()
        {
            throw new RPCError.GENERIC("FakeAddrMgrget_broadcast_client");
        }
    }
    public class FakeAddrMgrRmt : Object
    {
        public PeerToPeerAll peer_to_peer_all;
        public IPeerToPeer get_peer_to_peer_service(int pid)
        {
            return null;
        }
        public IOptionalPeerToPeer get_optional_peer_to_peer_service(int pid)
        {
            return null;
        }
    }
    public class AggregatedNeighbour : Object
    {
        public NIP nip;
        public FakeAddrMgrRmt tcp_client;
        public FakeAddrMgrRmt neighbour_client;
        public static bool equal_func(AggregatedNeighbour a, AggregatedNeighbour b)
        {
            throw new RPCError.GENERIC("AggregatedNeighbourequal_func");
        }
        public string to_string()
        {
            throw new RPCError.GENERIC("AggregatedNeighbourto_string");
        }
    }
    public class AggregatedNeighbourManager : Object
    {
        public Gee.List<AggregatedNeighbour> neighbour_list(bool? b=null)
        {
            return new ArrayList<AggregatedNeighbour>();
        }
    }
    public class Route : Object
    {
        public AggregatedNeighbour gw;
    }
    public class RouteNode : DataClass
    {
        private bool _free;
        public override void initialize(Object m, int lvl, int pos, bool its_me = false) {
            _free = true;
            // I am {1,2,3}
            // me is participant
            if (lvl == 0 && pos == 1) _free = false;
            if (lvl == 1 && pos == 2) _free = false;
            if (lvl == 2 && pos == 3) _free = false;
            // {2,2,3} is participant
            if (lvl == 0 && pos == 2) _free = false;
            // {*,1,3} is participant
            if (lvl == 1 && pos == 1) _free = false;
            // {*,0,3} is not
            // {*,*,1} is participant
            if (lvl == 2 && pos == 1) _free = false;
        }
        public override bool is_free() {return _free;}
        public Gee.List<Route> all_valid_routes()
        {
            return new ArrayList<Route>();
        }
    }
    public class MapRoute : Map<RouteNode>
    {
        public MapRoute(NIP me)
        {
            base(3, 4, me);
        }
        public FakeAddrMgr address_manager;
    }

    public string nip_to_str(int levels, int gsize, NIP nip) {return "";}
}

namespace Ntk.Test
{
    string logger;
    const bool output = false;

    public class PeerToPeerTester : Object
    {
        public void set_up ()
        {
            logger = "";
        }

        public void tear_down ()
        {
            logger = "";
        }

        public void test_search_participant()
        {
            AggregatedNeighbourManager anm = new AggregatedNeighbourManager();
            MapRoute maproute = new MapRoute(new NIP({1,2,3}));
            PeerToPeer ptp = new PeerToPeer(anm, maproute, 1);
            // I am {1,2,3} and me is participant
            // {2,2,3} is participant
            // {*,1,3} is participant
            // {*,*,1} is participant
            HCoord hc = ptp.search_participant(new NIP({2,2,1}));
            // {*,*,1} is (2,1) and participates itself
            if (output) stdout.printf(@"from 2,2,1 to $(hc)\n");
            assert(hc.is_equal(new HCoord(2, 1)));
            hc = ptp.search_participant(new NIP({1,1,2}));
            // {*,*,2} is (2,2) the nearest participant (going in direction +1) is {*,*,3}
            // which is myself, so look at next pos:
            // {*,1} is (1,1) and participates itself, but it's not myself, so stop here
            if (output) stdout.printf(@"from 1,1,2 to $(hc)\n");
            assert(hc.is_equal(new HCoord(1, 1)));
        }

        public void test_tpl()
        {
            PeerToPeerTracerPacketList x = new PeerToPeerTracerPacketList();
            // starts in node 1 of gnode 2 of ggnode 3
            MapRoute mr1 = new MapRoute(new NIP({1,2,3}));
            if (output) stdout.printf("executing mr1\n");
            PeerToPeer.execute(mr1, x);
            if (output) stdout.printf("done \n");
            // hops in ggnode 1
            MapRoute mr2 = new MapRoute(new NIP({2,3,1}));
            if (output) stdout.printf("executing mr2\n");
            PeerToPeer.execute(mr2, x);
            if (output) stdout.printf("done \n");
            // hops in ggnode 0
            MapRoute mr3 = new MapRoute(new NIP({0,0,0}));
            if (output) stdout.printf("executing mr3\n");
            PeerToPeer.execute(mr3, x);
            if (output) stdout.printf("done \n");
            // hops in ggnode 1 again (should detect loop and wait a bit)
            MapRoute mr4 = new MapRoute(new NIP({0,0,1}));
            if (output) stdout.printf("executing mr4 (should detect loop and wait a bit)\n");
            PeerToPeer.execute(mr4, x);
            if (output) stdout.printf("done \n");
        }

        public void test_map_peers()
        {
            MapPeerToPeer map = new MapPeerToPeer(3, 4, new NIP({1,2,3}), 2);
            // myself
            map.participate();
            assert(map.node_get(0, 1).participant);
            assert(map.node_get(1, 2).participant);
            assert(map.node_get(2, 3).participant);
            map.sit_out();
            assert(! map.node_get(0, 1).participant);
            assert(! map.node_get(1, 2).participant);
            assert(! map.node_get(2, 3).participant);
            // another node (a gnode of level 1)
            assert(! map.node_get(1, 1).participant);
            map.node_get(1, 1).participant = true;
            assert(map.node_get(1, 1).participant);
            map.node_get(1, 1).participant = false;
            assert(! map.node_get(1, 1).participant);
        }

        public void test_list_ids()
        {
            AggregatedNeighbourManager anm = new AggregatedNeighbourManager();
            MapRoute maproute = new MapRoute(new NIP({1,2,3})); // levels 3 gsize 4
            PeerToPeer ptp = new PeerToPeer(anm, maproute, 1);
            Gee.List<int> ret;
            // center 1, sign +1
            ret = ptp.list_ids(1, 1);
            assert(ret.remove_at(0) == 1);
            assert(ret.remove_at(0) == 2);
            assert(ret.remove_at(0) == 3);
            assert(ret.remove_at(0) == 0);
            assert(ret.is_empty);
            // center 2, sign -1
            ret = ptp.list_ids(2, -1);
            assert(ret.remove_at(0) == 2);
            assert(ret.remove_at(0) == 1);
            assert(ret.remove_at(0) == 0);
            assert(ret.remove_at(0) == 3);
            assert(ret.is_empty);
            // center 0, sign -1
            ret = ptp.list_ids(0, -1);
            assert(ret.remove_at(0) == 0);
            assert(ret.remove_at(0) == 3);
            assert(ret.remove_at(0) == 2);
            assert(ret.remove_at(0) == 1);
            assert(ret.is_empty);
        }

        public static int main(string[] args)
        {
            GLib.Test.init(ref args);
            Tasklet.init();
            GLib.Test.add_func ("/PeerToPeer/SearchParticipant", () => {
                var x = new PeerToPeerTester();
                x.set_up();
                x.test_search_participant();
                x.tear_down();
            });
            GLib.Test.add_func ("/PeerToPeer/TracerPacketList", () => {
                var x = new PeerToPeerTester();
                x.set_up();
                x.test_tpl();
                x.tear_down();
            });
            GLib.Test.add_func ("/PeerToPeer/MapPeers", () => {
                var x = new PeerToPeerTester();
                x.set_up();
                x.test_map_peers();
                x.tear_down();
            });
            GLib.Test.add_func ("/PeerToPeer/ListIDs", () => {
                var x = new PeerToPeerTester();
                x.set_up();
                x.test_list_ids();
                x.tear_down();
            });
            GLib.Test.run();
            Tasklet.kill();
            return 0;
        }
    }
}
