/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011-2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

/*   peerbuilder.in
errors:
 HookingError
  INEXISTENT_GNODE
  GENERIC
 QspnError
  NOT_YOUR_GATEWAY
  ALREADY_UP_TO_DATE
  GENERIC
 PeerRefuseServiceError
  GENERIC
 TunnelError
  GENERIC
 BorderNodesError
  WRONG_GNODE
  NOT_BORDER_NODE
  WILL_NOT_TUNNEL
  TIMEOUT
  GENERIC
 AndnaError
  GENERIC
serializables:
 TimeCapsule
 ParticipantNode
 PackedParticipantNodes
 OptionalServiceParticipants
 SetOptionalServiceParticipants
 PeerToPeerTracerPacketList
 PairNipDistance
 BnodeRecord
 BnodeList
 PairLvlNumberOfFreeNodes
 HCoord
 PartialNIP
 NIP
 REM
 NullREM
 DeadREM
 AlmostDeadREM
 RTT
 TracerPacketList
 RouteInSet
 PositionInRoutesSetPerLevel
 RoutesSetPerLevel
 RoutesSet
 ExtendedTracerPacket
 GNodeID
 NetworkID
 InfoNeighbour
 InfoRoute
 InfoNode
 QspnStats
 InfoBorderNode
 DHTKey
 DHTRecord
 PublicKey
 AndnaServiceKey
 AndnaServerRecord
 AndnaDomainRecord
 AndnaServer
 AndnaServers
 RegisterHostnameArguments
 CounterNipRecord
 CounterSetDataResponse
 CounterCheckHostnameResponse
 CounterGetCacheRecordsResponse
 AndnaConfirmPubkResponse
 AndnaRegisterMainResponse
 AndnaRegisterSpreadResponse
 AndnaGetServersResponse
 AndnaGetRegistrarResponse
 AndnaGetCacheRecordsResponse
 BroadcastID
 UnicastID
peers:
 Counter
  methods:
   CounterSetDataResponse set_data_for_pubk
    arguments:
     NIP nip
     Gee.List<string> hashed_domains
     PublicKey pubkey
     SerializableBuffer signature
     bool replicate
    throws:
     PeerRefuseServiceError
   CounterCheckHostnameResponse check_hostname
    arguments:
     NIP nip
     string hashed_domain
     PublicKey pubkey
    throws:
     PeerRefuseServiceError
   CounterNipRecord? get_hostnames
    arguments:
     NIP nip
    throws:
     PeerRefuseServiceError
   CounterGetCacheRecordsResponse get_cache_records
    arguments:
    throws:
     PeerRefuseServiceError
*/

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    public interface ICounterAsPeer : Object
    {
        public abstract  CounterSetDataResponse
                         set_data_for_pubk
                         (NIP nip,
                          Gee.List<string> hashed_domains,
                          PublicKey pubkey,
                          SerializableBuffer signature,
                          bool replicate=true)
                         throws RPCError, PeerRefuseServiceError;
        public abstract  CounterCheckHostnameResponse
                         check_hostname
                         (NIP nip,
                          string hashed_domain,
                          PublicKey pubkey)
                         throws RPCError, PeerRefuseServiceError;
        public abstract  CounterNipRecord?
                         get_hostnames
                         (NIP nip)
                         throws RPCError, PeerRefuseServiceError;
        public abstract  CounterGetCacheRecordsResponse
                         get_cache_records ()
                         throws RPCError, PeerRefuseServiceError;
    }

    public class RmtCounterPeer : RmtPeer, ICounterAsPeer
    {
        public RmtCounterPeer
               (PeerToPeer peer_to_peer_service,
                Object? key=null,
                NIP? hIP=null,
                AggregatedNeighbour? aggregated_neighbour=null)
        {
            base(peer_to_peer_service, key, hIP, aggregated_neighbour);
        }

        public  CounterSetDataResponse
                set_data_for_pubk
                (NIP nip,
                 Gee.List<string> hashed_domains,
                 PublicKey pubkey,
                 SerializableBuffer signature,
                 bool replicate)
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "set_data_for_pubk";
            rc.add_parameter(nip);
            ListString _hashed_domains = new ListString.with_backer(hashed_domains);
            rc.add_parameter(_hashed_domains);
            rc.add_parameter(pubkey);
            rc.add_parameter(signature);
            rc.add_parameter(new SerializableBool(replicate));
            try {
                return (CounterSetDataResponse)
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

        public  CounterCheckHostnameResponse
                check_hostname
                (NIP nip,
                 string hashed_domain,
                 PublicKey pubkey)
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "check_hostname";
            rc.add_parameter(nip);
            rc.add_parameter(new SerializableString(hashed_domain));
            rc.add_parameter(pubkey);
            try {
                return (CounterCheckHostnameResponse)
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

        public  CounterNipRecord?
                get_hostnames
                (NIP nip)
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "get_hostnames";
            rc.add_parameter(nip);
            try {
                ISerializable ret =
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
                if (ret.get_type().is_a(typeof(SerializableNone))) return null;
                else return (CounterNipRecord)ret;
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

        public  CounterGetCacheRecordsResponse
                get_cache_records ()
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "get_cache_records";
            try {
                return (CounterGetCacheRecordsResponse)
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

    }

    class ArgumentsForDuplicationSetData : Object
    {
        public NIP nip;
        public Gee.List<string> hashed_domains;
        public PublicKey pubkey;
        public SerializableBuffer signature;
        public NIP hashnode;
    }

    struct struct_helper_Counter_reset_my_counter_node
    {
        public Counter self;
    }

    public class Counter : PeerToPeer, ICounterAsPeer
    {
        public static const int mypid = 2;

        // The loaded configuration of this node
        private ArrayList<string> _hashednames;

        // get the configuration of this node
        private ArrayList<string> hashednames {
            get {
                if (_hashednames == null)
                {
                    _hashednames = new ArrayList<string>();
                    char buffer[256];
                    string? myhostname = null;
                    if (Posix.gethostname(buffer) == 0)
                    {
                        StringBuilder sb = new StringBuilder();
                        int pos = 0;
                        while (buffer[pos] != '\0') sb.append_c(buffer[pos++]);
                        sb.append_c('\0');
                        myhostname = sb.str;
                    }
                    if (myhostname != null) _hashednames.add(Andna.crypto_hash(myhostname));
                }
                return _hashednames;
            }
        }

        private PeerToPeerAll peer_to_peer_all;
        public bool memory_initialized;
        private Tasklet? reset_my_counter_node_tasklet;
        public Andna andna {private get; set;}
        private KeyPair my_keys;
        private PublicKey pubk;
        private HashMap<PairPublicKeyNIP, CounterNipRecord> cache;

        public signal void counter_ready();
        public signal void counter_registered();

        public Counter(KeyPair keypair, AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            base(aggregated_neighbour_manager, maproute, Counter.mypid);
            this.peer_to_peer_all = peer_to_peer_all;
            my_keys = keypair;
            pubk = my_keys.pub_key.to_pubkey();
            reset_my_counter_node_tasklet = null;

            // let's register ourself in peer_to_peer_all
            try
            {
                peer_to_peer_all.peer_to_peer_register(this);
            }
            catch (PeerToPeerError e)
            {
                error(@"Counter: registering failed: $(e.message)");
            }

            // Until I have my memory initialized I don't want to answer
            //  to registration/resolution requests.
            memory_initialized = false;
            // Start the init_memory phase of the peer-to-peer service when my map is valid.
            this.map_peer_to_peer_validated.connect(init_memory);

            // The key is the pubk
            //  and the value is the registered CounterNipRecord
            cache = new HashMap<PairPublicKeyNIP, CounterNipRecord>(
                             PairPublicKeyNIP.hash_func, PairPublicKeyNIP.equal_func);
        }

        public RmtCounterPeer
               peer
               (NIP? hIP=null,
                Object? key=null,
                AggregatedNeighbour? aggregated_neighbour=null)
        {
            assert(hIP != null || key != null);
            return new RmtCounterPeer(this, key, hIP, aggregated_neighbour);
        }

        /** This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public override ISerializable _dispatch(Object? caller, RemoteCall data) throws Error
        {
            string[] pieces = data.method_name.split(".");
            if (pieces[0] == "set_data_for_pubk")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "set_data_for_pubk is a function.");
                if (data.parameters.size != 5)
                    throw new RPCError.MALFORMED_PACKET(
                        "set_data_for_pubk wants 5 parameters.");
                ISerializable iser0 = data.parameters[0];
                if (! iser0.get_type().is_a(typeof(NIP)))
                    throw new RPCError.MALFORMED_PACKET(
                        "set_data_for_pubk parameter 1 is not a NIP.");
                NIP nip = (NIP)iser0;
                ISerializable iser1 = data.parameters[1];
                if (! iser1.get_type().is_a(typeof(ListString)))
                    throw new RPCError.MALFORMED_PACKET(
                        "set_data_for_pubk parameter 2 is not a List<string>.");
                ListString _hashed_domains = (ListString)iser1;
                Gee.List<string> hashed_domains = _hashed_domains.backed;
                ISerializable iser2 = data.parameters[2];
                if (! iser2.get_type().is_a(typeof(PublicKey)))
                    throw new RPCError.MALFORMED_PACKET(
                        "set_data_for_pubk parameter 3 is not a PublicKey.");
                PublicKey pubkey = (PublicKey)iser2;
                ISerializable iser3 = data.parameters[3];
                if (! iser3.get_type().is_a(typeof(SerializableBuffer)))
                    throw new RPCError.MALFORMED_PACKET(
                        "set_data_for_pubk parameter 4 is not a SerializableBuffer.");
                SerializableBuffer signature = (SerializableBuffer)iser3;
                ISerializable iser4 = data.parameters[4];
                if (! iser4.get_type().is_a(typeof(SerializableBool)))
                    throw new RPCError.MALFORMED_PACKET(
                        "set_data_for_pubk parameter 5 is not a bool.");
                bool replicate = ((SerializableBool)iser4).b;
                return set_data_for_pubk(nip, hashed_domains, pubkey, signature, replicate);
            }
            if (pieces[0] == "check_hostname")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "check_hostname is a function.");
                if (data.parameters.size != 3)
                    throw new RPCError.MALFORMED_PACKET(
                        "check_hostname wants 3 parameters.");
                ISerializable iser0 = data.parameters[0];
                if (! iser0.get_type().is_a(typeof(NIP)))
                    throw new RPCError.MALFORMED_PACKET(
                        "check_hostname parameter 1 is not a NIP.");
                NIP nip = (NIP)iser0;
                ISerializable iser1 = data.parameters[1];
                if (! iser1.get_type().is_a(typeof(SerializableString)))
                    throw new RPCError.MALFORMED_PACKET(
                        "check_hostname parameter 2 is not a string.");
                string hashed_domain = ((SerializableString)iser1).s;
                ISerializable iser2 = data.parameters[2];
                if (! iser2.get_type().is_a(typeof(PublicKey)))
                    throw new RPCError.MALFORMED_PACKET(
                        "check_hostname parameter 3 is not a PublicKey.");
                PublicKey pubkey = (PublicKey)iser2;
                return check_hostname(nip, hashed_domain, pubkey);
            }
            if (pieces[0] == "get_hostnames")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_hostnames is a function.");
                if (data.parameters.size != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_hostnames wants 1 parameter.");
                ISerializable iser0 = data.parameters[0];
                if (! iser0.get_type().is_a(typeof(NIP)))
                    throw new RPCError.MALFORMED_PACKET(
                        "get_hostnames parameter 1 is not a NIP.");
                NIP nip = (NIP)iser0;
                CounterNipRecord? ret = get_hostnames(nip);
                if (ret == null) return new SerializableNone();
                return ret;
            }
            if (pieces[0] == "get_cache_records")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_cache_records is a function.");
                if (data.parameters.size != 0)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_cache_records wants no parameters.");
                return get_cache_records();
            }
            return base._dispatch(caller, data);
        }

        private NIP nip_for_lvl_pos(int lvl, int pos)
        {
            int[] ret = maproute.me.get_positions();
            ret[lvl] = pos;
            return new NIP(ret);
        }

        private void impl_init_memory() throws Error
        {
            Tasklet.declare_self("Counter.init_memory");

            // clear old caches
            cache.clear();

            for (int lvl = 0; lvl < maproute.levels; lvl++)
            {
                int? first_forward;
                int? first_back;
                int? last_back;
                find_hook_peers(out first_forward,
                                out first_back,
                                out last_back,
                                lvl, COUNTER_DUPLICATION);
                if (first_forward == null)
                {
                    // no one in my gnode lvl+1 (except me)
                    // I need to go up one level.
                }
                else if (first_back == null)
                {
                    // my gnode lvl+1 has some participants but not
                    // enough to satisfy our replica COUNTER_DUPLICATION.
                    // So, get all.
                    NIP nip_first_forward = nip_for_lvl_pos(lvl, first_forward);
                    RmtCounterPeer peer_first_forward = peer(nip_first_forward);
                    // use peer_first_forward.get_cache_records() to
                    //  obtain records and save them all to my caches
                    CounterGetCacheRecordsResponse cache_first_forward =
                            peer_first_forward.get_cache_records();
                    foreach (PairPublicKeyNIP pk_nip in cache_first_forward.cache.keys)
                    {
                        if (! cache.has_key(pk_nip))
                        {
                            cache[pk_nip] = cache_first_forward.cache[pk_nip];
                        }
                    }
                    // and then no need to go up one level.
                    break;
                }
                else
                {
                    // my gnode lvl+1 has enough participant
                    if (first_back != last_back)
                    {
                        // (lvl, first_back) is a (g)node with some participants but not
                        // enough to satisfy our replica COUNTER_DUPLICATION.
                        // (lvl, last_back) + ... + (lvl, first_back) has enough
                        // participant, though.
                        // So I have to get a record from first_back IFF it is not in last_back.
                        NIP nip_first_back = nip_for_lvl_pos(lvl, first_back);
                        RmtCounterPeer peer_first_back = peer(nip_first_back);
                        NIP nip_last_back = nip_for_lvl_pos(lvl, last_back);
                        RmtCounterPeer peer_last_back = peer(nip_last_back);
                        // use peer_first_back.get_cache_records() to
                        //  obtain records from first and peer_last_back.get_cache_records() to
                        //  obtain records from last, then use the correct equal_func to
                        //  choose the ones to store in my caches
                        CounterGetCacheRecordsResponse recs_first =
                            peer_first_back.get_cache_records();
                        CounterGetCacheRecordsResponse recs_last =
                            peer_last_back.get_cache_records();
                        foreach (PairPublicKeyNIP pk_nip in recs_first.cache.keys)
                        {
                            if (! cache.has_key(pk_nip))
                            {
                                bool found = false;
                                foreach (PairPublicKeyNIP pk_nip2 in recs_last.cache.keys)
                                {
                                    if (PairPublicKeyNIP.equal_func(pk_nip2, pk_nip))
                                    {
                                        found = true;
                                        break;
                                    }
                                }
                                if (!found)
                                {
                                    cache[pk_nip] =
                                        recs_first.cache[pk_nip];
                                }
                            }
                        }
                    }
                    // Now the records that are in first forward and whose
                    // key is before me at this level.
                    NIP nip_first_forward = nip_for_lvl_pos(lvl, first_forward);
                    RmtCounterPeer peer_first_forward = peer(nip_first_forward);
                    // use peer_first_forward.get_cache_records() to
                    //  obtain records and foreach rec which is not in my cache:
                    //   use function h to compute the hashnode of rec.key;
                    //   store in hk the position at level lvl;
                    //   IF my position at level lvl comes before the
                    //      position of first_forward in list_ids(hk, 1):
                    //         save rec into my caches;
                    CounterGetCacheRecordsResponse cache_first_forward =
                            peer_first_forward.get_cache_records();
                    foreach (PairPublicKeyNIP pk_nip in cache_first_forward.cache.keys)
                    {
                        if (! cache.has_key(pk_nip))
                        {
                            bool check = false;
                            int hk = h(pk_nip.nip)
                                    .position_at(lvl);
                            Gee.List<int> ids = list_ids(hk, 1);
                            if (ids.index_of(first_forward) > 
                                ids.index_of(maproute.me.position_at(lvl)))
                            {
                                check = true;
                            }
                            if (check)
                            {
                                cache[pk_nip] =
                                    cache_first_forward.cache[pk_nip];
                            }
                        }
                    }
                    // Then no need to go up one level.
                    break;
                }
            }

            // Now I can answer to requests.
            memory_initialized = true;
            log_info("Counter service: memory ready.");
            counter_ready();
        }

        public void init_memory()
        {
            Tasklet.tasklet_callback(
                    () => {
                        while (true)
                        {
                            try
                            {
                                impl_init_memory();
                                break;
                            }
                            catch (Error e)
                            {
                                log_warn("Counter.init_memory: " +
                                         @"got $(e.domain.to_string()) $(e.code) $(e.message). " +
                                         "Trying again.\n");
                                ms_wait(100);
                            }
                        }
                    });
        }

        /** This is the function h:KEY-->hIP.
          */
        public override NIP h(Object key)
        {
            // NOTE: Given a key, a value for levels and a value for gsize, this
            // function should compute the same NIP in any platform
            // and any architecture.
            NIP _key = (NIP)key;
            uint8[] hash = Crypto.md5(@"$(_key)".data);
            // hash is a 128 bits = 16 octets value
            // We should use it all for an IPv6 implementation
            //  but for the moment we assume an IPv4, so a basic
            //  uint32 variable and basic calculation is enough.
            //  In future a more complex algorithm will enable up to
            //  128 bit values to be correctly handled.
            uint32 basic_hash = (uint32)hash[15] +
                                (uint32)hash[14] * (uint32)256 +
                                (uint32)hash[13] * (uint32)256 * (uint32)256 +
                                (uint32)hash[12] * (uint32)256 * (uint32)256 * (uint32)256;
            int[] positions = new int[maproute.levels];
            for (int lvl = 0; lvl < maproute.levels; lvl++)
            {
                uint pos = basic_hash % maproute.gsize;
                positions[lvl] = (int)pos;
                basic_hash /= maproute.gsize;
            }
            NIP ret = new NIP(positions);
            return ret;
        }

        private void impl_reset_my_counter_node() throws Error
        {
            Tasklet.declare_self("Counter.reset_my_counter_node");
            reset_my_counter_node_tasklet = Tasklet.self();
            while (true)
            {
                TimeCapsule ttl;
                try
                {
                    CounterSetDataResponse resp = ask_set_data_for_pubk();
                    // if all ok
                    if (resp.response == "OK")
                    {
                        // emit signal.
                        counter_registered();
                        // How long to wait?
                        ttl = resp.expires;
                        // At most 10 minutes before expiration
                        ttl = new TimeCapsule(ttl.get_msec_ttl() -
                                                (int64)1000 *
                                                (int64)60 *
                                                (int64)10 /* 10 minutes in millisec */
                                              );
                        // but no more than MAX_WAIT_REFRESH_COUNTER
                        TimeCapsule maxttl = new TimeCapsule(MAX_WAIT_REFRESH_COUNTER);
                        if (ttl.is_younger(maxttl)) ttl = maxttl;
                    }
                    // if not ok
                    else ttl = new TimeCapsule(
                                                (int64)1000 *
                                                (int64)60 *
                                                (int64)2 /* 2 minutes in millisec */
                                          );
                }
                catch (Error e)
                {
                    // the failure may be temporary
                    ttl = new TimeCapsule(
                                                (int64)1000 *
                                                (int64)60 *
                                                (int64)2 /* 2 minutes in millisec */
                                          );
                }
                while (! ttl.is_expired()) Tasklet.nap(0, 100000);
            }
        }

        private static void * helper_reset_my_counter_node(void *v) throws Error
        {
            struct_helper_Counter_reset_my_counter_node *tuple_p = (struct_helper_Counter_reset_my_counter_node *)v;
            // The caller function has to add a reference to the ref-counted instances
            Counter self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_reset_my_counter_node();
            // void method, return null
            return null;
        }

        public void reset_my_counter_node()
        {
            struct_helper_Counter_reset_my_counter_node arg = struct_helper_Counter_reset_my_counter_node();
            arg.self = this;
            Tasklet.spawn((Spawnable)helper_reset_my_counter_node, &arg);
        }

        public void stop_reset_my_counter_node()
        {
            // Stop updating my counter node
            if (reset_my_counter_node_tasklet != null)
                reset_my_counter_node_tasklet.abort();
            reset_my_counter_node_tasklet = null;
        }

        public bool reset_my_counter_node_ongoing()
        {
            // Returns true if we are keeping updating
            return reset_my_counter_node_tasklet != null;
        }

        CounterSetDataResponse ask_set_data_for_pubk() throws RPCError, PeerRefuseServiceError
        {
            // Declare I am the holder of this nip. I declare all the hostnames I will try and register.
            SerializableBuffer signature = _sign(hashednames);
            log_debug("possible uncaught error: Counter client: calling set_data_for_pubk: in progress...");
            var ret = peer(null, maproute.me)
                    .set_data_for_pubk
                        (maproute.me,
                         hashednames,
                         pubk,
                         signature);
            log_debug("done.");
            return ret;
        }

/*
        uchar[] prepare_msg(NIP nip, Gee.List<string> hashed_domains)
        {
            uchar[] message1 = nip.hash_for_signature();
            uchar[] message2 = (new ListString.with_backer(hashed_domains)).hash_for_signature();
            uchar[] message = new uchar[message1.length + message2.length];
            int im = 0;
            for (int i = 0; i < message1.length; i++) message[im++] = message1[i];
            for (int i = 0; i < message2.length; i++) message[im++] = message2[i];
            return message;
        }
*/

        SerializableBuffer _sign(Gee.List<string> hashed_domains)
        {
            uchar[] signed = my_keys.sign(
                    CounterUtilities.prepare_msg(maproute.me,
                                                 hashed_domains));
            return new SerializableBuffer((uint8[])(signed));
        }

        bool _verify(NIP nip, Gee.List<string> hashed_domains, PublicKey pubk, SerializableBuffer signature)
        {
            PublicKeyWrapper pkw = new PublicKeyWrapper.from_pubk(pubk);
            return pkw.verify(
                    CounterUtilities.prepare_msg(nip,
                                                 hashed_domains),
                    (uchar[])(signature.buffer));
        }

        public CounterCheckHostnameResponse ask_check_hostname
               (NIP nip,
                string hashed_domain,
                PublicKey pubkey)
               throws RPCError, PeerRefuseServiceError
        {
            return peer(null, nip).check_hostname(nip,
                                                  hashed_domain,
                                                  pubkey);
        }

        public CounterNipRecord? ask_get_hostnames(NIP nip)
        {
            RmtCounterPeer node = peer(null, nip);
            log_debug("possible uncaught error: Counter client: calling get_hostnames: in progress...");
            CounterNipRecord? ret =
                    node.get_hostnames(nip);
            log_debug("done.");
            // TODO save local cache
            return ret;
        }

        /** Helper methods used as a server
          */
        public void check_expirations_cache()
        {
            // Remove the expired entries from the ANDNA cache
            ArrayList<PairPublicKeyNIP> todel =
                    new ArrayList<PairPublicKeyNIP>(
                            PairPublicKeyNIP.equal_func);
            foreach (PairPublicKeyNIP k in cache.keys)
            {
                CounterNipRecord record = cache[k];
                if (record.expires.is_expired())
                {
                    // this has expired
                    todel.add(k);
                }
            }
            foreach (PairPublicKeyNIP k in todel)
            {
                cache.unset(k);
            }
        }

        // This is an overridable method just to accomodate for a testsuite.
        public virtual
        IAddressManagerRootDispatcher
        contact_registrar(string registrar_address)
        {
            return new AddressManagerTCPClient(registrar_address);
        }

        /** Remotable methods as peer()
          */
        public CounterSetDataResponse set_data_for_pubk
               (NIP nip,
                Gee.List<string> hashed_domains,
                PublicKey pubkey,
                SerializableBuffer signature,
                bool replicate)
               throws PeerRefuseServiceError
        {
            if (!memory_initialized)
            {
                if (replicate)
                {
                    // This is an original request. My memory is not ready, so I cant
                    // answer. Anyway, I can wait a bit before throwing
                    // an error to see if I get ready soon. But the client is waiting
                    // for an answer, thus after a bit I want to throw an exception
                    // in order for the client to try again.
                    Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                    while (!memory_initialized)
                    {
                        if (w.is_expired())
                            throw new PeerRefuseServiceError.GENERIC("Memory not ready yet");
                        Tasklet.nap(0, 1000);
                    }
                }
                else
                {
                    // This is a replica. So the client is not waiting and will not
                    // request again. Thus I will wait till I get ready to memorize.
                    while (!memory_initialized) Tasklet.nap(0, 1000);
                }
            }

            string response_msg = "Noooo";
            TimeCapsule response_ttl = new TimeCapsule(0);
            check_expirations_cache();
            NIP hashnode = h(nip);
            ArgumentsForDuplicationSetData rec = new ArgumentsForDuplicationSetData();
            rec.nip = nip;
            rec.hashed_domains = hashed_domains;
            rec.pubkey = pubkey;
            rec.signature = signature;
            rec.hashnode = hashnode;
            // I verify that I am the hash node for the key.
            check_hash_and_start_replica(this, hashnode, replicate, rec, COUNTER_DUPLICATION,
                    /*AcceptRecordCallback*/
                    () => {
                        // check signature
                        if (! _verify(nip, hashed_domains, pubkey, signature))
                                throw new PeerRefuseServiceError.GENERIC(
                                          "Not your signature.");

                        // check MAX_HOSTNAMES
                        if (hashed_domains.size > MAX_HOSTNAMES)
                                throw new PeerRefuseServiceError.GENERIC(
                                          "Too many names.");

                        // contact the registrar
                        string dest_addr = nip_to_str(maproute.levels, maproute.gsize, nip);
                        IAddressManagerRootDispatcher client = 
                                contact_registrar(dest_addr);
                        int to_be_signed = Random.int_range(0, 64000);
                        log_debug("possible uncaught error: Counter: contacting the registrar: in progress...");
                        AndnaConfirmPubkResponse resp = client.andna
                                .confirm_pubk(nip, pubkey, to_be_signed);
                        log_debug("done.");
                        if (resp.response == "OK")
                        {
                            PublicKeyWrapper pk = new PublicKeyWrapper.from_pubk(pubkey);
                            pk.verify((uchar[])(@"$(to_be_signed)".data),
                                      (uchar[])resp.signature.buffer);
                        }
                        else
                        {
                            throw new PeerRefuseServiceError.GENERIC(
                                          "Not your NIP.");
                        }

                        // If the record is in cache is going to be replaced.
                        // NOTE: in the cache I can put distinct records based
                        //  on pubkey+nip. It could be licit to have one keypair
                        //  for more than one node. But not to have more than one
                        //  keypair for a node.
                        //  Anyway, for simplicity, since at the moment I don't
                        //  see particular value in having one keypair on several
                        //  nodes, I assume that both the NIP alone or the PublicKey
                        //  alone can be used as valid unique ID.
                        PairPublicKeyNIP? to_del = null;
                        foreach (PairPublicKeyNIP pk_nip in cache.keys)
                        {
                            if (PublicKey.equal_func(pk_nip.pk, pubkey))
                            {
                                to_del = pk_nip;
                                break;
                            }
                        }
                        if (to_del != null) cache.unset(to_del);
                        var ttl = new TimeCapsule(MAX_TTL_COUNTER);
                        cache[new PairPublicKeyNIP(pubkey, nip)] =
                                new CounterNipRecord(
                                        pubkey,
                                        nip,
                                        new ListString.with_backer(hashed_domains),
                                        ttl);

                        response_msg = "OK";
                        response_ttl = ttl;
                        return true;
                    },
                    /*ForwardRecordCallback*/
                    (tasklet_obj1, tasklet_replica_nodes) => {
                        ArgumentsForDuplicationSetData tasklet_rec =
                                (ArgumentsForDuplicationSetData)tasklet_obj1;
                        // Here I am in a tasklet (the client has been served already)
                        if (tasklet_replica_nodes == null)
                        {
                            log_debug("possible uncaught error: Counter: finding replicas for set_data_for_pubk: in progress...");
                            tasklet_replica_nodes =
                                        find_nearest_to_register
                                        (tasklet_rec.hashnode, COUNTER_DUPLICATION);
                            log_debug("done.");
                        }
                        // For each node of the <n> nearest except myself...
                        foreach (NIP replica_node in tasklet_replica_nodes) if (!replica_node.is_equal(maproute.me))
                        {
                            // ... in another tasklet...
                            Tasklet.tasklet_callback(
                                (tpar1, tpar2) => {
                                    NIP tonip = (NIP)tpar1;
                                    ArgumentsForDuplicationSetData arec =
                                            (ArgumentsForDuplicationSetData)tpar2;
                                    Tasklet.declare_self("Counter.forward_record");
                                    // ... forward the record to the node.
                                    try
                                    {
                                        peer(tonip).set_data_for_pubk
                                                (arec.nip,
                                                 arec.hashed_domains,
                                                 arec.pubkey,
                                                 arec.signature,
                                                 false);
                                    }
                                    catch (RPCError e)
                                    {
                                        // report the error with some info on where it happened
                                        log_warn(@"Counter.forward_record: forwarding to $(tonip):"
                                            + @" got $(e.domain.to_string()) $(e.code) $(e.message)");
                                    }
                                },
                                replica_node,
                                tasklet_rec);
                        }
                    });

            return new CounterSetDataResponse(response_msg, response_ttl);
        }

        public CounterCheckHostnameResponse check_hostname
               (NIP nip,
                string hashed_domain,
                PublicKey pubkey)
               throws PeerRefuseServiceError
        {
            if (!memory_initialized)
            {
                // My memory is not ready, so I cant answer.
                // The client is waiting for a reply.
                // After a bit I will throw an exception.
                Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                while (!memory_initialized)
                {
                    if (w.is_expired())
                        throw new PeerRefuseServiceError.GENERIC("Memory not ready yet");
                    Tasklet.nap(0, 1000);
                }
            }

            check_expirations_cache();
            // First, I check if I am the exact best participant for the hashnode
            //  for the key of this request.
            NIP hashnode = h(nip);
            if (search_participant(hashnode) != null)
            {
                // Not me.
                throw new PeerRefuseServiceError.GENERIC("Not the correct hashnode");
            }

            bool response = false;
            TimeCapsule ttl = new TimeCapsule(0);
            PairPublicKeyNIP? found = null;
            foreach (PairPublicKeyNIP pk_nip in cache.keys)
            {
                if (PartialNIP.equal_func(pk_nip.nip, nip))
                {
                    found = pk_nip;
                    break;
                }
            }
            if (found != null)
            {
                response = true;
                ttl = cache[found].expires;
            }
            return new CounterCheckHostnameResponse(response, ttl);
        }

        public CounterNipRecord? get_hostnames
               (NIP nip)
               throws PeerRefuseServiceError
        {
            if (!memory_initialized)
            {
                // My memory is not ready, so I cant answer.
                // The client is waiting for a reply.
                // After a bit I will throw an exception.
                Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                while (!memory_initialized)
                {
                    if (w.is_expired())
                        throw new PeerRefuseServiceError.GENERIC("Memory not ready yet");
                    Tasklet.nap(0, 1000);
                }
            }

            check_expirations_cache();
            // First, I check if I am the exact best participant for the hashnode
            //  for the key of this request.
            NIP hashnode = h(nip);
            if (search_participant(hashnode) != null)
            {
                // Not me.
                throw new PeerRefuseServiceError.GENERIC("Not the correct hashnode");
            }

            PairPublicKeyNIP? found = null;
            foreach (PairPublicKeyNIP pk_nip in cache.keys)
            {
                if (PartialNIP.equal_func(pk_nip.nip, nip))
                {
                    found = pk_nip;
                    break;
                }
            }
            if (found != null) return cache[found];
            return null;
        }

        public CounterGetCacheRecordsResponse get_cache_records ()
               throws PeerRefuseServiceError
        {
            if (!memory_initialized)
            {
                // My memory is not ready, so I cant answer.
                // The client is waiting for a reply.
                // After a bit I will throw an exception.
                Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                while (!memory_initialized)
                {
                    if (w.is_expired())
                        throw new PeerRefuseServiceError.GENERIC("Memory not ready yet");
                    Tasklet.nap(0, 1000);
                }
            }

            return new CounterGetCacheRecordsResponse(cache);
        }
    }
}

