#!/bin/bash

# This file is part of mpd-hits.
# Copyright (C) 2010 Dmitry Samoyloff.
#
# mpd-hits is free software: you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# mpd-hits is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along
# with mpd-hits. If not, see <http://www.gnu.org/licenses/>.

PACKAGE_NAME=mpd-hits
SCRIPT_DIR=$(cd `dirname $0` && pwd)
VERSION=`cd $SCRIPT_DIR && cat VERSION`
OUTPUT_DIR=/tmp

print_usage() {
    echo "Usage: `basename $0` [-h | --help] TARGET"
    echo
    echo "TARGET is one of:"
    echo "  debian"
    echo "  gentoo"
    echo "  tarball"
    echo
}

not_implemented() {
    echo "Support for $1 is not implemented yet."
}

check_for() {
    if ! which $1 >/dev/null; then
        echo "Command \"$1\" not found."
        exit 1
    fi
}

create_tarball() {
    TARBALL_DIR_REL=$PACKAGE_NAME-$VERSION
    TARBALL_NAME=$PACKAGE_NAME-${VERSION}.tar.bz2

    cd $OUTPUT_DIR

    # Prepare project directory for distribution
    cp -a $SCRIPT_DIR $TARBALL_DIR_REL
    find $TARBALL_DIR_REL -name "CVS" | xargs rm -rf
    find $TARBALL_DIR_REL -name ".*" | xargs rm -rf
    find $TARBALL_DIR_REL -name "*~" | xargs rm -rf
    rm -rf $TARBALL_DIR_REL/debian/mpd-hits/ $TARBALL_DIR_REL/build-stamp
    make -C $TARBALL_DIR_REL clean >/dev/null

    echo "Tarball contents:"
    echo "-----------------"
    tar cvjf $TARBALL_NAME $TARBALL_DIR_REL
    echo

    rm -rf $TARBALL_DIR_REL # Remove temporary stuff
}

build_debian() {
    check_for dpkg-buildpackage

    DEB_DIR=$OUTPUT_DIR/mpd-hits-deb

    # Re-create build directory, removing a previous stuff.
    rm -rf $DEB_DIR
    mkdir -p $DEB_DIR

    # Prepare package directory.
    create_tarball
    cd $DEB_DIR
    mv $OUTPUT_DIR/$TARBALL_NAME .
    tar xjf $TARBALL_NAME
    cd $TARBALL_DIR_REL

    # Build package.
    echo "Building deb package:"
    echo "---------------------"
    dpkg-buildpackage -rfakeroot
    echo

    # Cleanup.
    cd ..
    rm -rf $TARBALL_DIR_REL
    rm -f $TARBALL_NAME

    echo "Saved files:"
    echo "------------"
    echo "$DEB_DIR/*"
    echo
}

build_gentoo() {
    create_tarball
    cp -f "$SCRIPT_DIR/gentoo/mpd-hits.ebuild" \
        "$OUTPUT_DIR/mpd-hits-${VERSION}.ebuild"

    echo "Saved files:"
    echo "------------"
    echo "$OUTPUT_DIR/$TARBALL_NAME"
    echo "$OUTPUT_DIR/mpd-hits-${VERSION}.ebuild"
}

build_tarball() {
    create_tarball

    echo "Saved files:"
    echo "------------"
    echo "$OUTPUT_DIR/$TARBALL_NAME"
}

if [ $# -ne 1 ] && [ $# -ne 2 ]; then
    print_usage
    exit 1
fi

if [ "$1" = "-h" ] || [ "$1" = "--help" ]; then
    print_usage
    exit 0
fi

case "$1" in
    debian | \
    gentoo | \
    tarball)
        FUNC_POSTFIX=`echo $1 | sed 's/-/_/'`
        eval build_$FUNC_POSTFIX
        ;;
    *)
        print_usage
        exit 1
        ;;
esac

exit 0
