// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "item.h"

namespace mobius
{
namespace model
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Check if attribute exists
//! \param id attribute ID
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
item::has_attribute (const std::string& id) const
{
  auto db = get_database ();

  auto stmt = db.new_statement (
                "SELECT * "
                  "FROM attribute "
                 "WHERE item_uid = ? "
                   "AND id = ?");

  stmt.bind (1, get_uid ());
  stmt.bind (2, id);

  return bool (stmt.fetch_row ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get attribute value
//! \param id attribute ID
//! \return attribute value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
item::get_attribute (const std::string& id) const
{
  auto db = get_database ();

  auto stmt = db.new_statement (
                "SELECT value "
                  "FROM attribute "
                 "WHERE item_uid = ? "
                   "AND id = ?");

  stmt.bind (1, get_uid ());
  stmt.bind (2, id);

  std::string value;

  if (stmt.fetch_row ())
    value = stmt.get_column_string (0);

  return value;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set attribute value
//! \param id attribute ID
//! \param value attribute value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::set_attribute (const std::string& id, const std::string& value)
{
  auto db = get_database ();
  mobius::database::statement stmt;

  if (has_attribute (id))
    {
      stmt = db.new_statement (
               "UPDATE attribute "
                  "SET value = ? "
                "WHERE item_uid = ? "
                  "AND id = ?");

      stmt.bind (1, value);
      stmt.bind (2, get_uid ());
      stmt.bind (3, id);
    }

  else
    {
      stmt = db.new_statement (
               "INSERT INTO attribute "
                    "VALUES (NULL, ?, ?, ?)");

      stmt.bind (1, get_uid ());
      stmt.bind (2, id);
      stmt.bind (3, value);
    }

  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove attribute
//! \param id attribute ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::remove_attribute (const std::string& id)
{
  auto db = get_database ();

  auto stmt = db.new_statement (
                "DELETE FROM attribute "
                      "WHERE item_uid = ? "
                        "AND id = ?");

  stmt.bind (1, get_uid ());
  stmt.bind (2, id);
  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get attributes
//! \return map containing attributes' IDs and values
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::map <std::string, std::string>
item::get_attributes () const
{
  auto db = get_database ();

  auto stmt = db.new_statement (
                "SELECT id, value "
                  "FROM attribute "
                 "WHERE item_uid = ?");

  stmt.bind (1, get_uid ());

  std::map <std::string, std::string> attributes;

  while (stmt.fetch_row ())
    {
      auto id = stmt.get_column_string (0);
      auto value = stmt.get_column_string (1);
      attributes[id] = value;
    }

  return attributes;
}

} // namespace model
} // namespace mobius
