#ifndef MOBIUS_FILESYSTEM_TSK_ENTRY_IMPL_H
#define MOBIUS_FILESYSTEM_TSK_ENTRY_IMPL_H

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/filesystem/entry_impl_base.h>
#include <tsk/libtsk.h>
#include <vector>
#include <memory>

namespace mobius
{
namespace filesystem
{
namespace tsk
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief entry_impl class
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class entry_impl : public entry_impl_base
{
public:
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // function prototypes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  explicit entry_impl (TSK_FS_FILE *);
  std::vector <std::shared_ptr <entry_impl_base>> get_children () const override;
  std::shared_ptr <entry_impl_base> get_parent () const override;
  std::shared_ptr <entry_impl_base> get_child_by_name (const std::string&, bool) const override;
  std::vector <std::shared_ptr <stream_impl_base>> get_streams () const override;
  mobius::io::reader new_reader () const override;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get i-node
  //! \return i-node
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint64_t
  get_inode () const override
  {
    if (fs_file_->name)
      _load_fs_name ();

    else
      _load_fs_meta ();

    return inode_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get deleted flag
  //! \return deleted flag
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  bool
  is_deleted () const override
  {
    _load_fs_name ();
    return is_deleted_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get name
  //! \return name
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::string
  get_name () const override
  {
    _load_fs_name ();
    return name_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get short name
  //! \return short name
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::string
  get_short_name () const override
  {
    _load_fs_name ();
    return short_name_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get type
  //! \return type
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  entry_type
  get_type () const override
  {
    if (fs_file_->name)
      _load_fs_name ();

    else
      _load_fs_meta ();

    return type_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // ! \brief get size in bytes
  //! \return size in bytes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint64_t
  get_size () const override
  {
    _load_fs_meta ();
    return size_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get access mode flags
  //! \return access mode flags
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  int
  get_mode () const override
  {
    _load_fs_meta ();
    return mode_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get reallocated flag
  //! \return reallocated flag
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  bool
  is_reallocated () const override
  {
    _load_fs_meta ();
    return is_reallocated_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get owner user id
  //! \return owner user id
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  int
  get_uid () const override
  {
    _load_fs_meta ();
    return uid_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get owner group id
  //! \return owner group id
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  int
  get_gid () const override
  {
    _load_fs_meta ();
    return gid_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get creation date/time
  //! \return creation date/time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::datetime::datetime
  get_creation_time () const override
  {
    _load_fs_meta ();
    return creation_time_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get last modification date/time
  //! \return last modification date/time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::datetime::datetime
  get_last_modification_time () const override
  {
    _load_fs_meta ();
    return last_modification_time_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get last access date/time
  //! \return last access date/time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::datetime::datetime
  get_last_access_time () const override
  {
    _load_fs_meta ();
    return last_access_time_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get last metadata modification date/time
  //! \return last metadata modification date/time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::datetime::datetime
  get_last_metadata_time () const override
  {
    _load_fs_meta ();
    return last_metadata_time_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get delection date/time
  //! \return delection date/time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::datetime::datetime
  get_deletion_time () const override
  {
    _load_fs_meta ();
    return deletion_time_;
  }

private:
  using fs_file_type = std::unique_ptr <TSK_FS_FILE, void(*)(TSK_FS_FILE *)>;
  using fs_dir_type = std::unique_ptr <TSK_FS_DIR, void(*)(TSK_FS_DIR *)>;

  //! \brief i-node
  mutable std::uint64_t inode_ = 0;

  //! \brief deleted flag
  mutable bool is_deleted_ = false;

  //! \brief name
  mutable std::string name_;

  //! \brief short name
  mutable std::string short_name_;

  //! \brief type
  mutable entry_type type_ = entry_type::unknown;

  //! \brief size in bytes
  mutable std::uint64_t size_ = 0;

  //! \brief access mode flags
  mutable int mode_ = 0;

  //! \brief reallocated flag
  mutable bool is_reallocated_ = false;

  //! \brief owner user id
  mutable int uid_ = -1;

  //! \brief owner group id
  mutable int gid_ = -1;

  //! \brief creation date/time
  mutable mobius::datetime::datetime creation_time_;

  //! \brief last modification date/time
  mutable mobius::datetime::datetime last_modification_time_;

  //! \brief last access date/time
  mutable mobius::datetime::datetime last_access_time_;

  //! \brief last metadata modification date/time
  mutable mobius::datetime::datetime last_metadata_time_;

  //! \brief delection date/time
  mutable mobius::datetime::datetime deletion_time_;

  //! \brief libtsk file structure pointer
  fs_file_type fs_file_;

  //! \brief libtsk dir structure pointer (cache)
  mutable fs_dir_type fs_dir_;

  //! \brief fs_name loaded flag
  mutable bool fs_name_loaded_ = false;

  //! \brief fs_meta loaded flag
  mutable bool fs_meta_loaded_ = false;

  //! \brief streams loaded flag
  mutable bool streams_loaded_ = false;

  // helper functions
  void _load_fs_name () const;
  void _load_fs_meta () const;
  void _load_fs_dir () const;
};

} // namespace tsk
} // namespace filesystem
} // namespace mobius

#endif
