# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import mobius
import pymobius
import pymobius.util
import pymobius.app.chrome
import sqlite3
import os.path

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Load data from History file
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def decode (path):

  # create data object
  data = pymobius.Data ()
  data.history = []
  data.downloads = []

  # get meta version
  db = sqlite3.connect (path)
  data.version = pymobius.app.chrome.get_meta_version (db)

  if data.version:
    mobius.core.log ('app.chrome: History.version = %s' % data.version) 

    decode_history (db, data)
    decode_downloads (db, data)

  db.close ()

  # return data object
  return data

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Load URL history
# Tested versions: 22, 32, 33, 39, 41
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def decode_history (db, data):
  cursor = db.cursor ()

  stmt = '''
       SELECT v.id,
              u.url,
              u.title,
              v.visit_time
         FROM urls u, visits v
        WHERE v.url = u.id
     ORDER BY v.visit_time'''

  for row in cursor.execute (stmt):
    entry = pymobius.Data ()
    entry.id = row[0]
    entry.url = row[1]
    entry.title = row[2]
    entry.timestamp = pymobius.util.get_nt_datetime (row[3])

    data.history.append (entry)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Load downloads
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def decode_downloads (db, data):
  if data.version <= 22:
    decode_downloads_22 (db, data)

  else:
    decode_downloads_32 (db, data)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Load downloads
# Tested versions: 22
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def decode_downloads_22 (db, data):
  cursor = db.cursor ()

  stmt = '''
       SELECT id,
              full_path,
              url,
              full_path,
              start_time,
              end_time,
              state,
              received_bytes,
              total_bytes
         FROM downloads'''

  for row in cursor.execute (stmt):
    entry = pymobius.Data ()
    entry.id = row[0]
    entry.name = os.path.basename (row[1].replace ('\\', '/'))
    entry.source = row[2]
    entry.target = row[3]
    entry.start_time = pymobius.util.get_nt_datetime (row[4])
    entry.end_time = pymobius.util.get_nt_datetime (row[5])
    entry.state = row[6]
    entry.bytes_downloaded = row[7]
    entry.size = row[8]
    entry.referrer = None
    entry.mimetype = None
      
    data.downloads.append (entry)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Load downloads
# Tested versions: 32, 33, 39, 41
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def decode_downloads_32 (db, data):
  cursor = db.cursor ()

  stmt = '''
       SELECT d.id,
              d.target_path,
              u.url,
              d.target_path,
              d.start_time,
              d.end_time,
              d.state,
              d.received_bytes,
              d.total_bytes,
              d.referrer,
              d.mime_type
         FROM downloads d, downloads_url_chains u
        WHERE d.id = u.id'''

  for row in cursor.execute (stmt):
    entry = pymobius.Data ()
    entry.id = len (data.downloads) + 1
    entry.name = os.path.basename (row[1].replace ('\\', '/'))
    entry.source = row[2]
    entry.target = row[3]
    entry.start_time = pymobius.util.get_nt_datetime (row[4])
    entry.end_time = pymobius.util.get_nt_datetime (row[5])
    entry.state = row[6]
    entry.bytes_downloaded = row[7]
    entry.size = row[8]
    entry.referrer = row[9]
    entry.mimetype = row[10]
      
    data.downloads.append (entry)
