// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \file item.cc C++ API <i>mobius.model.item</i> class wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "item.h"
#include "case.h"
#include "module.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create <i>item</i> Python object from C++ object
//! \param obj C++ object
//! \return new item object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
PyMobius_Model_Item_from_cpp (mobius::model::item obj)
{
  PyObject *ret = nullptr;

  if (obj)
    {
      ret = _PyObject_New (&model_item_t);

      if (ret)
        ((model_item_o *) ret)->obj = new mobius::model::item (obj);
    }
  else
    {
      Py_INCREF (Py_None);
      ret = Py_None;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>uid</i> attribute getter
//! \param self object
//! \return <i>uid</i> attribute
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_uid (model_item_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyLong_FromLongLong (self->obj->get_uid ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>category</i> attribute getter
//! \param self object
//! \return <i>category</i> attribute
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_category (model_item_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_FromString (self->obj->get_category ().c_str ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>case</i> attribute getter
//! \param self object
//! \return <i>case</i> attribute
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_case (model_item_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Model_Case_from_cpp (self->obj->get_case ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef tp_getset[] =
{
  {
    (char *) "uid",
    (getter) tp_getter_uid,
    (setter) 0,
    (char *) "Unique ID",
    NULL
  },
  {
    (char *) "category",
    (getter) tp_getter_category,
    (setter) 0,
    (char *) "category",
    NULL
  },
  {
    (char *) "case",
    (getter) tp_getter_case,
    (setter) 0,
    (char *) "case object",
    NULL
  },
  {NULL, NULL, NULL, NULL, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get_child_count</i> method implementation
//! \param self object
//! \param args argument list
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_child_count (model_item_o *self, PyObject *)
{
  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyInt_FromLong (self->obj->get_child_count ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get_children</i> method implementation
//! \param self object
//! \param args argument list
//! \return children
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_children (model_item_o *self, PyObject *)
{
  PyObject *ret = PyList_New (0);

  if (!ret)
    return nullptr;

  try
    {
      for (const auto& item : self->obj->get_children ())
        {
          PyObject *py_item = PyMobius_Model_Item_from_cpp (item);

          if (!py_item)
            {
              Py_CLEAR (ret);
              return nullptr;
            }

          PyList_Append (ret, py_item);
          Py_DECREF (py_item);
        }
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      Py_CLEAR (ret);
      ret = nullptr;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get_parent</i> method implementation
//! \param self object
//! \param args argument list
//! \return parent, if exists
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_parent (model_item_o *self, PyObject *)
{
  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Model_Item_from_cpp (self->obj->get_parent ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>new_child</i> method implementation
//! \param self object
//! \param args argument list
//! \return new item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_new_child (model_item_o *self, PyObject *args)
{
  // parse input args
  const char * arg_category;
  int arg_idx = -1;

  if (!PyArg_ParseTuple (args, "s|i", &arg_category, &arg_idx))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Model_Item_from_cpp (self->obj->new_child (arg_category, arg_idx));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>remove</i> method implementation
//! \param self object
//! \param args argument list
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_remove (model_item_o *self, PyObject *)
{
  // execute C++ function
  try
    {
      self->obj->remove ();
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>move</i> method implementation
//! \param self object
//! \param args argument list
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_move (model_item_o *self, PyObject *args)
{
  // parse input args
  int arg_idx;
  PyObject *arg_parent;

  if (!PyArg_ParseTuple (args, "iO", &arg_idx, &arg_parent))
    return nullptr;

  // execute C++ function
  try
    {
      self->obj->move (arg_idx, *(((model_item_o *) arg_parent)->obj));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>has_attribute</i> method implementation
//! \param self object
//! \param args argument list
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_has_attribute (model_item_o *self, PyObject *args)
{
  // parse input args
  const char * arg_id;

  if (!PyArg_ParseTuple (args, "s", &arg_id))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyBool_FromLong (self->obj->has_attribute (arg_id));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get_attribute</i> method implementation
//! \param self object
//! \param args argument list
//! \return attribute value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_attribute (model_item_o *self, PyObject *args)
{
  // parse input args
  const char * arg_id;

  if (!PyArg_ParseTuple (args, "s", &arg_id))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_FromString (self->obj->get_attribute (arg_id).c_str ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>set_attribute</i> method implementation
//! \param self object
//! \param args argument list
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_set_attribute (model_item_o *self, PyObject *args)
{
  // parse input args
  const char * arg_id;
  const char * arg_value;

  if (!PyArg_ParseTuple (args, "ss", &arg_id, &arg_value))
    return nullptr;

  // execute C++ function
  try
    {
      self->obj->set_attribute (arg_id, arg_value);
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>remove_attribute</i> method implementation
//! \param self object
//! \param args argument list
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_remove_attribute (model_item_o *self, PyObject *args)
{
  // parse input args
  const char * arg_id;

  if (!PyArg_ParseTuple (args, "s", &arg_id))
    return nullptr;

  // execute C++ function
  try
    {
      self->obj->remove_attribute (arg_id);
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get_attributes</i> method implementation
//! \param self object
//! \param args argument list
//! \return map containing attributes' IDs and values
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_attributes (model_item_o *self, PyObject *)
{
  PyObject *ret = PyDict_New ();

  if (!ret)
    return nullptr;

  try
    {
      for (const auto& p : self->obj->get_attributes ())
        {
          PyObject *py_key = PyString_FromString (p.first.c_str ());

          if (!py_key)
            {
              Py_CLEAR (ret);
              return nullptr;
            }

          PyObject *py_value = PyString_FromString (p.second.c_str ());

          if (!py_value)
            {
              Py_DECREF (py_key);
              Py_CLEAR (ret);
              return nullptr;
            }

          PyDict_SetItem (ret, py_key, py_value);
          Py_DECREF (py_key);
          Py_DECREF (py_value);
        }
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      Py_CLEAR (ret);
      ret = nullptr;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>expand_masks</i> method implementation
//! \param self object
//! \param args argument list
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_expand_masks (model_item_o *self, PyObject *)
{
  // execute C++ function
  try
    {
      self->obj->expand_masks ();
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get_data_path</i> method implementation
//! \param self object
//! \param args argument list
//! \return full path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_data_path (model_item_o *self, PyObject *args)
{
  // parse input args
  const char * arg_rpath;

  if (!PyArg_ParseTuple (args, "s", &arg_rpath))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_FromString (self->obj->get_data_path (arg_rpath).c_str ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>create_data_path</i> method implementation
//! \param self object
//! \param args argument list
//! \return full path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_create_data_path (model_item_o *self, PyObject *args)
{
  // parse input args
  const char * arg_rpath;

  if (!PyArg_ParseTuple (args, "s", &arg_rpath))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_FromString (self->obj->create_data_path (arg_rpath).c_str ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef tp_methods[] =
{
  {
    (char *) "get_child_count",
    (PyCFunction) tp_f_get_child_count,
    METH_VARARGS,
    "Get number of child items"
  },
  {
    (char *) "get_children",
    (PyCFunction) tp_f_get_children,
    METH_VARARGS,
    "Get children items"
  },
  {
    (char *) "get_parent",
    (PyCFunction) tp_f_get_parent,
    METH_VARARGS,
    "Get parent item"
  },
  {
    (char *) "new_child",
    (PyCFunction) tp_f_new_child,
    METH_VARARGS,
    "Create new child item"
  },
  {
    (char *) "remove",
    (PyCFunction) tp_f_remove,
    METH_VARARGS,
    "Remove item"
  },
  {
    (char *) "move",
    (PyCFunction) tp_f_move,
    METH_VARARGS,
    "Move item"
  },
  {
    (char *) "has_attribute",
    (PyCFunction) tp_f_has_attribute,
    METH_VARARGS,
    "Check if attribute exists"
  },
  {
    (char *) "get_attribute",
    (PyCFunction) tp_f_get_attribute,
    METH_VARARGS,
    "Get attribute value"
  },
  {
    (char *) "set_attribute",
    (PyCFunction) tp_f_set_attribute,
    METH_VARARGS,
    "Set attribute value"
  },
  {
    (char *) "remove_attribute",
    (PyCFunction) tp_f_remove_attribute,
    METH_VARARGS,
    "Remove attribute"
  },
  {
    (char *) "get_attributes",
    (PyCFunction) tp_f_get_attributes,
    METH_VARARGS,
    "Get attributes"
  },
  {
    (char *) "expand_masks",
    (PyCFunction) tp_f_expand_masks,
    METH_VARARGS,
    "Expand value masks using item's attributes"
  },
  {
    (char *) "get_data_path",
    (PyCFunction) tp_f_get_data_path,
    METH_VARARGS,
    "get item data path"
  },
  {
    (char *) "create_data_path",
    (PyCFunction) tp_f_create_data_path,
    METH_VARARGS,
    "create item data path"
  },
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>item</i> constructor
//! \param type type object
//! \param args argument list
//! \param kwds keywords dict
//! \return new <i>item</i> object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_new (PyTypeObject *type, PyObject *, PyObject *)
{
  model_item_o *self = (model_item_o *) type->tp_alloc (type, 0);

  if (self)
    self->obj = new mobius::model::item ();

  return (PyObject *) self;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>item</i> deallocator
//! \param self object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
tp_dealloc (model_item_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject model_item_t =
{
  PyVarObject_HEAD_INIT (NULL, 0)
  "mobius.model.item",                     		// tp_name
  sizeof (model_item_o),                   		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) tp_dealloc,                 		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "item class",                            		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  tp_methods,                              		// tp_methods
  0,                                       		// tp_members
  tp_getset,                               		// tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  tp_new,                                  		// tp_new
  0,                                       		// tp_free
  0,                                       		// tp_is_gc
  0,                                       		// tp_bases
  0,                                       		// tp_mro
  0,                                       		// tp_cache
  0,                                       		// tp_subclasses
  0,                                       		// tp_weaklist
  0,                                       		// tp_del
  0,                                       		// tp_version_tag
};
