// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/data_decoder.h>
#include <mobius/datetime/conv_nt_timestamp.h>
#include <mobius/charset.h>
#include <cstdio>

namespace mobius
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param in reader object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
data_decoder::data_decoder (mobius::io::reader in)
 : in_ (in)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief skip n bytes
//! \param size size in bytes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_decoder::skip (size_type size)
{
  in_.skip (size);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief jump to position pos
//! \param pos position from the start of data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_decoder::seek (size_type pos)
{
  in_.seek (pos);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get current position
//! \return position from the start of data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
data_decoder::size_type
data_decoder::tell () const
{
  return in_.tell ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an int32 (le)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int32_t
data_decoder::get_int32_le ()
{
  return int (get_uint32_le ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint16 (le)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint16_t
data_decoder::get_uint16_le ()
{
  bytearray data = in_.read (2);

  return data[0] | (std::uint16_t (data[1]) << 8);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint16 (be)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint16_t
data_decoder::get_uint16_be ()
{
  bytearray data = in_.read (2);

  return data[1] | (std::uint16_t (data[0]) << 8);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint32 (le)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint32_t
data_decoder::get_uint32_le ()
{
  bytearray data = in_.read (4);

  return data[0] | (std::uint32_t (data[1]) << 8) |
                   (std::uint32_t (data[2]) << 16) |
                   (std::uint32_t (data[3]) << 24);  
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint32 (be)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint32_t
data_decoder::get_uint32_be ()
{
  bytearray data = in_.read (4);

  return data[3] | (std::uint32_t (data[2]) << 8) |
                   (std::uint32_t (data[1]) << 16) |
                   (std::uint32_t (data[0]) << 24);  
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint64 (le)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint64_t
data_decoder::get_uint64_le ()
{
  bytearray data = in_.read (8);

  return data[0] | (std::uint64_t (data[1]) << 8) |
                   (std::uint64_t (data[2]) << 16) |
                   (std::uint64_t (data[3]) << 24) |
                   (std::uint64_t (data[4]) << 32) |
                   (std::uint64_t (data[5]) << 40) |
                   (std::uint64_t (data[6]) << 48) |
                   (std::uint64_t (data[7]) << 56);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint64 (be)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint64_t
data_decoder::get_uint64_be ()
{
  bytearray data = in_.read (8);

  return data[7] | (std::uint64_t (data[6]) << 8) |
                   (std::uint64_t (data[5]) << 16) |
                   (std::uint64_t (data[4]) << 24) |
                   (std::uint64_t (data[3]) << 32) |
                   (std::uint64_t (data[2]) << 40) |
                   (std::uint64_t (data[1]) << 48) |
                   (std::uint64_t (data[0]) << 56);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode NT timestamp
//! \return datetime object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::datetime::datetime
data_decoder::get_nt_datetime ()
{
  return mobius::datetime::datetime_from_nt_timestamp (get_uint64_le ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode bytearray by size
//! \param size size in bytes
//! \return bytearray
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::bytearray
data_decoder::get_bytearray_by_size (std::size_t size)
{
  return in_.read (size);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode string by size
//! \param size size in bytes
//! \return string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
data_decoder::get_string_by_size (std::size_t size, const std::string& encoding)
{
  bytearray data = in_.read (size);
  std::string result;

  if (encoding == "ASCII" || encoding == "UTF-8")
    result = std::string (data.begin (), data.end ());

  else
    result = conv_charset_to_utf8 (data, encoding);

  auto pos = result.find ('\0');
  if (pos != std::string::npos)
    result.erase (pos);

  return result;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode GUID
//! \return GUID as formatted string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
data_decoder::get_guid ()
{
  uint32_t guid1 = get_uint32_le ();
  uint16_t guid2 = get_uint16_le ();
  uint16_t guid3 = get_uint16_le ();
  uint16_t guid4 = get_uint16_be ();
  bytearray guid5 = in_.read (6);

  char buffer[64];
  sprintf (buffer, "%08x-%04x-%04x-%04x-%02x%02x%02x%02x%02x%02x",
           guid1, guid2, guid3, guid4,
           guid5[0],
           guid5[1],
           guid5[2],
           guid5[3],
           guid5[4],
           guid5[5]);

  return buffer;
}

} // namespace mobius
