;;; upad - A program for debugging, and uploading code to embedded devices.
;;; Copyright (C) 2016 John Darrington

;;; This program is free software: you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation, either version 3 of the License, or
;;; (at your option) any later version.

;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.

;;; You should have received a copy of the GNU General Public License
;;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

	.include "s12z/flash.s"
	.include "s12z/cop.s"

	#define BIT_CCIF 7
	#define BIT_ACCERR 5
	#define BIT_FPVIOL 4
	#define BIT_MSGSTAT0 0
	#define BIT_MSGSTAT1 1

#define MASK(X) (0x1 << X)

	#define CMD_VERIFY_PFLASH 3
	#define CMD_PROGRAM_PFLASH 6

write_pflash:
;;; Write the "phrase" (ie. 8 bytes) pointed to by the X register
;;; into the location pointed to by the Y register.  The number of bytes
;;; to write should be placed into the D7 register.
;;; On completion D0 will contain zero if the operation was successfull.

	;; Clear the COP.
	;; We don't want to get reset in the middle of a write!
	clr.b  CPMU_COP

	tfr     y, d6

	;; Force Y (the source address) to be 8 byte aligned.
	and     d6, #0xFFFFF8

	;; Put the offset of Y (from its 8 byte alignment) into D2
	tfr     y,  d2
	and     d2, #0x0007

	neg.w  d2
	tfr     d2, d3

	;; Restore Y
	tfr     d6, y

	;; Copy the D2 bytes from the memory immediately before  (Y) into
	;; (X).
	;; These D2 bytes aren't part of the requested range, but we can't
	;; avoid flashing them.  Thus we must ensure they are flashed with
	;; their original values.
.l0:
	tbeq d3, .l1
	mov.b (d3, y), (d3, x)
	inc d3
	bra .l0
.l1:

	;; Decrement the source address by the offset
	lea x, (d2, x)

	add     d7, d6

	;; Set the clock
	mov.b   #5, FTMRZ_FCLKDIV

	;; Wait until any existing command is finished
	brclr.b FTMRZ_FSTAT, #BIT_CCIF, *+0

	;; Clear any error status
	mov.b   #(MASK(BIT_ACCERR) | MASK(BIT_FPVIOL)), FTMRZ_FSTAT

.loop:
	;; Set the number of words to be loaded in FCCOB
	mov.b   #5, FTMRZ_FCCOBIX

	;; Set the command id
	mov.b   #CMD_PROGRAM_PFLASH, FTMRZ_FCCOB

	;; Set the destination address
	st      y,  FTMRZ_FCCOB+1

	mov.l   (0, x), FTMRZ_FCCOB+4
	mov.l   (4, x), FTMRZ_FCCOB+8

	;; Pull the trigger
	bset.b  FTMRZ_FSTAT, #BIT_CCIF

	;; Wait until the command finishes
	brclr.b FTMRZ_FSTAT, #BIT_CCIF, *+0

	ld d0, FTMRZ_FSTAT
	;; Consider only the FPVIOL and ACCERR bits.
	;; Normally the MSGSTATn bits will also indicate errors, however
	;; they can be set under circumstances which are not "errors". For
	;; example when writing to the security bits.
	and d0, #(MASK (BIT_ACCERR) | MASK(BIT_FPVIOL))

	bne err

	lea x, (8, x)
	lea y, (8, y)

	cmp y, d7

	blt .loop
;;; end loop

err:
	bgnd
