/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <config.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <getopt.h>
#include <errno.h>
#include <stdarg.h>
#include <termios.h>
#include <assert.h>

#include "uploader.h"
#include "user-interface.h"
#include "misc.h"
#include "sep.h"
#include "binary-uploader.h"
#include "elf-uploader.h"
#include "srecord-uploader.h"
#include "lib/md5.h"
#include "transaction.h"

static void
confidence_message (FILE *fp, const char *format, ...)
{
  if (!isatty (fileno (fp)))
    return;

  fputs ("\r", fp);
  va_list ap;
  va_start (ap, format);
  vfprintf (fp, format, ap);
  va_end (ap);
  //  fputs ("\n", fp);
  fflush (fp);
}

void
upad_msg (enum msglevel lvl, const char *fmt, ...)
{
  if (lvl > MSG_INFORMATION)
    return;
  va_list ap;
  va_start (ap, fmt);
  vprintf (fmt, ap);
  va_end (ap);
}

void
upad_errno (enum msglevel lvl, const char *fmt, ...)
{
#define BUFMAX 512
  char buf[BUFMAX];
  char *err = strerror (errno);
  va_list ap;
  va_start (ap, fmt);
  vsnprintf (buf, BUFMAX, fmt, ap);
  va_end (ap);
  upad_msg (lvl, "%s: %s\n", buf, err);
}

/* Search for a device whose name matches TEMPL.
   If found, returns a pointer to NAME.
   If no match is found, returns NULL. */
static char *
search_serial_device (const char *templ)
{
  int d;
  static char name[64];
  for (d = 0; d < 8; ++d)
    {
      struct stat sb;
      snprintf (name, 64, templ, d);
      if (0 != stat (name, &sb))
	{
          int err = errno;
          if (err != ENOENT && err != ENXIO)
	    {
              upad_msg (MSG_ERROR,
			   "%s exists but cannot be statted: %s\n",
			   name, strerror (err));
	    }
	}
      else if (sb.st_rdev != 0)
	{
          return name;
	}
    }
  return NULL;
}

static void
usage (const char *progname)
{
  fprintf (stderr, "Usage: %s [-p port] [-e] [-n] [file]\n", progname);
  exit (1);
}

/* Upload the format UF using the interface DES */
int
upload_format (int des, struct upload_format *uf, const struct bounce_code_usage *bcu)
{
  uint32_t address;
  uint8_t buffer[MAX_PAYLOAD_LENGTH * 2]; // FIXME: Buffer overflow ....
  size_t n_data;
  size_t total_data = 0;
  uint32_t start_address = -1;
  bool first = true;
  size_t data_offset =  1 + sizeof (struct write_mem_params);
  if (uf == NULL)
    return 1;
  while (uf->get_data_fragment (uf, &address, buffer + data_offset + total_data, MAX_PAYLOAD_LENGTH - 8, &n_data))
    {
      if (first)
	start_address = address;
      if (start_address + total_data != address || total_data + n_data > MAX_PAYLOAD_LENGTH - 8)
	{
	  confidence_message (stdout, "Start address: 0x%08x", start_address);
	  if (0 > write_fragment (des, start_address, buffer, total_data, bcu, first))
	    {
	      confidence_message (stdout, "\n");
	      upad_msg (MSG_ERROR, "Error programming target.\n");
	      return 1;
	    }
	  memcpy (buffer + data_offset,
		  buffer + data_offset + total_data, n_data);
	  start_address = address;
	  total_data = 0;
	}
      total_data += n_data;
      first = false;
    }
  confidence_message (stdout, "Start address: 0x%08x", start_address);
  if (0 > write_fragment (des, start_address, buffer, total_data, bcu, first))
    {
      confidence_message (stdout, "\n");
      upad_msg (MSG_ERROR, "Error programming target.\n");
      return 1;
    }
  confidence_message (stdout, "                                                \r");

  return post_final (des, bcu, start_address + total_data);
}




/* Attempt to open the serial port called PORT, initialise it according
   to our needs and return a file descriptor to the opened port.
   On error -1 is returned. */
static int
initialise_terminal (const char *port)
{
  int des = open (port, O_RDWR);
  if (des < 0)
    {
      char *emsg = strerror (errno);
      upad_msg (MSG_ERROR, "Cannot open serial port `%s': %s\n", port, emsg);
      return -1;
    }

  struct termios termios;
  if (0 != cfsetspeed (&termios, B115200))
    {
      upad_msg (MSG_ERROR, "Baud rate of 115200 seems not to be supported by this system\n");
    }
  cfmakeraw (&termios);
  if (0 != tcsetattr (des, TCSANOW, &termios))
    {
      perror ("Cannot set serial port attributes");
      return -1;
    }

  /* Discard any rubbish which might be in the buffer */
  tcflush (des, TCIOFLUSH);

  return des;
}

bool
process_command (const char *cmd, size_t cmd_length, int des)
{
  bool result = false;
  char *str = strdup (cmd);

  char *semicolon_r;
  char *stmt = strtok_r (str, ";", &semicolon_r);
  do
    {
      char *space_r;
      char *s = strtok_r (stmt, " ", &space_r);
      if (s == NULL)
        {
	  goto done;
        }
      result = false;
      int c;
      for (c = 0; c < N_COMMANDS; ++c)
        {
	  if (0 == strncmp (s, cmds[c].name, cmd_length))
            {
	      size_t arg_capacity = 4;
	      char **args = safe_realloc (NULL, sizeof (*args) * arg_capacity);
	      int arg = 0;
	      while ((args[arg] = strtok_r (NULL, " ", &space_r)))
                {
		  if (args[arg] == NULL)
		    break;

		  arg++;
		  if (arg >= arg_capacity)
                    {
		      arg_capacity *= 2;
		      args = safe_realloc (args, sizeof (*args) * arg_capacity);
                    }
                }
	      result = (0 == cmds[c].call (des, arg, args));
	      free (args);
            }
        }
    }
  while ((stmt = strtok_r (NULL, ";", &semicolon_r)));

 done:
  free (str);
  return result;
}

static int
run_script (int des, const char *script)
{
  int ret = 0;
  FILE *sfp = fopen (script, "r");
  if (sfp == NULL)
    {
      const char *se = strerror (errno);
      upad_msg (MSG_FATAL, "Cannot open file \"%s\": %s\n", script, se);
      return 1;
    }
  char *lineptr = NULL;
  size_t line_len = 0;
  while (!feof (sfp))
    {
      int n = getline (&lineptr, &line_len, sfp);
      if (n <= 0)
	continue;
      /* Discard the trailing delimiter */
      lineptr[n - 1] = '\0';
      if (!process_command (lineptr, n, des))
	{
	  upad_msg (MSG_ERROR, "Command \"%s\"failed\n", lineptr);
          ret = 1;
	  break;
	}
    }
  if (sfp)
    fclose (sfp);
  return ret;
}


struct user_options
{
  /* The port to which the controller is connected */
  const char *port;

  /* Set if the only operation to be performed is a bulk erase */
  bool erase_only;

  /* Set if the target should not be reset after programming */
  bool no_reset;

  /* The script file name to be run, or NULL */
  const char *script;

  /* The name of the file to be uploaded */
  const char *file;

  /* True if the file to be uploaded is a binary file (ie not S-record and not elf) */
  bool binary;

  /* The address where to load the file  (relevant for binary files only).  */
  uint32_t load_address;

  /* The name of the file which contains the bounce code */
  const char *bounce_code;
};

enum ftype
  {
    FTYPE_SREC,
    FTYPE_BINARY,
    FTYPE_ELF
  };


static bool
is_elf (FILE *fp)
{
  bool ret = false;
  uint8_t buf[4];
  int n = 0;
  while (!feof (fp))
    {
      n += fread (buf + n, 1, 4, fp);
      if (n >= 4)
	break;
    }

  if (buf[0] == 0x7F &&
      buf[1] == 0x45 &&
      buf[2] == 0x4c &&
      buf[3] == 0x46)
    {
      ret = true;
    }
  fseeko (fp, 0, SEEK_SET);
  return ret;
}


/* Attempts to autoidentify the format of the file associated with FP.
   FP must be a stream opened for reading and positioned at the start
   of the file.
   This function will always succeed.  If the file format cannot be recognised,
   then FTYPE_BINARY will be returned.
*/
enum ftype
identify_format (FILE *fp)
{
  char *lineptr = NULL;
  size_t size = 0;

  /* If we cannot recognise what it is, then default to BINARY */
  enum ftype ret = FTYPE_BINARY;

  /* Is it an elf file? */
  if (is_elf (fp))
    {
      ret = FTYPE_ELF;
      goto end;
    }

  /* A very short file can't be an S-record */
  if (getline (&lineptr, &size, fp) < 6)
    goto end;

  if (lineptr[0] != 'S')
    goto end;

  if (lineptr[1] < '0' || lineptr[1] > '9')
    goto end;

  ret = FTYPE_SREC;

 end:
  free (lineptr);
  fseeko (fp, 0, SEEK_SET);
  return ret;
}

static int
bulk_erase (int des)
{
  if (0 != execute_poll (des))
    return 1;

  if (0 != execute_reset_bdm (des))
    return 1;

  if (0 != execute_sync (des))
    return 1;

  if (0 != execute_bulk_erase (des))
    return 1;

  return 0;
}


static int
run_upload (int des, const struct user_options *opts)
{
  struct bounce_code_usage bcu;
  int ret = 0;

  FILE *fp = fopen (opts->file, "r");
  if (fp == NULL)
    {
      char *emsg = strerror (errno);
      upad_msg (MSG_ERROR, "Cannot open file `%s': %s\n", opts->file, emsg);
      return 1;
    }

  execute_poll (des);

  FILE *fpb = NULL;
  if (opts->bounce_code)
    {
      fpb = fopen (opts->bounce_code, "r");
      if (fpb == NULL)
	{
	  char *emsg = strerror (errno);
	  upad_msg (MSG_ERROR,
		    "Cannot open bounce code file `%s': %s. Falling back to direct programming.\n",
		    opts->bounce_code, emsg);
	}
      else
	{
	  fseek (fpb, 0, SEEK_END);
	  bcu.address = 0x1000;
	  bcu.size = ftell (fpb);
	  fseek (fpb, 0, SEEK_SET);
	  bcu.size_register = 7;    // D7
	  bcu.source_register = 8;  // X
	  bcu.dest_register = 9;    // Y
	  bcu.pc_register = 11;     // PC
	  bcu.err_register = 0;     // D0
	}
    }

  uint16_t csr = 1;
  do
    {
      execute_reset_bdm (des);
      execute_sync (des);
      if (0 != execute_bulk_erase (des))
	return 1;

      csr = execute_read_bdccsr (des);
    }
  while (csr & 0x0001);

  if (fpb)
    {
      struct upload_format *uf = create_binary_uploader (fpb, bcu.address);
      int ret = upload_format (des, uf, NULL);
      uf->destroy (uf);
      if (ret != 0)
	printf ("Upload of bounce code failed\n");
    }

  enum ftype ft = opts->binary ? FTYPE_BINARY : identify_format (fp);

  if (ft == FTYPE_BINARY && !opts->binary)
    {
      upad_msg (MSG_ERROR, "A binary format was detected, but no address for upload was specified.\n");
      return 1;
    }

  struct upload_format *uf = NULL;
  switch (ft)
    {
    default:
      uf = create_binary_uploader (fp, opts->load_address);
      break;
    case FTYPE_ELF:
      uf = create_elf_uploader (fp);
      break;
    case FTYPE_SREC:
      uf = create_srec_uploader (fp);
      break;
    }
  ret = upload_format (des, uf, fpb ? &bcu : NULL);
  uf->destroy (uf);
  fclose (fp);

  if (! opts->no_reset)
    execute_reset (des);

  return ret;
}

/* Parses the command line options and populates OPTS accordingly. */
static void
parse_options (int argc, char **argv, struct user_options *opts)
{
  bool no_bounce = false;
  #define FLAG_BOUNCE 1
  #define FLAG_NO_BOUNCE 2
  /*-
@menu
* @samp{--version} or @samp{-v}::
* @samp{--port} or @samp{-p}::
* @samp{--erase-only} or @samp{-e}::
* @samp{--no-reset} or @samp{-n}::
* @samp{--script} or @samp{-T}::
* @samp{--binary} or @samp{-b}::
* @samp{--bounce}::
* @samp{--no-bounce}::
@end menu
  */
  for (;;)
    {
      int long_flag;
      int option_index = 0;
      const struct option long_options[] = {
        /*-
          @node @samp{--version} or @samp{-v}
          @section @samp{--version} or @samp{-v}

          Displays the version of @upad{}.
         */
        {"version",     0,                 0,  'v' },
        /*-
          @node @samp{--port} or @samp{-p}
          @section @samp{--port} or @samp{-p}

          Specifies the serial port to use for communication between
          the local computer and the controller.
	  Normally is is not necessary to use this option because it
	  is automatically detected.
         */
        {"port",        required_argument, 0,  'p' },
        /*-
          @node @samp{--erase-only} or @samp{-e}
          @section @samp{--erase-only} or @samp{-e}

          This option species that the controller should perform a bulk
          erase of the target's non-volatile memory and not program
	  anything.
          If you specify this option, then you may not also specify a
	  filename.
        */
        {"erase-only",  0,                 0,  'e' },
        /*-
          @node @samp{--no-reset} or @samp{-n}
          @section @samp{--no-reset} or @samp{-n}

          By default, @upad{} will reset the target device immediately after
          the programming operation has completed.
	  This option requests that no reset should happen.
        */
        {"no-reset",    0,                 0,  'n' },
        /*-
          @node @samp{--script} or @samp{-T}
          @section @samp{--script} or @samp{-T}

          The @samp{--script} specifies that @upad{} should go into
	  scripting mode.
	  This option requires an argument which is the name of the
          script file to run.
        */
        {"script",      required_argument, 0,  'T' },
        /*-
          @node @samp{--binary} or @samp{-b}
          @section @samp{--binary} or @samp{-b}

          The file to be uploaded should be considered a raw binary file.
          Its contents should be uploaded verbatim.   This option requires
          an argument which is the address where to load the contents.
        */
        {"binary",      required_argument, 0,  'b' },
        /*-
          @node @samp{--bounce}
          @section @samp{--bounce}
	  @cindex bounce

          The @samp{--bounce} option specifies an alternative bounce code
          file to use, instead of the default one.   It requires an argument
          which is the name of the file containing the bounce code.  The file
          should be a raw binary file.  This option is mutually exclusive with
          the @samp{--no-bounce} option.
        */
	{"bounce",      required_argument, &long_flag, FLAG_BOUNCE },
        /*-
          @node @samp{--no-bounce}
          @section @samp{--no-bounce}

          The @samp{--no-bounce} option specifies that programming operations
          should not be bounced (@pxref{Bouncing}) through the target device's RAM.  Using this
          option will mean that programming will be somewhat slower.  This option
          may be useful if you suspect that the bounce algorithm is faulty.
        */
	{"no-bounce",   0,                 &long_flag, FLAG_NO_BOUNCE },
        {0,             0,                 0,   0  }
      };

      int c = getopt_long (argc, argv, "b:vp:enT:", long_options, &option_index);

      if (c == -1)
        break;

      switch (c)
	{
	case 'p':
          opts->port = optarg;
          break;
	case 'e':
          opts->erase_only = true;
          break;
	case 'b':
	  opts->binary = true;
	  opts->load_address = strtol (optarg, NULL, 0);
          break;
	case 'n':
          opts->no_reset = true;
          break;
	case 'v':
          printf ("%s - %s\n", PACKAGE_NAME, PACKAGE_VERSION);
          exit (0);
          break;
        case 'T':
          opts->script = optarg;
          break;
        case 0:
          switch (long_flag)
            {
            case FLAG_BOUNCE:
              if (no_bounce)
                {
                  fprintf (stderr, "--bounce and --no-bounce are mutually exclusive\n");
                  exit (1);
                }
              opts->bounce_code = optarg;
              break;
            case FLAG_NO_BOUNCE:
              if (opts->bounce_code)
                {
                  fprintf (stderr, "--bounce and --no-bounce are mutually exclusive\n");
                  exit (1);
                }
              no_bounce = true;
              break;
            default:
              assert (false);
              break;
            }
          break;
	default:
          usage (argv[0]);
          break;
	};
    }

  if (no_bounce)
    {
      opts->bounce_code = NULL;
    }
  else if (opts->bounce_code == NULL)
  {
      opts->bounce_code = TARGETDIR "/bounce.bin";
  }

  opts->file = (optind < argc) ? argv[optind] : NULL;

  if (opts->erase_only && opts->file)
    {
      fprintf (stderr,
	       "A filename should not be specified together with -e .\n");
      exit (1);
    }
}


int
main (int argc, char **argv)
{
  struct user_options opts = {0};
  parse_options (argc, argv, &opts);

  /* If a port wasn't specified in the command line, then
     search for a suitable default. */
  if (opts.port == NULL)
    {
      opts.port = search_serial_device ("/dev/ttyACM%d");
      if (opts.port == NULL)
        opts.port = search_serial_device ("/dev/ttyS%d");
      if (opts.port == NULL)
	{
          fprintf (stderr,
                   "No serial port was specified and no suitable candidate could be found\n");
          return 1;
	}
    }

  int des = initialise_terminal (opts.port);

  if (des < 0)
    return 1;

  int ret = 0;
  if (opts.script)
    ret = run_script (des, opts.script);
  else if (!opts.file && !opts.erase_only)
    ret = run_interactive (des);
  else if (! opts.erase_only)
    ret = run_upload (des, &opts);
  else
    ret = bulk_erase (des);

  if (des >= 0)
    close (des);

  return ret;
}
