/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016 John Darrington

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/
#include <config.h>

#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "periph/gpio.h"
#include "periph/uart.h"
#include "hashes/md5.h"
#include "board.h"
#include "sema.h"

#include "sep.h"
#include "commands.h"
#include "reset.h"
#include "bdc.h"
#include "math.h"

#include "upad.h"
#include "flash.h"
#include "target/s12z/cop.h"

#include "xtimer.h"

static sema_t byte_received;
static uint8_t byte;

#define PHRASE_LENGTH 8

enum status
  {
    STATE_IDLE,
    STATE_START0,
    STATE_START1,
    STATE_DATA,
    STATE_CHECKSUM0,
    STATE_CHECKSUM1
  };

static enum status state = STATE_IDLE;

static short flag = 0;
static void
cb (void *arg __attribute__ ((unused)), unsigned char x)
{
  byte = x;
  if (state == STATE_IDLE)
    {
      sema_post (&byte_received);
      flag = 0;
      return;
    }

  if (x == SEP_START)
    flag = !flag;
  else
    flag = 0;

  if (!flag)
    {
      sema_post (&byte_received);
    }
}



/* Insert BYTE into BUF + X, padding if necessary.
   Returns a new value which can be used as X in the
   next call to this function.
*/
static int
byte_insert (int x, uint8_t *buf, uint8_t byte)
{
  buf[x++] = byte;
  if (buf[x-1] == SEP_START)
    buf[x++] = SEP_START;

  return x;
}

static int
send_rep (int uart_dev, const uint8_t *payload, int n)
{
  uint8_t buf[MAX_PACKET_LENGTH] ;

  if (n > MAX_PAYLOAD_LENGTH)
    return -1;

  uint16_t crc = 0;

  buf[0] = SEP_START;
  int x = 1;

  /* length high byte */
  crc = crc16_update (crc, n >> 8);
  x = byte_insert (x, buf, n >> 8);

  /* length low byte */
  crc = crc16_update (crc, n);
  x = byte_insert (x, buf, n);

  for (int i = 0; i < n; ++i)
    {
      crc = crc16_update (crc, payload[i]);
      x = byte_insert (x, buf, payload[i]);
    }

  /* Set the CRC high byte */
  x = byte_insert (x, buf, crc >> 8);

  /* Set the CRC low byte */
  x = byte_insert (x, buf, crc & 0x00FF);

  uart_write  (uart_dev, buf, x);

  return 0;
}

/* Write the data in BUFFER to the NVM using PCMD as the algorithm */
bool
flash_buffer (const uint8_t *buffer, uint8_t pcmd, struct cache *cache)
{
  struct write_mem_params wmp;
  memcpy (&wmp, buffer + 1, sizeof (wmp));

  bool ok = true;
  uint32_t alignment_offset = wmp.address % PHRASE_LENGTH;
  wmp.address -= alignment_offset;
  wmp.quantity += alignment_offset;
  for (uint32_t a = wmp.address; a < wmp.address + wmp.quantity; a += PHRASE_LENGTH)
    {
      uint16_t words[5];
      int len = wmp.address + wmp.quantity - a;
      if (len > PHRASE_LENGTH)
	len = PHRASE_LENGTH;
      words[0] = a & 0xFFFF;
      words[1] = 0xFFFF; words[2] = 0xFFFF;
      words[3] = 0xFFFF; words[4] = 0xFFFF;
      bool blank = true;
      for (int i = 0; i < len; ++i)
	{
	  const uint32_t preamble_length = 1 + sizeof (wmp);
	  uint32_t idx = preamble_length + i + a - wmp.address - alignment_offset;
	  if (idx < preamble_length)
	    continue;
	  if (buffer[idx] != 0xFF)
	    blank = false;
	  if (0 == (i % 2))
	    {
	      words[1 + i/2] = (buffer[idx] << 8) | 0xFF;
	    }
	  else
	    {
	      words[1 + i/2] &= 0xFF00;
	      words[1 + i/2] |= buffer[idx];
	    }
	}
      if (!blank)
	{
	  if (pcmd == Z12FL_PROGRAM_PFLASH)
	    len = PHRASE_LENGTH;
	  ok = flash_command (pcmd,  a >> 16, 1 + len / 2, words, cache);
	}
      if (!ok)
	break;
    }
  return ok;
}

static const char *vers = PACKAGE_VERSION;


typedef void (*what_to_do) (int dev, uint8_t *buffer, struct cache *cache);

static void
reply_poll (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  buffer[0] = CMD_ACK;
  memcpy (buffer + 1, vers, strlen (vers));
  send_rep (uart_dev, (uint8_t *) buffer, 1 + strlen (vers));
}

static void
reply_echo (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  /* Echo the same data back.  Primarily useful for debug and test. */
  uint16_t len;
  memcpy (&len, buffer + 1, sizeof (len));
  buffer[2] = CMD_ACK;
  send_rep (uart_dev, (uint8_t *) buffer + 2, 1 + len);
}

static void
reply_bkdm (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  background_debug ();
  buffer[0] = CMD_ACK;
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}

static void
reply_reset (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  reset ();
  buffer[0] = CMD_ACK;
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}

static void
reply_sync (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  uint32_t sync_time;
  int result = bdc_sync (&sync_time);
  if (result == 0)
    {
      uint32_t period = round (sync_time / 8.0);
      bdc_calibrate (period);
      /* Disable the watchdog */
      uint8_t cop;
      bdc_read_byte ((uint32_t) &CPMU->COP, &cop);
      bdc_write_octet ((uint32_t) &CPMU->COP, cop | 0x40);

      buffer[0] = CMD_ACK;
      int r = htonl (sync_time);
      memcpy (buffer + 1, &r, sizeof (sync_time));
      send_rep (uart_dev, (uint8_t *) buffer, 1 + sizeof (r));
    }
  else
    {
      buffer[0] = CMD_NACK;
      send_rep (uart_dev, (uint8_t *) buffer, 1);
    }
}

static void
reply_bulk_erase (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  bdc_bulk_erase ();

  uint16_t csr = bdc_read_bdccsr ();
  buffer[0] = CMD_ACK;
  uint16_t r = htons (csr);
  memcpy (buffer + 1, &r, sizeof (r));
  flash_setup ();
  send_rep (uart_dev, (uint8_t *) buffer, 1 + sizeof (r));
}

static void
reply_eeprom (int uart_dev, uint8_t *buffer, struct cache *cache)
{
  bool ok = flash_buffer (buffer, Z12FL_PROGRAM_EEPROM, cache);
  buffer[0] = ok ? CMD_ACK : CMD_NACK;
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}

static void
reply_pflash (int uart_dev, uint8_t *buffer, struct cache *cache)
{
  bool ok = flash_buffer (buffer, Z12FL_PROGRAM_PFLASH, cache);
  buffer[0] = ok ? CMD_ACK : CMD_NACK;
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}

static void
reply_read_bdccsr (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  uint16_t csr = bdc_read_bdccsr ();
  buffer[0] = CMD_ACK;
  uint16_t r = htons (csr);
  memcpy (buffer + 1, &r, sizeof (r));
  send_rep (uart_dev, (uint8_t *) buffer, 1 + sizeof (r));
}

static void
reply_write_bdccsr (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  uint16_t csr;
  uint8_t *data = buffer + 1;
  memcpy (&csr, data, sizeof (csr));
  bdc_write_bdccsr (csr);
  buffer[0] = CMD_ACK;
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}

static void
reply_write_regs (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  int n_regs = buffer[1];
  uint8_t *data = buffer + 2;
  for (int i = 0; i < n_regs; ++i)
    {
      uint32_t value;
      uint8_t reg;
      memcpy (&reg, data, sizeof (reg));
      data++;
      memcpy (&value, data, sizeof (value));
      data += sizeof (value);
      bdc_write_register (reg, value);
    }
  buffer[0] = CMD_ACK;
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}


static void
reply_read_all_regs (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  buffer[0] = CMD_ACK;
  uint8_t *data = buffer + 1;
  for (int reg = 0; reg < 13; ++reg)
    {
      uint32_t rv = bdc_read_register (reg);
      memcpy (data, &rv, sizeof (rv));
      data += sizeof (rv);
    }
  send_rep (uart_dev, (uint8_t *) buffer, data - buffer);
}

static void
reply_read_register (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  buffer[0] = CMD_ACK;
  uint8_t reg = buffer[1];
  uint8_t *data = buffer + 1;
  uint32_t rv = bdc_read_register (reg);
  memcpy (data, &rv, sizeof (rv));
  data += sizeof (rv);
  send_rep (uart_dev, (uint8_t *) buffer, data - buffer);
}

static void
reply_erase_eeprom_sector (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  flash_setup ();
  uint32_t address;
  memcpy (&address, buffer + 1, sizeof (address));
  address &= 0x00FFFFFFE;

  uint16_t a = address;

  bool ok = flash_command (Z12FL_ERASE_EEPROM_SECTOR,
			   address >> 16, 1, &a,  cache);

  buffer[0] = ok ? CMD_ACK : CMD_NACK;
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}

static void
reply_erase_pflash_sector (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  flash_setup ();
  uint32_t address;
  memcpy (&address, buffer + 1, sizeof (address));
  address &= 0x00FFFFF8;

  uint16_t a = address;

  bool ok = flash_command (Z12FL_ERASE_PFLASH_SECTOR,
			   address >> 16, 1, &a,  cache);

  buffer[0] = ok ? CMD_ACK : CMD_NACK;
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}

static int
read_memory_range (uint32_t address, int quantity, uint8_t *data)
{
  int i = 0;
  while (quantity - i > 0)
    {
      if (quantity - i >= 4 && ((address + i) % 4 == 0))
	{
	  if (i == 0)
	    bdc_read_quad_byte (address + i, data + i);
	  else
	    bdc_dump_quad_byte (data + i);
	  i += 4;
	}
      else if (quantity - i  >= 2 && ((address + i) % 2 == 0))
	{
	  if (i == 0)
	    bdc_read_double_byte (address + i, data + i);
	  else
	    bdc_dump_double_byte (data + i);
	  i += 2;
	}
      else
	{
	  if (i == 0)
	    bdc_read_byte (address + i, data + i);
	  else
	    bdc_dump_byte (data + i);
	  i += 1;
	}
    }

  return 0;
}


static int
md5_memory_range (uint32_t address, int quantity, md5_ctx_t *ctx)
{
  uint8_t data[4];
  int i = 0;
  while (quantity - i > 0)
    {
      if (quantity - i >= 4 && ((address + i) % 4 == 0))
	{
	  if (i == 0)
	    bdc_read_quad_byte (address + i, data);
	  else
	    bdc_dump_quad_byte (data);

	  md5_update (ctx, data, 4);
	  i += 4;
	}
      else if (quantity - i  >= 2 && ((address + i) % 2 == 0))
	{
	  if (i == 0)
	    bdc_read_double_byte (address + i, data);
	  else
	    bdc_dump_double_byte (data);
	  md5_update (ctx, data, 2);
	  i += 2;
	}
      else
	{
	  if (i == 0)
	    bdc_read_byte (address + i, data);
	  else
	    bdc_dump_byte (data);

	  md5_update (ctx, data, 1);
	  i += 1;
	}
    }

  return 0;
}


static void
reply_md5_check (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  buffer[0] = CMD_ACK;
  uint32_t address;
  uint32_t quantity;
  memcpy (&address, buffer + 1, sizeof (address));
  memcpy (&quantity, buffer + 1 + sizeof (address), sizeof (quantity));

  md5_ctx_t md5_ctx;
  md5_init (&md5_ctx);
  md5_memory_range (address, quantity, &md5_ctx);
  md5_final (&md5_ctx, buffer + 1);

  send_rep (uart_dev, (uint8_t *) buffer, MD5_DIGEST_LENGTH + 1);
}



static void
reply_read_mem (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  buffer[0] = CMD_ACK;
  uint32_t address;
  uint16_t quantity;
  memcpy (&address, buffer + 1, sizeof (address));
  memcpy (&quantity, buffer + 1 + sizeof (address), sizeof (quantity));

  memset (buffer, 0, quantity); // DELETE THIS
  uint8_t *data = buffer + 1;

  read_memory_range (address, quantity, data);

  send_rep (uart_dev, (uint8_t *) buffer, 1 + quantity);
}

static void
reply_write_mem (int uart_dev, uint8_t *buffer, struct cache *cache __attribute__((unused)))
{
  uint8_t *data = buffer + 1;
  struct write_mem_params wmp;
  memcpy (&wmp, data, sizeof (wmp));
  data += sizeof (wmp);

  int i = 0;
  while (wmp.quantity - i > 0)
    {
      if (wmp.quantity - i >= 4 && ((wmp.address + i) % 4 == 0))
	{
	  if (i == 0)
	    bdc_write_quad_byte (wmp.address + i, data + i);
	  else
	    bdc_fill_quad_byte (data + i);
	  i += 4;
	}
      else if (wmp.quantity - i  >= 2 && ((wmp.address + i) % 2 == 0))
	{
	  if (i == 0)
	    bdc_write_double_byte (wmp.address + i, data + i);
	  else
	    bdc_fill_double_byte (data + i);
	  i += 2;
	}
      else
	{
	  if (i == 0)
	    bdc_write_byte (wmp.address + i, data + i);
	  else
	    bdc_fill_byte (data + i);
	  i += 1;
	}
    }

  buffer[0] = CMD_ACK;
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}

static void
reply_go (int uart_dev, uint8_t *buffer,
	  struct cache *cache __attribute__((unused)))
{
  buffer[0] = CMD_ACK;
  bdc_go ();
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}

static void
reply_step (int uart_dev, uint8_t *buffer,
	    struct cache *cache __attribute__((unused)))
{
  buffer[0] = CMD_ACK;
  bdc_step ();
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}


static void
reply_background (int uart_dev, uint8_t *buffer,
		  struct cache *cache __attribute__((unused)))

{
  buffer[0] = CMD_ACK;
  bdc_background ();
  send_rep (uart_dev, (uint8_t *) buffer, 1);
}

static const what_to_do lookup[n_CMDS] =
  {
    /* ACK */                  0,
    /* NACK */                 0,
    /* POLL */                 reply_poll,
    /* ECHO */                 reply_echo,
    /* BKDM */                 reply_bkdm,
    /* SYNC */                 reply_sync,
    /* BULK_ERASE */           reply_bulk_erase,
    /* EEPROM */               reply_eeprom,
    /* PFLASH */               reply_pflash,
    /* RESET */                reply_reset,
    /* READ_BDCCSR */          reply_read_bdccsr,
    /* READ_MEM */             reply_read_mem,
    /* READ_ALL_REGS */        reply_read_all_regs,
    /* READ_REG */             reply_read_register,
    /* GO */                   reply_go,
    /* BACKGROUND */           reply_background,
    /* STEP */                 reply_step,
    /* WRITE_MEM */            reply_write_mem,
    /* WRITE_BDCCSR */         reply_write_bdccsr,
    /* WRITE_REGS */           reply_write_regs,
    /* MD5_CHECK */            reply_md5_check,
    /* ERASE_EEPROM_SECTOR */  reply_erase_eeprom_sector,
    /* ERASE_PFLASH_SECTOR */  reply_erase_pflash_sector
  };

static void
despatch_payload (int uart_dev, uint8_t *buffer, struct cache *cache)
{
  what_to_do operation = lookup [buffer[0]];

  if (operation)
    {
      operation (uart_dev, buffer, cache);
    }
  else
    {
      buffer[0] = CMD_NACK;
      send_rep (uart_dev, (uint8_t *) buffer, 1);
    }
}

static uint8_t payload[MAX_PAYLOAD_LENGTH];

int
sep_start (int uart_dev)
{
  int res = uart_init  (uart_dev,
  			115200,
  			cb,
  			NULL);
  if (res != 0)
    {
      return -1;
    }

  int length = 0;
  int count = 0;
  state = STATE_IDLE;
  xtimer_init ();
  struct cache cache;
  cache.fstat = 0xFF;
  uint16_t crc = 0;
  uint16_t rx_crc = 0;
  while (1)
    {
      sema_wait (&byte_received);
      uint8_t rx_byte = byte;
      switch (state)
	{
	default:
	  break;
	case STATE_IDLE:
	  if (rx_byte == SEP_START)
	    state = STATE_START0;
	  break;
	case STATE_START0:
	  /* length high byte */
	  state = STATE_START1;
	  crc = crc16_update (0, rx_byte);
	  length = rx_byte;
	  length <<= 8;
	  break;
	case STATE_START1:
	  /* length low byte */
	  length |= rx_byte;
	  state = STATE_DATA;
	  count = 0;
	  crc = crc16_update (crc, rx_byte);
	  break;
	case STATE_DATA:
	  payload[count++] = rx_byte;
	  if (count >= length)
	    state = STATE_CHECKSUM0;
	  crc = crc16_update (crc, rx_byte);
	  break;
	case STATE_CHECKSUM0:
	  state = STATE_CHECKSUM1;
	  rx_crc = rx_byte;
	  rx_crc <<= 8;
	  break;
	case STATE_CHECKSUM1:
	  rx_crc |= rx_byte;
	  if (crc == rx_crc)
	    {
	      despatch_payload (uart_dev, payload, &cache);
	    }
	  else
	    {
	      payload[0] = CMD_NACK;
	      send_rep (uart_dev, (uint8_t *) payload, 1);
	    }
	  state = STATE_IDLE;
	  break;
	}
    }

  return 0;
}
