# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 96 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

#-- imports --------------------------------------------------------------------
#-- Qt imports
from qt import SIGNAL, QListView, QListViewItem, QSize, QTimer
from qt import PYSIGNAL, QApplication


#-- Python imports
import logging

from types import ListType


#-- own imports
from Settings import getPreferences
from AbstractFile import AbstractFileRecord


#-- preferences ----------------------------------------------------------------
pref = getPreferences(__name__)



class ListViewRecord(QListView):
    def __init__(self, parent=None, name=None, fl=0):
        """displays records in a list view and allows incremental search"""
        QListView.__init__(self, parent, name, fl)

        # create logging
        self.log = logging.getLogger( self.__class__.__name__ )
        self.log.setLevel(pref['logLevel'])

        # define layout
        self.setMinimumSize(QSize(180,0))
        self.addColumn("No")
        self.addColumn("Title")

        # rudimentry properties
        self.setAllColumnsShowFocus( True )
        self.setSorting( pref['sortColumn'] )
        self.setShowSortIndicator( True )
        self.setColumnWidthMode( 0, QListView.Manual )
        self.setColumnWidthMode( 1, QListView.Maximum )
        self.setResizeMode( QListView.LastColumn )
        self.setSelectionMode( QListView.Single )
        self.setDefaultRenameAction( QListView.Accept )

        # runtime variables
        self.obj = None           # keeps track of the last item added
        self.items = {}                 # maps ListViewItem -> record

        # incremental search string
        self.incSearch = ""
        self.incSearchHistory = []
        self.wrappedSearch = False
        self.searchOffTimer = QTimer()

        # connect signals
        self.connect( self.searchOffTimer, SIGNAL("timeout()"), self.onSearchTimeout )
        self.connect( self, SIGNAL("itemRenamed(QListViewItem*,int,const QString&)"), self.onItemRenamed )


    #-- python interface -------------------------------------------------------
    def __delitem__(self, y):
        """removes ListViewItem(s) by reference"""
        self.log.debug("deleting record %s", y)
        if (not type(y) is ListType): y = [y]
        for obj in y:
            self.takeItem(obj)
            del self.items[obj]


    def values(self):
        """returns an unsorted list of all records"""
        return self.items.values()

    def __len__(self):
        """returns number of elements in the list view"""
        return len(self.items.keys())
    

    #-- own interface ----------------------------------------------------------
    def add(self, rec):
        """W.add(rec) -- append record(s)"""
        # add all new records
        if (not type(rec) is ListType): rec = [rec]
        obj = None
        for r in rec:
            if (not isinstance(r, AbstractFileRecord)):
                raise(TypeError("Record not instance of AbstractFileRecord"))
            obj = QListViewItem( self, *self.compileTitles(r) )
            obj.setRenameEnabled(0, True)
            self.items[obj] = r
        # keep track of last item added
        self.obj = obj


    def getCurrentRecord(self):
        """returns the current record"""
        return self.items[self.currentItem()]


    def setCurrentRecord(self, record):
        """sets values of the current record"""
        currentObj = self.currentItem()
        self.items[currentObj] = record        
        self.onModified()


    def focusFirst(self):
        """sets focus on the first record in the listview"""
        self.focus(self.firstChild())


    def focusNew(self):
        """sets focus on the last item added"""
        self.focus(self.obj)
    

    #-- helper functions -------------------------------------------------------
    def focus(self, obj):
        """sets focus on one particular object"""
        if (obj != None):
            self.ensureItemVisible(obj)
            self.setCurrentItem(obj)    # sends currentChanged SIGNAL
            self.emit(SIGNAL("currentChanged(QListViewItem*)"), (obj,))
        self.setFocus()
          

    def compileTitles(self, record):
        """return a text-tuple what forms the ListViewItem entries """
        num = unicode(record['number'])
        if (pref['zeroPrefix_patch']):
            num = u'0'*(4-len(num))+num
        return (num,
                record['mediaLabel'] or record['originalTitle'] or record['translatedTitle'])


    #-- slots ------------------------------------------------------------------
    def onModified(self):
        """called when a record is modified in the form"""
        # update ListViewItem title
        index = 0
        record = self.items[self.currentItem()]
        for text in self.compileTitles(record):
            self.currentItem().setText(index, text)
            index += 1


    def onItemRenamed(self, obj, col, text):
        """called when record number is modified in the list view"""
        try:
            number = int(unicode(text))
        except ValueError:
            pass
        else:
            self.items[obj]['number'] = number
        # update list view
        self.onModified()
        self.sort()


    #-- incremental search 
    def onSearchTimeout(self):
        """called when timeout on incremental search"""
        self.searchOffTimer.stop()      # stop running timer
        self.emit(PYSIGNAL("incSearchTextChanged"), ("Stopping incremental search",))
        self.log.debug("Clearing incremental search history")
        self.incSearch = ""
        self.incSearchHistory = []
        self.wrappedSearch = False
      

    def keyPressEvent(self, e):
        """reacs on key-press-events for incremental search"""      
        # ignore keypressed when renaming
        if (self.isRenaming()):
            e.ignore()
            return

        if (e.text() == ""):
            QListView.keyPressEvent(self, e)
            return
       
        # react on backspace
        if (e.key() == QListView.Key_Backspace):
            if (len(self.incSearchHistory) > 0):
                (self.incSearch, prevItem) = self.incSearchHistory.pop()
                self.emit(PYSIGNAL("incSearchTextChanged"), ("Text found: \"%s\""%self.incSearch,))
                self.setCurrentItem( prevItem )
                self.ensureItemVisible( prevItem )
            else:
                QApplication.beep()
                self.emit(PYSIGNAL("incSearchTextChanged"), ("Search text empty",))
            return

        # abort on escape
        elif (e.key() == QListView.Key_Escape):
            self.onSearchTimeout()
            return

        # reset timer
        self.searchOffTimer.start(5000, True)

        currItem = self.currentItem()
        currText = self.incSearch

        # next search on enter or return
        if (e.key() in [QListView.Key_Enter, QListView.Key_Return]):
            # test on rollover
            self.setCurrentItem(currItem.nextSibling())
            nextItem = self.findItem(self.incSearch, 1, QListView.Contains)

            # enter on empty search string behaves like DOWN key
            if (currText == ""):
                self.ensureItemVisible(self.currentItem())
                return
            
            # rollover
            if (not self.wrappedSearch and (self.itemPos(nextItem) < self.itemPos(currItem))):
                self.setCurrentItem(currItem)
                self.wrappedSearch = True
                self.emit(PYSIGNAL("incSearchTextChanged"), ("End of list: wrap over?",))
                QApplication.beep()
                return
            self.wrappedSearch = False

        else:
            self.incSearch += unicode(e.text())

        self.incSearchHistory.append( (currText,currItem) ) # remember current item for back-travel
        item = self.findItem(self.incSearch, 1, QListView.Contains)
            
        # text does not match
        if (item == None):
            self.emit(PYSIGNAL("incSearchTextChanged"), ("Text not found: \"%s\""%self.incSearch,))
            QApplication.beep()
            self.log.info("No match for \"%s\"", self.incSearch)
        # match found
        else:
            self.emit(PYSIGNAL("incSearchTextChanged"), ("Text found: \"%s\""%self.incSearch,))
            self.setCurrentItem(item)
            self.ensureItemVisible(item)
    


#-- testarea -------------------------------------------------------------------
if (__name__ == "__main__"):
    pass
