;;;; FILE IDENTIFICATION
;;;; 
;;;; Name:		operators.lisp
;;;; Purpose:		The structure of LKalculus with operators
;;;; Programmer:	Raffaele Arecchi
;;;; Date Started:	11 Sep 2007
;;;;
;;;; Copyright (C) 2007 Raffaele Arecchi
;;;;
;;;; This program is free software; you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation; either version 2 of the License, or
;;;; (at your option) any later version.


(defpackage :operators
	(:use :cl :utility)
	(:export #:true1
		#:false1
		#:root
		#:parameters
		#:non
		#:et
		#:vel
		#:implies
		#:find-propositions
		#:find-propositions-quote
		#:set-parameters
		#:forall
		#:exist
		#:check-absurd
		#:reset-parameters
		#:proof))

(in-package :operators)

(defvar root)				; the set of formulas with branches and true-false lists
(setf root '((NIL NIL)))		; and set it empty
(defvar parameters)			; the set of variables and parameters
(setf parameters NIL)			; and set it empty

(defun true1 (x) (first (first x)))						; the true list

(defun false1 (x) (second (first x)))						; the false list

(defmacro non (x)
	`(cond  ( (not (equal (member-rec ''(non ,x) (car (car root))) NIL))	; if (non x) is in the true list 
			(progn
				(nremove ''(non ,x) (car (car root)))		; remove x from the true list
				(push ',x (second (car root)))))		; put x in the false list			
		( (not (equal (member-rec ''(non ,x) (second (car root))) NIL))	; if (non x) is in the false list
			(progn
				(nremove ''(non ,x) (second (car root)))		; remove x from the false list
				(push ',x (car (car root)))))))			; put x in the true list

(defmacro et (&rest x)
	`(cond  ( (not (equal (member-rec ''(et ,@x) (car (car root))) NIL))	; if (et x) is in the true list
			(progn
				(nremove ''(et ,@x) (car (car root)))		; remove (et x) from the true list
		  		(dolist (y '(,@x)) (push y (car (car root))))))	; put each x in the true list
		( (not (equal (member-rec ''(et ,@x) (second (car root))) NIL))	; if (et x) is in the false list
			(progn
				(nremove ''(et ,@x) (second (car root)))	; remove (et x) from the false list
				(let ((a (copy-list (car root))))
					(push (car '(,@x)) (second (car root)))	; set the first argument
					(dolist (y (cdr '(,@x)))
						(push a root) (push y (second (car root))))))))) ; set remaining arguments

(defmacro vel (&rest x)
	`(cond  ( (not (equal (member-rec ''(vel ,@x) (second (car root))) NIL)); if (vel x) is in the false list
			(progn
				(nremove ''(vel ,@x) (second (car root)))	; remove (vel x) from the false list
		  		(dolist (y '(,@x)) (push y (second (car root))))))	; put each x in the false list
		( (not (equal (member-rec ''(vel ,@x) (car (car root))) NIL))	; if (vel x) is in the true list
			(progn
				(nremove ''(vel ,@x) (car (car root)))	; remove (car x) from the true list
				(let ((a (copy-list (car root))))
					(push (car '(,@x)) (car (car root)))	; set the first argument
					(dolist (y (cdr '(,@x)))
						(push a root) (push y (car (car root))))))))) ; set remaining arguments

(defmacro implies (a b)
	`(cond	( (not (equal (member-rec ''(implies ,a ,b) (second (car root))) NIL))	; as before
			(progn
				(nremove ''(implies ,a ,b) (second (car root)))
				(push ',a (car (car root)))
				(push ',b (second (car root)))))
		( (not (equal (member-rec ''(implies ,a ,b) (car (car root))) NIL))
			(progn
				(nremove ''(implies ,a ,b) (car (car root)))
				(let ((y (copy-list (car root))))
					(push ',b (car (car root)))
					(push y root) (push ',a (second (car root))))))))

; this function finds all propositions of the form P(x,y,...) and get them into a list
(defun find-propositions () (mapcar #'eval (find-propositions-quote)))

(defun find-propositions-quote ()
	(let ((y (append (true1 root) (false1 root))))
		(while 
			(eval (cons 'or (mapcar #'(lambda (x) (memberp (car (eval x)) '(non et vel implies forall exist))) y)))
; that is, when every member of y  has a list which begins with 'et' or 'vel' or 'implies' or..
			(dolist (i y)
				(cond ((memberp (car (eval i)) '(non et vel implies))
					(progn (setf y (set-difference y (list i) :test #'equal)) (setf y (append (cdr (eval i)) y))))
				      ((memberp (car (eval i)) '(forall exist))
					(progn (setf y (set-difference y (list i) :test #'equal)) (push (nth 2 (eval i)) y))))))
		y))

; now we set the variables and constants before processing formulas
; this function cheks all variables and sets them in parameters

(defun set-parameters ()
	(let* ((q (find-propositions))
		(y (apply #'append (mapcar #'cdr q))))
	       (dolist (i y)
			  (if (symbolp i)
				(progn
					(nremove i y)
					(push i y))		; prevent repetitions
			      (nremove i y)))			; y become the list of all variables
		(dolist (i y)
			(push (list i) parameters))		; set them in parameters
; now we set the constants
		(while (not (equal q NIL))
			(let* ((k (car (car q)))			
				(j (remove NIL (mapcar #'(lambda (x) (if (equal (car x) k) x NIL)) q)))); j is the list of all 'P' propositions
			      (do ((m 1 (+ m 1)))
				  ((equal m (length (car j))) NIL)
				  (let ((h (mapcar #'(lambda (x) (nth m x)) j)))	; returns all symbols in m position
					(if (equal (find-sym h) NIL)	
					; if there are no variables, it doesn't matter for the theory, think about it
					    NIL
					    (progn
						(nconc h (assoc (find-sym h) parameters)) ; get the existing constants, or NIL
						(nremove (assoc (find-sym h) parameters) parameters) ; remove that parameter
						(push (append (list (find-sym h)) (find-num h)) parameters))))
						;move to parameters
			      (dolist (w j) (nremove w q)))))))		; remove all prop in j from q

(defmacro forall (x y)
	`(cond ( (not (equal (member-rec ''(forall ,x ,y) (true1 root)) NIL))	; if (forall x y) is in the true list
			(dolist (j (cdr (assoc ',x parameters)))	; for j in the list of constants associated to x
				(nconc (true1 root) (list (subst j ',x ',y))))) ; set P( ..,j,..) in the true list
		((not (equal (member-rec ''(forall ,x ,y) (false1 root)) NIL))	; if (forall x y) is in false list
		 (let ((j (+ (car (reverse (assoc ',x parameters))) 1))		; set new constant for x
			(w (remove ''(forall ,x ,y) (false1 root) :test #'equal)))
			(nconc (assoc ',x parameters) (list j))			; insert new constant in parameters
			(nremove (false1 root) (car root))
			(nconc (car root) (list (append w (list (subst j ',x ',y)))))))))

(defmacro exist (x y)
	`(cond ( (not (equal (member-rec ''(exist ,x ,y) (false1 root)) NIL))	; if (exist x y) is in the false list
			(dolist (j (cdr (assoc ',x parameters)))	; for j in the list of constants associated to x
				(nconc (false1 root) (list (subst j ',x ',y))))) ; set P( ..,j,..) in the false list
		((not (equal (member-rec ''(exist ,x ,y) (true1 root)) NIL))	; if (exist x y) is in true list
		 (let ((j (+ (car (reverse (assoc ',x parameters))) 1))		; set new constant for x
			(w (remove ''(exist ,x ,y) (true1 root) :test #'equal)))
			(nconc (assoc ',x parameters) (list j))			; insert new constant in parameters
			(nremove (true1 root) (car root))
			(push (append w (list (subst j ',x ',y))) (car root))))))

(defun check-absurd ()
	(if (equal (intersection (true1 root) (false1 root) :test #'equal) NIL)
		NIL
		(nremove (car root) root)))			; remove first branch

(defun reset-parameters () (setf root '((NIL NIL))) (setf parameters NIL))

(defun proof (ipothesis thesis)
; where ipothesis is a list of (true) statements and thesis a list of (false) statements
	(setf (car (car root)) ipothesis)
	(setf (second (car root)) thesis)
	(set-parameters)
	(do	()
		((equal root NIL) (format t "a proof is found!!") (reset-parameters))
		(check-absurd)
		(let ((y (set-difference (append (true1 root) (false1 root)) (find-propositions-quote) :test #'equal)))
			(cond ( (and (equal (true1 root) NIL) (equal (false1 root) NIL)) NIL)
			      ( (equal y NIL) (progn (format t "a proof cannot be found: there is a finite-countermodel!") (reset-parameters) (return NIL)))
			      ( (not (equal (intersection (mapcar #'(lambda (x) (car (eval x))) y) '(non et vel implies)) NIL))
				(eval (eval (car (remove NIL (mapcar #'(lambda (x) (if (memberp (car (eval x)) '(et non vel implies)) x NIL)) y))))))

	      		  (t (eval (eval (car (remove NIL (mapcar #'(lambda (x) (if (memberp (car (eval x)) '(forall exist)) x NIL)) y))))))))))
; finally is necessary to write a parser
