/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "map.h"

/*
 * This is used to stamp maps as they are created.
 * Note that there's no race condition due to the fact that this
 * is global here, in fact even when 2 processes would share
 * this sequence id, it would not matter for they would then
 * try and identify the objects in their on per-process lists,
 * structures, Guile object, whatever they use.
 */
static u_int32_t seq_id = 0;

/**
 * lw6map_new
 *
 * Creates a new empty map. This object is perfectly unusable as is,
 * since it has a 0x0 size, and many things set to "NULL". Still, it's
 * used internally and is the canonical way to create the object, it
 * ensures later calls that set up default parameters, for instance,
 * will succeed.
 *
 * Return value: a newly allocated pointer.
 */
lw6map_level_t *
lw6map_new ()
{
  lw6map_level_t *level = NULL;

  level = (lw6map_level_t *) LW6SYS_CALLOC (sizeof (lw6map_level_t));

  if (level)
    {
      level->id = 0;
      while (!level->id)
	{
	  level->id = ++seq_id;
	}
    }

  return level;
}

/**
 * lw6map_defaults
 *
 * @nb_layers: the number of layers of the map
 * @w: the width of the map
 * @h: the height of the map
 *
 * Creates a map, set to defaults. This is usefull mostly for testing,
 * it will just create a dull rectangle plain and uninteresting map,
 * however it's a quick way to have a working object and test stuff on it.
 *
 * Return value: a newly allocated map. 
 */
lw6map_level_t *
lw6map_defaults (int nb_layers, int w, int h)
{
  lw6map_level_t *level = NULL;
  lw6map_color_couple_t color;

  color.fg = LW6SYS_COLOR_8_WHITE;
  color.bg = LW6SYS_COLOR_8_BLACK;

  level = (lw6map_level_t *) LW6SYS_CALLOC (sizeof (lw6map_level_t));
  if (level)
    {
      level->id = 0;
      while (!level->id)
	{
	  level->id = ++seq_id;
	}

      lw6map_body_defaults (&level->body, nb_layers, w, h);
      lw6map_texture_from_body (&level->texture, &level->body, &color);
      lw6map_param_defaults (&level->param);
      lw6map_metadata_defaults (&level->metadata);
    }

  return level;
}

/**
 * lw6map_free
 *
 * @map: the map to free
 *
 * Frees a map and releases all its internal ressources.
 *
 * Return value: none.
 */
void
lw6map_free (lw6map_level_t * level)
{
  if (level)
    {
      lw6map_texture_clear (&level->texture);
      lw6map_body_clear (&level->body);
      lw6map_param_clear (&level->param);
      lw6map_metadata_clear (&level->metadata);

      LW6SYS_FREE (level);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, _("trying to free NULL map"));
    }
}

/**
 * lw6map_memory_footprint
 *
 * @map: the map to query
 *
 * Reports how many bytes the map needs, in memory. Note that this is
 * not contiguous memory, it involves a bunch of pointers, and possibly
 * much more...
 */
int
lw6map_memory_footprint (lw6map_level_t * level)
{
  int memory_footprint = 0;

  if (level)
    {
      memory_footprint += sizeof (lw6map_level_t);
      memory_footprint += strlen (level->metadata.title) + 1;
      memory_footprint += strlen (level->metadata.readme) + 1;
      memory_footprint +=
	level->body.shape.w * level->body.shape.h * level->body.shape.d *
	sizeof (u_int8_t);
      memory_footprint +=
	level->texture.w * level->texture.h * sizeof (lw6sys_color_8_t);

    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("calling memory_footprint on NULL map"));
    }

  return memory_footprint;
}

/**
 * lw6map_repr
 *
 * @map: the map to describe
 *
 * Returns a string describing the map. This is a very short description,
 * use it for logs, and to debug stuff. By no means it's a complete exhaustive
 * description. Still, the string returned should be unique.
 *
 * Return value: a dynamically allocated string.
 */
char *
lw6map_repr (lw6map_level_t * level)
{
  char *ret = NULL;

  if (level)
    {
      char *title = NULL;

      title = level->metadata.title;
      if (!title)
	{
	  title = _("No title");
	}

      ret =
	lw6sys_new_sprintf ("%u \"%s\" (%dx%dx%d)", level->id, title,
			    level->body.shape.w,
			    level->body.shape.h, level->body.shape.d);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("can't generate string id for NULL level"));
    }

  return ret;
}
