/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "liquidwar6.h"

/*
 * Helper function, creates a 0 terminated string
 * from a Guile string. A very common task.
 * Returned pointer must be freed.
 */
static char *
to_0str (SCM string)
{
  char *c_string;
  int length;

  SCM_ASSERT (scm_is_string (string), string, SCM_ARG1, "");

  /*
   * See the comment in sys/sys-str.c to see why we use
   * 2 trailing '\0' at the end of the string.
   */
  length = scm_c_string_length (string);
  c_string = (char *) LW6SYS_MALLOC ((length + 2) * sizeof (char));

  if (c_string)
    {
      memcpy ((void *) c_string, (void *) scm_i_string_chars (string),
	      length * sizeof (char));
      c_string[length] = c_string[length + 1] = '\0';
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _
		  ("unable to convert a guile SCM string to a standard C \"'\\0' terminated\" string"));
    }

  return c_string;
}

static SCM
to_scm_str_list (lw6sys_list_t * c_list)
{
  SCM ret = SCM_EOL;
  lw6sys_list_t *c_item = NULL;

  for (c_item = c_list; c_item; c_item = lw6sys_list_next (c_item))
    {
      if (c_item->data)
	{
	  ret = scm_cons (scm_makfrom0str ((char *) c_item->data), ret);
	}
    }

  ret = scm_reverse (ret);

  return ret;
}

static lw6sys_assoc_t *
to_sys_str_assoc (SCM assoc)
{
  lw6sys_assoc_t *c_assoc = NULL;

  c_assoc = lw6sys_assoc_new (lw6sys_free_callback);
  if (c_assoc)
    {
      if (SCM_CONSP (assoc))
	{
	  int i, n;
	  char *key, *value;
	  SCM item;

	  n = scm_to_int (scm_length (assoc));
	  for (i = 0; i < n; ++i)
	    {
	      item = scm_list_ref (assoc, scm_int2num (i));
	      if (SCM_CONSP (item))
		{
		  key = to_0str (scm_car (item));
		  value = to_0str (scm_cdr (item));
		  if (key && value)
		    {
		      lw6sys_assoc_set (&c_assoc, key, (void *) value);
		      LW6SYS_FREE (key);
		      // value must not be freed now
		    }
		}
	    }
	}
    }

  return c_assoc;
}

/*
 * For a GNU gettext-like behavior of scheme code.
 */
static SCM
_scm_gettext (SCM string)
{
  char *c_string;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_string (string), string, SCM_ARG1, __FUNCTION__);

  c_string = to_0str (string);
  if (c_string)
    {
      ret = scm_makfrom0str (gettext (c_string));
      LW6SYS_FREE (c_string);
    }

  return ret;
}

/*
 * In liquidwar6sys
 */

/*
 * In bazooka.c
 */
static SCM
_scm_lw6sys_set_memory_bazooka_size (SCM size)
{
  SCM ret = SCM_BOOL_F;
  int c_size;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (size), size, SCM_ARG1, __FUNCTION__);

  c_size = scm_to_int (size);

  ret = lw6sys_set_memory_bazooka_size (c_size) ? SCM_BOOL_T : SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_set_memory_bazooka_eraser (SCM state)
{
  SCM ret = SCM_BOOL_F;
  int c_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_bool (state), state, SCM_ARG1, __FUNCTION__);

  c_state = scm_to_bool (state);

  ret = lw6sys_set_memory_bazooka_eraser (c_state) ? SCM_BOOL_T : SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * build.c
 */
static SCM
_scm_lw6sys_build_get_package_tarname ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_package_tarname ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_package_name ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_package_name ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_package_string ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_package_string ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_version ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_version ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_codename ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_codename ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_stamp ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_stamp ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_md5sum ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_md5sum ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_copyright ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_copyright ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_license ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_license ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_configure_args ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_configure_args ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_gcc_version ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_gcc_version ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_cflags ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_cflags ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_ldflags ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_ldflags ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_hostname ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_hostname ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_date ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_date ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_time ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_time ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_target_cpu ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_target_cpu ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_endianness ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_endianness ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_pointer_size ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_int2num (lw6sys_build_get_pointer_size ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_target_os ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_target_os ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_is_ms_windows ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = lw6sys_build_is_ms_windows ()? SCM_BOOL_T : SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_top_srcdir ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_top_srcdir ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_prefix ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_prefix ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_datadir ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_datadir ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_libdir ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_libdir ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_includedir ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_includedir ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_localedir ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_localedir ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_console ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_console ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_mod_gl ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_mod_gl ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_mod_csound ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_mod_csound ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_mod_ogg ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_mod_ogg ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_mod_http ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_mod_http ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_optimize ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_optimize ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_allinone ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_allinone ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_fullstatic ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_fullstatic ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_gprof ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_gprof ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_gcov ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_gcov ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_build_get_enable_valgrind ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_makfrom0str (lw6sys_build_get_enable_valgrind ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * in debug.c
 */

static SCM
_scm_lw6sys_debug_get ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = lw6sys_debug_get ()? SCM_BOOL_T : SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_debug_set (SCM mode)
{
  int c_mode;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_bool (mode), mode, SCM_ARG1, __FUNCTION__);

  c_mode = scm_to_bool (mode);

  lw6sys_debug_set (c_mode);

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In dump.c
 */

static SCM
_scm_lw6sys_dump (SCM content)
{
  char *c_content;
  char *user_dir = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (content), content, SCM_ARG1, __FUNCTION__);

  c_content = to_0str (content);
  if (c_content)
    {
      user_dir =
	lw6cfg_unified_get_user_dir (lw6_global.argc, lw6_global.argv);
      if (user_dir)
	{
	  lw6sys_dump (user_dir, c_content);
	  LW6SYS_FREE (user_dir);
	}
      LW6SYS_FREE (c_content);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6sys_dump_clear ()
{
  char *user_dir = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  user_dir = lw6cfg_unified_get_user_dir (lw6_global.argc, lw6_global.argv);
  if (user_dir)
    {
      lw6sys_dump_clear (user_dir);
      LW6SYS_FREE (user_dir);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In env.c
 */
static SCM
_scm_lw6sys_get_username ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_username ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_hostname ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_hostname ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In id.c
 */
static SCM
_scm_lw6sys_generate_id_16 ()
{
  SCM ret = SCM_BOOL_F;
  char *c_ret = NULL;

  c_ret = lw6sys_id_ltoa (lw6sys_generate_id_16 ());
  if (c_ret)
    {
      ret = scm_makfrom0str (c_ret);
      LW6SYS_FREE (c_ret);
    }

  return ret;
}

static SCM
_scm_lw6sys_generate_id_32 ()
{
  SCM ret = SCM_BOOL_F;
  char *c_ret = NULL;

  c_ret = lw6sys_id_ltoa (lw6sys_generate_id_32 ());
  if (c_ret)
    {
      ret = scm_makfrom0str (c_ret);
      LW6SYS_FREE (c_ret);
    }

  return ret;
}

static SCM
_scm_lw6sys_generate_id_64 ()
{
  SCM ret = SCM_BOOL_F;
  char *c_ret = NULL;

  c_ret = lw6sys_id_ltoa (lw6sys_generate_id_64 ());
  if (c_ret)
    {
      ret = scm_makfrom0str (c_ret);
      LW6SYS_FREE (c_ret);
    }

  return ret;
}

/*
 * In log.c
 */
static SCM
_scm_lw6sys_log (SCM level, SCM message)
{
  int c_level;
  char *c_message;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (level), level, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (message), message, SCM_ARG2, __FUNCTION__);

  c_level = scm_to_int (level);
  c_message = to_0str (message);
  if (c_message)
    {
      lw6sys_log (c_level, __FILE__, __LINE__, "%s", c_message);
      LW6SYS_FREE (c_message);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6sys_log_get_level ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_int2num (lw6sys_log_get_level ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_log_set_level (SCM level)
{
  int c_level;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (level), level, SCM_ARG1, __FUNCTION__);

  c_level = scm_to_int (level);
  lw6sys_log_set_level (c_level);

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In option.c
 */
static SCM
_scm_lw6sys_get_default_user_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_default_user_dir ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_default_config_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_default_config_file ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_default_log_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_default_log_file ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_default_prefix ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_default_prefix ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_default_mod_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_default_mod_dir ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_default_data_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_default_data_dir ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_default_map_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_default_map_dir ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_default_map_path ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_default_map_path ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_default_script_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_default_script_file ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_cwd ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_cwd ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_run_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_run_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_user_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_user_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_config_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_config_file (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_log_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_log_file (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_prefix ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_prefix (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_mod_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_mod_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_data_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_data_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_map_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_map_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_map_path ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_map_path (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_get_script_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6sys_get_script_file (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In path.c
 */
static SCM
_scm_lw6sys_path_concat (SCM path1, SCM path2)
{
  SCM ret = SCM_BOOL_F;
  char *c_path1 = NULL;
  char *c_path2 = NULL;
  char *c_ret = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (path1), path1, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (path2), path2, SCM_ARG2, __FUNCTION__);

  c_path1 = to_0str (path1);
  if (c_path1)
    {
      c_path2 = to_0str (path2);
      if (c_path2)
	{
	  c_ret = lw6sys_path_concat (c_path1, c_path2);
	  if (c_ret)
	    {
	      ret = scm_makfrom0str (c_ret);
	      LW6SYS_FREE (c_ret);
	    }
	  LW6SYS_FREE (c_path2);
	}
      LW6SYS_FREE (c_path1);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_path_parent (SCM path)
{
  SCM ret = SCM_BOOL_F;
  char *c_path = NULL;
  char *c_ret = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (path), path, SCM_ARG1, __FUNCTION__);

  c_path = to_0str (path);
  if (c_path)
    {
      c_ret = lw6sys_path_parent (c_path);
      if (c_ret)
	{
	  ret = scm_makfrom0str (c_ret);
	  LW6SYS_FREE (c_ret);
	}
      LW6SYS_FREE (c_path);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_path_split (SCM path)
{
  SCM ret = SCM_BOOL_F;
  char *c_path = NULL;
  lw6sys_list_t *c_ret = NULL;
  lw6sys_list_t *tmp = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (path), path, SCM_ARG1, __FUNCTION__);

  c_path = to_0str (path);
  if (c_path)
    {
      ret = SCM_LIST0;
      c_ret = lw6sys_path_split (c_path);
      if (c_ret)
	{
	  for (tmp = c_ret; tmp != NULL;
	       tmp = (lw6sys_list_t *) tmp->next_item)
	    {
	      if (tmp->data)
		{
		  ret = scm_cons (scm_makfrom0str ((char *) tmp->data), ret);
		}
	    }
	  lw6sys_list_free (c_ret);
	  ret = scm_reverse (ret);
	}
      LW6SYS_FREE (c_path);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In time.c
 */
static SCM
_scm_lw6sys_timestamp ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = scm_long_long2num (lw6sys_timestamp ());

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6sys_sleep (SCM seconds)
{
  SCM ret = SCM_BOOL_F;
  float c_seconds;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_number (seconds), seconds, SCM_ARG1, __FUNCTION__);

  c_seconds = scm_to_double (seconds);

  lw6sys_sleep (c_seconds);

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * lw6hlp
 */
static SCM
_scm_lw6hlp_list_quick ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_quick ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_doc ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_doc ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_show ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_show ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_path ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_path ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_players ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_players ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_input ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_input ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_graphics ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_graphics ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_sound ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_sound ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_network ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_network ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_map ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_map ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_map_rules ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_map_rules ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_map_hints ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_map_hints ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_map_style ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_map_style ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_funcs ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_funcs ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_hooks ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_hooks ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_tuning ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_tuning ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list_aliases ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list_aliases ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6hlp_list ()
{
  lw6sys_list_t *c_list = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_list = lw6hlp_list ();
  if (c_list)
    {
      ret = to_scm_str_list (c_list);
      lw6sys_list_free (c_list);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In load.c
 */
static SCM
_scm_lw6cfg_load (SCM filename)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_filename;

      SCM_ASSERT (scm_is_string (filename), filename, SCM_ARG1, __FUNCTION__);

      c_filename = to_0str (filename);
      if (c_filename)
	{
	  if (lw6cfg_load (lw6_global.cfg_context, c_filename))
	    {
	      ret = SCM_BOOL_T;
	    }
	  else
	    {
	      ret = SCM_BOOL_F;
	    }
	  LW6SYS_FREE (c_filename);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In option.c
 */
static SCM
_scm_lw6cfg_option_exists (SCM key)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_key;

      SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, __FUNCTION__);

      c_key = to_0str (key);
      if (c_key)
	{
	  if (lw6cfg_option_exists (lw6_global.cfg_context, c_key))
	    {
	      ret = SCM_BOOL_T;
	    }
	  else
	    {
	      ret = SCM_BOOL_F;
	    }
	  LW6SYS_FREE (c_key);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6cfg_get_option (SCM key)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_key;

      SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, __FUNCTION__);

      c_key = to_0str (key);
      if (c_key)
	{
	  char *value = NULL;

	  value = lw6cfg_get_option (lw6_global.cfg_context, c_key);
	  if (value)
	    {
	      ret =
		scm_makfrom0str (lw6cfg_get_option
				 (lw6_global.cfg_context, c_key));
	    }

	  LW6SYS_FREE (c_key);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6cfg_set_option (SCM key, SCM value)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_key;
      char *c_value;

      SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_string (value), value, SCM_ARG2, __FUNCTION__);

      c_key = to_0str (key);
      if (c_key)
	{
	  c_value = to_0str (value);
	  if (c_value)
	    {
	      lw6cfg_set_option (lw6_global.cfg_context, c_key, c_value);

	      LW6SYS_FREE (c_value);
	    }
	  LW6SYS_FREE (c_key);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In save.c
 */
static SCM
_scm_lw6cfg_save (SCM filename)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_filename;

      SCM_ASSERT (scm_is_string (filename), filename, SCM_ARG1, __FUNCTION__);

      c_filename = to_0str (filename);
      if (c_filename)
	{
	  if (lw6cfg_save (lw6_global.cfg_context, c_filename))
	    {
	      ret = SCM_BOOL_T;
	    }
	  else
	    {
	      ret = SCM_BOOL_F;
	    }
	  LW6SYS_FREE (c_filename);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In setup.c
 */
static SCM
_scm_lw6cfg_init ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  lw6_global.cfg_context = lw6cfg_init (lw6_global.argc, lw6_global.argv);
  lw6_global.cfg_initialized = lw6_global.cfg_context ? 1 : 0;
  ret = lw6_global.cfg_context ? SCM_BOOL_T : SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6cfg_quit ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.cfg_context != NULL)
    {
      lw6cfg_quit (lw6_global.cfg_context);
    }

  lw6_global.cfg_context = NULL;
  lw6_global.cfg_initialized = 0;

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * Unified
 */

static SCM
_scm_lw6cfg_unified_get_user_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6cfg_unified_get_user_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6cfg_unified_get_log_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6cfg_unified_get_log_file (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6cfg_unified_get_map_path ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  buf = lw6cfg_unified_get_map_path (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}



/*
 * In liquidwar6opt
 */

/*
 * In static.c
 */
static SCM
_scm_lw6map_rules_get_default (SCM key)
{
  SCM ret = SCM_BOOL_F;
  char *c_key = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, __FUNCTION__);

  c_key = to_0str (key);
  if (c_key)
    {
      ret = scm_int2num (lw6map_rules_get_default (c_key));

      LW6SYS_FREE (c_key);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6map_rules_get_min (SCM key)
{
  SCM ret = SCM_BOOL_F;
  char *c_key = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, __FUNCTION__);

  c_key = to_0str (key);
  if (c_key)
    {
      ret = scm_int2num (lw6map_rules_get_min (c_key));

      LW6SYS_FREE (c_key);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6map_rules_get_max (SCM key)
{
  SCM ret = SCM_BOOL_F;
  char *c_key = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, __FUNCTION__);

  c_key = to_0str (key);
  if (c_key)
    {
      ret = scm_int2num (lw6map_rules_get_max (c_key));

      LW6SYS_FREE (c_key);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6map_rules_get_int (SCM game_struct, SCM key)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_struct_t *c_game_struct;
      char *c_key;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_struct,
		   game_struct), game_struct, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_string (key), key, SCM_ARG2, __FUNCTION__);

      c_game_struct = lw6_scm_to_game_struct (game_struct);
      if (c_game_struct)
	{
	  c_key = to_0str (key);
	  if (c_key)
	    {
	      ret =
		scm_int2num (lw6map_rules_get_int
			     (&(c_game_struct->rules), c_key));
	      LW6SYS_FREE (c_key);
	    }
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6map_style_get_default (SCM key)
{
  SCM ret = SCM_BOOL_F;
  char *c_key = NULL;
  char *c_value = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, __FUNCTION__);

  c_key = to_0str (key);
  if (c_key)
    {
      c_value = lw6map_style_get_default (c_key);
      if (c_value)
	{
	  ret = scm_makfrom0str (c_value);
	  LW6SYS_FREE (c_value);
	}
      LW6SYS_FREE (c_key);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In liquidwar6gui
 */
static SCM
_scm_lw6gui_menu_new (SCM title)
{
  char *c_title;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (title), title, SCM_ARG1, __FUNCTION__);

  c_title = to_0str (title);
  if (c_title)
    {
      lw6gui_menu_t *c_menu;

      c_menu = lw6gui_menu_new (c_title);
      if (c_menu)
	{
	  ret = lw6_make_scm_menu (c_menu);
	}
      LW6SYS_FREE (c_title);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gui_menu_append (SCM menu, SCM menuitem)
{
  char *c_label;
  int c_value;
  int c_enabled;
  int c_selected;
  int c_colored;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.menu,
	       menu), menu, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (SCM_CONSP (menuitem)
	      || menuitem == SCM_EOL, menuitem, SCM_ARG2, __FUNCTION__);

  c_label = to_0str (scm_assoc_ref (menuitem, scm_makfrom0str ("label")));

  if (c_label)
    {
      lw6gui_menu_t *c_menu;
      int now = 0;

      c_value =
	scm_to_int (scm_assoc_ref (menuitem, scm_makfrom0str ("value")));
      c_enabled =
	scm_to_bool (scm_assoc_ref (menuitem, scm_makfrom0str ("enabled")));
      c_selected =
	scm_to_bool (scm_assoc_ref (menuitem, scm_makfrom0str ("selected")));
      c_colored =
	scm_to_bool (scm_assoc_ref (menuitem, scm_makfrom0str ("colored")));

      c_menu = lw6_scm_to_menu (menu);

      if (lw6_global.gfx_backend != NULL)
	{
	  now = lw6gfx_get_ticks (lw6_global.gfx_backend);
	}

      ret =
	scm_int2num (lw6gui_menu_append_for_id_use
		     (c_menu, c_label, c_value, c_enabled, c_selected,
		      c_colored, now));

      LW6SYS_FREE (c_label);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gui_menu_sync (SCM menu, SCM menuitem)
{
  int c_id;
  char *c_label;
  int c_value;
  int c_enabled;
  int c_selected;
  int c_colored;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.menu,
	       menu), menu, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (SCM_CONSP (menuitem)
	      || menuitem == SCM_EOL, menuitem, SCM_ARG2, __FUNCTION__);

  c_label = to_0str (scm_assoc_ref (menuitem, scm_makfrom0str ("label")));

  if (c_label)
    {
      lw6gui_menu_t *c_menu;
      int now = 0;

      c_id = scm_to_int (scm_assoc_ref (menuitem, scm_makfrom0str ("id")));
      c_value =
	scm_to_int (scm_assoc_ref (menuitem, scm_makfrom0str ("value")));
      c_enabled =
	scm_to_bool (scm_assoc_ref (menuitem, scm_makfrom0str ("enabled")));
      c_selected =
	scm_to_bool (scm_assoc_ref (menuitem, scm_makfrom0str ("selected")));
      c_colored =
	scm_to_bool (scm_assoc_ref (menuitem, scm_makfrom0str ("colored")));

      c_menu = lw6_scm_to_menu (menu);

      if (lw6_global.gfx_backend != NULL)
	{
	  now = lw6gfx_get_ticks (lw6_global.gfx_backend);
	}

      lw6gui_menu_sync_using_id
	(c_menu, c_id, c_label, c_value, c_enabled, c_selected,
	 c_colored, now);

      LW6SYS_FREE (c_label);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gui_menu_select (SCM menu, SCM position)
{
  lw6gui_menu_t *c_menu;
  int c_position;
  int now = 0;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.menu,
	       menu), menu, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (position), position, SCM_ARG2, __FUNCTION__);

  c_menu = lw6_scm_to_menu (menu);
  c_position = scm_to_int (position);

  if (lw6_global.gfx_backend != NULL)
    {
      now = lw6gfx_get_ticks (lw6_global.gfx_backend);
    }

  ret =
    lw6gui_menu_select (c_menu, c_position, now) ? SCM_BOOL_T : SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gui_menu_scroll_up (SCM menu)
{
  lw6gui_menu_t *c_menu;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.menu,
	       menu), menu, SCM_ARG1, __FUNCTION__);

  c_menu = lw6_scm_to_menu (menu);

  ret = lw6gui_menu_scroll_up (c_menu) ? SCM_BOOL_T : SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gui_menu_scroll_down (SCM menu)
{
  lw6gui_menu_t *c_menu;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.menu,
	       menu), menu, SCM_ARG1, __FUNCTION__);

  c_menu = lw6_scm_to_menu (menu);

  ret = lw6gui_menu_scroll_down (c_menu) ? SCM_BOOL_T : SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gui_default_look ()
{
  lw6gui_look_t *c_look;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  c_look = lw6gui_look_new (NULL);
  if (c_look)
    {
      ret = lw6_make_scm_look (c_look);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gui_look_set (SCM look, SCM key, SCM value)
{
  lw6gui_look_t *c_look;
  char *c_key;
  char *c_value;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.look,
	       look), look, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (key), key, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_string (value), value, SCM_ARG3, __FUNCTION__);

  c_look = lw6_scm_to_look (look);
  c_key = to_0str (key);
  if (c_key)
    {
      c_value = to_0str (value);
      if (c_value)
	{
	  ret =
	    lw6gui_look_set (c_look, c_key,
			     c_value) ? SCM_BOOL_T : SCM_BOOL_F;
	  LW6SYS_FREE (c_value);
	}
      LW6SYS_FREE (c_key);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gui_look_get (SCM look, SCM key)
{
  lw6gui_look_t *c_look;
  char *c_key;
  char *c_value;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.look,
	       look), look, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (key), key, SCM_ARG2, __FUNCTION__);

  c_look = lw6_scm_to_look (look);
  c_key = to_0str (key);
  if (c_key)
    {
      c_value = lw6gui_look_get (c_look, c_key);
      if (c_value)
	{
	  ret = scm_makfrom0str (c_value);
	  LW6SYS_FREE (c_value);
	}
      LW6SYS_FREE (c_key);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In liquidwar6gfx
 */

/*
 * In backend.c
 */
static SCM
_scm_lw6gfx_get_backends ()
{
  SCM ret = SCM_BOOL_F;
  lw6sys_assoc_t *backends;
  lw6sys_list_t *keys;
  lw6sys_list_t *key;
  char *module_id;
  char *module_name;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  backends = lw6gfx_get_backends (lw6_global.argc, lw6_global.argv);
  if (backends)
    {
      keys = lw6sys_assoc_keys (backends);
      if (keys)
	{
	  ret = SCM_EOL;
	  key = keys;
	  while (key)
	    {
	      if (key->data)
		{
		  module_id = (char *) key->data;
		  module_name =
		    (char *) lw6sys_assoc_get (backends, module_id);
		  ret =
		    scm_cons (scm_cons
			      (scm_makfrom0str (module_id),
			       scm_makfrom0str (module_name)), ret);
		}
	      key = lw6sys_list_next (key);
	    }
	  lw6sys_list_free (keys);
	  ret = scm_reverse (ret);
	}
      lw6sys_assoc_free (backends);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_create_backend (SCM backend_name)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend == NULL)
    {
      char *c_backend_name;

      SCM_ASSERT (scm_is_string (backend_name),
		  backend_name, SCM_ARG1, __FUNCTION__);

      c_backend_name = to_0str (backend_name);
      if (c_backend_name)
	{
	  lw6_global.gfx_backend =
	    lw6gfx_create_backend (lw6_global.argc, lw6_global.argv,
				   c_backend_name);
	  LW6SYS_FREE (c_backend_name);
	}
    }

  if (lw6_global.gfx_backend)
    {
      ret = SCM_BOOL_T;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_destroy_backend ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      if (lw6_global.gfx_initialized)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _
		      ("destroying gfx backend, but backend has not been previously cleaned up"));
	  lw6gfx_quit (lw6_global.gfx_backend);
	}

      lw6gfx_destroy_backend (lw6_global.gfx_backend);
    }
  lw6_global.gfx_backend = NULL;
  lw6_global.gfx_initialized = 0;

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In background.c
 */
static SCM
_scm_lw6gfx_display_background (SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG1, __FUNCTION__);

      c_look = lw6_scm_to_look (look);

      lw6gfx_display_background (lw6_global.gfx_backend, c_look);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In event.c
 */
static SCM
_scm_lw6gfx_poll_key ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_keypress_t keypress;

      if (lw6gfx_poll_key (lw6_global.gfx_backend, &keypress))
	{
	  ret =
	    scm_list_3 (scm_cons
			(scm_makfrom0str ("keysym"),
			 scm_int2num (keypress.keysym)),
			scm_cons (scm_makfrom0str ("unicode"),
				  scm_int2num (keypress.unicode)),
			scm_cons (scm_makfrom0str ("label"),
				  scm_makfrom0str (keypress.label)));
	}
      else
	{
	  ret = SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_is_key_esc (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1, __FUNCTION__);

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_esc (lw6_global.gfx_backend,
			   c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_is_key_enter (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1, __FUNCTION__);

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_enter (lw6_global.gfx_backend,
			     c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_is_key_up (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1, __FUNCTION__);

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_up (lw6_global.gfx_backend,
			  c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_is_key_down (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1, __FUNCTION__);

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_down (lw6_global.gfx_backend,
			    c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_is_key_left (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1, __FUNCTION__);

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_left (lw6_global.gfx_backend,
			    c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_is_key_right (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1, __FUNCTION__);

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_right (lw6_global.gfx_backend,
			     c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_poll_quit ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_quit (lw6_global.gfx_backend) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_send_quit ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gfx_send_quit (lw6_global.gfx_backend);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6gfx_get_default_keys_state ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_up = 0;
      int c_down = 0;
      int c_left = 0;
      int c_right = 0;
      int c_esc = 0;
      int c_enter = 0;

      lw6gfx_get_default_keys_state (lw6_global.gfx_backend, &c_up, &c_down,
				     &c_left, &c_right, &c_esc, &c_enter);
      ret =
	scm_cons (scm_cons
		  (scm_makfrom0str ("up"), c_up ? SCM_BOOL_T : SCM_BOOL_F),
		  scm_cons (scm_cons (scm_makfrom0str ("down"),
				      c_down ? SCM_BOOL_T : SCM_BOOL_F),
			    scm_cons (scm_cons (scm_makfrom0str ("left"),
						c_left ? SCM_BOOL_T :
						SCM_BOOL_F),
				      scm_cons (scm_cons
						(scm_makfrom0str ("right"),
						 c_right ? SCM_BOOL_T :
						 SCM_BOOL_F),
						scm_cons (scm_cons
							  (scm_makfrom0str
							   ("esc"),
							   c_esc ? SCM_BOOL_T
							   : SCM_BOOL_F),
							  scm_cons (scm_cons
								    (scm_makfrom0str
								     ("enter"),
								     c_enter ?
								     SCM_BOOL_T
								     :
								     SCM_BOOL_F),
								    SCM_LIST0))))));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_get_key_state (SCM keysym)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;
      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1, __FUNCTION__);
      c_keysym = scm_to_int (keysym);
      ret =
	lw6gfx_get_key_state (lw6_global.gfx_backend,
			      c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_poll_mouse_move ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_x;
      int c_y;

      if (lw6gfx_poll_mouse_move (lw6_global.gfx_backend, &c_x, &c_y))
	{
	  ret = scm_list_2 (scm_cons
			    (scm_makfrom0str ("x"), scm_int2num (c_x)),
			    scm_cons (scm_makfrom0str ("y"),
				      scm_int2num (c_y)));
	}
      else
	{
	  ret = SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_poll_mouse_button_left ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_mouse_button_left (lw6_global.gfx_backend) ? SCM_BOOL_T :
	SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_poll_mouse_button_right ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_mouse_button_right (lw6_global.gfx_backend) ? SCM_BOOL_T :
	SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_poll_mouse_wheel_up ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_mouse_wheel_up (lw6_global.gfx_backend) ? SCM_BOOL_T :
	SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_poll_mouse_wheel_down ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_mouse_wheel_down (lw6_global.gfx_backend) ? SCM_BOOL_T :
	SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_get_mouse_state ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_x;
      int c_y;
      int c_button_left;
      int c_button_right;
      lw6gfx_get_mouse_state (lw6_global.gfx_backend,
			      &c_x, &c_y, &c_button_left, &c_button_right);
      ret =
	scm_list_4 (scm_cons
		    (scm_makfrom0str ("x"),
		     scm_int2num (c_x)),
		    scm_cons (scm_makfrom0str ("y"),
			      scm_int2num (c_y)),
		    scm_cons (scm_makfrom0str
			      ("button-left"),
			      c_button_left ? SCM_BOOL_T :
			      SCM_BOOL_F),
		    scm_cons (scm_makfrom0str
			      ("button-right"),
			      c_button_right ? SCM_BOOL_T : SCM_BOOL_F));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_warp_mouse (SCM x, SCM y)
{
  SCM ret = SCM_BOOL_F;
  int c_x = 0;
  int c_y = 0;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      SCM_ASSERT (scm_is_integer (x), x, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_integer (y), y, SCM_ARG2, __FUNCTION__);

      c_x = scm_to_int (x);
      c_y = scm_to_int (y);

      ret =
	lw6gfx_warp_mouse (lw6_global.gfx_backend, c_x,
			   c_y) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_show_mouse (SCM state)
{
  int c_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      SCM_ASSERT (SCM_BOOLP (state), state, SCM_ARG1, __FUNCTION__);

      c_state = SCM_NFALSEP (state);

      lw6gfx_show_mouse (lw6_global.gfx_backend, c_state);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6gfx_poll_joystick_up (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1, __FUNCTION__);

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      ret =
	lw6gfx_poll_joystick_up (lw6_global.gfx_backend,
				 c_i) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_poll_joystick_down (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1, __FUNCTION__);

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      ret =
	lw6gfx_poll_joystick_down (lw6_global.gfx_backend,
				   c_i) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_poll_joystick_left (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1, __FUNCTION__);

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      ret =
	lw6gfx_poll_joystick_left (lw6_global.gfx_backend,
				   c_i) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_poll_joystick_right (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1, __FUNCTION__);

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      ret =
	lw6gfx_poll_joystick_right (lw6_global.gfx_backend,
				    c_i) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_poll_joystick_button (SCM i, SCM b)
{
  SCM ret = SCM_BOOL_F;
  int c_i, c_b;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (b), b, SCM_ARG2, __FUNCTION__);

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      c_b = scm_to_int (b);
      ret =
	lw6gfx_poll_joystick_button (lw6_global.gfx_backend,
				     c_i, c_b) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_get_joystick_state (SCM i, SCM b1, SCM b2)
{
  SCM ret = SCM_BOOL_F;
  int c_i, c_b1, c_b2;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (b1), b2, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (b1), b2, SCM_ARG3, __FUNCTION__);

  if (lw6_global.gfx_backend != NULL)
    {
      int c_x = 0;
      int c_y = 0;
      int c_button_a = 0;
      int c_button_b = 0;

      c_i = scm_to_int (i);
      c_b1 = scm_to_int (b1);
      c_b2 = scm_to_int (b2);
      lw6gfx_get_joystick_state (lw6_global.gfx_backend,
				 &c_x, &c_y, &c_button_a, &c_button_b, c_i,
				 c_b1, c_b2);
      ret =
	scm_list_4 (scm_cons (scm_makfrom0str ("x"), scm_int2num (c_x)),
		    scm_cons (scm_makfrom0str ("y"), scm_int2num (c_y)),
		    scm_cons (scm_makfrom0str ("button-a"),
			      c_button_a ? SCM_BOOL_T : SCM_BOOL_F),
		    scm_cons (scm_makfrom0str ("button-b"),
			      c_button_b ? SCM_BOOL_T : SCM_BOOL_F));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_set_key_repeat (SCM key_repeat_delay, SCM key_repeat_interval)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (key_repeat_delay), key_repeat_delay, SCM_ARG1,
	      __FUNCTION__);
  SCM_ASSERT (scm_is_integer (key_repeat_interval), key_repeat_interval,
	      SCM_ARG2, __FUNCTION__);

  if (lw6_global.gfx_backend != NULL)
    {
      int c_key_repeat_delay;
      int c_key_repeat_interval;

      c_key_repeat_delay = scm_to_int (key_repeat_delay);
      c_key_repeat_interval = scm_to_int (key_repeat_interval);
      ret =
	lw6gfx_set_key_repeat (lw6_global.gfx_backend, c_key_repeat_delay,
			       c_key_repeat_interval) ? SCM_BOOL_T :
	SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In hud.c
 */
static SCM
_scm_lw6gfx_display_hud (SCM game_state, SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, __FUNCTION__);

      c_game_state = lw6_scm_to_game_state (game_state);
      c_look = lw6_scm_to_look (look);
      lw6gfx_display_hud (lw6_global.gfx_backend, c_look, c_game_state);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In info.c
 */
static SCM
_scm_lw6gfx_display_log (SCM timeout, SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_timeout;
      lw6gui_look_t *c_look;

      SCM_ASSERT (scm_is_number (timeout), timeout, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, __FUNCTION__);

      c_look = lw6_scm_to_look (look);
      c_timeout = scm_to_int (timeout);
      if (c_timeout)
	{
	  char **c_log = NULL;

	  c_log = lw6sys_history_get (c_timeout);
	  if (c_log)
	    {
	      lw6gfx_display_log (lw6_global.gfx_backend, c_look, c_log);
	      lw6sys_history_free (c_log);
	    }
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}


static SCM
_scm_lw6gfx_display_fps (SCM fps, SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_fps;
      lw6gui_look_t *c_look;

      SCM_ASSERT (scm_is_integer (fps), fps, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, __FUNCTION__);

      c_fps = scm_to_int (fps);
      c_look = lw6_scm_to_look (look);

      lw6gfx_display_fps (lw6_global.gfx_backend, c_look, c_fps);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In map.c
 */
static SCM
_scm_lw6gfx_display_map_preview (SCM level, SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6map_level_t *c_level;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, level), level,
		  SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, __FUNCTION__);

      c_level = lw6_scm_to_map (level);
      c_look = lw6_scm_to_look (look);

      lw6gfx_display_map_preview (lw6_global.gfx_backend, c_look, c_level);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6gfx_display_map_zones (SCM game_struct, SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_struct_t *c_game_struct;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_struct,
		   game_struct), game_struct, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, __FUNCTION__);

      c_game_struct = lw6_scm_to_game_struct (game_struct);
      c_look = lw6_scm_to_look (look);

      lw6gfx_display_map_zones (lw6_global.gfx_backend, c_look,
				c_game_struct);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6gfx_display_map_gradient (SCM game_state, SCM team_id, SCM layer_id,
				  SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      int c_team_id;
      int c_layer_id;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2, __FUNCTION__);
      SCM_ASSERT (scm_is_integer (layer_id), layer_id, SCM_ARG3,
		  __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE (lw6_global.smob_types.look, look), look,
		  SCM_ARG4, __FUNCTION__);

      c_game_state = lw6_scm_to_game_state (game_state);
      c_team_id = scm_to_int (team_id);
      c_layer_id = scm_to_int (layer_id);
      c_look = lw6_scm_to_look (look);

      lw6gfx_display_map_gradient (lw6_global.gfx_backend, c_look,
				   c_game_state, c_team_id, c_layer_id);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6gfx_display_game (SCM game_state, SCM main_cursor_id, SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      char *c_main_cursor_id_str = NULL;
      u_int16_t c_main_cursor_id_int = 0;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_string
		  (main_cursor_id), main_cursor_id, SCM_ARG2, __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG3, __FUNCTION__);

      c_game_state = lw6_scm_to_game_state (game_state);
      c_main_cursor_id_str = to_0str (main_cursor_id);
      if (c_main_cursor_id_str)
	{
	  c_main_cursor_id_int = lw6sys_id_atol (c_main_cursor_id_str);
	  LW6SYS_FREE (c_main_cursor_id_str);
	}
      c_look = lw6_scm_to_look (look);
      lw6gfx_display_game (lw6_global.gfx_backend, c_look, c_game_state,
			   c_main_cursor_id_int);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6gfx_display_cursors (SCM game_state, SCM main_cursor_id, SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      char *c_main_cursor_id_str = NULL;
      u_int16_t c_main_cursor_id_int = 0;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_string
		  (main_cursor_id), main_cursor_id, SCM_ARG2, __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG3, __FUNCTION__);

      c_game_state = lw6_scm_to_game_state (game_state);
      c_main_cursor_id_str = to_0str (main_cursor_id);
      if (c_main_cursor_id_str)
	{
	  c_main_cursor_id_int = lw6sys_id_atol (c_main_cursor_id_str);
	  LW6SYS_FREE (c_main_cursor_id_str);
	}
      c_look = lw6_scm_to_look (look);
      lw6gfx_display_cursors (lw6_global.gfx_backend, c_look, c_game_state,
			      c_main_cursor_id_int);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In menu.c
 */
static SCM
_scm_lw6gfx_display_menu (SCM menu, SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_menu_t *c_menu;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.menu,
		   menu), menu, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, __FUNCTION__);

      c_menu = lw6_scm_to_menu (menu);
      c_look = lw6_scm_to_look (look);

      lw6gfx_display_menu (lw6_global.gfx_backend, c_look, c_menu,
			   lw6_global.progress);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6gfx_pick_menuitem (SCM menu, SCM screen_x, SCM screen_y, SCM look)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_position = 0;
      int c_scroll = 0;
      lw6gui_menu_t *c_menu;
      int c_screen_x;
      int c_screen_y;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.menu,
		   menu), menu, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_integer (screen_x), screen_x, SCM_ARG2,
		  __FUNCTION__);
      SCM_ASSERT (scm_is_integer (screen_y), screen_y, SCM_ARG3,
		  __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG4, __FUNCTION__);

      c_menu = lw6_scm_to_menu (menu);
      c_screen_x = scm_to_int (screen_x);
      c_screen_y = scm_to_int (screen_y);
      c_look = lw6_scm_to_look (look);

      lw6gfx_pick_menuitem (lw6_global.gfx_backend, c_look, &c_position,
			    &c_scroll, c_menu, c_screen_x, c_screen_y);

      ret = scm_list_2
	(scm_cons
	 (scm_makfrom0str ("position"),
	  scm_int2num (c_position)),
	 scm_cons (scm_makfrom0str ("scroll"), scm_int2num (c_scroll)));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_warp_mouse_on_menuitem (SCM menu, SCM position, SCM look)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_menu_t *c_menu;
      int c_position = 0;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.menu,
		   menu), menu, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_integer (position), position, SCM_ARG2,
		  __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG3, __FUNCTION__);

      c_menu = lw6_scm_to_menu (menu);
      c_position = scm_to_int (position);
      c_look = lw6_scm_to_look (look);

      ret =
	lw6gfx_warp_mouse_on_menuitem (lw6_global.gfx_backend, c_look, c_menu,
				       c_position) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_center_menu (SCM menu, SCM position, SCM look)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_menu_t *c_menu;
      int c_position = 0;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.menu,
		   menu), menu, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_integer (position), position, SCM_ARG2,
		  __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG3, __FUNCTION__);

      c_menu = lw6_scm_to_menu (menu);
      c_position = scm_to_int (position);
      c_look = lw6_scm_to_look (look);

      lw6gfx_center_menu (lw6_global.gfx_backend, c_look, c_menu, c_position);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In render.c
 */
static SCM
_scm_lw6gfx_prepare_buffer (SCM look)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG1, __FUNCTION__);

      c_look = lw6_scm_to_look (look);

      lw6gfx_prepare_buffer (lw6_global.gfx_backend, c_look);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6gfx_swap_buffers ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gfx_swap_buffers (lw6_global.gfx_backend);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In setup.c
 */
static SCM
_scm_lw6gfx_init (SCM width, SCM height, SCM fullscreen, SCM ticks,
		  SCM key_repeat_delay, SCM key_repeat_interval)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_width;
      int c_height;
      int c_fullscreen;
      int c_ticks;
      int c_key_repeat_delay;
      int c_key_repeat_interval;

      SCM_ASSERT (scm_is_integer (width), width, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_integer (height), height, SCM_ARG2, __FUNCTION__);
      SCM_ASSERT (SCM_BOOLP (fullscreen), fullscreen, SCM_ARG3, __FUNCTION__);
      SCM_ASSERT (scm_is_integer (ticks), ticks, SCM_ARG4, __FUNCTION__);
      SCM_ASSERT (scm_is_integer (key_repeat_delay), ticks, SCM_ARG5,
		  __FUNCTION__);
      SCM_ASSERT (scm_is_integer (key_repeat_interval), ticks, SCM_ARG6,
		  __FUNCTION__);
      c_width = scm_to_int (width);
      c_height = scm_to_int (height);
      c_fullscreen = SCM_NFALSEP (fullscreen);
      c_ticks = scm_to_int (ticks);
      c_key_repeat_delay = scm_to_int (key_repeat_delay);
      c_key_repeat_interval = scm_to_int (key_repeat_interval);
      if (lw6gfx_init (lw6_global.gfx_backend, c_width,
		       c_height, c_fullscreen, lw6_resize_callback, c_ticks,
		       c_key_repeat_delay, c_key_repeat_interval))
	{
	  lw6_global.gfx_initialized = 1;
	  ret = SCM_BOOL_T;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_get_low_resolution ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_width;
      int c_height;

      lw6gfx_get_low_resolution (lw6_global.gfx_backend, &c_width, &c_height);
      ret =
	scm_list_2 (scm_cons
		    (scm_makfrom0str (LW6DEF_WIDTH),
		     scm_int2num (c_width)),
		    scm_cons (scm_makfrom0str (LW6DEF_HEIGHT),
			      scm_int2num (c_height)));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_get_standard_resolution ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_width;
      int c_height;

      lw6gfx_get_standard_resolution (lw6_global.gfx_backend,
				      &c_width, &c_height);
      ret =
	scm_list_2 (scm_cons
		    (scm_makfrom0str (LW6DEF_WIDTH),
		     scm_int2num (c_width)),
		    scm_cons (scm_makfrom0str (LW6DEF_HEIGHT),
			      scm_int2num (c_height)));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_get_high_resolution ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_width;
      int c_height;

      lw6gfx_get_high_resolution (lw6_global.gfx_backend,
				  &c_width, &c_height);
      ret =
	scm_list_2 (scm_cons
		    (scm_makfrom0str (LW6DEF_WIDTH),
		     scm_int2num (c_width)),
		    scm_cons (scm_makfrom0str (LW6DEF_HEIGHT),
			      scm_int2num (c_height)));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_set_video_mode (SCM width, SCM height, SCM fullscreen)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_width;
      int c_height;
      int c_fullscreen;
      SCM_ASSERT (scm_is_integer (width), width, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (scm_is_integer (height), height, SCM_ARG2, __FUNCTION__);
      c_width = scm_to_int (width);
      c_height = scm_to_int (height);
      c_fullscreen = SCM_NFALSEP (fullscreen);
      ret =
	lw6gfx_set_video_mode (lw6_global.gfx_backend,
			       c_width, c_height,
			       c_fullscreen,
			       lw6_resize_callback) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_get_video_mode ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_width;
      int c_height;
      int c_fullscreen;

      if (lw6gfx_get_video_mode (lw6_global.gfx_backend,
				 &c_width, &c_height, &c_fullscreen))
	{
	  ret =
	    scm_list_3 (scm_cons
			(scm_makfrom0str (LW6DEF_WIDTH),
			 scm_int2num (c_width)),
			scm_cons (scm_makfrom0str (LW6DEF_HEIGHT),
				  scm_int2num (c_height)),
			scm_cons (scm_makfrom0str
				  (LW6DEF_FULLSCREEN),
				  c_fullscreen ? SCM_BOOL_T : SCM_BOOL_F));
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6gfx_quit ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gfx_quit (lw6_global.gfx_backend);
    }
  lw6_global.gfx_initialized = 0;

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In timer.c
 */
static SCM
_scm_lw6gfx_delay (SCM ms)
{
  int c_ms;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      SCM_ASSERT (scm_is_integer (ms), ms, SCM_ARG1, __FUNCTION__);
      c_ms = scm_to_int (ms);
      lw6gfx_delay (lw6_global.gfx_backend, c_ms);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6gfx_get_ticks ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      ret = scm_int2num (lw6gfx_get_ticks (lw6_global.gfx_backend));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}




/*
 * In liquidwar6ldr
 */


static SCM
_scm_lw6ldr_get_entries (SCM map_path, SCM relative_path)
{
  lw6sys_list_t *c_maps;
  lw6ldr_entry_t *c_entry;
  char *c_map_path;
  char *c_relative_path;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (map_path), relative_path, SCM_ARG1,
	      __FUNCTION__);
  SCM_ASSERT (scm_is_string (relative_path), relative_path, SCM_ARG2,
	      __FUNCTION__);

  ret = SCM_LIST0;
  c_map_path = to_0str (map_path);
  if (c_map_path)
    {
      c_relative_path = to_0str (relative_path);
      if (c_relative_path)
	{
	  c_maps = lw6ldr_get_entries (c_map_path, c_relative_path);
	  if (c_maps)
	    {
	      while (!lw6sys_list_is_empty (c_maps))
		{
		  c_entry = (lw6ldr_entry_t *) lw6sys_lifo_pop (&c_maps);
		  if (c_entry)
		    {
		      ret = scm_cons (scm_list_4
				      (scm_cons
				       (scm_makfrom0str ("title"),
					scm_makfrom0str (c_entry->title)),
				       scm_cons (scm_makfrom0str
						 ("absolute-path"),
						 scm_makfrom0str
						 (c_entry->absolute_path)),
				       scm_cons (scm_makfrom0str
						 ("relative-path"),
						 scm_makfrom0str
						 (c_entry->relative_path)),
				       scm_cons (scm_makfrom0str
						 ("has-subdirs"),
						 c_entry->has_subdirs ?
						 SCM_BOOL_T : SCM_BOOL_F)),
				      ret);
		      lw6ldr_free_entry (c_entry);
		    }
		}
	      lw6sys_list_free (c_maps);
	    }
	  LW6SYS_FREE (c_relative_path);
	}
      LW6SYS_FREE (c_map_path);
    }

  ret = scm_reverse (ret);
  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ldr_read (SCM dirname, SCM default_param, SCM forced_param,
		  SCM display_width, SCM display_height)
{
  char *c_dirname;
  int c_display_width;
  int c_display_height;
  SCM ret = SCM_BOOL_F;
  lw6sys_progress_t progress;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (dirname), dirname, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (SCM_CONSP (default_param)
	      || default_param == SCM_EOL, default_param, SCM_ARG2,
	      __FUNCTION__);
  SCM_ASSERT (SCM_CONSP (forced_param)
	      || forced_param == SCM_EOL, forced_param, SCM_ARG3,
	      __FUNCTION__);
  SCM_ASSERT (scm_is_integer (display_width), display_width, SCM_ARG4,
	      __FUNCTION__);
  SCM_ASSERT (scm_is_integer (display_height), display_height, SCM_ARG5,
	      __FUNCTION__);

  progress.min = 0.0f;
  progress.max = 1.0f;
  progress.value = &(lw6_global.progress);

  c_dirname = to_0str (dirname);
  if (c_dirname)
    {
      lw6map_level_t *c_level;

      c_display_width = scm_to_int (display_width);
      c_display_height = scm_to_int (display_height);

      // todo : get options from SCM obj
      c_level =
	lw6ldr_read (c_dirname, NULL, NULL, c_display_width, c_display_height,
		     &progress);
      if (c_level)
	{
	  ret = lw6_make_scm_map (c_level);
	}
      LW6SYS_FREE (c_dirname);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ldr_read_relative (SCM map_path, SCM relative_path, SCM default_param,
			   SCM forced_param, SCM display_width,
			   SCM display_height)
{
  char *c_map_path;
  char *c_relative_path;
  int c_display_width;
  int c_display_height;
  SCM ret = SCM_BOOL_F;
  lw6sys_progress_t progress;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (map_path), map_path, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (relative_path), relative_path, SCM_ARG2,
	      __FUNCTION__);
  SCM_ASSERT (SCM_CONSP (default_param)
	      || default_param == SCM_EOL, default_param, SCM_ARG3,
	      __FUNCTION__);
  SCM_ASSERT (SCM_CONSP (forced_param)
	      || forced_param == SCM_EOL, forced_param, SCM_ARG4,
	      __FUNCTION__);
  SCM_ASSERT (scm_is_integer (display_width), display_width, SCM_ARG5,
	      __FUNCTION__);
  SCM_ASSERT (scm_is_integer (display_height), display_height, SCM_ARG6,
	      __FUNCTION__);

  progress.min = 0.0f;
  progress.max = 1.0f;
  progress.value = &(lw6_global.progress);

  c_map_path = to_0str (map_path);
  if (c_map_path)
    {
      c_relative_path = to_0str (relative_path);
      if (c_relative_path)
	{
	  lw6map_level_t *c_level;

	  c_display_width = scm_to_int (display_width);
	  c_display_height = scm_to_int (display_height);

	  // todo : get options from SCM obj
	  c_level =
	    lw6ldr_read_relative (c_map_path, c_relative_path, NULL, NULL,
				  c_display_width, c_display_height,
				  &progress);
	  if (c_level)
	    {
	      ret = lw6_make_scm_map (c_level);
	    }
	  LW6SYS_FREE (c_relative_path);
	}
      LW6SYS_FREE (c_map_path);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ldr_print_examples ()
{
  char *user_dir = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  user_dir = lw6cfg_unified_get_user_dir (lw6_global.argc, lw6_global.argv);
  if (user_dir)
    {
      lw6ldr_print_examples (user_dir);
      LW6SYS_FREE (user_dir);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6ldr_hints_get_default (SCM key)
{
  SCM ret = SCM_BOOL_F;
  char *c_key = NULL;
  char *c_value = NULL;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, __FUNCTION__);

  c_key = to_0str (key);
  if (c_key)
    {
      c_value = lw6ldr_hints_get_default (c_key);
      if (c_value)
	{
	  ret = scm_makfrom0str (c_value);
	  LW6SYS_FREE (c_value);
	}
      LW6SYS_FREE (c_key);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In liquidwar6map
 */

static SCM
_scm_lw6map_get_look (SCM level)
{
  SCM ret = SCM_BOOL_F;
  lw6map_level_t *c_level;
  lw6gui_look_t *c_look;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.map, level), level,
	      SCM_ARG1, __FUNCTION__);

  c_level = lw6_scm_to_map (level);
  if (c_level)
    {
      c_look = lw6gui_look_new (&(c_level->param.style));
      if (c_look)
	{
	  ret = lw6_make_scm_look (c_look);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6map_team_color_index_to_key (SCM index)
{
  SCM ret = SCM_BOOL_F;

  int c_index;

  SCM_ASSERT (scm_is_integer (index), index, SCM_ARG1, __FUNCTION__);

  c_index = scm_to_int (index);
  ret = scm_makfrom0str (lw6map_team_color_index_to_key (c_index));

  return ret;
}

static SCM
_scm_lw6map_team_color_key_to_index (SCM key)
{
  SCM ret = SCM_BOOL_F;

  char *c_key = NULL;

  SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, __FUNCTION__);

  c_key = to_0str (key);
  if (c_key)
    {
      ret = scm_int2num (lw6map_team_color_key_to_index (c_key));
      LW6SYS_FREE (c_key);
    }

  return ret;
}

/*
 * In liquidwar6ker
 */
static SCM
_scm_lw6ker_build_game_struct (SCM level)
{
  lw6map_level_t *c_level;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.map, level), level,
	      SCM_ARG1, __FUNCTION__);
  c_level = lw6_scm_to_map (level);
  if (c_level)
    {
      lw6ker_game_struct_t *c_game_struct;

      c_game_struct = lw6ker_game_struct_new (c_level, NULL);
      if (c_game_struct)
	{
	  ret = lw6_make_scm_game_struct (c_game_struct, level);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_build_game_state (SCM game_struct)
{
  lw6ker_game_struct_t *c_game_struct;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_struct,
	       game_struct), game_struct, SCM_ARG1, __FUNCTION__);
  c_game_struct = lw6_scm_to_game_struct (game_struct);
  if (c_game_struct)
    {
      lw6ker_game_state_t *c_game_state;
      c_game_state = lw6ker_game_state_new (c_game_struct, NULL);
      if (c_game_state)
	{
	  ret = lw6_make_scm_game_state (c_game_state, game_struct);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_sync_game_state (SCM dst, SCM src)
{
  lw6ker_game_state_t *c_dst;
  lw6ker_game_state_t *c_src;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       dst), dst, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       src), src, SCM_ARG2, __FUNCTION__);
  c_dst = lw6_scm_to_game_state (dst);
  c_src = lw6_scm_to_game_state (src);
  if (c_dst && c_src)
    {
      ret = lw6ker_game_state_sync (c_dst, c_src) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_dup_game_state (SCM game_struct, SCM game_state)
{
  lw6ker_game_struct_t *c_game_struct;
  lw6ker_game_state_t *c_game_state;
  lw6ker_game_state_t *c_ret;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_struct,
	       game_struct), game_struct, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG2, __FUNCTION__);

  c_game_struct = lw6_scm_to_game_struct (game_struct);
  if (c_game_struct)
    {
      c_game_state = lw6_scm_to_game_state (game_state);
      if (c_game_state)
	{
	  c_ret = lw6ker_game_state_new (c_game_struct, NULL);
	  if (c_ret)
	    {
	      lw6ker_game_state_sync (c_ret, c_game_state);
	      ret = lw6_make_scm_game_state (c_ret, game_struct);
	    }
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_game_struct_checksum (SCM game_struct)
{
  lw6ker_game_struct_t *c_game_struct;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_struct,
	       game_struct), game_struct, SCM_ARG1, __FUNCTION__);
  c_game_struct = lw6_scm_to_game_struct (game_struct);
  if (c_game_struct)
    {
      ret = scm_int2num (lw6ker_game_struct_checksum (c_game_struct));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_game_state_checksum (SCM game_state)
{
  lw6ker_game_state_t *c_game_state;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      ret = scm_int2num (lw6ker_game_state_checksum (c_game_state));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_register_server (SCM game_state, SCM server_id)
{
  SCM ret = SCM_BOOL_F;
  char *c_server_id_str;
  u_int64_t c_server_id_int = 0LL;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (server_id), server_id, SCM_ARG2, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_server_id_str = to_0str (server_id);
      if (c_server_id_str)
	{
	  c_server_id_int = lw6sys_id_atol (c_server_id_str);
	  LW6SYS_FREE (c_server_id_str);
	}

      if (lw6sys_check_id (c_server_id_int))
	{
	  ret =
	    lw6ker_game_state_register_server (c_game_state,
					       c_server_id_int) ? SCM_BOOL_T :
	    SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_unregister_server (SCM game_state, SCM server_id)
{
  SCM ret = SCM_BOOL_F;
  char *c_server_id_str;
  u_int64_t c_server_id_int = 0LL;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (server_id), server_id, SCM_ARG2, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_server_id_str = to_0str (server_id);
      if (c_server_id_str)
	{
	  c_server_id_int = lw6sys_id_atol (c_server_id_str);
	  LW6SYS_FREE (c_server_id_str);
	}

      if (lw6sys_check_id (c_server_id_int))
	{
	  ret =
	    lw6ker_game_state_unregister_server (c_game_state,
						 c_server_id_int) ? SCM_BOOL_T
	    : SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_server_exists (SCM game_state, SCM server_id)
{
  SCM ret = SCM_BOOL_F;
  char *c_server_id_str;
  u_int64_t c_server_id_int = 0;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (server_id), server_id, SCM_ARG2, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_server_id_str = to_0str (server_id);
      if (c_server_id_str)
	{
	  c_server_id_int = lw6sys_id_atol (c_server_id_str);
	  LW6SYS_FREE (c_server_id_str);
	}

      if (lw6sys_check_id (c_server_id_int))
	{
	  ret =
	    lw6ker_game_state_server_exists (c_game_state,
					     c_server_id_int) ? SCM_BOOL_T :
	    SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_add_cursor (SCM game_state, SCM server_id, SCM cursor_id,
			SCM team_color)
{
  SCM ret = SCM_BOOL_F;
  char *c_server_id_str;
  char *c_cursor_id_str;
  char *c_team_color_str;
  u_int64_t c_server_id_int = 0LL;
  u_int16_t c_cursor_id_int = 0;
  int c_team_color_int = LW6MAP_TEAM_COLOR_INVALID;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (server_id), server_id, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_string (cursor_id), cursor_id, SCM_ARG3, __FUNCTION__);
  SCM_ASSERT (scm_is_string (team_color), team_color, SCM_ARG4, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_server_id_str = to_0str (server_id);
      if (c_server_id_str)
	{
	  c_server_id_int = lw6sys_id_atol (c_server_id_str);
	  LW6SYS_FREE (c_server_id_str);
	}
      c_cursor_id_str = to_0str (cursor_id);
      if (c_cursor_id_str)
	{
	  c_cursor_id_int = lw6sys_id_atol (c_cursor_id_str);
	  LW6SYS_FREE (c_cursor_id_str);
	}
      c_team_color_str = to_0str (team_color);
      if (c_team_color_str)
	{
	  c_team_color_int =
	    lw6map_team_color_key_to_index (c_team_color_str);
	  LW6SYS_FREE (c_team_color_str);
	}

      if (lw6sys_check_id (c_server_id_int)
	  && lw6sys_check_id (c_cursor_id_int)
	  && c_team_color_int != LW6MAP_TEAM_COLOR_INVALID)
	{
	  ret =
	    lw6ker_game_state_add_cursor (c_game_state, c_server_id_int,
					  c_cursor_id_int,
					  c_team_color_int) ? SCM_BOOL_T :
	    SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_remove_cursor (SCM game_state, SCM server_id, SCM cursor_id)
{
  SCM ret = SCM_BOOL_F;
  char *c_server_id_str;
  char *c_cursor_id_str;
  u_int64_t c_server_id_int = 0LL;
  u_int16_t c_cursor_id_int = 0;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (server_id), server_id, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_string (cursor_id), cursor_id, SCM_ARG3, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_server_id_str = to_0str (server_id);
      if (c_server_id_str)
	{
	  c_server_id_int = lw6sys_id_atol (c_server_id_str);
	  LW6SYS_FREE (c_server_id_str);
	}
      c_cursor_id_str = to_0str (cursor_id);
      if (c_cursor_id_str)
	{
	  c_cursor_id_int = lw6sys_id_atol (c_cursor_id_str);
	  LW6SYS_FREE (c_cursor_id_str);
	}

      if (lw6sys_check_id (c_server_id_int)
	  && lw6sys_check_id (c_cursor_id_int))
	{
	  ret =
	    lw6ker_game_state_remove_cursor (c_game_state, c_server_id_int,
					     c_cursor_id_int) ? SCM_BOOL_T :
	    SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_cursor_exists (SCM game_state, SCM cursor_id)
{
  SCM ret = SCM_BOOL_F;
  char *c_cursor_id_str;
  u_int16_t c_cursor_id_int = 0;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (cursor_id), cursor_id, SCM_ARG2, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_cursor_id_str = to_0str (cursor_id);
      if (c_cursor_id_str)
	{
	  c_cursor_id_int = lw6sys_id_atol (c_cursor_id_str);
	  LW6SYS_FREE (c_cursor_id_str);
	}

      if (lw6sys_check_id (c_cursor_id_int))
	{
	  ret =
	    lw6ker_game_state_cursor_exists (c_game_state,
					     c_cursor_id_int) ? SCM_BOOL_T :
	    SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_get_cursor_info (SCM game_state, SCM cursor_id)
{
  SCM ret = SCM_BOOL_F;
  char *c_cursor_id_str;
  u_int16_t c_cursor_id_int = 0;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (cursor_id), cursor_id, SCM_ARG2, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_cursor_id_str = to_0str (cursor_id);
      if (c_cursor_id_str)
	{
	  c_cursor_id_int = lw6sys_id_atol (c_cursor_id_str);
	  LW6SYS_FREE (c_cursor_id_str);
	}

      if (lw6sys_check_id (c_cursor_id_int))
	{
	  int c_team_color_int = 0;
	  char *c_team_color_str = NULL;
	  u_int64_t c_server_id_int = 0LL;
	  char *c_server_id_str = NULL;
	  char c_letter_char = '\0';
	  char c_letter_str[2] = { 0, 0 };
	  int32_t c_x = 0;
	  int32_t c_y = 0;

	  if (lw6ker_game_state_get_cursor_info (c_game_state,
						 c_cursor_id_int,
						 &c_server_id_int,
						 &c_letter_char,
						 &c_team_color_int, &c_x,
						 &c_y))
	    {
	      c_team_color_str =
		lw6map_team_color_index_to_key (c_team_color_int);
	      if (c_team_color_str)
		{
		  c_server_id_str = lw6sys_id_ltoa (c_server_id_int);
		  if (c_server_id_str)
		    {
		      c_letter_str[0] = c_letter_char;
		      ret = scm_list_5 (scm_cons
					(scm_makfrom0str ("server-id"),
					 scm_makfrom0str (c_server_id_str)),
					scm_cons
					(scm_makfrom0str ("letter"),
					 scm_makfrom0str (c_letter_str)),
					scm_cons
					(scm_makfrom0str ("team-color"),
					 scm_makfrom0str (c_team_color_str)),
					scm_cons (scm_makfrom0str ("x"),
						  scm_int2num (c_x)),
					scm_cons (scm_makfrom0str ("y"),
						  scm_int2num (c_y)));
		      LW6SYS_FREE (c_server_id_str);
		    }
		}
	    }
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_set_cursor (SCM game_state, SCM server_id, SCM cursor_id, SCM x,
			SCM y)
{
  SCM ret = SCM_BOOL_F;
  char *c_server_id_str;
  char *c_cursor_id_str;
  u_int64_t c_server_id_int = 0LL;
  u_int16_t c_cursor_id_int = 0;
  int32_t c_x = 0;
  int32_t c_y = 0;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (server_id), server_id, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_string (cursor_id), cursor_id, SCM_ARG3, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (x), x, SCM_ARG4, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (y), y, SCM_ARG5, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_server_id_str = to_0str (server_id);
      if (c_server_id_str)
	{
	  c_server_id_int = lw6sys_id_atol (c_server_id_str);
	  LW6SYS_FREE (c_server_id_str);
	}
      c_cursor_id_str = to_0str (cursor_id);
      if (c_cursor_id_str)
	{
	  c_cursor_id_int = lw6sys_id_atol (c_cursor_id_str);
	  LW6SYS_FREE (c_cursor_id_str);
	}
      c_x = scm_to_int (x);
      c_y = scm_to_int (y);

      if (lw6sys_check_id (c_server_id_int)
	  && lw6sys_check_id (c_cursor_id_int))
	{
	  ret =
	    lw6ker_game_state_set_cursor (c_game_state, c_server_id_int,
					  c_cursor_id_int, c_x,
					  c_y) ? SCM_BOOL_T : SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_do_round (SCM game_state)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, __FUNCTION__);
      c_game_state = lw6_scm_to_game_state (game_state);
      if (c_game_state)
	{
	  lw6ker_game_state_do_round (c_game_state);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6ker_get_moves (SCM game_state)
{
  SCM ret = SCM_BOOL_F;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      ret = scm_int2num (lw6ker_game_state_get_moves (c_game_state));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_get_spreads (SCM game_state)
{
  SCM ret = SCM_BOOL_F;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      ret = scm_int2num (lw6ker_game_state_get_spreads (c_game_state));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6ker_get_rounds (SCM game_state)
{
  SCM ret = SCM_BOOL_F;
  lw6ker_game_state_t *c_game_state;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      ret = scm_int2num (lw6ker_game_state_get_rounds (c_game_state));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In liquidwar6pil
 */
static SCM
_scm_lw6pil_build_pilot (SCM game_state, SCM sleep, SCM ticks)
{
  lw6ker_game_state_t *c_game_state;
  float c_sleep;
  int c_ticks;
  lw6pil_pilot_t *c_ret;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_number (sleep), sleep, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (ticks), ticks, SCM_ARG3, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_sleep = scm_to_double (sleep);
      c_ticks = scm_to_int (ticks);
      c_ret = lw6pil_pilot_new (c_game_state, c_sleep, c_ticks);
      if (c_ret)
	{
	  ret = lw6_make_scm_pilot (c_ret);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_send_command (SCM pilot, SCM command_text, SCM verified)
{
  lw6pil_pilot_t *c_pilot = NULL;
  char *c_command_text = NULL;
  int c_verified;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (command_text), command_text, SCM_ARG2,
	      __FUNCTION__);
  SCM_ASSERT (SCM_BOOLP (verified), verified, SCM_ARG3, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      c_command_text = to_0str (command_text);
      if (c_command_text)
	{
	  c_verified = SCM_NFALSEP (verified);
	  ret =
	    lw6pil_pilot_send_command (c_pilot, c_command_text,
				       c_verified) ? SCM_BOOL_T : SCM_BOOL_F;
	  LW6SYS_FREE (c_command_text);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_commit (SCM pilot)
{
  lw6pil_pilot_t *c_pilot = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      ret = lw6pil_pilot_commit (c_pilot) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_fix_coords (SCM game_state, SCM x, SCM y, SCM z)
{
  lw6ker_game_state_t *c_game_state = NULL;
  float c_x = 0.0f;
  float c_y = 0.0f;
  float c_z = 0.0f;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_number (x), x, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_number (y), y, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_number (z), z, SCM_ARG3, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_x = scm_to_double (x);
      c_y = scm_to_double (y);
      c_z = scm_to_double (z);

      lw6pil_coords_fix (&(c_game_state->game_struct->rules),
			 &(c_game_state->map_state.shape), &c_x, &c_y, &c_z);
      ret =
	scm_list_3 (scm_cons (scm_makfrom0str ("x"), scm_double2num (c_x)),
		    scm_cons (scm_makfrom0str ("y"), scm_double2num (c_y)),
		    scm_cons (scm_makfrom0str ("z"), scm_double2num (c_z)));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_execute_command (SCM game_state, SCM command_text)
{
  lw6ker_game_state_t *c_game_state = NULL;
  char *c_command_text = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (command_text), command_text, SCM_ARG2,
	      __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_command_text = to_0str (command_text);
      if (c_command_text)
	{
	  ret =
	    lw6pil_command_execute_text (c_game_state,
					 c_command_text) ? SCM_BOOL_T :
	    SCM_BOOL_F;
	  LW6SYS_FREE (c_command_text);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_make_backup (SCM pilot)
{
  lw6pil_pilot_t *c_pilot = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      ret = lw6pil_pilot_make_backup (c_pilot) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_sync_from_backup (SCM target, SCM pilot)
{
  lw6ker_game_state_t *c_target = NULL;
  lw6pil_pilot_t *c_pilot = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       target), target, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG2, __FUNCTION__);

  c_target = lw6_scm_to_game_state (target);
  if (c_target)
    {
      c_pilot = lw6_scm_to_pilot (pilot);
      if (c_pilot)
	{
	  ret =
	    lw6pil_pilot_sync_from_backup (c_target,
					   c_pilot) ? SCM_BOOL_T : SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_sync_from_reference (SCM target, SCM pilot)
{
  lw6ker_game_state_t *c_target = NULL;
  lw6pil_pilot_t *c_pilot = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       target), target, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG2, __FUNCTION__);

  c_target = lw6_scm_to_game_state (target);
  if (c_target)
    {
      c_pilot = lw6_scm_to_pilot (pilot);
      if (c_pilot)
	{
	  ret =
	    lw6pil_pilot_sync_from_reference (c_target,
					      c_pilot) ? SCM_BOOL_T :
	    SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_sync_from_draft (SCM target, SCM pilot)
{
  lw6ker_game_state_t *c_target = NULL;
  lw6pil_pilot_t *c_pilot = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       target), target, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG2, __FUNCTION__);

  c_target = lw6_scm_to_game_state (target);
  if (c_target)
    {
      c_pilot = lw6_scm_to_pilot (pilot);
      if (c_pilot)
	{
	  ret =
	    lw6pil_pilot_sync_from_draft (c_target,
					  c_pilot) ? SCM_BOOL_T : SCM_BOOL_F;
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_calibrate (SCM pilot, SCM ticks, SCM round)
{
  lw6pil_pilot_t *c_pilot = NULL;
  int c_ticks;
  int c_round;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (ticks), ticks, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (round), round, SCM_ARG3, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      c_ticks = scm_to_int (ticks);
      c_round = scm_to_int (round);

      lw6pil_pilot_calibrate (c_pilot, c_ticks, c_round);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6pil_speed_up (SCM pilot, SCM round_inc)
{
  lw6pil_pilot_t *c_pilot = NULL;
  int c_round_inc;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (round_inc), round_inc, SCM_ARG2, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      c_round_inc = scm_to_int (round_inc);

      lw6pil_pilot_speed_up (c_pilot, c_round_inc);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6pil_slow_down (SCM pilot, SCM round_dec)
{
  lw6pil_pilot_t *c_pilot = NULL;
  int c_round_dec;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (round_dec), round_dec, SCM_ARG2, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      c_round_dec = scm_to_int (round_dec);

      lw6pil_pilot_slow_down (c_pilot, c_round_dec);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6pil_get_next_round (SCM pilot, SCM ticks)
{
  lw6pil_pilot_t *c_pilot = NULL;
  int c_ticks;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (ticks), ticks, SCM_ARG2, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      c_ticks = scm_to_int (ticks);

      ret = scm_int2num (lw6pil_pilot_get_next_round (c_pilot, c_ticks));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_get_last_commit_round (SCM pilot)
{
  lw6pil_pilot_t *c_pilot = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      ret = scm_int2num (lw6pil_pilot_get_last_commit_round (c_pilot));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_get_reference_current_round (SCM pilot)
{
  lw6pil_pilot_t *c_pilot = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      ret = scm_int2num (lw6pil_pilot_get_reference_current_round (c_pilot));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_get_reference_target_round (SCM pilot)
{
  lw6pil_pilot_t *c_pilot = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      ret = scm_int2num (lw6pil_pilot_get_reference_target_round (c_pilot));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6pil_get_max_round (SCM pilot)
{
  lw6pil_pilot_t *c_pilot = NULL;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.pilot,
	       pilot), pilot, SCM_ARG1, __FUNCTION__);

  c_pilot = lw6_scm_to_pilot (pilot);
  if (c_pilot)
    {
      ret = scm_int2num (lw6pil_pilot_get_max_round (c_pilot));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In liquidwar6snd
 */

/*
 * In backend.c
 */
static SCM
_scm_lw6snd_get_backends ()
{
  SCM ret = SCM_BOOL_F;
  lw6sys_assoc_t *backends;
  lw6sys_list_t *keys;
  lw6sys_list_t *key;
  char *module_id;
  char *module_name;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  backends = lw6snd_get_backends (lw6_global.argc, lw6_global.argv);
  if (backends)
    {
      keys = lw6sys_assoc_keys (backends);
      if (keys)
	{
	  ret = SCM_EOL;
	  key = keys;
	  while (key)
	    {
	      if (key->data)
		{
		  module_id = (char *) key->data;
		  module_name =
		    (char *) lw6sys_assoc_get (backends, module_id);
		  ret =
		    scm_cons (scm_cons
			      (scm_makfrom0str (module_id),
			       scm_makfrom0str (module_name)), ret);
		}
	      key = lw6sys_list_next (key);
	    }
	  lw6sys_list_free (keys);
	  ret = scm_reverse (ret);
	}
      lw6sys_assoc_free (backends);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6snd_create_backend (SCM backend_name)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.snd_backend == NULL)
    {
      char *c_backend_name;

      SCM_ASSERT (scm_is_string (backend_name),
		  backend_name, SCM_ARG1, __FUNCTION__);

      c_backend_name = to_0str (backend_name);
      if (c_backend_name)
	{
	  lw6_global.snd_backend =
	    lw6snd_create_backend (lw6_global.argc, lw6_global.argv,
				   c_backend_name);
	  LW6SYS_FREE (c_backend_name);
	}
    }

  if (lw6_global.snd_backend)
    {
      ret = SCM_BOOL_T;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6snd_destroy_backend ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.snd_backend != NULL)
    {
      if (lw6_global.snd_initialized)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _
		      ("destroying snd backend, but backend has not been previously cleaned up"));
	  lw6snd_quit (lw6_global.snd_backend);
	}

      lw6snd_destroy_backend (lw6_global.snd_backend);
    }
  lw6_global.snd_backend = NULL;
  lw6_global.snd_initialized = 0;

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In setup.c
 */
static SCM
_scm_lw6snd_init (SCM sound_volume, SCM music_volume)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_REALP (sound_volume), sound_volume, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (SCM_REALP (music_volume), music_volume, SCM_ARG2, __FUNCTION__);

  if (lw6_global.snd_backend != NULL)
    {
      float c_sound_volume;
      float c_music_volume;

      c_sound_volume = scm_num2float (sound_volume, 0, NULL);
      c_music_volume = scm_num2float (music_volume, 0, NULL);

      lw6snd_init (lw6_global.snd_backend, c_sound_volume, c_music_volume);
      lw6_global.snd_initialized = 1;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6snd_quit ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.snd_backend != NULL)
    {
      lw6snd_quit (lw6_global.snd_backend);
    }
  lw6_global.snd_initialized = 0;

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6snd_play_sound (SCM sound_id)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_integer (sound_id), sound_id, SCM_ARG1, __FUNCTION__);

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      int c_sound_id;

      c_sound_id = scm_to_int (sound_id);

      ret =
	scm_int2num (lw6snd_play_sound (lw6_global.snd_backend, c_sound_id));
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6snd_set_sound_volume (SCM sound_volume)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_REALP (sound_volume), sound_volume, SCM_ARG1, __FUNCTION__);

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      float c_sound_volume;

      c_sound_volume = scm_num2float (sound_volume, 0, NULL);
      lw6snd_set_sound_volume (lw6_global.snd_backend, c_sound_volume);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6snd_play_music (SCM music_name)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (music_name), music_name, SCM_ARG1, __FUNCTION__);

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      char *c_music_name;

      c_music_name = to_0str (music_name);
      if (c_music_name)
	{
	  ret =
	    scm_int2num (lw6snd_play_music
			 (lw6_global.snd_backend, c_music_name));
	  LW6SYS_FREE (c_music_name);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6snd_set_music_volume (SCM music_volume)
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_REALP (music_volume), music_volume, SCM_ARG1, __FUNCTION__);

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      float c_music_volume;

      c_music_volume = scm_num2float (music_volume, 0, NULL);
      lw6snd_set_music_volume (lw6_global.snd_backend, c_music_volume);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In liquidwar6cns
 */

/*
 * In support.c
 */
static SCM
_scm_lw6cns_support ()
{
  SCM ret = SCM_BOOL_F;
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  ret = lw6cns_support ()? SCM_BOOL_T : SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In backend.c
 */
static SCM
_scm_lw6cns_init ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (!lw6_global.cns_initialized)
    {
      lw6cns_handler_install (lw6_cns_handler);
      lw6_global.cns_initialized = 1;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6cns_quit ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.cns_initialized)
    {
      lw6cns_handler_remove ();
      lw6_global.cns_initialized = 0;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6cns_poll ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.cns_initialized)
    {
      lw6cns_handler_poll ();
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In liquidwar6tsk
 */
static SCM
_scm_lw6tsk_loader_new (SCM sleep)
{
  float c_sleep;
  lw6tsk_loader_t *c_loader;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_number (sleep), sleep, SCM_ARG1, __FUNCTION__);

  c_sleep = scm_to_double (sleep);
  c_loader = lw6tsk_loader_new (c_sleep, &(lw6_global.progress));
  if (c_loader)
    {
      ret = lw6_make_scm_loader (c_loader);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6tsk_loader_push (SCM loader, SCM map_path, SCM relative_path,
			 SCM default_param, SCM forced_param,
			 SCM display_width, SCM display_height)
{
  lw6tsk_loader_t *c_loader;
  char *c_map_path;
  char *c_relative_path;
  lw6sys_assoc_t *c_default_param;
  lw6sys_assoc_t *c_forced_param;
  int c_display_width;
  int c_display_height;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.loader,
	       loader), loader, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (map_path), map_path, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_string (relative_path), relative_path, SCM_ARG3,
	      __FUNCTION__);
  SCM_ASSERT (SCM_CONSP (default_param)
	      || default_param == SCM_EOL, default_param, SCM_ARG4,
	      __FUNCTION__);
  SCM_ASSERT (SCM_CONSP (forced_param)
	      || forced_param == SCM_EOL, forced_param, SCM_ARG5,
	      __FUNCTION__);
  SCM_ASSERT (scm_is_integer (display_width), display_width, SCM_ARG6,
	      __FUNCTION__);
  SCM_ASSERT (scm_is_integer (display_height), display_height, SCM_ARG7,
	      __FUNCTION__);

  c_loader = lw6_scm_to_loader (loader);
  c_map_path = to_0str (map_path);
  if (c_map_path)
    {
      c_relative_path = to_0str (relative_path);
      if (c_relative_path)
	{
	  c_default_param = to_sys_str_assoc (default_param);
	  if (c_default_param)
	    {
	      c_forced_param = to_sys_str_assoc (forced_param);
	      if (c_forced_param)
		{
		  c_display_width = scm_to_int (display_width);
		  c_display_height = scm_to_int (display_height);

		  lw6tsk_loader_push (c_loader, c_map_path, c_relative_path,
				      c_default_param, c_forced_param,
				      c_display_width, c_display_height);
		  ret = SCM_BOOL_T;

		  lw6sys_assoc_free (c_forced_param);
		}
	      lw6sys_assoc_free (c_default_param);
	    }
	  LW6SYS_FREE (c_relative_path);
	}
      LW6SYS_FREE (c_map_path);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6tsk_loader_pop (SCM loader)
{
  lw6tsk_loader_t *c_loader;
  lw6map_level_t *c_level = NULL;
  lw6ker_game_struct_t *c_game_struct = NULL;
  lw6ker_game_state_t *c_game_state = NULL;
  SCM level = SCM_BOOL_F;
  SCM game_struct = SCM_BOOL_F;
  SCM game_state = SCM_BOOL_F;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.loader,
	       loader), loader, SCM_ARG1, __FUNCTION__);

  c_loader = lw6_scm_to_loader (loader);
  if (c_loader)
    {
      if (lw6tsk_loader_pop
	  (&c_level, &c_game_struct, &c_game_state, c_loader))
	{
	  if (c_level)
	    {
	      level = lw6_make_scm_map (c_level);
	      if (c_game_struct)
		{
		  game_struct =
		    lw6_make_scm_game_struct (c_game_struct, level);
		  if (c_game_state)
		    {
		      game_state =
			lw6_make_scm_game_state (c_game_state, game_struct);
		    }
		}
	    }
	  ret =
	    scm_list_3 (scm_cons
			(scm_makfrom0str ("level"), level),
			scm_cons
			(scm_makfrom0str ("game-struct"), game_struct),
			scm_cons (scm_makfrom0str ("game-state"),
				  game_state));
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6tsk_loader_get_stage (SCM loader)
{
  lw6tsk_loader_t *c_loader;
  int c_ret = 0;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.loader,
	       loader), loader, SCM_ARG1, __FUNCTION__);

  c_loader = lw6_scm_to_loader (loader);
  if (c_loader)
    {
      c_ret = lw6tsk_loader_get_stage (c_loader);
      ret = scm_int2num (c_ret);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In liquidwar6net
 */

/*
 * In setup.c
 */
static SCM
_scm_lw6net_init ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6net_init (lw6_global.argc, lw6_global.argv))
    {
      lw6_global.net_initialized = 1;
      ret = SCM_BOOL_F;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6net_quit ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.net_initialized)
    {
      lw6net_quit ();
    }
  lw6_global.net_initialized = 0;

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

static SCM
_scm_lw6net_server_start (SCM ip, SCM port)
{
  SCM ret = SCM_BOOL_F;
  char *c_ip;
  int c_port;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (scm_is_string (ip), ip, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (port), port, SCM_ARG2, __FUNCTION__);

  if (lw6_global.net_initialized)
    {
      c_ip = to_0str (ip);
      if (c_ip)
	{
	  c_port = scm_to_int (port);
	  /*
	     ret =
	     lw6net_server_start (lw6_global.net_context, c_ip,
	     c_port) ? SCM_BOOL_T : SCM_BOOL_F;
	   */
	  ret = SCM_BOOL_T;

	  LW6SYS_FREE (c_ip);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6net_server_poll_idle ()
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.net_initialized)
    {
      /*
         ret =
         lw6net_server_poll (lw6_global.net_context,
         NULL, NULL) ? SCM_BOOL_T : SCM_BOOL_F;
       */
      ret = SCM_BOOL_T;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6net_server_poll_playing (SCM level, SCM game_state)
{
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.net_initialized)
    {
      lw6map_level_t *c_level;
      lw6ker_game_state_t *c_game_state;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map,
		   level), level, SCM_ARG1, __FUNCTION__);
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG2, __FUNCTION__);

      c_level = lw6_scm_to_map (level);
      c_game_state = lw6_scm_to_game_state (game_state);

      /*
         ret =
         lw6net_server_poll (lw6_global.net_context,
         c_level, c_game_state) ? SCM_BOOL_T : SCM_BOOL_F;
       */
      ret = SCM_BOOL_T;
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6net_server_stop ()
{
  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  if (lw6_global.net_initialized)
    {
      /*
         lw6net_server_stop (lw6_global.net_context);
       */
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return SCM_UNDEFINED;
}

/*
 * In liquidwar6cli
 */

/*
 * In backend.c
 */
static SCM
_scm_lw6cli_get_backends ()
{
  SCM ret = SCM_BOOL_F;
  lw6sys_assoc_t *backends;
  lw6sys_list_t *keys;
  lw6sys_list_t *key;
  char *module_id;
  char *module_name;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  backends = lw6cli_get_backends (lw6_global.argc, lw6_global.argv);
  if (backends)
    {
      keys = lw6sys_assoc_keys (backends);
      if (keys)
	{
	  ret = SCM_EOL;
	  key = keys;
	  while (key)
	    {
	      if (key->data)
		{
		  module_id = (char *) key->data;
		  module_name =
		    (char *) lw6sys_assoc_get (backends, module_id);
		  ret =
		    scm_cons (scm_cons
			      (scm_makfrom0str (module_id),
			       scm_makfrom0str (module_name)), ret);
		}
	      key = lw6sys_list_next (key);
	    }
	  lw6sys_list_free (keys);
	  ret = scm_reverse (ret);
	}
      lw6sys_assoc_free (backends);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In liquidwar6srv
 */

/*
 * In backend.c
 */
static SCM
_scm_lw6srv_get_backends ()
{
  SCM ret = SCM_BOOL_F;
  lw6sys_assoc_t *backends;
  lw6sys_list_t *keys;
  lw6sys_list_t *key;
  char *module_id;
  char *module_name;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  backends = lw6srv_get_backends (lw6_global.argc, lw6_global.argv);
  if (backends)
    {
      keys = lw6sys_assoc_keys (backends);
      if (keys)
	{
	  ret = SCM_EOL;
	  key = keys;
	  while (key)
	    {
	      if (key->data)
		{
		  module_id = (char *) key->data;
		  module_name =
		    (char *) lw6sys_assoc_get (backends, module_id);
		  ret =
		    scm_cons (scm_cons
			      (scm_makfrom0str (module_id),
			       scm_makfrom0str (module_name)), ret);
		}
	      key = lw6sys_list_next (key);
	    }
	  lw6sys_list_free (keys);
	  ret = scm_reverse (ret);
	}
      lw6sys_assoc_free (backends);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * In liquidwar6bot
 */

/*
 * In backend.c
 */
static SCM
_scm_lw6bot_get_backends ()
{
  SCM ret = SCM_BOOL_F;
  lw6sys_assoc_t *backends;
  lw6sys_list_t *keys;
  lw6sys_list_t *key;
  char *module_id;
  char *module_name;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  backends = lw6bot_get_backends (lw6_global.argc, lw6_global.argv);
  if (backends)
    {
      keys = lw6sys_assoc_keys (backends);
      if (keys)
	{
	  ret = SCM_EOL;
	  key = keys;
	  while (key)
	    {
	      if (key->data)
		{
		  module_id = (char *) key->data;
		  module_name =
		    (char *) lw6sys_assoc_get (backends, module_id);
		  ret =
		    scm_cons (scm_cons
			      (scm_makfrom0str (module_id),
			       scm_makfrom0str (module_name)), ret);
		}
	      key = lw6sys_list_next (key);
	    }
	  lw6sys_list_free (keys);
	  ret = scm_reverse (ret);
	}
      lw6sys_assoc_free (backends);
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6bot_build (SCM game_state, SCM cursor_id, SCM backend_name, SCM speed,
		   SCM iq)
{
  lw6ker_game_state_t *c_game_state;
  char *c_backend_name;
  char *c_cursor_id_str = NULL;
  u_int16_t c_cursor_id_int;
  float c_speed;
  int c_iq;
  lw6bot_backend_t *c_ret;
  lw6bot_param_t c_param;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, __FUNCTION__);
  SCM_ASSERT (scm_is_string (cursor_id), cursor_id, SCM_ARG2, __FUNCTION__);
  SCM_ASSERT (scm_is_string (backend_name), backend_name, SCM_ARG3,
	      __FUNCTION__);
  SCM_ASSERT (scm_is_number (speed), speed, SCM_ARG4, __FUNCTION__);
  SCM_ASSERT (scm_is_integer (iq), iq, SCM_ARG5, __FUNCTION__);

  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      c_backend_name = to_0str (backend_name);
      if (backend_name)
	{
	  c_cursor_id_str = to_0str (cursor_id);
	  if (c_cursor_id_str)
	    {
	      c_cursor_id_int = lw6sys_id_atol (c_cursor_id_str);
	      c_speed = scm_to_double (speed);
	      c_iq = scm_to_int (iq);
	      c_ret =
		lw6bot_create_backend (lw6_global.argc, lw6_global.argv,
				       c_backend_name);
	      if (c_ret)
		{
		  memset (&c_param, 0, sizeof (lw6bot_param_t));
		  c_param.game_state = c_game_state;
		  c_param.cursor_id = c_cursor_id_int;
		  c_param.speed = c_speed;
		  c_param.iq = c_iq;
		  if (lw6bot_init (c_ret, &c_param))
		    {
		      ret = lw6_make_scm_bot (c_ret, game_state);
		    }
		  else
		    {
		      lw6bot_destroy_backend (c_ret);
		    }
		}
	      LW6SYS_FREE (c_cursor_id_str);
	    }
	  LW6SYS_FREE (c_backend_name);
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

static SCM
_scm_lw6bot_next_move (SCM bot)
{
  lw6bot_backend_t *c_bot;
  int c_x = 0;
  int c_y = 0;
  SCM ret = SCM_BOOL_F;

  LW6SYS_SCRIPT_FUNCTION_BEGIN;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.bot, bot), bot, SCM_ARG1, __FUNCTION__);

  c_bot = lw6_scm_to_bot (bot);
  if (c_bot)
    {
      if (lw6bot_next_move (c_bot, &c_x, &c_y))
	{
	  ret = scm_list_2 (scm_cons
			    (scm_makfrom0str ("x"), scm_int2num (c_x)),
			    scm_cons (scm_makfrom0str ("y"),
				      scm_int2num (c_y)));
	}
    }

  LW6SYS_SCRIPT_FUNCTION_END;

  return ret;
}

/*
 * Register all the functions, make them callable from Guile.
 */
int
lw6_register_funcs ()
{
  int ret = 1;
  scm_c_define_gsubr ("c-gettext", 1, 0, 0, (SCM (*)())_scm_gettext);
  scm_c_define_gsubr ("_", 1, 0, 0, (SCM (*)())_scm_gettext);

  /*
   * In liquidwar6sys
   */
  /*
   * In bazooka.c
   */
  scm_c_define_gsubr ("c-lw6sys-set-memory-bazooka-size", 1, 0, 0,
		      (SCM (*)())_scm_lw6sys_set_memory_bazooka_size);
  scm_c_define_gsubr ("c-lw6sys-set-memory-bazooka-eraser", 1, 0, 0,
		      (SCM (*)())_scm_lw6sys_set_memory_bazooka_eraser);
  /*
   * In build.c
   */
  scm_c_define_gsubr ("c-lw6sys-build-get-package-tarname", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_package_tarname);
  scm_c_define_gsubr ("c-lw6sys-build-get-package-name", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_package_name);
  scm_c_define_gsubr ("c-lw6sys-build-get-package-string", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_package_string);
  scm_c_define_gsubr ("c-lw6sys-build-get-version", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_version);
  scm_c_define_gsubr ("c-lw6sys-build-get-codename", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_codename);
  scm_c_define_gsubr ("c-lw6sys-build-get-stamp", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_stamp);
  scm_c_define_gsubr ("c-lw6sys-build-get-md5sum", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_md5sum);
  scm_c_define_gsubr ("c-lw6sys-build-get-copyright", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_copyright);
  scm_c_define_gsubr ("c-lw6sys-build-get-license", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_license);
  scm_c_define_gsubr ("c-lw6sys-build-get-configure-args", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_configure_args);
  scm_c_define_gsubr ("c-lw6sys-build-get-gcc-version", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_gcc_version);
  scm_c_define_gsubr ("c-lw6sys-build-get-cflags", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_cflags);
  scm_c_define_gsubr ("c-lw6sys-build-get-ldflags", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_ldflags);
  scm_c_define_gsubr ("c-lw6sys-build-get-hostname", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_hostname);
  scm_c_define_gsubr ("c-lw6sys-build-get-date", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_date);
  scm_c_define_gsubr ("c-lw6sys-build-get-time", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_time);
  scm_c_define_gsubr ("c-lw6sys-build-get-target-cpu", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_target_cpu);
  scm_c_define_gsubr ("c-lw6sys-build-get-endianness", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_endianness);
  scm_c_define_gsubr ("c-lw6sys-build-get-pointer-size", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_pointer_size);
  scm_c_define_gsubr ("c-lw6sys-build-get-target-os", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_target_os);
  scm_c_define_gsubr ("c-lw6sys-build-is-ms-windows", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_is_ms_windows);
  scm_c_define_gsubr ("c-lw6sys-build-get-top-srcdir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_top_srcdir);
  scm_c_define_gsubr ("c-lw6sys-build-get-prefix", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_prefix);
  scm_c_define_gsubr ("c-lw6sys-build-get-datadir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_datadir);
  scm_c_define_gsubr ("c-lw6sys-build-get-libdir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_libdir);
  scm_c_define_gsubr ("c-lw6sys-build-get-includedir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_includedir);
  scm_c_define_gsubr ("c-lw6sys-build-get-localedir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_localedir);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-console", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_console);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-mod-gl", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_mod_gl);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-mod-csound", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_mod_csound);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-mod-ogg", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_mod_ogg);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-mod-http", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_mod_http);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-optimize", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_optimize);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-allinone", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_allinone);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-fullstatic", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_fullstatic);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-gprof", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_gprof);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-gcov", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_gcov);
  scm_c_define_gsubr ("c-lw6sys-build-get-enable-valgrind", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_build_get_enable_valgrind);
  /*
   * In debug.c
   */
  scm_c_define_gsubr ("c-lw6sys-debug-get", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_debug_get);
  scm_c_define_gsubr ("c-lw6sys-debug-set", 1, 0, 0,
		      (SCM (*)())_scm_lw6sys_debug_set);
  /*
   * In dump.c
   */
  scm_c_define_gsubr ("c-lw6sys-dump", 1, 0, 0, (SCM (*)())_scm_lw6sys_dump);
  scm_c_define_gsubr ("c-lw6sys-dump-clear", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_dump_clear);
  /*
   * In env.c
   */
  scm_c_define_gsubr ("c-lw6sys-get-username", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_username);
  scm_c_define_gsubr ("c-lw6sys-get-hostname", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_hostname);

  /*
   * In id.c
   */
  scm_c_define_gsubr ("c-lw6sys-generate-id-16", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_generate_id_16);
  scm_c_define_gsubr ("c-lw6sys-generate-id-32", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_generate_id_32);
  scm_c_define_gsubr ("c-lw6sys-generate-id-64", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_generate_id_64);

  /*
   * In log.c
   */
  scm_c_define_gsubr ("c-lw6sys-log", 2, 0, 0, (SCM (*)())_scm_lw6sys_log);
  scm_c_define_gsubr ("c-lw6sys-log-get-level", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_log_get_level);
  scm_c_define_gsubr ("c-lw6sys-log-set-level", 1, 0, 0,
		      (SCM (*)())_scm_lw6sys_log_set_level);
  /*
   * In options.c
   */
  scm_c_define_gsubr ("c-lw6sys-get-default-user-dir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_default_user_dir);
  scm_c_define_gsubr ("c-lw6sys-get-default-config-file", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_default_config_file);
  scm_c_define_gsubr ("c-lw6sys-get-default-log-file", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_default_log_file);
  scm_c_define_gsubr ("c-lw6sys-get-default-prefix", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_default_prefix);
  scm_c_define_gsubr ("c-lw6sys-get-default-mod-dir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_default_mod_dir);
  scm_c_define_gsubr ("c-lw6sys-get-default-data-dir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_default_data_dir);
  scm_c_define_gsubr ("c-lw6sys-get-default-map-dir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_default_map_dir);
  scm_c_define_gsubr ("c-lw6sys-get-default-map-path", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_default_map_path);
  scm_c_define_gsubr ("c-lw6sys-get-default-script-file", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_default_script_file);
  scm_c_define_gsubr ("c-lw6sys-get-cwd", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_cwd);
  scm_c_define_gsubr ("c-lw6sys-get-run-dir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_run_dir);
  scm_c_define_gsubr ("c-lw6sys-get-user-dir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_user_dir);
  scm_c_define_gsubr ("c-lw6sys-get-config-file", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_config_file);
  scm_c_define_gsubr ("c-lw6sys-get-log-file", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_log_file);
  scm_c_define_gsubr ("c-lw6sys-get-prefix", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_prefix);
  scm_c_define_gsubr ("c-lw6sys-get-mod-dir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_mod_dir);
  scm_c_define_gsubr ("c-lw6sys-get-data-dir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_data_dir);
  scm_c_define_gsubr ("c-lw6sys-get-map-dir", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_map_dir);
  scm_c_define_gsubr ("c-lw6sys-get-map-path", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_map_path);
  scm_c_define_gsubr ("c-lw6sys-get-script-file", 0, 0, 0,
		      (SCM (*)())_scm_lw6sys_get_script_file);
  /*
   * In path.c
   */
  scm_c_define_gsubr ("c-lw6sys-path-concat", 2, 0, 0,
		      (SCM (*)())_scm_lw6sys_path_concat);
  scm_c_define_gsubr ("c-lw6sys-path-parent", 1, 0, 0,
		      (SCM (*)())_scm_lw6sys_path_parent);
  scm_c_define_gsubr ("c-lw6sys-path-split", 1, 0, 0,
		      (SCM (*)())_scm_lw6sys_path_split);
  /*
   * In time.c
   */
  scm_c_define_gsubr ("c-lw6sys-timestamp",
		      0, 0, 0, (SCM (*)())_scm_lw6sys_timestamp);
  scm_c_define_gsubr ("c-lw6sys-sleep", 1, 0, 0,
		      (SCM (*)())_scm_lw6sys_sleep);

  /*
   * In liquidwar6hlp
   */
  scm_c_define_gsubr ("c-lw6hlp-list-quick",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_quick);
  scm_c_define_gsubr ("c-lw6hlp-list-doc",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_doc);
  scm_c_define_gsubr ("c-lw6hlp-list-show",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_show);
  scm_c_define_gsubr ("c-lw6hlp-list-path",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_path);
  scm_c_define_gsubr ("c-lw6hlp-list-players",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_players);
  scm_c_define_gsubr ("c-lw6hlp-list-input",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_input);
  scm_c_define_gsubr ("c-lw6hlp-list-graphics",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_graphics);
  scm_c_define_gsubr ("c-lw6hlp-list-sound",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_sound);
  scm_c_define_gsubr ("c-lw6hlp-list-network",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_network);
  scm_c_define_gsubr ("c-lw6hlp-list-map",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_map);
  scm_c_define_gsubr ("c-lw6hlp-list-map-rules",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_map_rules);
  scm_c_define_gsubr ("c-lw6hlp-list-map-hints",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_map_hints);
  scm_c_define_gsubr ("c-lw6hlp-list-map-style",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_map_style);
  scm_c_define_gsubr ("c-lw6hlp-list-funcs",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_funcs);
  scm_c_define_gsubr ("c-lw6hlp-list-hooks",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_hooks);
  scm_c_define_gsubr ("c-lw6hlp-list-tuning",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_tuning);
  scm_c_define_gsubr ("c-lw6hlp-list-aliases",
		      0, 0, 0, (SCM (*)())_scm_lw6hlp_list_aliases);
  scm_c_define_gsubr ("c-lw6hlp-list", 0, 0, 0, (SCM (*)())_scm_lw6hlp_list);

  /*
   * In liquidwar6cfg
   */
  /*
   * In load.c
   */
  scm_c_define_gsubr ("c-lw6cfg-load", 1, 0, 0, (SCM (*)())_scm_lw6cfg_load);
  /*
   * In option.c
   */
  scm_c_define_gsubr ("c-lw6cfg-option-exists",
		      1, 0, 0, (SCM (*)())_scm_lw6cfg_option_exists);
  scm_c_define_gsubr ("c-lw6cfg-get-option", 1, 0, 0,
		      (SCM (*)())_scm_lw6cfg_get_option);
  scm_c_define_gsubr ("c-lw6cfg-set-option", 2, 0, 0,
		      (SCM (*)())_scm_lw6cfg_set_option);
  /*
   * In save.c
   */
  scm_c_define_gsubr ("c-lw6cfg-save", 1, 0, 0, (SCM (*)())_scm_lw6cfg_save);
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6cfg-init", 0, 0, 0, (SCM (*)())_scm_lw6cfg_init);
  scm_c_define_gsubr ("c-lw6cfg-quit", 0, 0, 0, (SCM (*)())_scm_lw6cfg_quit);
  /*
   * In unified.c
   */
  scm_c_define_gsubr ("c-lw6cfg-unified-get-user-dir", 0, 0, 0,
		      (SCM (*)())_scm_lw6cfg_unified_get_user_dir);
  scm_c_define_gsubr ("c-lw6cfg-unified-get-log-file", 0, 0, 0,
		      (SCM (*)())_scm_lw6cfg_unified_get_log_file);
  scm_c_define_gsubr ("c-lw6cfg-unified-get-map-path", 0, 0, 0,
		      (SCM (*)())_scm_lw6cfg_unified_get_map_path);

  /*
   * In liquidwar6gui
   */
  scm_c_define_gsubr ("c-lw6gui-menu-new", 1, 0, 0,
		      (SCM (*)())_scm_lw6gui_menu_new);
  scm_c_define_gsubr ("c-lw6gui-menu-append", 2, 0, 0,
		      (SCM (*)())_scm_lw6gui_menu_append);
  scm_c_define_gsubr ("c-lw6gui-menu-sync", 2, 0, 0,
		      (SCM (*)())_scm_lw6gui_menu_sync);
  scm_c_define_gsubr ("c-lw6gui-menu-select", 2, 0, 0,
		      (SCM (*)())_scm_lw6gui_menu_select);
  scm_c_define_gsubr ("c-lw6gui-menu-scroll-up", 1, 0, 0,
		      (SCM (*)())_scm_lw6gui_menu_scroll_up);
  scm_c_define_gsubr ("c-lw6gui-menu-scroll-down", 1, 0, 0,
		      (SCM (*)())_scm_lw6gui_menu_scroll_down);
  scm_c_define_gsubr ("c-lw6gui-default-look",
		      0, 0, 0, (SCM (*)())_scm_lw6gui_default_look);
  scm_c_define_gsubr ("c-lw6gui-look-set",
		      3, 0, 0, (SCM (*)())_scm_lw6gui_look_set);
  scm_c_define_gsubr ("c-lw6gui-look-get",
		      2, 0, 0, (SCM (*)())_scm_lw6gui_look_get);

  /*
   * In liquidwar6gfx
   */
  /*
   * In backend.c
   */
  scm_c_define_gsubr ("c-lw6gfx-get-backends", 0, 0, 0,
		      (SCM (*)())_scm_lw6gfx_get_backends);
  scm_c_define_gsubr ("c-lw6gfx-create-backend",
		      1, 0, 0, (SCM (*)())_scm_lw6gfx_create_backend);
  scm_c_define_gsubr ("c-lw6gfx-destroy-backend", 0,
		      0, 0, (SCM (*)())_scm_lw6gfx_destroy_backend);
  /*
   * In background.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-background",
		      1, 0, 0, (SCM (*)())_scm_lw6gfx_display_background);
  /*
   * In event.c
   */
  scm_c_define_gsubr ("c-lw6gfx-poll-key",
		      0, 0, 0, (SCM (*)())_scm_lw6gfx_poll_key);
  scm_c_define_gsubr ("c-lw6gfx-is-key-esc", 1, 0, 0,
		      (SCM (*)())_scm_lw6gfx_is_key_esc);
  scm_c_define_gsubr ("c-lw6gfx-is-key-enter", 1, 0,
		      0, (SCM (*)())_scm_lw6gfx_is_key_enter);
  scm_c_define_gsubr ("c-lw6gfx-is-key-up", 1, 0, 0,
		      (SCM (*)())_scm_lw6gfx_is_key_up);
  scm_c_define_gsubr ("c-lw6gfx-is-key-down", 1, 0,
		      0, (SCM (*)())_scm_lw6gfx_is_key_down);
  scm_c_define_gsubr ("c-lw6gfx-is-key-left", 1, 0,
		      0, (SCM (*)())_scm_lw6gfx_is_key_left);
  scm_c_define_gsubr ("c-lw6gfx-is-key-right", 1, 0,
		      0, (SCM (*)())_scm_lw6gfx_is_key_right);
  scm_c_define_gsubr ("c-lw6gfx-poll-quit", 0, 0, 0,
		      (SCM (*)())_scm_lw6gfx_poll_quit);
  scm_c_define_gsubr ("c-lw6gfx-send-quit", 0, 0, 0,
		      (SCM (*)())_scm_lw6gfx_send_quit);
  scm_c_define_gsubr ("c-lw6gfx-get-default-keys-state", 0,
		      0, 0, (SCM (*)())_scm_lw6gfx_get_default_keys_state);
  scm_c_define_gsubr ("c-lw6gfx-get-key-state", 1,
		      0, 0, (SCM (*)())_scm_lw6gfx_get_key_state);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-move",
		      0, 0, 0, (SCM (*)())_scm_lw6gfx_poll_mouse_move);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-button-left",
		      0, 0, 0, (SCM (*)())_scm_lw6gfx_poll_mouse_button_left);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-button-right",
		      0, 0, 0,
		      (SCM (*)())_scm_lw6gfx_poll_mouse_button_right);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-wheel-up", 0, 0, 0,
		      (SCM (*)())_scm_lw6gfx_poll_mouse_wheel_up);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-wheel-down", 0, 0, 0,
		      (SCM (*)())_scm_lw6gfx_poll_mouse_wheel_down);
  scm_c_define_gsubr ("c-lw6gfx-get-mouse-state", 0, 0, 0,
		      (SCM (*)())_scm_lw6gfx_get_mouse_state);
  scm_c_define_gsubr ("c-lw6gfx-warp-mouse", 2, 0, 0,
		      (SCM (*)())_scm_lw6gfx_warp_mouse);
  scm_c_define_gsubr ("c-lw6gfx-show-mouse", 1, 0, 0,
		      (SCM (*)())_scm_lw6gfx_show_mouse);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-up", 1, 0, 0,
		      (SCM (*)())_scm_lw6gfx_poll_joystick_up);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-down", 1, 0, 0,
		      (SCM (*)())_scm_lw6gfx_poll_joystick_down);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-left", 1, 0, 0,
		      (SCM (*)())_scm_lw6gfx_poll_joystick_left);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-right", 1, 0, 0,
		      (SCM (*)())_scm_lw6gfx_poll_joystick_right);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-button", 2, 0, 0,
		      (SCM (*)())_scm_lw6gfx_poll_joystick_button);
  scm_c_define_gsubr ("c-lw6gfx-get-joystick-state", 3, 0, 0,
		      (SCM (*)())_scm_lw6gfx_get_joystick_state);
  scm_c_define_gsubr ("c-lw6gfx-set-key-repeat", 2, 0, 0,
		      (SCM (*)())_scm_lw6gfx_set_key_repeat);

  /*
   * In hud.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-hud",
		      2, 0, 0, (SCM (*)())_scm_lw6gfx_display_hud);

  /*
   * In info.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-log", 2, 0,
		      0, (SCM (*)())_scm_lw6gfx_display_log);
  scm_c_define_gsubr ("c-lw6gfx-display-fps", 2, 0,
		      0, (SCM (*)())_scm_lw6gfx_display_fps);
  /*
   * In map.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-map-preview",
		      2, 0, 0, (SCM (*)())_scm_lw6gfx_display_map_preview);
  scm_c_define_gsubr ("c-lw6gfx-display-map-zones",
		      2, 0, 0, (SCM (*)())_scm_lw6gfx_display_map_zones);
  scm_c_define_gsubr
    ("c-lw6gfx-display-map-gradient", 4, 0, 0,
     (SCM (*)())_scm_lw6gfx_display_map_gradient);
  scm_c_define_gsubr ("c-lw6gfx-display-game", 3, 0,
		      0, (SCM (*)())_scm_lw6gfx_display_game);
  scm_c_define_gsubr ("c-lw6gfx-display-cursors", 3, 0,
		      0, (SCM (*)())_scm_lw6gfx_display_cursors);
  scm_c_define_gsubr ("c-lw6map-rules-get-default", 1, 0, 0,
		      (SCM (*)())_scm_lw6map_rules_get_default);
  scm_c_define_gsubr ("c-lw6map-rules-get-min", 1, 0, 0,
		      (SCM (*)())_scm_lw6map_rules_get_min);
  scm_c_define_gsubr ("c-lw6map-rules-get-max", 1, 0, 0,
		      (SCM (*)())_scm_lw6map_rules_get_max);
  scm_c_define_gsubr ("c-lw6map-rules-get-int", 2, 0, 0,
		      (SCM (*)())_scm_lw6map_rules_get_int);

  scm_c_define_gsubr ("c-lw6map-style-get-default", 1, 0, 0,
		      (SCM (*)())_scm_lw6map_style_get_default);

  /*
   * In menu.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-menu",
		      2, 0, 0, (SCM (*)())_scm_lw6gfx_display_menu);
  scm_c_define_gsubr ("c-lw6gfx-pick-menuitem",
		      4, 0, 0, (SCM (*)())_scm_lw6gfx_pick_menuitem);
  scm_c_define_gsubr ("c-lw6gfx-warp-mouse-on-menuitem",
		      3, 0, 0, (SCM (*)())_scm_lw6gfx_warp_mouse_on_menuitem);
  scm_c_define_gsubr ("c-lw6gfx-center-menu",
		      3, 0, 0, (SCM (*)())_scm_lw6gfx_center_menu);

  /* 
   * In render.c
   */
  scm_c_define_gsubr ("c-lw6gfx-prepare-buffer",
		      1, 0, 0, (SCM (*)())_scm_lw6gfx_prepare_buffer);
  scm_c_define_gsubr ("c-lw6gfx-swap-buffers", 0, 0,
		      0, (SCM (*)())_scm_lw6gfx_swap_buffers);
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6gfx-init", 6, 0, 0, (SCM (*)())_scm_lw6gfx_init);
  scm_c_define_gsubr ("c-lw6gfx-get-low-resolution", 0,
		      0, 0, (SCM (*)())_scm_lw6gfx_get_low_resolution);
  scm_c_define_gsubr ("c-lw6gfx-get-standard-resolution", 0,
		      0, 0, (SCM (*)())_scm_lw6gfx_get_standard_resolution);
  scm_c_define_gsubr ("c-lw6gfx-get-high-resolution", 0,
		      0, 0, (SCM (*)())_scm_lw6gfx_get_high_resolution);
  scm_c_define_gsubr ("c-lw6gfx-set-video-mode", 3,
		      0, 0, (SCM (*)())_scm_lw6gfx_set_video_mode);
  scm_c_define_gsubr ("c-lw6gfx-get-video-mode", 0,
		      0, 0, (SCM (*)())_scm_lw6gfx_get_video_mode);
  scm_c_define_gsubr ("c-lw6gfx-quit", 0, 0, 0, (SCM (*)())_scm_lw6gfx_quit);
  /*
   * In timer.c
   */
  scm_c_define_gsubr ("c-lw6gfx-delay", 1, 0, 0,
		      (SCM (*)())_scm_lw6gfx_delay);
  scm_c_define_gsubr ("c-lw6gfx-get-ticks", 0, 0, 0,
		      (SCM (*)())_scm_lw6gfx_get_ticks);
  /*
   * In liquidwar6ldr
   */
  scm_c_define_gsubr ("c-lw6ldr-get-entries", 2, 0, 0,
		      (SCM (*)())_scm_lw6ldr_get_entries);
  scm_c_define_gsubr ("c-lw6ldr-read", 5, 0, 0, (SCM (*)())_scm_lw6ldr_read);
  scm_c_define_gsubr ("c-lw6ldr-read-relative", 6, 0, 0,
		      (SCM (*)())_scm_lw6ldr_read_relative);
  scm_c_define_gsubr ("c-lw6ldr-print-examples", 0, 0, 0,
		      (SCM (*)())_scm_lw6ldr_print_examples);
  scm_c_define_gsubr ("c-lw6ldr-hints-get-default", 1, 0, 0,
		      (SCM (*)())_scm_lw6ldr_hints_get_default);

  /*
   * In liquidwar6map
   */
  scm_c_define_gsubr
    ("c-lw6map-get-look", 1, 0, 0, (SCM (*)())_scm_lw6map_get_look);
  scm_c_define_gsubr
    ("c-lw6map-team-color-index-to-key", 1, 0, 0,
     (SCM (*)())_scm_lw6map_team_color_index_to_key);
  scm_c_define_gsubr ("c-lw6map-team-color-key-to-index", 1, 0, 0,
		      (SCM (*)())_scm_lw6map_team_color_key_to_index);

  /*
   * In liquidwar6ker
   */
  scm_c_define_gsubr ("c-lw6ker-build-game-struct",
		      1, 0, 0, (SCM (*)())_scm_lw6ker_build_game_struct);
  scm_c_define_gsubr ("c-lw6ker-build-game-state", 1,
		      0, 0, (SCM (*)())_scm_lw6ker_build_game_state);
  scm_c_define_gsubr ("c-lw6ker-sync-game-state", 2,
		      0, 0, (SCM (*)())_scm_lw6ker_sync_game_state);
  scm_c_define_gsubr ("c-lw6ker-dup-game-state", 2,
		      0, 0, (SCM (*)())_scm_lw6ker_dup_game_state);
  scm_c_define_gsubr
    ("c-lw6ker-game-struct-checksum", 1, 0, 0,
     (SCM (*)())_scm_lw6ker_game_struct_checksum);
  scm_c_define_gsubr ("c-lw6ker-game-state-checksum",
		      1, 0, 0, (SCM (*)())_scm_lw6ker_game_state_checksum);
  scm_c_define_gsubr ("c-lw6ker-register-server",
		      2, 0, 0, (SCM (*)())_scm_lw6ker_register_server);
  scm_c_define_gsubr ("c-lw6ker-unregister-server",
		      2, 0, 0, (SCM (*)())_scm_lw6ker_unregister_server);
  scm_c_define_gsubr ("c-lw6ker-server-exists",
		      2, 0, 0, (SCM (*)())_scm_lw6ker_server_exists);
  scm_c_define_gsubr ("c-lw6ker-add-cursor",
		      4, 0, 0, (SCM (*)())_scm_lw6ker_add_cursor);
  scm_c_define_gsubr ("c-lw6ker-remove-cursor",
		      3, 0, 0, (SCM (*)())_scm_lw6ker_remove_cursor);
  scm_c_define_gsubr ("c-lw6ker-cursor-exists",
		      2, 0, 0, (SCM (*)())_scm_lw6ker_cursor_exists);
  scm_c_define_gsubr ("c-lw6ker-get-cursor-info",
		      2, 0, 0, (SCM (*)())_scm_lw6ker_get_cursor_info);
  scm_c_define_gsubr ("c-lw6ker-set-cursor",
		      5, 0, 0, (SCM (*)())_scm_lw6ker_set_cursor);
  scm_c_define_gsubr ("c-lw6ker-do-round",
		      1, 0, 0, (SCM (*)())_scm_lw6ker_do_round);
  scm_c_define_gsubr ("c-lw6ker-get-moves", 1, 0, 0,
		      (SCM (*)())_scm_lw6ker_get_moves);
  scm_c_define_gsubr ("c-lw6ker-get-spreads", 1, 0, 0,
		      (SCM (*)())_scm_lw6ker_get_spreads);
  scm_c_define_gsubr ("c-lw6ker-get-rounds", 1, 0, 0,
		      (SCM (*)())_scm_lw6ker_get_rounds);

  /*
   * In liquidwar6pil
   */
  scm_c_define_gsubr ("c-lw6pil-build-pilot",
		      3, 0, 0, (SCM (*)())_scm_lw6pil_build_pilot);
  scm_c_define_gsubr ("c-lw6pil-send-command",
		      3, 0, 0, (SCM (*)())_scm_lw6pil_send_command);
  scm_c_define_gsubr ("c-lw6pil-commit",
		      1, 0, 0, (SCM (*)())_scm_lw6pil_commit);
  scm_c_define_gsubr ("c-lw6pil-fix-coords",
		      4, 0, 0, (SCM (*)())_scm_lw6pil_fix_coords);
  scm_c_define_gsubr ("c-lw6pil-execute-command",
		      2, 0, 0, (SCM (*)())_scm_lw6pil_execute_command);
  scm_c_define_gsubr ("c-lw6pil-make-backup",
		      1, 0, 0, (SCM (*)())_scm_lw6pil_make_backup);
  scm_c_define_gsubr ("c-lw6pil-sync-from-backup",
		      2, 0, 0, (SCM (*)())_scm_lw6pil_sync_from_backup);
  scm_c_define_gsubr ("c-lw6pil-sync-from-reference",
		      2, 0, 0, (SCM (*)())_scm_lw6pil_sync_from_reference);
  scm_c_define_gsubr ("c-lw6pil-sync-from-draft",
		      2, 0, 0, (SCM (*)())_scm_lw6pil_sync_from_draft);
  scm_c_define_gsubr ("c-lw6pil-calibrate",
		      3, 0, 0, (SCM (*)())_scm_lw6pil_calibrate);
  scm_c_define_gsubr ("c-lw6pil-speed-up",
		      2, 0, 0, (SCM (*)())_scm_lw6pil_speed_up);
  scm_c_define_gsubr ("c-lw6pil-slow-down",
		      2, 0, 0, (SCM (*)())_scm_lw6pil_slow_down);
  scm_c_define_gsubr ("c-lw6pil-get-next-round",
		      2, 0, 0, (SCM (*)())_scm_lw6pil_get_next_round);
  scm_c_define_gsubr ("c-lw6pil-get-last-commit-round",
		      1, 0, 0, (SCM (*)())_scm_lw6pil_get_last_commit_round);
  scm_c_define_gsubr ("c-lw6pil-get-reference-current-round",
		      1, 0, 0,
		      (SCM (*)())_scm_lw6pil_get_reference_current_round);
  scm_c_define_gsubr ("c-lw6pil-get-reference-target-round", 1, 0, 0,
		      (SCM (*)())_scm_lw6pil_get_reference_target_round);
  scm_c_define_gsubr ("c-lw6pil-get-max-round", 1, 0, 0,
		      (SCM (*)())_scm_lw6pil_get_max_round);

  /*
   * In liquidwar6snd
   */
  /*
   * In backend.c
   */
  scm_c_define_gsubr ("c-lw6snd-get-backends", 0, 0, 0,
		      (SCM (*)())_scm_lw6snd_get_backends);
  scm_c_define_gsubr ("c-lw6snd-create-backend",
		      1, 0, 0, (SCM (*)())_scm_lw6snd_create_backend);
  scm_c_define_gsubr ("c-lw6snd-destroy-backend", 0,
		      0, 0, (SCM (*)())_scm_lw6snd_destroy_backend);
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6snd-init", 2, 0, 0, (SCM (*)())_scm_lw6snd_init);
  scm_c_define_gsubr ("c-lw6snd-quit", 0, 0, 0, (SCM (*)())_scm_lw6snd_quit);

  /*
   * In sound.c
   */
  scm_c_define_gsubr ("c-lw6snd-play-sound", 1,
		      0, 0, (SCM (*)())_scm_lw6snd_play_sound);
  scm_c_define_gsubr ("c-lw6snd-set-sound-volume", 1,
		      0, 0, (SCM (*)())_scm_lw6snd_set_sound_volume);

  /*
   * In music.c
   */
  scm_c_define_gsubr ("c-lw6snd-play-music", 1,
		      0, 0, (SCM (*)())_scm_lw6snd_play_music);
  scm_c_define_gsubr ("c-lw6snd-set-music-volume", 1,
		      0, 0, (SCM (*)())_scm_lw6snd_set_music_volume);

  /*
   * In liquidwar6con
   */
  /*
   * In support.c
   */
  scm_c_define_gsubr ("c-lw6cns-support", 0, 0, 0,
		      (SCM (*)())_scm_lw6cns_support);
  /*
   * In handler.c
   */
  scm_c_define_gsubr ("c-lw6cns-init", 0, 0, 0, (SCM (*)())_scm_lw6cns_init);
  scm_c_define_gsubr ("c-lw6cns-quit", 0, 0, 0, (SCM (*)())_scm_lw6cns_quit);
  scm_c_define_gsubr ("c-lw6cns-poll", 0, 0, 0, (SCM (*)())_scm_lw6cns_poll);

  /*
   * In liquidwar6tsk
   */
  scm_c_define_gsubr ("c-lw6tsk-loader-new", 1, 0, 0,
		      (SCM (*)())_scm_lw6tsk_loader_new);
  scm_c_define_gsubr ("c-lw6tsk-loader-push", 7, 0, 0,
		      (SCM (*)())_scm_lw6tsk_loader_push);
  scm_c_define_gsubr ("c-lw6tsk-loader-pop", 1, 0, 0,
		      (SCM (*)())_scm_lw6tsk_loader_pop);
  scm_c_define_gsubr ("c-lw6tsk-loader-get-stage", 1, 0, 0,
		      (SCM (*)())_scm_lw6tsk_loader_get_stage);

  /*
   * In liquidwar6net
   */
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6net-init", 0, 0, 0, (SCM (*)())_scm_lw6net_init);
  scm_c_define_gsubr ("c-lw6net-quit", 0, 0, 0, (SCM (*)())_scm_lw6net_quit);
  scm_c_define_gsubr ("c-lw6net-server-start", 2, 0, 0,
		      (SCM (*)())_scm_lw6net_server_start);
  scm_c_define_gsubr ("c-lw6net-server-poll-idle", 0, 0, 0,
		      (SCM (*)())_scm_lw6net_server_poll_idle);
  scm_c_define_gsubr ("c-lw6net-server-poll-playing", 2, 0, 0,
		      (SCM (*)())_scm_lw6net_server_poll_playing);
  scm_c_define_gsubr ("c-lw6net-server-stop", 0, 0, 0,
		      (SCM (*)())_scm_lw6net_server_stop);

  /*
   * In liquidwar6cli
   */
  /*
   * In backend.c
   */
  scm_c_define_gsubr ("c-lw6cli-get-backends", 0, 0, 0,
		      (SCM (*)())_scm_lw6cli_get_backends);

  /*
   * In liquidwar6srv
   */
  /*
   * In backend.c
   */
  scm_c_define_gsubr ("c-lw6srv-get-backends", 0, 0, 0,
		      (SCM (*)())_scm_lw6srv_get_backends);

  /*
   * In liquidwar6bot
   */
  /*
   * In backend.c
   */
  scm_c_define_gsubr ("c-lw6bot-get-backends", 0, 0, 0,
		      (SCM (*)())_scm_lw6bot_get_backends);
  scm_c_define_gsubr ("c-lw6bot-build", 5, 0, 0,
		      (SCM (*)())_scm_lw6bot_build);
  scm_c_define_gsubr ("c-lw6bot-next-move", 1, 0, 0,
		      (SCM (*)())_scm_lw6bot_next_move);

  return ret;
}

extern void
lw6_cns_handler (char *c_line)
{
  SCM line;
  SCM func_symbol;
  SCM func;

  if (c_line)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("interpreting console input \"%s\""),
		  c_line);

      lw6cns_history_add_if_needed (c_line);
      line = scm_makfrom0str (c_line);

      func_symbol = scm_c_lookup ("lw6-console-try-catch");
      func = scm_variable_ref (func_symbol);
      scm_call_1 (func, line);
      free (c_line);		// exceptionnally, don't use LW6SYS_FREE
    }
  else
    {
      printf (_("(quit) or CTRL-C to quit"));
      printf ("\n");
    }
}
