/*
Copyright (C) 2015 John Tse

This file is part of Libknit.

Libknit is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Libknit is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Libknit.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <stdint.h>

#define rol16(x, n) (((x << n) | (x >> (16 - n))) & 0xffff)
#define ror16(x, n) (((x >> n) | (x << (16 - n))) & 0xffff)
#define mod4(x)     (x < 0 ? x + 4 : x)

static uint8_t PITABLE[] = {
	0xd9, 0x78, 0xf9, 0xc4, 0x19, 0xdd, 0xb5, 0xed, 0x28, 0xe9, 0xfd, 0x79, 0x4a, 0xa0, 0xd8, 0x9d,
	0xc6, 0x7e, 0x37, 0x83, 0x2b, 0x76, 0x53, 0x8e, 0x62, 0x4c, 0x64, 0x88, 0x44, 0x8b, 0xfb, 0xa2,
	0x17, 0x9a, 0x59, 0xf5, 0x87, 0xb3, 0x4f, 0x13, 0x61, 0x45, 0x6d, 0x8d, 0x09, 0x81, 0x7d, 0x32,
	0xbd, 0x8f, 0x40, 0xeb, 0x86, 0xb7, 0x7b, 0x0b, 0xf0, 0x95, 0x21, 0x22, 0x5c, 0x6b, 0x4e, 0x82,
	0x54, 0xd6, 0x65, 0x93, 0xce, 0x60, 0xb2, 0x1c, 0x73, 0x56, 0xc0, 0x14, 0xa7, 0x8c, 0xf1, 0xdc,
	0x12, 0x75, 0xca, 0x1f, 0x3b, 0xbe, 0xe4, 0xd1, 0x42, 0x3d, 0xd4, 0x30, 0xa3, 0x3c, 0xb6, 0x26,
	0x6f, 0xbf, 0x0e, 0xda, 0x46, 0x69, 0x07, 0x57, 0x27, 0xf2, 0x1d, 0x9b, 0xbc, 0x94, 0x43, 0x03,
	0xf8, 0x11, 0xc7, 0xf6, 0x90, 0xef, 0x3e, 0xe7, 0x06, 0xc3, 0xd5, 0x2f, 0xc8, 0x66, 0x1e, 0xd7,
	0x08, 0xe8, 0xea, 0xde, 0x80, 0x52, 0xee, 0xf7, 0x84, 0xaa, 0x72, 0xac, 0x35, 0x4d, 0x6a, 0x2a,
	0x96, 0x1a, 0xd2, 0x71, 0x5a, 0x15, 0x49, 0x74, 0x4b, 0x9f, 0xd0, 0x5e, 0x04, 0x18, 0xa4, 0xec,
	0xc2, 0xe0, 0x41, 0x6e, 0x0f, 0x51, 0xcb, 0xcc, 0x24, 0x91, 0xaf, 0x50, 0xa1, 0xf4, 0x70, 0x39,
	0x99, 0x7c, 0x3a, 0x85, 0x23, 0xb8, 0xb4, 0x7a, 0xfc, 0x02, 0x36, 0x5b, 0x25, 0x55, 0x97, 0x31,
	0x2d, 0x5d, 0xfa, 0x98, 0xe3, 0x8a, 0x92, 0xae, 0x05, 0xdf, 0x29, 0x10, 0x67, 0x6c, 0xba, 0xc9,
	0xd3, 0x00, 0xe6, 0xcf, 0xe1, 0x9e, 0xa8, 0x2c, 0x63, 0x16, 0x01, 0x3f, 0x58, 0xe2, 0x89, 0xa9,
	0x0d, 0x38, 0x34, 0x1b, 0xab, 0x33, 0xff, 0xb0, 0xbb, 0x48, 0x0c, 0x5f, 0xb9, 0xb1, 0xcd, 0x2e,
	0xc5, 0xf3, 0xdb, 0x47, 0xe5, 0xa5, 0x9c, 0x77, 0x0a, 0xa6, 0x20, 0x68, 0xfe, 0x7f, 0xc1, 0xad
};

static uint16_t j;
static uint16_t s[] = { 1, 2, 3, 5 };
static uint16_t *K;

static void Initialize(uint64_t input, uint16_t *R) {
        R[0] = ((input >> 56) & 0xff) + 256 * ((input >> 48) & 0xff);
        R[1] = ((input >> 40) & 0xff) + 256 * ((input >> 32) & 0xff);
        R[2] = ((input >> 24) & 0xff) + 256 * ((input >> 16) & 0xff);
        R[3] = ((input >> 8)  & 0xff) + 256 * ((input)       & 0xff);
}

static void MixUp(uint16_t *R, uint16_t i) {
	R[i] = R[i] + K[j++] + (R[mod4(i - 1)] & R[mod4(i - 2)]) + ((~R[mod4(i - 1)]) & R[mod4(i - 3)]);
	R[i] = rol16(R[i], s[i]);
}

static void Mash(uint16_t *R, uint16_t i) {
	R[i] = R[i] + K[R[mod4(i - 1)] & 63];
}

static void MixRound(uint16_t *R) {
	MixUp(R, 0);
	MixUp(R, 1);
	MixUp(R, 2);
	MixUp(R, 3);
}

static void MashRound(uint16_t *R) {
	Mash(R, 0);
	Mash(R, 1);
	Mash(R, 2);
	Mash(R, 3);
}

static void RMixUp(uint16_t *R, uint16_t i) {
	R[i] = ror16(R[i], s[i]);
	R[i] = R[i] - K[j--] - (R[mod4(i - 1)] & R[mod4(i - 2)]) - ((~R[mod4(i - 1)]) & R[mod4(i - 3)]);
}

static void RMash(uint16_t *R, uint16_t i) {
	R[i] = R[i] - K[R[mod4(i - 1)] & 63];
}

static void RMixRound(uint16_t *R) {
	RMixUp(R, 3);
	RMixUp(R, 2);
	RMixUp(R, 1);
	RMixUp(R, 0);
}

static void RMashRound(uint16_t *R) {
	RMash(R, 3);
	RMash(R, 2);
	RMash(R, 1);
	RMash(R, 0);
}

void rc2_key_expansion(uint8_t *k, uint32_t T, uint32_t T1) {
	uint8_t *L;
	uint32_t T8;
	uint32_t TM;
	int32_t i;

	L = (uint8_t *)  calloc(128, 1);
	K = (uint16_t *) calloc(64,  2);

	for (i = 0; i < T; i++)
		L[i] = k[i];

	T8 = (T1 + 7) / 8;
	TM = 255 % (uint32_t) pow(2, 8 + T1 - 8 * T8);

	for (i = T; i < 128; i++)
		L[i] = PITABLE[(L[i - 1] + L[i - T]) % 256];

	L[128 - T8] = PITABLE[L[128 - T8] & TM];

        for (i = 127 - T8; i >= 0; i--)
                L[i] = PITABLE[L[i + 1] ^ L[i + T8]];

	for (i = 0; i < 64; i++)
		K[i] = L[2 * i] + 256 * L[2 * i + 1];

	free(L);
}

void rc2_encrypt(uint64_t input, uint64_t *output) {
	uint16_t R[4];
	uint16_t i;

	Initialize(input, R);

	j = 0;

	for (i = 0; i < 5; i++)
		MixRound(R);

	MashRound(R);

	for (i = 0; i < 6; i++)
		MixRound(R);

	MashRound(R);

	for (i = 0; i < 5; i++)
		MixRound(R);

	*output  = 0;
	*output |= (uint64_t) ((R[0] & 0xff) * 256 + ((R[0] >> 8) & 0xff)) << 48;
	*output |= (uint64_t) ((R[1] & 0xff) * 256 + ((R[1] >> 8) & 0xff)) << 32;
	*output |= (uint64_t) ((R[2] & 0xff) * 256 + ((R[2] >> 8) & 0xff)) << 16;
	*output |= (uint64_t) ((R[3] & 0xff) * 256 + ((R[3] >> 8) & 0xff));
}

void rc2_decrypt(uint64_t input, uint64_t *output) {
	uint16_t R[4];
	uint16_t i;

	Initialize(input, R);

	j = 63;

	for (i = 0; i < 5; i++)
		RMixRound(R);

	RMashRound(R);

	for (i = 0; i < 6; i++)
		RMixRound(R);

	RMashRound(R);

	for (i = 0; i < 5; i++)
		RMixRound(R);

	*output  = 0;
	*output |= (uint64_t) ((R[0] & 0xff) * 256 + ((R[0] >> 8) & 0xff)) << 48;
	*output |= (uint64_t) ((R[1] & 0xff) * 256 + ((R[1] >> 8) & 0xff)) << 32;
	*output |= (uint64_t) ((R[2] & 0xff) * 256 + ((R[2] >> 8) & 0xff)) << 16;
	*output |= (uint64_t) ((R[3] & 0xff) * 256 + ((R[3] >> 8) & 0xff));
}
