{- This source file is part of mmisar - a tool for converting
Metamath proofs to Isabelle/Isar ZF logic.
Copyright 2006 Slawomir Kolodynski

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
-}
module Outer (
              MMIinput(..),
              MMIoutput(..),
              makeMMscript,
              mmiIn2out,
              mmiIn2outDeps
             )
   where

import Data.List 
import Utils
import Inner


commentPrefs = ["#"," "]

-- makes Metamath commands that show statements and proofs of theorems
makeMMscript :: String -> String -> String
makeMMscript logFile ts = "set scroll continuous\nset width 999\nopen log " ++ logFile ++ "\n" ++ 
   (unlines $ map makeCommand tsc) ++ "close log\nquit\n"
      where 
         tsc = filter (not . null) (rmLinesWithPrefs ts commentPrefs)
         makeCommand t = "show statement " ++ t ++ "\nshow proof " ++ t ++ " /essential /lemmon /renumber"
         {- temporarily disable flagging error to be able to prove axltrrn and axlttri 
         makeCommand t = if "ax" `isPrefixOf` t then error (t ++ " is an axiom, add to MMI_prelude.thy by hand\n")
                            else "show statement " ++ t ++ "\nshow proof " ++ t ++ " /essential /lemmon /renumber"
         -}


-- MMI input is two strings - the string with known theorems and a string resulting
--- from show statement, show proof commands
data MMIinput =
   MMIinput { knownThms :: String, -- "\n" separated list of known theorems 
              mmout     :: String  -- output from show statement, show proof commands
            } deriving Show

data MMIoutput =
   MMIoutput { newKnown :: String, -- '\n' separated list of known theorems
               newDeps  :: String, -- '\n' separated list of new dependents
               isarThms :: String  -- string to be added to the MMIsar_ZF.thy
             } deriving Show


-- main function - converts the struct that represents the input of the to the output program
mmiIn2out :: MMIinput -> MMIoutput
mmiIn2out = stage2 . stage1 

-- dependency processing is simpler
mmiIn2outDeps :: String -> String
mmiIn2outDeps = stage2deps . stage1deps

--------------------------- Level 1 -----------------------------------------------------
-- this is an internal representation of a theorem
data MMITheorem =
   MMITheorem { thName   :: String,
                premises :: [Premise],
                thesis   :: String,
                proof    :: [ProofStep],
                qed      :: ProofStep
              } deriving Show

-- stage1 converts the string read from the file with proofs exported from 
-- Metamath to internal representation
stage1 :: MMIinput -> ([String],[MMITheorem])
stage1 mmiIn = ( lines (knownThms mmiIn),
               map (str2MMITheorem . makeMMTh) (splitIntoTheorems $ mmout mmiIn) )

-- stage2 converts the internal representation to a list of
-- Isar theorems
stage2 :: ([String],[MMITheorem]) -> MMIoutput
stage2 (knThms, mmiThrms) =  MMIoutput { newKnown = unlines (knThms `union` nd `union` nt),
                                         newDeps =  unlines nd,
                                         isarThms = unlines (map mmi2IsarTh mmiThrms) 
                                       }
                                       where 
                                          nd = findNewDeps knThms mmiThrms
                                          nt = map thName mmiThrms


-- converts the input string into internal representation when processing depemdencies 
stage1deps :: String -> [MMITheorem]
stage1deps s = map (str2MMITheorem . makeMMTh) (splitIntoTheorems s)

stage2deps :: [MMITheorem] -> String
stage2deps thms = unlines (map mmi2IsarThSimple thms)

--------------------------- Level 2 ----------------------------------------------------
-- convert a string representation of a theorem to internal representation
str2MMITheorem :: MMTheorem -> MMITheorem
str2MMITheorem th = MMITheorem { thName   = name,
                                 premises = map str2Premise (premisesStr th),
                                 thesis   = extrThesis $ thesisStr th,
                                 proof    = map str2ProofStep (proofStr th),
                                 qed      = str2ProofStep (qedStr th)
                               }
                               where name = thrmName th

-- finds all unsatisfied dependencies by extracting all deps and removing known
-- theorems and theorems being proven in this batch
findNewDeps :: [String] -> [MMITheorem] -> [String]
findNewDeps known thms = ((unionAll (map extrDeps thms)) \\ known) \\ (map thName thms)

-- We start from a string resulting from a series of outputs from doing s
-- show statement, show proof commands in Metamath. 
-- The next function first splits this string into lines, removing some meaningless
-- lines (recognized by their prefixes), then joins the lines broken by Metamath 
-- because they were too long (done by joinStrsStartingWSpace), then splits
-- the lines into groups representing single theorems
splitIntoTheorems :: String -> [[String]]
splitIntoTheorems s = 
   splitGroups isShowStatementLine (rmLinesWithPrefs s excludedprefs)

-- this is a representation of an MM theorem as a bunch of strings
data MMTheorem =
   MMTheorem { premisesStr :: [String],
               thesisStr   :: String,
               proofStr    :: [String],
               qedStr      :: String
             } deriving (Eq, Show)

-- fills in an MMTheorem "structure" with strings that form a theorem
makeMMTh :: [String] -> MMTheorem
makeMMTh strthrm = MMTheorem{ premisesStr = init (fst z),
                              thesisStr   = last (fst z),
                              proofStr    = init (snd z),
                              qedStr      = last (snd z)
                             } where z =  breakAtDrop isShowProofLine strthrm


-- synthetizes a full Isar theorem string from the internal representation
mmi2IsarTh :: MMITheorem -> String
mmi2IsarTh thrm = (mmi2IsarThHeader thrm) ++ "\nproof -\n" ++ 
                   (mmi2IsarProof thrm) ++ "qed\n"

-- a simpler version of synthetizing an Isar theorem, the proof is by auto
-- if we have assumptions, then we say "using premises by auto
mmi2IsarThSimple :: MMITheorem -> String
mmi2IsarThSimple thrm = "lemma MMI_" ++ (convName $ thName thrm) ++ ": " ++ 
   ( if (not $ null $ premises thrm) then (mmi2IsarPrems thrm) else "" ) ++ 
   "\n   shows " ++ (mmForm2Isar $ thesis thrm) ++ 
   ( if (not $ null $ premises thrm) then "\n   using prems by auto\n" else "\n  by auto\n") 

------------------------------ Level 3 ----------------------------------------

-- a type for internal reoresentation a a premise. It consists of a label and 
-- a formula, represented as strings
data Premise = 
   Premise { premiseLabel   :: String,
             premiseFormula :: String
           } deriving (Eq, Show)

-- a type for internal representation of a proof step
-- it has step label, a (possibly empty) list of proof steps references
-- a (possibly empty) premise reference, a (possibly empty) theorem reference
-- and a formula that is the proof step conclusion
data ProofStep =
   ProofStep { stepLabel      :: String,
               stepRefs :: [String],
               premRef  :: String,
               thrmRef  :: String,
               stepConc :: String
             } deriving (Eq, Show)


-- these are prefixes of lines that we don't want in the initial string
excludedprefs = ["The","MM> close log","$d "]

-- joins any string starting with a space with the previous one
{-joinStrsStartingWSpace :: [String] -> [String]
joinStrsStartingWSpace strs = 
   map rmLeadSpacesConcatenate (groupBy secndStrStartsW2Spaces strs)
-}

-- extracts dependencies from a theorem, a list of all nonempty references in the proof steps,
-- including the last one
extrDeps :: MMITheorem -> [String]
extrDeps th = (thrmRef $ qed th) `pne` 
              [thrmRef st | st <- (proof th), not $ null $ thrmRef st]

--breaks the string into lines and
--removes lines with prefixes on the list of forbidden prefixes
rmLinesWithPrefs :: String -> [String] -> [String]
str `rmLinesWithPrefs` prefs = [ s | s <- (lines str), s `isNotPrefixedByAnyOf` prefs]

-- Some long formulae are broken in the Metamath output into shorter lines
-- The next function fixes that - it joins the lines starting with a space
-- with the previous one


-- a predicate that detects the line that resulted from the "show statement"
-- command. The theorem follows that line until the next such line
isShowStatementLine :: String -> Bool
isShowStatementLine  s = "MM> show statement" `isPrefixOf` s

-- a predicate that detects the line that resulted from the "show proof"
-- command. 
isShowProofLine :: String -> Bool
isShowProofLine s = "MM> show proof" `isPrefixOf` s

-- splits a list into groups using the elements that satisfy a predicate
-- as separators. In addition, it removes leading emty groups (corresponding
-- to the situation that there are several separators at the start
splitGroups :: (a -> Bool) -> [a] -> [[a]]
splitGroups pred x = (dropWhile null) (unfoldr (splitGroup pred) x) 

-- convert a string representation of a premise to internal representation
str2Premise :: String -> Premise
str2Premise s = Premise { premiseLabel   = extrPremiseLabel s,
                          premiseFormula = extrPremiseFormula s
                        }

-- extracts the formula that constitutes thesis of the theorem
-- all words after "|-", but before the "$=" -- example:  
-- extrThesis "2 1 sseli        $p |- ( A e. RR -> A e. CC ) " == A e. RR -> A e. CC
extrThesis :: String -> String
extrThesis s = unwords $ stripParens z 
   where z = takeWhile (/= "$=") $ tail $ dropWhile (/= "|-") (words s)

-- extract name from a MM theorem. This is the second word in the thesis
thrmName :: MMTheorem -> String
thrmName = head . tail . words . thesisStr

-- converts proof step from a string to internal representation
str2ProofStep :: String -> ProofStep
str2ProofStep s = ProofStep { stepLabel = extrStepLabel s,
                                   stepRefs  = extrStepRefs s,
                                   premRef   = extrPremRef s,
                                   thrmRef   = extrThRef s,
                                   stepConc  = extrConcl s
                                 }

-- synthetizes a theorem header string
mmi2IsarThHeader :: MMITheorem -> String
mmi2IsarThHeader th = 
   "lemma (in MMIsar0) MMI_" ++ (convName $ thName th) ++ ": " ++ 
   (if (not $ null $ premises th) then (mmi2IsarPrems th) else "")
   ++ "\n   shows " ++ (mmForm2Isar $ thesis th)

-- translates a Metamath proof into Isar proof

mmi2IsarProof :: MMITheorem -> String
mmi2IsarProof th = (unlines (map mmi2IsarProofStep (proof th))) ++ 
                   mmi2IsarProofConcl (qed th)


------------------------- Level 4 ---------------------------------------------------------

-- extracts premise label from a string representing a premise
-- this is the string after the last dot in the second word
-- extrPremiseLabel "17241 recnd.1 $e |- ( ph -> A e. RR ) $." == "1"
-- extrPremiseLabel "4965 r19.23aiv.1 $e |- ( x e. A -> ( ph -> ps ) ) $." == "1"
extrPremiseLabel :: String -> String
extrPremiseLabel s = reverse $ takeWhile (/='.') $ reverse $ sndEl $ words s
--extrPremiseLabel s = tail $ dropWhile (/='.') $ head $ tail $ words s

-- extracts formula from the string representing a premise
-- all words after "|-", but before the "$.". Example: 
-- example: 
-- extrPremiseFormula "17241 recnd.1 $e |- ( ph -> A e. RR ) $." == "ph -> A e. RR"
extrPremiseFormula :: String -> String
extrPremiseFormula s = unwords $ stripParens z 
   where z = takeWhile (/= "$.") $ tail $ dropWhile (/= "|-") (words s)

-- returns true if the second string out of two starts with a space
--secndStrStartsW2Spaces :: String -> String -> Bool
--secndStrStartsW2Spaces s1 s2 = "  " `isPrefixOf` s2

-- concatenates a list of strings, removing leading spaces, (except one) from them
--rmLeadSpacesConcatenate :: [String] -> String
--rmLeadSpacesConcatenate strs = foldr1 (++) (map dropSpacesButOne strs)

-- remove parentheses around a formula if any
stripParens :: [String] -> [String]
stripParens ss = if (head ss) == "(" && (minNestLevel $ unwords t) == 1 
                 then (init t) else ss
                    where t = tail ss


-- extracts a step label from a string as the first word of that string
-- example extrStepLabel "2 recn.1         $e |- A e. RR" == "2"
extrStepLabel :: String -> String
extrStepLabel = head . words

-- extracts a possibly empty list of references to steps
-- This is a comma separated list in the second word, 
-- if that second word consists of digits and commas only Examples:
-- extrStepRefs "3 1,2 syl        $p |- ( ph -> A e. CC )" == ["1","2"]
-- extrStepRefs "2 recnt          $p |- ( A e. RR -> A e. CC )" == []
-- it may be also a labeled step reference
-- extrStepRefs "7 @6             $p |- ( y e. -u A <-> y e. ( 0 - A ) )" == ["6"]
extrStepRefs :: String -> [String]
extrStepRefs s = 
   let z = sndEl (words s) in
      if (head z) == '@' then [tail z] 
         else if (all isADigitOrComma z) then splitGroups (==',') z
            else []

-- extracts a premise reference. If the there is a word "$e" in the string, then 
-- this is the string after the last dot in the second, otherwise the string does not have a premise
-- reference and we calculate empty
-- Examples: extrPremRef "1 recnd.1        $e |- ( ph -> A e. RR )" == "1"
-- extrPremRef "2 recnt          $p |- ( A e. RR -> A e. CC )" == []
-- extrPremRef "28 r19.22        $p |- ( A. x e. CC ( x = ( y + B ) -> ( A + x ) = B ) -> ( E. x e. CC x = ( y + B ) -> E. x e. CC ( A + x ) = B ) )"
-- = []"
extrPremRef :: String -> String
extrPremRef s = if "$e" `elem` ws then reverse $ takeWhile (/='.') $ reverse $ sndEl ws
                     else []
                        where ws = words s

{-extrPremRef :: String -> String -> String
extrPremRef name s = if (name ++ ".") `isPrefixOf` sndw then reverse $ takeWhile (/='.') $ reverse sndw 
                     else []
                        where sndw = sndEl $ words s -}

-- extracts a theorem reference in a proof step
-- If the string contains "$e" or the second word starts with a '@' then it
-- is a statement with a premise or step reference and we calculate null,
-- otherwise
-- this maybe the second word, except when consists only of digits and commas, then
-- the second word is in fact a list of proof step references and we 
-- return the third word) 
-- Examples:
-- extrThRef "1 axresscn       $p |- RR (_ CC" == "axresscn"
-- extrThRef "2 recn.1         $e |- A e. RR" = []
-- extrThRef "3 1,2 syl        $p |- ( ph -> A e. CC )" == "syl"
-- extrThRef "5 1cn            $p |- 1 e. CC" == "1cn"
-- extrThRef "28 r19.22        $p |- ( A. x e. CC ( x = ( y + B ) -> ( A + x ) = B ) -> ( E. x e. CC x = ( y + B ) -> E. x e. CC ( A + x ) = B ) )"
-- == "r19.22"
-- extrThRef "7 @6             $p |- ( y e. -u A <-> y e. ( 0 - A ) )" == [] 
extrThRef :: String -> String
extrThRef s = if ("$e" `elem` ws) || (head sndw == '@') then ""
              else if all isADigitOrComma sndw then thirdEl ws
                 else sndw
              where ws = words s
                    sndw = sndEl ws

{-extrThRef :: String -> String -> String
extrThRef name s =
   let sndw = sndEl (words s) in
      if (all isADigitOrComma sndw) then thirdEl (words s)
      else if ( (name ++ ".") `isPrefixOf` sndw) || (head sndw == '@') then [] 
           else sndw -}

-- extracts step conclusion
-- This is all stuff that is after "|-", with stripped parantheses
-- Examples 
-- extrConcl "1 axresscn       $p |- RR (_ CC" == "RR (_ CC""
-- extrConcl "1 recnd.1        $e |- ( ph -> A e. RR )" == "ph -> A e. RR"
extrConcl :: String -> String
extrConcl s = unwords $ stripParens $ tail $ dropWhile (/= "|-") (words s)

-- takes a list of premises from the internal representation and converts them to a string
-- representing premises in the Isar format
mmi2IsarPrems :: MMITheorem -> String
mmi2IsarPrems th = 
   "assumes " ++ (unwords $ intersperse "and\n   " (map mmi2IsarPrem (premises th))) ++ "   "

-- translates a proof step from internal reprezentation to Isar
mmi2IsarProofStep :: ProofStep -> String
mmi2IsarProofStep ps | s && t = "   from " ++ unwords (map ("S"++) (stepRefs ps)) ++ 
                           " have S" ++ (stepLabel ps) ++ ": " ++
                           (mmForm2Isar $ stepConc ps) ++ " by (rule MMI_" ++ thref ++ ")"
                     | s = "   from " ++ unwords (map ("S"++) (stepRefs ps)) ++ 
                           " have S" ++ (stepLabel ps) ++ ": " ++
                           (mmForm2Isar $ stepConc ps) ++ " ."
                     | t = "   have S" ++ (stepLabel ps) ++ ": " ++ 
                           (mmForm2Isar $ stepConc ps) ++ " by (rule MMI_" ++ thref ++ ")"
                     | p = "   from A" ++ (premRef ps) ++ " have S" ++ (stepLabel ps) ++ ": " ++
                           (mmForm2Isar $ stepConc ps) ++ "."
                     | otherwise = error "mmi2IsarProofStep: don't know " ++ (stepLabel ps) ++ (":") ++ (stepConc ps)
                     where 
                        s = not $ null $ stepRefs ps
                        t = not $ null $ thrmRef ps
                        p = not $ null $ premRef ps
                        thref = convName $ thrmRef ps

{-mmi2IsarProofStep ps = 
    if not $ null $ stepRefs ps then
      "   from " ++ unwords (map ("S"++) (stepRefs ps)) ++ 
      " have S" ++ (stepLabel ps) ++ ": " ++
      (mmForm2Isar $ stepConc ps) ++ " by (rule MMI_" ++ thref ++ ")"
    else if  not $ null $ premRef ps then
       "   from A" ++ (premRef ps) ++ " have S" ++ (stepLabel ps) ++ ": " ++
       (mmForm2Isar $ stepConc ps) ++ "."
        else "   have S" ++ (stepLabel ps) ++ ": " ++ 
           (mmForm2Isar $ stepConc ps) ++ " by (rule MMI_" ++ thref ++ ")"
    where thref = convName $ thrmRef ps-}

-- translates a proof conclusion. 
mmi2IsarProofConcl :: ProofStep -> String
mmi2IsarProofConcl ps = 
   if  not $ null $ stepRefs ps then
      "   from " ++ unwords (map ("S"++) (stepRefs ps)) ++ " show " ++
      (mmForm2Isar $ stepConc ps) ++ " by (rule MMI_" ++ thref ++ ")\n"
   else if not $ null $ premRef ps then
      "   from S" ++ (premRef ps) ++ " show " ++ (mmForm2Isar $ stepConc ps) ++ " .\n"
      else "   show " ++ (mmForm2Isar $ stepConc ps) ++ 
           "  by (rule MMI_" ++ thref ++ ")\n"
   where thref = convName $ thrmRef ps


--------------------------- Level 5 ------------------------------------------

-- The second word in a step can be a comma separated list of step numbers or 
-- or a theorem reference, possibly starting with a digit. The next function 
-- evaluates to true if a string represents step references
-- Examples: 
-- isStepRefs "1,2" == True
-- isStepRefs "1cn" == False
-- isStepRefs "200" == True
-- isStepRefs :: String -> Bool
-- isStepRefs s = all isADigitOrComma s

-- If a string starts with a space, then drops spaces all off them but one,
-- otherwise does nothing. String has to be nonempty
dropSpacesButOne :: String -> String
dropSpacesButOne s = if (head s) == ' ' then ' ':(dropWhile (== ' ') s)
                     else s

-- finds the minimal nest level as the string is traversed from right to left
minNestLevel :: String -> Int
minNestLevel s = minimum [nestLevel ps | ps <- init (tails s)]

-- takes a premise in the internal representation and converts to an Isar premise
mmi2IsarPrem :: Premise -> String
mmi2IsarPrem p = "A" ++ (premiseLabel p) ++ ": " ++ mmForm2Isar (premiseFormula p)

--------------------------- Level 6 -------------------------------------------
-- tests if a character in a string are digits or commas
isADigitOrComma :: Char -> Bool
isADigitOrComma c = any (c==) ['0','1','2','3','4','5','6','7','8','9',',']


-- calculates the nest level in a string, i. e. number of closed paratheses 
-- minus open parantheses
nestLevel :: String -> Int
nestLevel "" = 0
nestLevel (')':t) = (nestLevel t) + 1
nestLevel ('(':t) = (nestLevel t) - 1
nestLevel s = nestLevel (tail s)

-- some theorem names in Metamath contain dashes (minus signs) or points. Isar
-- doesn't like to have those in names - we replace them with underscores 
convName :: String -> String
convName = replaceAll [("-", "_"), ("." ,"_")]
