(*   This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{OrderedField\_ZF.thy}*}

theory OrderedField_ZF imports OrderedRing_ZF Field_ZF

begin

text{*This theory covers basic facts about ordered fiels.*}

section{*Definition and basic properties*}

text{*Ordered field is a notrivial ordered ring such that all 
  non-zero elements have an inverse. We define the notion of being a ordered 
  field as
  a statement about four sets. The first set, denoted @{text "K"} is the 
  carrier of the field. The second set, denoted @{text "A"} represents the 
  additive operation on @{text "K"} (recall that in ZF set theory functions 
  are sets). The third set @{text "M"} represents the multiplicative operation 
  on @{text "K"}. The fourth set @{text "r"} is the order 
  relation on @{text "K"}.*}

constdefs
  "IsAnOrdField(K,A,M,r) \<equiv> (IsAnOrdRing(K,A,M,r) \<and>
  (M {is commutative on} K) \<and>
  TheNeutralElement(K,A) \<noteq> TheNeutralElement(K,M) \<and>
  (\<forall>a\<in>K. a\<noteq>TheNeutralElement(K,A)\<longrightarrow>
  (\<exists>b\<in>K. M`\<langle>a,b\<rangle> = TheNeutralElement(K,M))))"

text{*The next context (locale) defines notation used for ordered fields. 
  We do that by extending the notation defined in the 
  @{text "ring1"} context that is used for oredered rings and 
  adding some assumptions to make sure we are 
  talking about ordered fields in this context. 
  We should rename the carrier from $R$ used in the @{text "ring1"} 
  context to $K$, more appriopriate for fields. Theoretically the Isar locale
  facility supports such renaming, but we experienced diffculties using
  some lemmas from @{text "ring1"} locale after renaming. 
  *}

locale field1 = ring1 +

  assumes mult_commute: "M {is commutative on} R"
  
  assumes not_triv: "\<zero> \<noteq> \<one>"

  assumes inv_exists: "\<forall>a\<in>R. a\<noteq>\<zero> \<longrightarrow> (\<exists>b\<in>R. a\<cdot>b = \<one>)"

  fixes non_zero ("R\<^isub>0")
  defines non_zero_def[simp]: "R\<^isub>0 \<equiv> R-{\<zero>}"

  fixes inv ("_\<inverse> " [96] 97)
  defines inv_def[simp]: "a\<inverse> \<equiv> GroupInv(R\<^isub>0,restrict(M,R\<^isub>0\<times>R\<^isub>0))`(a)"

text{*The next lemma assures us that we are talking fields 
  in the @{text "field1"} context.*}

lemma (in field1) OrdField_ZF_1_L1: shows "IsAnOrdField(R,A,M,r)"
  using OrdRing_ZF_1_L1 mult_commute not_triv inv_exists IsAnOrdField_def
  by simp;

text{*Ordered field is a field, of course.*}

lemma OrdField_ZF_1_L1A: assumes "IsAnOrdField(K,A,M,r)"
  shows "IsAfield(K,A,M)"
  using prems IsAnOrdField_def IsAnOrdRing_def IsAfield_def
  by simp;

text{*Theorems proven in @{text "field0"} (about fields) context are valid
  in the @{text "field1"} context (about ordered fields). *}

lemma (in field1) OrdField_ZF_1_L1B: shows "field0(R,A,M)"
  using  OrdField_ZF_1_L1 OrdField_ZF_1_L1A Field_ZF_1_L2 
  by simp;

text{*We can use theorems proven in the @{text "field1"} context whenever we
  talk about an ordered field.*}

lemma OrdField_ZF_1_L2: assumes "IsAnOrdField(K,A,M,r)"
  shows "field1(K,A,M,r)"
  using prems IsAnOrdField_def OrdRing_ZF_1_L2 ring1_def
    IsAnOrdField_def field1_axioms_def field1_def
  by auto;

text{*In ordered rings the existence of a right inverse for all positive
  elements implies the existence of an inverse for all non zero elements.*}

lemma (in ring1) OrdField_ZF_1_L3: 
  assumes A1: "\<forall>a\<in>R\<^isub>+. \<exists>b\<in>R. a\<cdot>b = \<one>" and A2: "c\<in>R"  "c\<noteq>\<zero>"
  shows "\<exists>b\<in>R. c\<cdot>b = \<one>"
proof (cases "c\<in>R\<^isub>+");
  assume "c\<in>R\<^isub>+"
  with A1 show "\<exists>b\<in>R. c\<cdot>b = \<one>" by simp;
next assume "c\<notin>R\<^isub>+"
  with A2 have "(\<rm>c) \<in> R\<^isub>+"
    using OrdRing_ZF_3_L2A by simp;
  with A1 obtain b where "b\<in>R" and "(\<rm>c)\<cdot>b = \<one>"
    by auto;
  with A2 have "(\<rm>b) \<in> R"  "c\<cdot>(\<rm>b) = \<one>"
    using Ring_ZF_1_L3 Ring_ZF_1_L7 by auto;
  then show "\<exists>b\<in>R. c\<cdot>b = \<one>" by auto;
qed;
  
text{*Ordered fields are easier to deal with, because it is sufficient 
  to show the existence of an inverse for the set of positive elements.*}

lemma (in ring1) OrdField_ZF_1_L4: 
  assumes "\<zero> \<noteq> \<one>" and "M {is commutative on} R" 
  and "\<forall>a\<in>R\<^isub>+. \<exists>b\<in>R. a\<cdot>b = \<one>"
  shows "IsAnOrdField(R,A,M,r)"
  using prems OrdRing_ZF_1_L1 OrdField_ZF_1_L3 IsAnOrdField_def
  by simp;

text{*The set of positive field elements is closed under multiplication.*}

lemma (in field1) OrdField_ZF_1_L5: shows "R\<^isub>+ {is closed under} M"
  using OrdField_ZF_1_L1B field0.field_has_no_zero_divs OrdRing_ZF_3_L3
  by simp;

text{*The set of positive field elements is closed under multiplication:
  the explicit version.*}

lemma (in field1) pos_mul_closed: 
  assumes A1: "\<zero> \<ls> a"  "\<zero> \<ls> b"
  shows "\<zero> \<ls> a\<cdot>b"
proof -
  from A1 have "a \<in> R\<^isub>+" and  "b \<in> R\<^isub>+"
    using OrdRing_ZF_3_L14 by auto;
  then show "\<zero> \<ls> a\<cdot>b" 
    using OrdField_ZF_1_L5 IsOpClosed_def PositiveSet_def
    by simp;
qed;


text{*In fields square of a nonzero element is positive. *}

lemma (in field1) OrdField_ZF_1_L6: assumes "a\<in>R"  "a\<noteq>\<zero>"
  shows "a\<twosuperior> \<in> R\<^isub>+"
  using prems OrdField_ZF_1_L1B field0.field_has_no_zero_divs
    OrdRing_ZF_3_L15 by simp;

text{*The next lemma restates the fact @{text "Field_ZF"} that out notation
  for the field inverse means what it is supposed to mean.*}

lemma (in field1) OrdField_ZF_1_L7: assumes "a\<in>R"  "a\<noteq>\<zero>"
  shows "a\<cdot>(a\<inverse>) = \<one>"  "(a\<inverse>)\<cdot>a = \<one>"
  using prems OrdField_ZF_1_L1B field0.Field_ZF_1_L6
  by auto;

text{*A simple lemma about multiplication and cancelling of a positive field
   element.*}

lemma (in field1) OrdField_ZF_1_L7A: 
  assumes A1: "a\<in>R"  "b \<in> R\<^isub>+"
  shows 
  "a\<cdot>b\<cdot>b\<inverse> = a"
  "a\<cdot>b\<inverse>\<cdot>b = a"
proof -
  from A1 have "b\<in>R"  "b\<noteq>\<zero>" using PositiveSet_def
    by auto
  with A1 show  "a\<cdot>b\<cdot>b\<inverse> = a" and "a\<cdot>b\<inverse>\<cdot>b = a"
    using OrdField_ZF_1_L1B field0.Field_ZF_1_L7
    by auto;
qed;
    
text{*Some properties of the inverse of a positive element.*}

lemma (in field1) OrdField_ZF_1_L8: assumes A1: "a \<in> R\<^isub>+"
  shows "a\<inverse> \<in> R\<^isub>+"  "a\<cdot>(a\<inverse>) = \<one>"  "(a\<inverse>)\<cdot>a = \<one>"
proof -
  from A1 have I: "a\<in>R"  "a\<noteq>\<zero>" using PositiveSet_def 
    by auto;
  with A1 have "a\<cdot>(a\<inverse>)\<twosuperior> \<in> R\<^isub>+" 
    using OrdField_ZF_1_L1B field0.Field_ZF_1_L5 OrdField_ZF_1_L6
      OrdField_ZF_1_L5 IsOpClosed_def by simp;
  with I show "a\<inverse> \<in> R\<^isub>+"
    using OrdField_ZF_1_L1B field0.Field_ZF_2_L1
    by simp;
  from I show  "a\<cdot>(a\<inverse>) = \<one>"  "(a\<inverse>)\<cdot>a = \<one>"
    using OrdField_ZF_1_L7 by auto
qed;

text{*If $a<b$, then $(b-a)^{-1}$ is positive.*}

lemma (in field1) OrdField_ZF_1_L9: assumes "a\<ls>b"
  shows  "(b\<rs>a)\<inverse> \<in> R\<^isub>+"  
  using prems OrdRing_ZF_1_L14 OrdField_ZF_1_L8
  by simp;

text{*In ordered fields if at least one of $a,b$ is not zero, then
  $a^2+b^2 > 0$, in particular $a^2+b^2\neq 0$ and exists the 
  (multiplicative) inverse of $a^2+b^2$. *}

lemma (in field1) OrdField_ZF_1_L10: 
  assumes A1: "a\<in>R"  "b\<in>R" and A2: "a \<noteq> \<zero> \<or> b \<noteq> \<zero>"
  shows "\<zero> \<ls> a\<twosuperior> \<ra> b\<twosuperior>"  and "\<exists>c\<in>R. (a\<twosuperior> \<ra> b\<twosuperior>)\<cdot>c = \<one>"
proof -
  from A1 A2 show "\<zero> \<ls> a\<twosuperior> \<ra> b\<twosuperior>"
    using OrdField_ZF_1_L1B field0.field_has_no_zero_divs 
      OrdRing_ZF_3_L19 by simp;
  then have 
    "(a\<twosuperior> \<ra> b\<twosuperior>)\<inverse> \<in> R" and "(a\<twosuperior> \<ra> b\<twosuperior>)\<cdot>(a\<twosuperior> \<ra> b\<twosuperior>)\<inverse> = \<one>"
    using OrdRing_ZF_1_L3 PositiveSet_def OrdField_ZF_1_L8
    by auto;
  then show "\<exists>c\<in>R. (a\<twosuperior> \<ra> b\<twosuperior>)\<cdot>c = \<one>" by auto;
qed;
  
section{*Inequalities*}

text{*In this section we develop tools to deal inequalities in fields.*}

text{*We can multiply strict inequality by a positive element.*}

lemma (in field1) OrdField_ZF_2_L1: 
  assumes "a\<ls>b" and "c\<in>R\<^isub>+"
  shows "a\<cdot>c \<ls> b\<cdot>c"
  using prems OrdField_ZF_1_L1B field0.field_has_no_zero_divs
    OrdRing_ZF_3_L13
  by simp;

text{*A special case of @{text "OrdField_ZF_2_L1"} when we multiply
  an inverse by an element.*}

lemma (in field1) OrdField_ZF_2_L2: 
  assumes A1: "a\<in>R\<^isub>+" and A2: "a\<inverse> \<ls> b"
  shows "\<one> \<ls> b\<cdot>a"
proof -
  from A1 A2 have "(a\<inverse>)\<cdot>a \<ls> b\<cdot>a"
    using OrdField_ZF_2_L1 by simp;
  with A1 show "\<one> \<ls> b\<cdot>a"
    using OrdField_ZF_1_L8 by simp
qed;

text{*We can multiply an inequality by the inverse of a positive element.*}

lemma (in field1) OrdField_ZF_2_L3:
  assumes "a\<lsq>b"  and "c\<in>R\<^isub>+" shows "a\<cdot>(c\<inverse>) \<lsq> b\<cdot>(c\<inverse>)"
  using prems OrdField_ZF_1_L8 OrdRing_ZF_1_L9A
  by simp;

text{*We can multiply a strict inequality by a  positive element
  or its inverse.*}

lemma (in field1) OrdField_ZF_2_L4:
  assumes "a\<ls>b" and "c\<in>R\<^isub>+"
  shows 
  "a\<cdot>c \<ls> b\<cdot>c"
  "c\<cdot>a \<ls> c\<cdot>b"
  "a\<cdot>c\<inverse> \<ls> b\<cdot>c\<inverse>"
   using prems OrdField_ZF_1_L1B field0.field_has_no_zero_divs
    OrdField_ZF_1_L8 OrdRing_ZF_3_L13 by auto;

text{*We can put a positive factor on the other side of an inequality,
  changing it to its inverse.*}

lemma (in field1) OrdField_ZF_2_L5:
  assumes A1: "a\<in>R"  "b\<in>R\<^isub>+" and A2: "a\<cdot>b \<lsq> c"
  shows "a \<lsq> c\<cdot>b\<inverse>"
proof -
  from A1 A2 have "a\<cdot>b\<cdot>b\<inverse> \<lsq> c\<cdot>b\<inverse>"
    using OrdField_ZF_2_L3 by simp;
  with A1 show "a \<lsq> c\<cdot>b\<inverse>" using OrdField_ZF_1_L7A
    by simp;
qed;

text{*We can put a positive factor on the other side of an inequality,
  changing it to its inverse, version with a product initially on the 
  right hand side.*}

lemma (in field1) OrdField_ZF_2_L5A:
  assumes A1: "b\<in>R"  "c\<in>R\<^isub>+" and A2: "a \<lsq> b\<cdot>c"
  shows "a\<cdot>c\<inverse> \<lsq> b"
proof -
  from A1 A2 have "a\<cdot>c\<inverse> \<lsq> b\<cdot>c\<cdot>c\<inverse>"
    using OrdField_ZF_2_L3 by simp
  with A1 show "a\<cdot>c\<inverse> \<lsq> b" using OrdField_ZF_1_L7A
    by simp
qed;

text{*We can put a positive factor on the other side of a strict
  inequality, changing it to its inverse, version with a product
  initially on the left hand side.*}

lemma (in field1) OrdField_ZF_2_L6:
  assumes A1: "a\<in>R"  "b\<in>R\<^isub>+" and A2: "a\<cdot>b \<ls> c"
  shows "a \<ls> c\<cdot>b\<inverse>"
proof -
  from A1 A2 have "a\<cdot>b\<cdot>b\<inverse> \<ls> c\<cdot>b\<inverse>"
    using OrdField_ZF_2_L4 by simp
  with A1 show "a \<ls> c\<cdot>b\<inverse>" using OrdField_ZF_1_L7A
    by simp;
qed;

text{*We can put a positive factor on the other side of a strict
  inequality, changing it to its inverse, version with a product
  initially on the right hand side.*}

lemma (in field1) OrdField_ZF_2_L6A:
  assumes A1: "b\<in>R"  "c\<in>R\<^isub>+" and A2: "a \<ls> b\<cdot>c"
  shows "a\<cdot>c\<inverse> \<ls> b"
proof -
  from A1 A2 have "a\<cdot>c\<inverse> \<ls> b\<cdot>c\<cdot>c\<inverse>"
    using OrdField_ZF_2_L4 by simp
  with A1 show "a\<cdot>c\<inverse> \<ls> b" using OrdField_ZF_1_L7A
    by simp
qed;

text{*Sometimes we can reverse an inequality by taking inverse
  on both sides.*}

lemma (in field1) OrdField_ZF_2_L7: 
  assumes A1: "a\<in>R\<^isub>+" and A2: "a\<inverse> \<lsq> b"
  shows "b\<inverse> \<lsq> a"
proof -
  from A1 have "a\<inverse> \<in> R\<^isub>+" using OrdField_ZF_1_L8
    by simp;
  with A2 have "b \<in> R\<^isub>+" using  OrdRing_ZF_3_L7
    by blast;
  then have T: "b \<in> R\<^isub>+"  "b\<inverse> \<in> R\<^isub>+" using OrdField_ZF_1_L8
    by auto
  with A1 A2 have "b\<inverse>\<cdot>a\<inverse>\<cdot>a \<lsq> b\<inverse>\<cdot>b\<cdot>a"
    using OrdRing_ZF_1_L9A by simp;
  moreover 
  from A1 A2 T have
    "b\<inverse> \<in> R"  "a\<in>R" "a\<noteq>\<zero>"  "b\<in>R"  "b\<noteq>\<zero>"
    using PositiveSet_def OrdRing_ZF_1_L3 by auto;
  then have "b\<inverse>\<cdot>a\<inverse>\<cdot>a = b\<inverse>" and  "b\<inverse>\<cdot>b\<cdot>a = a"
    using OrdField_ZF_1_L1B field0.Field_ZF_1_L7 
      field0.Field_ZF_1_L6 Ring_ZF_1_L3
    by auto;
  ultimately show "b\<inverse> \<lsq> a" by simp;
qed;

text{*Sometimes we can reverse a strict inequality by taking inverse
  on both sides.*}

lemma (in field1) OrdField_ZF_2_L8: 
  assumes A1: "a\<in>R\<^isub>+" and A2: "a\<inverse> \<ls> b"
  shows "b\<inverse> \<ls> a"
proof -
  from A1 A2 have "a\<inverse> \<in> R\<^isub>+"  "a\<inverse> \<lsq>b"
    using OrdField_ZF_1_L8 by auto;
  then have "b \<in> R\<^isub>+" using OrdRing_ZF_3_L7
    by blast;
  then have "b\<in>R"  "b\<noteq>\<zero>" using PositiveSet_def by auto;
  with A2 have "b\<inverse> \<noteq> a"
    using OrdField_ZF_1_L1B field0.Field_ZF_2_L4
    by simp;
  with A1 A2 show "b\<inverse> \<ls> a"
    using OrdField_ZF_2_L7 by simp;
qed;
    
text{*A technical lemma about solving a strict inequality with three
  field elements and inverse of a difference.*}

lemma (in field1) OrdField_ZF_2_L9: 
  assumes A1: "a\<ls>b" and A2: "(b\<rs>a)\<inverse> \<ls> c"
  shows "\<one> \<ra> a\<cdot>c \<ls> b\<cdot>c"
proof -
  from A1 A2 have "(b\<rs>a)\<inverse> \<in> R\<^isub>+"  "(b\<rs>a)\<inverse> \<lsq> c" 
    using OrdField_ZF_1_L9 by auto;
  then have T1: "c \<in> R\<^isub>+" using OrdRing_ZF_3_L7 by blast;
  with A1 A2 have T2: 
    "a\<in>R"  "b\<in>R"  "c\<in>R"  "c\<noteq>\<zero>"   "c\<inverse> \<in> R"
    using OrdRing_ZF_1_L3 OrdField_ZF_1_L8 PositiveSet_def 
    by auto;
  with A1 A2  have "c\<inverse> \<ra> a \<ls> b\<rs>a \<ra> a"
    using OrdRing_ZF_1_L14 OrdField_ZF_2_L8 ring_strict_ord_trans_inv
    by simp;
  with T1 T2 have "(c\<inverse> \<ra> a)\<cdot>c \<ls> b\<cdot>c"
    using Ring_ZF_2_L1A OrdField_ZF_2_L1 by simp;
  with T1 T2 show "\<one> \<ra> a\<cdot>c \<ls> b\<cdot>c"
    using ring_oper_distr OrdField_ZF_1_L8
    by simp;
qed;

section{*Definition of real numbers*}

text{*The only purpose of this section is to define what does it mean
  to be a model of real numbers.*}

text{*We define model of real numbers as any quadruple (?) of sets $(K,A,M,r)$ 
  such that $(K,A,M,r)$ is an ordered field and the order relation $r$
  is complete, that is every set that is nonempty and bounded above in this 
  relation has a supremum. *}

constdefs
  "IsAmodelOfReals(K,A,M,r) \<equiv> IsAnOrdField(K,A,M,r) \<and> (r {is complete})";


  
end