/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.model;

import java.util.Vector;
import javax.mail.URLName;
import org.apache.log4j.Logger;

/**
 * Encapsulation of overview information about a mailbox, to be used
 * in the mailbox model
 */
public class MailboxOverview
{
    private final Logger log = Logger.getLogger(MailboxOverview.class);
    private URLName url;
    private String user;
    private Vector mailFolderHeaders;

    // TO DO: replace Vector in parameter with Collection and
    // navigate through
    public MailboxOverview(URLName mailboxUrl, Vector mailFolderHeaders)
	throws IllegalArgumentException
    {
	if (mailboxUrl == null)
	    throw new IllegalArgumentException("mailboxUrl may not be null");

	url = mailboxUrl;
	user = url.getUsername();
	this.mailFolderHeaders = mailFolderHeaders;
    }


    public URLName getUrl() {
	return url;
    }

    public String getUser() {
	return user;
    }

    public Vector getMailFolderHeaders() {
	return mailFolderHeaders;
    }

    public MailFolderHeader getMailFolderHeader(int nr) {
	return (MailFolderHeader)mailFolderHeaders.get(nr);
    }

    /**
     * Given a folder index, return as array the indices of all
     * SELECTABLE folders, indexed by their position in the mailbox, excluding
     * the parameter nr
     * (this could be useful for example in a "move to folder" operation ...)
     */
    public int[] getOtherSelectableFolderIndices(int nr) 
    {
	log.debug("getOtherFolderIndices() called with nr " + nr);
	if (nr < 0 || nr > (mailFolderHeaders.size() - 1)) {
	    log.warn("getOtherFolderIndices() called with nr " + nr + " but this number is not in range, returning empty list");
	    return new int[0];
	}

	Vector indices = new Vector(mailFolderHeaders.size() - 1);
	for (int i = 0; i < mailFolderHeaders.size(); i++) {
	    if (i != nr) {
		MailFolderHeader mfh = getMailFolderHeader(i);
		if (mfh.isSelectable())
		    indices.add(new Integer(i));
	    }
	}
	int[] ret = new int[indices.size()];
	for (int i = 0; i < indices.size(); i++) 
	    ret[i] = ((Integer)indices.get(i)).intValue();

	return ret;
    }

    /**
     * Given a folder index, return as array the indices of all
     * other folders, indexed by their position in the mailbox.
     * (this could be useful for example in a "move to folder" operation ...)
     */
    public int[] getOtherFolderIndices(int nr) 
    {
	log.debug("getOtherFolderIndices() called with nr " + nr);
	if (nr < 0 || nr > (mailFolderHeaders.size() - 1)) {
	    log.warn("getOtherFolderIndices() called with nr " + nr + " but this number is not in range, returning empty list");
	    return new int[0];
	}
	int[] ret = new int[mailFolderHeaders.size() - 1];
	int offset = 0;
	for (int i = 0; i < mailFolderHeaders.size(); i++) {
	    if (i == nr)
		offset++;
	    else {
		ret[i - offset] = i;
	    }
	}

	return ret;
    }

    /** 
     * Given a folder name, return the index of this folder in the mailbox model,
     * or return -1 if it does not exist.
     */
    public int getFolderIndex(String folderName) 
    {
	int ret = -1;
	if (log.isDebugEnabled())
	    log.debug("getFolderIndex() called for " + folderName);

	if (folderName != null && ! folderName.equals(""))
	    for (int i = 0; i < mailFolderHeaders.size(); i++) {
		MailFolderHeader mfh = (MailFolderHeader)mailFolderHeaders.get(i);
		if (folderName.equals(mfh.getName())) {
		    ret = i;
		    break;
		}
	    }

	if (log.isDebugEnabled())
	    log.debug("getFolderIndex() returning " + ret);
	return ret;
    }

    public String toString() {
	StringBuffer ret = new StringBuffer();
	ret.append("Mailbox for " + url.getUsername() + " at " + url.getHost() + " has " + mailFolderHeaders.size() + " folders.\n");
	for (int i = 0; i < mailFolderHeaders.size(); i++) {
	    MailFolderHeader el = (MailFolderHeader)mailFolderHeaders.get(i);
	    ret.append("\t" + el.toString() + "\n");
	}

	return ret.toString();
    }


}

