/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.model;

import javax.mail.Folder;
import javax.mail.FolderNotFoundException;
import javax.mail.MessagingException;
import org.apache.log4j.Logger;

/**
 * Encapsulation of header information about a folder, to be used
 * in the mailbox model
 */
public class MailFolderHeader
{
    private final Logger log = Logger.getLogger(MailFolderHeader.class);
    public static final String PRIMARY_FOLDER_NAME = "INBOX";

    private String name;
    private int nbMessages;
    private int nbNewMessages;
    private Folder folder;
    private boolean isDeleteable;
    private boolean isEmpty;
    private boolean holdsMessages;
    private boolean holdsFolders;
    private boolean isSelectable;
    private boolean isPrimary;

    public MailFolderHeader(Folder f) 
	throws FolderNotFoundException, MessagingException, ModelException
    {
	log.debug("MailFolderHeader() called");
	folder = f;

	name = f.getFullName();
	int type = f.getType();

	if (log.isDebugEnabled())
	    log.debug("MailFolderHeader() got folder name=" + name 
		      + ", type=" + type 
		      + ", full name is " + f.getFullName()
		      + ", url name is " + f.getURLName()
		);
	holdsMessages = (type & f.HOLDS_MESSAGES) > 0;
	holdsFolders = (type & f.HOLDS_FOLDERS) > 0;

	// Note: the following calls can take a long time, as it usually 
	// implies that folder will actually be opened.
	isSelectable = checkSelectable();
	log.debug("MailFolderHeader() isSelectable=" + isSelectable);
	if (isSelectable) {
	    nbMessages = f.getMessageCount();
	    log.debug("MailFolderHeader() got message count: " + nbMessages);
	    nbNewMessages = f.getUnreadMessageCount();
	    log.debug("MailFolderHeader() got new message count: " + nbNewMessages);
	}

	isPrimary = (name != null &&
		     (name.toUpperCase().equals(PRIMARY_FOLDER_NAME)));

	isDeleteable = ! isPrimary;

	isEmpty = (nbMessages <= 0);

	log.debug("MailFolderHeader() done, was folder opened? " + folder.isOpen());
    }

    /**
     * Indicates whether this folder may hold subfolders.
     * If not, it should not be attempted (nor possible) to
     * create a subfolder.
     * NOTE: this information does not seem to be accurately set by IMAP backend
     */
    public boolean holdsFolders() {
	return holdsFolders;
    }

    /**
     * Indicates whether this folder may hold messages.
     * If not, it should not be attempted (nor possible) to
     * create a subfolder.
     * NOTE: this information does not seem to be accurately set by IMAP backend
     */
    public boolean holdsMessages() {
	return holdsMessages;
    }

    public String getName() {
	return name;
    }

    public int getNbMessages() {
	return nbMessages;
    }

    public int getNbNewMessages() {
	return nbNewMessages;
    }

    public Folder getFolder() {
	return folder;
    }

    public boolean isDeleteable() {
	return isDeleteable;
    }

    public boolean isEmpty() {
	return isEmpty;
    }

    public boolean isPrimary() {
	return isPrimary;
    }

    public boolean isSelectable() {
	log.debug("isSelectable() called for " + name + ", returning " + isSelectable);
	return isSelectable;
    }

    private final static String BACKEND_UNCLEAN_EXCEPTION = "not a selectable mailbox";

    /** 
     * Implementation note: I am not sure what exactly a
     * "selectable" folder is. But I need to check for it,
     * because the Sun mail implementation throws a mean
     * exception when attempting to access certain folders,
     * so here is a hack to avoid that
     */
    private boolean checkSelectable() throws MessagingException {
	try {
	    // attempt some action which actually tries to open the folder
	    int dummy = folder.getMessageCount();
	    // no error, so it must be selectable
	    return true;
	}
	catch (MessagingException e) {
	    if (e.getMessage().indexOf(BACKEND_UNCLEAN_EXCEPTION) >= 0) {
		log.info("working around strange backend behaviour, trying to open folder " + name + " raises exception, instead just setting selectable to false");
		return false;
	    }
	    else
		// unexpected shit: propagate
		throw e;
	}
    }


    public String toString() {
	return name + "\t" + nbMessages + "/" + nbNewMessages;
    }


}

