/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller.action;

import java.util.Map;
import javax.servlet.ServletRequest;
import org.apache.cocoon.environment.Redirector;
import org.apache.log4j.Logger;

import net.jatec.ironmailer.controller.ControllerException;
import net.jatec.ironmailer.controller.FolderSetupException;
import net.jatec.ironmailer.controller.MailWorkerBean;
import net.jatec.ironmailer.framework.ServletTools;
import net.jatec.ironmailer.framework.StringTools;

/**
 * If something is screwed when setting up folders, the attribute
 * error.folder.name
 * is set in the request.
 * The attributes error.content and error.stack are set as usual.
 * 
 * If something completely unexpected happens, the usual
 * ControllerException mechanism is used.
 */
class SetupFoldersAction implements ActionDispatcher.ActionStrategy
{
    private final Logger log = Logger.getLogger(SetupFoldersAction.class);

    public void process(ServletRequest req, MailWorkerBean wb, Redirector redirector, Map objectModel) 
	throws ControllerException
    {
	// if "add" or "delete" is set, we need to do something.
	// if not, nothing to be done.
	String add = req.getParameter("add");
	String delete = req.getParameter("delete");
	log.debug("process() called, add=" + add + " and delete=" + delete);

	if (add == null && delete == null) {
	    log.debug("process() nothing special to do");
	}
	else if (add != null && delete == null) {
	    log.debug("process() called to add folder");
	    handleAdd(req, wb);
	}
	else if (delete != null && add == null) {
	    log.debug("process() called to delete folder");
	    handleDelete(req, wb);
	}
	else
	    throw new ControllerException("setup error: two submit names were given", null);
    }

    private void handleDelete(ServletRequest req, MailWorkerBean wb)
	throws ControllerException
    {
	int folderNr = ServletTools.getInteger(req,"folderNr",true);
	log.debug("process() to delete folder nr " + folderNr);
	
	try {
	    wb.deleteFolder(folderNr);
	}
	catch (FolderSetupException fe) {
	    // note folder name involved
	    // TO DO: maybe encapsulate writing of error attributes somehow
	    req.setAttribute("error.folder.name", fe.getFolderName());
	    req.setAttribute("error.content", fe.toString());
	    Throwable t = fe.getChild();
	    req.setAttribute("error.stack", 
			     t != null ? StringTools.getStackTraceAsString(t) : "");
	}
	catch (ControllerException e) {
	    // unknow shit ... just propagate
	    throw e;
	}

    }

    private void handleAdd(ServletRequest req, MailWorkerBean wb)
	throws ControllerException
    {
	String name = ServletTools.getString(req,"name");
	if (name == null)
	    throw new ControllerException("setup error: want to add folder but no folder parameter was given", null);

	boolean isMessagesFolder = ServletTools.getBoolean(req,"messagefolder");
	log.debug("process() to add folder named " + name);
	if (name.equals("")) return;

	try {
	    wb.addFolder(name, isMessagesFolder);
	}
	catch (FolderSetupException fe) {
	    // note folder name involved
	    // TO DO: maybe encapsulate writing of error attributes somehow
	    req.setAttribute("error.folder.name", fe.getFolderName());
	    req.setAttribute("error.content", fe.toString());
	    Throwable t = fe.getChild();
	    req.setAttribute("error.stack", 
			     t != null ? StringTools.getStackTraceAsString(t) : "");		
	}
	catch (ControllerException e) {
	    // unknow shit ... just propagate
	    throw e;
	}
    }
}
