/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller.action;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Map;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.servlet.ServletRequest;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Redirector;
import org.apache.log4j.Logger;
import net.jatec.ironmailer.controller.ControllerException;
import net.jatec.ironmailer.controller.MailWorkerBean;
import net.jatec.ironmailer.controller.UserInput;
import net.jatec.ironmailer.framework.ServletTools;
import net.jatec.ironmailer.model.Contact;
import net.jatec.ironmailer.model.ContactDuplicateException;


public class ContactAction implements ActionDispatcher.ActionStrategy
{
    private final Logger log = Logger.getLogger(ContactAction.class);
    public final static String PARAM_EMAIL = "email";
    public final static String PARAM_NAME = "name";
    public final static String PARAM_NICKNAME = "nickname";
    private final static String[] CONTACTS_PARAMS
    = {PARAM_EMAIL,PARAM_NAME,PARAM_NICKNAME};

    public void process(ServletRequest req, MailWorkerBean wb, 
			Redirector redirector, Map objectModel) 
	throws ControllerException
    {
	log.debug("process() called");

	if (ServletTools.getBoolean(req,"new"))
	    handleNew(req,wb);
	else
	    if (ServletTools.getBoolean(req,"add"))
		handleAdd(req,wb,redirector);
	    else
		if (ServletTools.getBoolean(req,"edit"))
		    handleEdit(req,wb);
		else
		    if (ServletTools.getBoolean(req,"update"))
			handleUpdate(req,wb,redirector);
		    else
			throw new ControllerException("setup error - what are you trying to do ?");

	log.debug("process completed");
    }

    private void handleEdit(ServletRequest req, MailWorkerBean wb) 
	throws ControllerException
    {
	if (log.isDebugEnabled())
	    log.debug("handleEdit() called");
	int contactNr = ServletTools.getInteger(req,"contactNr",true);
	if (log.isDebugEnabled())
	    log.debug("handleEdit() editing contactNr=" + contactNr);
	Contact current = wb.getContactListController().getContactList().getContact(contactNr);
	if (current == null)
	    throw new ControllerException("setup error: trying to edit non-existing contact");

	// OK, let s get busy.
	req.setAttribute("contact.edit", "true");
	req.setAttribute("contact.nr", String.valueOf(contactNr));

	UserInput initInput = new UserInput();
	initInput.add(PARAM_EMAIL, current.getEmailAddress().getAddress());
	initInput.add(PARAM_NAME, current.getEmailAddress().getPersonal());
	initInput.add(PARAM_NICKNAME, current.getNickname());
	// simulate an initial user input
	wb.setUserInput(initInput);
    }

    private void handleNew(ServletRequest req, MailWorkerBean wb) {
	log.debug("handleNew() for new contact");
	req.setAttribute("contact.new", "true");
	// check if an initial address is given
	String s_initAddress = ServletTools.getString(req, "address");
	if (s_initAddress != null) {
	    UserInput initInput = new UserInput();
	    if (log.isDebugEnabled())
		log.debug("handleNew() new contact, got initial address " + s_initAddress);
	    try {
		InternetAddress initAddress = new InternetAddress(s_initAddress);
		initInput.add(PARAM_EMAIL, initAddress.getAddress());
		initInput.add(PARAM_NAME, initAddress.getPersonal());
	    }
	    catch (AddressException e) {
		initInput.add(PARAM_NAME, s_initAddress);
	    }
	    // simulate an initial user input
	    wb.setUserInput(initInput);
	}
    }

    private void handleUpdate(ServletRequest req, MailWorkerBean wb, Redirector redirector) 
	throws ControllerException
    {
	if (log.isDebugEnabled())
	    log.debug("handleUpdate called");

	// read parameters
	int contactNr = ServletTools.getInteger(req,"contactNr",true);
	UserInput input = new UserInput(req, CONTACTS_PARAMS);
	try {
	    Contact contact = getContact(input, wb);
	    if (log.isDebugEnabled())
		log.debug("handleUpdate setting contact");

	    wb.getContactListController().setContact(contactNr, contact);
	    if (log.isDebugEnabled())
		log.debug("handleUpdate done setting contact");

	    // seems OK, set a flag
	    req.setAttribute("contact.updated", "true");
	    req.setAttribute("contact.updated.name", contact.getEmailAddress().toString());
	    // set new as well, so new contact can be added
	    req.setAttribute("contact.new", "true");

	    
	    String redirect = ServletTools.getString(req, "redirect");
	    if (redirect != null) {
		if (log.isDebugEnabled())
		    log.debug("handleUpdate() done, calling redirect");
		redirector.redirect(false,redirect);
	    }
	}
	catch (AddressException e) {
	    log.debug("handleAdd() setting AddressException");
	    wb.setUserInput(input);
	    wb.setLastException(e);
	    req.setAttribute("contact.new", "true");
	}
	catch (UnsupportedEncodingException e) {
	    throw new ControllerException("cannot create address due to encoding, application encoding is set to " + wb.getApplicationConfiguration().getEncoding(),e);
	}
	catch (IOException e) {
	    // exception from Cocoon API !?
	    throw new ControllerException("got IOException when trying to redirect after a successful update",e);
	}
	catch (ProcessingException e) {
	    // exception from Cocoon API !?
	    throw new ControllerException("got IOException when trying to redirect after a successful update",e);
	}
    }

    private void handleAdd(ServletRequest req, MailWorkerBean wb, Redirector redirector) 
	throws ControllerException
    {
	// read parameters
	UserInput input = new UserInput(req, CONTACTS_PARAMS);
	try {
	    Contact contact = getContact(input, wb);
	    wb.getContactListController().addContact(contact);

	    // seems OK, set a flag
	    req.setAttribute("contact.added", "true");
	    req.setAttribute("contact.added.name", contact.getEmailAddress().toString());
	    // set new as well, so new contact can be added
	    req.setAttribute("contact.new", "true");

	    String redirect = ServletTools.getString(req, "redirect");
	    if (redirect != null) {
		if (log.isDebugEnabled())
		    log.debug("handleUpdate() done, calling redirect");
		redirector.redirect(false,redirect);
	    }
	}
	catch (ContactDuplicateException e) {
	    log.debug("handleAdd() setting ContactDuplicateException");
	    wb.setUserInput(input);
	    wb.setLastException(e);
	    req.setAttribute("contact.new", "true");
	}
	catch (AddressException e) {
	    log.debug("handleAdd() setting AddressException");
	    wb.setUserInput(input);
	    wb.setLastException(e);
	    req.setAttribute("contact.new", "true");
	}
	catch (UnsupportedEncodingException e) {
	    throw new ControllerException("cannot create address due to encoding, application encoding is set to " + wb.getApplicationConfiguration().getEncoding(),e);
	}
	catch (IOException e) {
	    // exception from Cocoon API !?
	    throw new ControllerException("got IOException when trying to redirect after a successful add",e);
	}
	catch (ProcessingException e) {
	    // exception from Cocoon API !?
	    throw new ControllerException("got IOException when trying to redirect after a successful add",e);
	}
	
    }

    private Contact getContact(UserInput input, MailWorkerBean wb)
	throws AddressException, UnsupportedEncodingException
    {
	// try to create a contact
	String s_email = input.get(PARAM_EMAIL);
	String name = input.get(PARAM_NAME);
	Contact contact = new Contact();
	new InternetAddress(s_email); // trick because of stupid API
	InternetAddress email = new InternetAddress(s_email,name,wb.getApplicationConfiguration().getEncoding());
	contact.setEmailAddress(email);
	contact.setNickname(input.get(PARAM_NICKNAME));
	return contact;
    }


}
