/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller;


import org.apache.log4j.Logger;

import net.jatec.ironmailer.model.ApplicationConfiguration;
import net.jatec.ironmailer.model.Contact;
import net.jatec.ironmailer.model.ContactDuplicateException;
import net.jatec.ironmailer.model.ContactList;

public class ContactListController
{
    private final Logger log = Logger.getLogger(ContactListController.class);
    private ContactList userContactList;
    private GeneralStore userContactListStore;
    private MailServerConnection serverConnection;
    private ApplicationConfiguration applicationConfiguration;

    public ContactListController(MailServerConnection serverConnection,
				 ApplicationConfiguration applicationConfiguration) {
	this.serverConnection = serverConnection;
	this.applicationConfiguration = applicationConfiguration;
    }

    public ContactList getContactList() {
	if (log.isDebugEnabled())
	    log.debug("getContactList() called, is null? " + (userContactList == null));
	return userContactList;
    }

    public void close() throws ControllerException {
	if (userContactListStore != null)
	    userContactListStore.close();
    }

    public void addContact(Contact c)
	throws ControllerException, ContactDuplicateException
    {
	log.debug("addContact() called");
	// ensure the list is loaded
	initContactList();
	// add the contact
	userContactList.addContact(c);

	// write it straight away to backend
	store();
    }

    public void setContact(int contactNr, Contact contact)
	throws ControllerException
    {
	if (log.isDebugEnabled())
	    log.debug("setContact() called with contactNr=" + contactNr);

	userContactList.setContact(contactNr, contact);

	// write it straight away to backend
	store();
    }

    public void deleteContact(int index )
	throws ControllerException
    {
	log.debug("deleteContact() called");
	// sanity checks
	if (userContactList == null)
	    throw new ControllerException("setup error: calling delete contact before reading contact list");

	// delete the contact
	userContactList.deleteContact(index);

	// store straight away to backend
	store();
    }

    private void store() 
	throws ControllerException
    {
	if (log.isDebugEnabled())
	    log.debug("store() called, contact list has " + userContactList.getLength() + " elements.");
	getUserContactListStore().writeAsXml(ContactList.KEY, 
					     userContactList, 
					     ContactList.TYPES);
    }
    
    public void initContactList() 
	throws ControllerException
    {
	if (userContactList == null) {
	    userContactList = 
		(ContactList)getUserContactListStore().
		getObject(ContactList.KEY, ContactList.class);
	    if (userContactList != null) { 
		log.debug("getUserContactList() : read an existing list from store");
	    }
	    else {
		log.debug("getUserContactList() : no existing list in store, creating a new (empty) one");
		userContactList = new ContactList();
	    }
	}
    }

    private GeneralStore getUserContactListStore() 
	throws ControllerException
    {
	if (userContactListStore == null)
	    userContactListStore = new GeneralStore(serverConnection, applicationConfiguration.getUserContactsFolder());
	return userContactListStore;
    }


}
