//  Gnomoradio - rainbow/hub-client.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "hub-client.h"
#include "rainbow/util.h"

extern "C" {
#include "rainbow/sha1.h"
}

#include <fcntl.h>
#include <sys/stat.h>

#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <unistd.h>
#include <cstdio>
#include <string>

#include <cstdlib>
#include <sstream>

#define CRLF "\r\n"

namespace Rainbow
{
	class Checksum : public SigC::Object
	{
	public:
		static void verify (const SigC::Slot2<void,bool,Resource*> &slot,
				    Resource *resource);

	private:
		Checksum (const SigC::Slot2<void,bool,Resource*> &slot,
			  Resource *r);
		void verify_thread ();
		void verify_do ();
		void dispatch ();
		
		SigC::Signal2<void,bool,Resource*> signal;
		Glib::Dispatcher dispatcher;
		bool success;
		Resource *resource;
		Glib::ustring filename;
		std::vector<guint8> checksum;
	};

	class HubConnector : public SigC::Object
	{
	public:
		static void connect (const SigC::Slot1<void,int> &slot,
				     const Glib::ustring &hub_server);

	private:
		HubConnector (const SigC::Slot1<void,int> &slot,
			      const Glib::ustring &hub_server);
		void thread ();
		void dispatch ();

		SigC::Signal1<void,int> signal;
		Glib::Dispatcher dispatcher;
		int sock;
		sockaddr_in hub;
		Glib::ustring server;
	};
}

using namespace std;
using namespace SigC;
using namespace Glib;
using namespace Rainbow;

Rainbow::Checksum::Checksum (const SigC::Slot2<void,bool,Resource*> &slot,
			     Resource *r)
	: resource(r),
	  filename(r->get_filename()),
	  checksum(r->checksum),
	  success(false)
{
	signal.connect(slot);
	dispatcher.connect(SigC::slot(*this, &Checksum::dispatch));

	if (!Glib::thread_supported())
		Glib::thread_init();
	Glib::Thread::create(SigC::slot(*this, &Checksum::verify_thread), false);
}

void Rainbow::Checksum::verify (const SigC::Slot2<void,bool,Resource*> &slot,
				Resource *resource)
{
	new Checksum(slot, resource);
}

void Rainbow::Checksum::verify_thread ()
{
	verify_do();
	dispatcher();
}

void Rainbow::Checksum::verify_do ()
{
	std::string fn;
	try {
		fn = filename_from_utf8(filename);
	} catch (...) {
		return;
	}

        FILE *file = fopen(fn.c_str(), "rb");
	if (file) {
		const int buf_size = 4096;
		unsigned char buf[buf_size];

		SHA1Context cxt;
		if (SHA1Reset(&cxt))
			return;

		size_t r;
		for (;;) {
			r = fread(buf, 1, buf_size, file);
			if (r == 0)
				break;
			if (SHA1Input(&cxt, buf, r))
				return;
		}
		
		fclose(file);

		guint8 sum[SHA1HashSize];
		if (SHA1Result(&cxt, sum))
			return;

		for (int i = 0; i < SHA1HashSize; ++i)
			if (sum[i] != checksum[i])
				return;
		success = true;
	}
}

void Rainbow::Checksum::dispatch ()
{
	signal(success, resource);
	delete this;
}

void Rainbow::HubConnector::connect (const SigC::Slot1<void,int> &slot,
				     const Glib::ustring &hub_server)
{
	new HubConnector(slot, hub_server);
}

Rainbow::HubConnector::HubConnector (const SigC::Slot1<void,int> &slot,
				     const Glib::ustring &hub_server)
	: server(hub_server),
	  sock(0)
{
	signal.connect(slot);
	dispatcher.connect(SigC::slot(*this, &HubConnector::dispatch));

	struct hostent *host = gethostbyname(hub_server.c_str());
	if (!host) {
		logmsg << "HubClient: Could not create host entry for " << hub_server << endl;
	} else {
		sock = socket(PF_INET, SOCK_STREAM, 0);
		if (sock < 1)
			logmsg << "HubClient: Could not create socket" << endl;
	}

	if (sock > 0) {
		hub.sin_addr = *(struct in_addr*) host->h_addr;
		hub.sin_family = AF_INET;
		hub.sin_port = htons(18373);
		
		if (!Glib::thread_supported())
			Glib::thread_init();
		Glib::Thread::create(SigC::slot(*this, &HubConnector::thread), false);
	} else {
		signal(sock);
		delete this;
	}
}

void Rainbow::HubConnector::thread ()
{
	if (::connect(sock, (struct sockaddr*) &hub, sizeof(hub))) {
		close(sock);
		sock = 0;
		logmsg << "HubClient: Could not connect to host" << endl;
	} else {
		// connected, now send handshake line
		string send = "RAINBOW/1.0 " "4617" CRLF; // FIXME
		if (!send_data_on_socket(sock, send)) {
			close(sock);
			sock = 0;
			logmsg << "HubClient: Could not initiate handshake (broken socket)" << endl;
		}
	}
	
	dispatcher();
}

void Rainbow::HubConnector::dispatch ()
{
	signal(sock);
	delete this;
}

static inline guint8 hex_to_int (char c)
{
	if (c >= '0' && c <= '9')
		return guint8(c - '0');
	if (c >= 'A' && c <= 'F')
		return guint8(c - 'A') + 10;
	if (c >= 'a' && c <= 'f')
		return guint8(c - 'a') + 10;
	return 0;
}

void Rainbow::HubClient::parse_resource (xmlpp::Element *xml, ref_ptr<RdfResource> rdf, Resource *r)
{
	r->obtained_info = true;

	if (xml) {
		xmlpp::Node::NodeList children = xml->get_children();
		for (xmlpp::Node::NodeList::iterator i = children.begin(); i != children.end(); ++i) {
			xmlpp::Element *el = dynamic_cast<xmlpp::Element*>(*i);
			if (!el)
				continue;
			xmlpp::TextNode *content = el->get_child_text();
			xmlpp::Attribute *resource_link = el->get_attribute("resource");
			
			if (el->get_name() == "available") {
				if (resource_link)
					r->available_locations.push_back(resource_link->get_value());
			} else if (el->get_name() == "license") {
				if (resource_link) {
					Glib::ustring absolute = RdfResource::absolute_uri(resource_link->get_value(),
										    rdf->get_base_uri());
					License::get_and_do(absolute,
							    bind(slot(*this, &HubClient::license_obtained_callback), r),
							    rdf->get_secondary_info(absolute));
				}
			} else if (el->get_name() == "checksum") {
				if (content && content->get_content().length()) {
					r->checksum.resize(SHA1HashSize);
					try {
						string chk = content->get_content();
						int n = chk.length() / 2;
						if (n > SHA1HashSize) // we only have 20 bytes to work with
							n = SHA1HashSize;
						for (int i = 0; i < n; ++i) {
							r->checksum[i] = hex_to_int(chk[i*2]) * 16;
							r->checksum[i] += hex_to_int(chk[i*2+1]);
						}
					} catch (...) {
					}
				}
			} else if (el->get_name() == "representationOf") {
				
			}
		}
	}

	if (r->obtained_info) {
		if (r->prepare_once_info_is_obtained)
			prepare_resource(r);
		r->signal_found_info(bool(xml));
	}
}

void Rainbow::HubClient::license_obtained_callback (Rainbow::ref_ptr<Rainbow::License> license, Resource *r)
{
	r->sharable = license->sharable();
}

Rainbow::HubClient::HubClient (const Glib::ustring &hub)
	: server(this),
	  hub_host(hub),
	  hubsock(0),
	  cache_size(50),
	  total_allocated_size(0)
{
	alarm.signal_alarm().connect(slot(*this, &HubClient::connect));

	load_database();
	save_alarm.signal_alarm().connect(slot(*this, &HubClient::on_save_alarm));
	save_alarm.set_relative(save_alarm_interval);

	// FIXME: we could wait until connected to the hub to start http services
	if (!server.start(http_server_port)) {
		logmsg << "HubClient: Could not start http server" << endl;
	}
	connect();
}

Rainbow::HubClient::~HubClient ()
{
	disconnect();
	save_database();
}

void Rainbow::HubClient::on_save_alarm ()
{
	save_database();
	save_alarm.set_relative(save_alarm_interval);
}

void Rainbow::HubClient::load_database ()
{
	Mutex::Lock lock(db_mutex);
	xmlpp::DomParser tree;
	try {
		tree.parse_file(string(getenv("HOME")) + "/.rainbow-db.xml");
	} catch (...) {
		return;
	}

	xmlpp::Node *root = tree.get_document()->get_root_node();
	xmlpp::Node::NodeList headings = root->get_children();

	for (xmlpp::Node::NodeList::iterator heading = headings.begin();
	     heading != headings.end(); ++heading) {
		if ((*heading)->get_name() == "resources") {
			xmlpp::Node::NodeList resources = (*heading)->get_children();
			for (xmlpp::Node::NodeList::iterator i = resources.begin();
			     i != resources.end(); ++i) {
				xmlpp::Element *resource = dynamic_cast<xmlpp::Element*>(*i);
				if (!resource)
					continue;

				xmlpp::Attribute *url = resource->get_attribute("url");
				xmlpp::Attribute *filename = resource->get_attribute("filename");
				if (!url || !filename)
					continue;

				Resource *r = new Resource(url->get_value(), filename->get_value());

				xmlpp::Attribute *sz = resource->get_attribute("size");
				if (sz)
					set_allocated_size(r, atoi(sz->get_value().c_str()));
				else
					check_allocated_size(r);

				xmlpp::Attribute *sharable = resource->get_attribute("sharable");
				if (sharable && sharable->get_value() == "true")
					r->sharable = true;

				db.insert(make_pair(url->get_value(),
						    ref_ptr<Resource>(r)));
			}
		}
	}
}

void Rainbow::HubClient::save_database ()
{
	Mutex::Lock lock(db_mutex);
	xmlpp::Document tree;
	tree.create_root_node("rainbow");
	xmlpp::Node *root = tree.get_root_node();

	xmlpp::Node *resources = root->add_child("resources");
	map<Glib::ustring,ref_ptr<Resource> >::iterator p;
	for (p = db.begin(); p != db.end(); ++p) {
		if (!p->second->prepared)
			continue;
		xmlpp::Element *node = resources->add_child("resource");
		node->set_attribute("url", p->second->get_url());
		node->set_attribute("filename", p->second->get_filename());
		node->set_attribute("sharable", p->second->sharable ? "true" : "false");
		if (p->second->allocated_size) {
			ostringstream as;
			as << p->second->allocated_size;
			node->set_attribute("size", as.str());
		}
	}

	tree.write_to_file_formatted(string(getenv("HOME")) + "/.rainbow-db.xml");
}

void Rainbow::HubClient::connect ()
{
	if (hub_host.length() == 0)
		return;

	if (hubsock > 0) {
		// connection is already/still established
		alarm.set_relative(reconnect_interval);
	} else {
		HubConnector::connect(slot(*this, &HubClient::on_connect), hub_host);
	}
}

void Rainbow::HubClient::disconnect ()
{
	if (hubsock > 0) {
		close(hubsock);
	}
	hubsock = 0;
}

void Rainbow::HubClient::on_connect (int socket)
{
	hubsock = socket;

	// reset alarm
	alarm.set_relative(reconnect_interval);

	// send database to server
	if (socket > 0) {
		string send;
		map<Glib::ustring,ref_ptr<Resource> >::iterator i;
		for (i = db.begin(); i != db.end(); ++i)
			if (i->second->sharable)
				send.append('+' + i->first + CRLF);
		
		if (!send_data_on_socket(hubsock, send))
			disconnect();
	}
}

ref_ptr<Resource> Rainbow::HubClient::find (const Glib::ustring &url)
{
	Mutex::Lock lock(db_mutex);
	map<Glib::ustring,ref_ptr<Resource> >::iterator p;
	p = db.find(url);
	if (p != db.end())
		return p->second;
	else
		return ref_ptr<Resource>();
}

ref_ptr<Resource> Rainbow::HubClient::create (const Glib::ustring &url)
{
	ref_ptr<Resource> r = find(url);
	if (!&*r) {
		Mutex::Lock lock(db_mutex);
		r = ref_ptr<Resource>(new Resource(url));
		RdfResource::get_and_do(url, bind(slot(*this, &HubClient::parse_resource), &*r));
		db.insert(make_pair(url, r));
	}
	return r;
}

bool Rainbow::HubClient::get_filename_threadsafe (const Glib::ustring &url,
						  Glib::ustring &filename,
						  bool only_if_sharable)
{
	ref_ptr<Resource> r = find(url);
	if (!&*r)
		return false;
	if (only_if_sharable && !r->sharable)
		return false;

	Mutex::Lock lock(db_mutex);
	filename = r->get_filename();
	return true;
}

size_t Rainbow::HubClient::get_size_threadsafe (const Glib::ustring &url)
{
	ref_ptr<Resource> r = find(url);
	if (!&*r)
		return 0;

	Mutex::Lock lock(db_mutex);
	size_t s = r->allocated_size;
	return s;
}

static inline bool check_path (const char *path)
{
	struct stat s;

	if (!stat(path, &s)) {
		if (!S_ISDIR(s.st_mode)) {
			//g_warning("%s exists but is not a directory", path);
			return false;
		}
	} else {
		if (mkdir(path, 0755)) {
			//g_warning("%s cannot be created", path);
			return false;
		}
	}

	return true;
}

Glib::ustring Rainbow::HubClient::random_filename ()
{
	string basedir = getenv("HOME") + string("/.rainbow-cache");
	check_path(basedir.c_str()); // error out if false?

	string filename;
	struct stat s;
	do {
		char rnd[] = "        ";
		for (int i = 0; i < 8; ++i)
			rnd[i] = rand() % 26 + 'a';
		filename = basedir + string("/") + rnd;
	} while (!stat(filename.c_str(), &s));

	return filename_to_utf8(filename);
}

void Rainbow::HubClient::prepare_resource (Resource *r)
{
	if (r->prepared || r->downloading)
		return;
	if (!r->obtained_info) {
		r->prepare_once_info_is_obtained = true;
		return;
	}

	r->downloading = true;

	query_hub(r); // FIXME: make this asynchronous
	start_download(r);
}

void Rainbow::HubClient::query_hub (Resource *r)
{
	string query('?' + r->get_url() + CRLF);
	string buffer;
	int lines_remaining = -1;
	if (hubsock <= 0 || !send_data_on_socket(hubsock, query)) {
		disconnect();
	} else {
		while (lines_remaining) {
			const size_t buf_size = 2048;
			char buf[buf_size];
			ssize_t rb = read(hubsock, buf, buf_size);
			if (rb <= 0)
				break;
			buffer.append(buf, rb);
			string::size_type p;
			while (lines_remaining
			       && (p = buffer.find(CRLF)) != string::npos) {
				string line = buffer.substr(0, p);
				buffer = buffer.substr(p + 2);
				if (lines_remaining == -1) {
					// first line
					lines_remaining = atoi(line.c_str());
					if (lines_remaining < 0)
						return;
				} else {
					// add to array
					try {
						r->mirrors.push_back(Glib::ustring(line));
					} catch (...) {
					}
					--lines_remaining;
				}
			}
		}
	}
}

void Rainbow::HubClient::start_download (Resource *r)
{
	if (!r->filename.size() && (r->mirrors.size() || r->available_locations.size()))
		r->filename = random_filename();

	if (r->mirrors.size()) {
		Glib::ustring mirror = r->mirrors[r->mirrors.size() - 1];
		r->mirrors.pop_back();
		unsigned short port = 80;
		Glib::ustring::size_type colon = mirror.find(':');
		if (colon != Glib::ustring::npos) {
			Glib::ustring prt = mirror.substr(colon + 1);
			mirror = mirror.substr(0, colon);
			port = atoi(prt.c_str());
		}
		// FIXME: use different DL
		r->dl.reset(new HttpClient(mirror, port, false));
		if (r->get_url().substr(0, 6) != "http:/") {
			start_download(r); // try, try again
			return;
		}
		r->dl->get(r->get_url().substr(6), r->get_filename());
		r->dl->signal_request_done.connect(bind(slot(*this, &HubClient::file_download_done_callback), r));
		r->dl->signal_download_percent.connect(bind(slot(*this, &HubClient::file_download_percent_callback), r));
	} else if (r->available_locations.size() > r->available_locations_cursor) {
		Glib::ustring remote_host, remote_file;
		unsigned short remote_port;
		if (!HttpClient::parse_url(r->available_locations[r->available_locations_cursor++],
					   remote_host, remote_port, remote_file)) {
			start_download(r); // try, try again
			return;
		}
		r->dl.reset(new HttpClient(remote_host, remote_port, false));
		r->dl->get(remote_file, r->get_filename());
		r->dl->signal_request_done.connect(bind(slot(*this, &HubClient::file_download_done_callback), r));
		r->dl->signal_download_percent.connect(bind(slot(*this, &HubClient::file_download_percent_callback), r));
	} else {
		// not available anywhere now
		logmsg << "HubClient: Resource not available currently: " << r->get_url() << endl;
		r->downloading = false;
		r->signal_download_done(false);
	}
}

void Rainbow::HubClient::set_hub (const Glib::ustring &hub)
{
	if (hub == hub_host)
		return;

	hub_host = hub;
	alarm.reset();
	disconnect();
	connect();
}

void Rainbow::HubClient::file_download_percent_callback (unsigned int percent, Resource *r)
{
	r->signal_downloading(percent);
}

void Rainbow::HubClient::download_success (Resource *r)
{
	r->prepared = true;
	r->downloading = false;
	check_allocated_size(r);
	r->signal_download_done(true);
}

void Rainbow::HubClient::file_download_done_callback (bool success, Resource *r)
{
	r->prepared = success;
	r->dl.reset();

	if (success) {
		// verify checksum
		if (r->checksum.size() != 0)
			Checksum::verify(slot(*this, &HubClient::verify_checksum_callback), r);
		else
			download_success(r);
	} else
		start_download(r);
}

void Rainbow::HubClient::verify_checksum_callback (bool success, Resource *r)
{
	if (success) {
		download_success(r);
		// tell hub we have this file
		if (hubsock > 0) {
			string send(' ' + r->get_url() + CRLF);
			if (r->sharable)
				send[0] = '+';
			else
				send[0] = '-'; // fixme: is the necessary?

			if (!send_data_on_socket(hubsock, send))
				disconnect();
		}
	} else
		start_download(r);
}

void Rainbow::HubClient::check_allocated_size (Resource *r)
{
	struct stat s;
	try {
		if (!stat(filename_from_utf8(r->get_filename()).c_str(), &s))
			set_allocated_size(r, s.st_size);
	} catch (...) {
	}
}

void Rainbow::HubClient::set_allocated_size (Resource *r, size_t s)
{
	total_allocated_size -= r->allocated_size / 1024;
	r->allocated_size = s;
	total_allocated_size += r->allocated_size / 1024;

	if (s != 0)
		review_cache_size();
}

void Rainbow::HubClient::review_cache_size ()
{
	const size_t allowed_allocated_size = cache_size * 1024;
	if (total_allocated_size > allowed_allocated_size)
		signal_must_uncache(total_allocated_size - allowed_allocated_size);
}

void Rainbow::HubClient::set_cache_size (unsigned int megabytes)
{
	cache_size = megabytes;
	review_cache_size();
}

void Rainbow::HubClient::uncache_resource (Resource *r)
{
	if (!r->prepared)
		return;

	set_allocated_size(r, 0);
	r->prepared = false;
	r->obtained_info = false;
	r->download_percent = 0;
	r->available_locations_cursor = 0;
	r->available_locations.clear();
	r->mirrors.clear();
	r->signal_file_deleted();

	unlink(r->filename.c_str());

	// fixme: this should not be necessary now, but it needs to be done at some point in case you decide to download again
	RdfResource::get_and_do(r->get_url(), bind(slot(*this, &HubClient::parse_resource), r));
}

// fixme: maybe size should be stored, with a boolean whether it's allocated or used
