// -*- C++ -*-

/* 
 * GChemPaint
 * view.cc
 *
 * Copyright (C) 2001-2003
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"
#include "widgetdata.h"
#include "view.h"
#include "settings.h"
#include "document.h"
#include "tool.h"
#include "tools.h"
#include "globals.h"
#include "text.h"
#include "selectiontool.h"
#include "libgcpcanvas/gprintable.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include <math.h>
#include <pango/pango-context.h>

/*
Derivation of a new widget from gnome_canvas with an event for updating canvas size
*/

typedef struct _GnomeCanvasGCP           GnomeCanvasGCP;
typedef struct _GnomeCanvasGCPClass      GnomeCanvasGCPClass;

#define GNOME_TYPE_CANVAS_GCP            (gnome_canvas_gcp_get_type ())
#define GNOME_CANVAS_GCP(obj)            (GTK_CHECK_CAST ((obj), GNOME_TYPE_CANVAS_GCP, GnomeCanvasGCP))
#define GNOME_CANVAS_CLASS_GCP(klass)    (GTK_CHECK_CLASS_CAST ((klass), GNOME_TYPE_CANVAS_GCP, GnomeCanvasGCPClass))
#define GNOME_IS_CANVAS_GCP(obj)         (GTK_CHECK_TYPE ((obj), GNOME_TYPE_CANVAS_GCP))
#define GNOME_IS_CANVAS_GCP_CLASS(klass) (GTK_CHECK_CLASS_TYPE ((klass), GNOME_TYPE_CANVAS_GCP))
#define GNOME_CANVAS_GCP_GET_CLASS(obj)  (GTK_CHECK_GET_CLASS ((obj), GNOME_TYPE_CANVAS_GCP, GnomeCanvasGCPClass))

enum {
  UPDATE_BOUNDS,
  LAST_SIGNAL
};

struct _GnomeCanvasGCP {
	GnomeCanvas canvas;
};

struct _GnomeCanvasGCPClass {
	GnomeCanvasClass parent_class;

	void (* update_bounds) (GnomeCanvasGCP *canvas);
};

GType gnome_canvas_gcp_get_type (void) G_GNUC_CONST;
static void gnome_canvas_gcp_class_init(GnomeCanvasGCPClass *Class);
static void gnome_canvas_gcp_init(GnomeCanvasGCP *canvas);
static void gnome_canvas_gcp_update_bounds(GnomeCanvasGCP *canvas);
static GtkWidget *gnome_canvas_gcp_new (void);

static guint gnome_canvas_gcp_signals[LAST_SIGNAL] = { 0 };

GType
gnome_canvas_gcp_get_type (void)
{
	static GType canvas_gcp_type;

	if (!canvas_gcp_type) {
		static const GTypeInfo object_info = {
			sizeof(GnomeCanvasGCPClass),
			(GBaseInitFunc)NULL,
			(GBaseFinalizeFunc)NULL,
			(GClassInitFunc)gnome_canvas_gcp_class_init,
			(GClassFinalizeFunc)NULL,
			NULL,			/* class_data */
			sizeof(GnomeCanvasGCP),
			0,			/* n_preallocs */
			(GInstanceInitFunc)gnome_canvas_gcp_init,
			NULL			/* value_table */
		};

		canvas_gcp_type = g_type_register_static(GNOME_TYPE_CANVAS, "GnomeCanvasGCP",
						      &object_info, (GTypeFlags)0);
	}

	return canvas_gcp_type;
}

GtkWidget *gnome_canvas_gcp_new(void)
{
	return GTK_WIDGET(g_object_new(GNOME_TYPE_CANVAS_GCP, "aa", TRUE, NULL));
}

void gnome_canvas_gcp_class_init(GnomeCanvasGCPClass *Class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (Class);
	gnome_canvas_gcp_signals[UPDATE_BOUNDS] =
	g_signal_new ("update_bounds",
				  G_TYPE_FROM_CLASS(gobject_class),
				  G_SIGNAL_RUN_LAST,
				  G_STRUCT_OFFSET(GnomeCanvasGCPClass, update_bounds),
				  NULL, NULL,
				  g_cclosure_marshal_VOID__VOID,
				  G_TYPE_NONE, 0
				  );
	Class->update_bounds = gnome_canvas_gcp_update_bounds;
}

void gnome_canvas_gcp_init(GnomeCanvasGCP *canvas)
{
}

void gnome_canvas_gcp_update_bounds(GnomeCanvasGCP *canvas)
{
	while (gtk_events_pending()) gtk_main_iteration();
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(canvas), "data");
	double x1, y1, x2, y2;
	gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(pData->Group), &x1, &y1, &x2, &y2);
	gcpView *pView = (gcpView*)g_object_get_data(G_OBJECT(canvas), "view");
	pView->UpdateSize(x1, y1, x2, y2);
}

bool on_event(GnomeCanvasItem *item, GdkEvent *event, GtkWidget* widget)
{
	gcpView* pView = (gcpView*) g_object_get_data(G_OBJECT(widget), "view");
	return pView->OnEvent(item, event, widget);
}

static bool on_destroy(GtkWidget *widget, gcpView * pView)
{
	pView->OnDestroy(widget);
	return true;
}

static bool on_size(GtkWidget *widget, GtkAllocation *alloc, gcpView * pView)
{
	pView->OnSize(alloc->width, alloc->height);
	return true;
}

void on_receive(GtkClipboard *clipboard, GtkSelectionData *selection_data, gcpView * pView)
{
	pView->OnReceive(clipboard, selection_data);
}

using namespace std;

gcpView::gcpView(gcpDocument *pDoc)
{
	m_pDoc = pDoc;
	m_PangoFontDesc = pango_font_description_new();
	pango_font_description_set_family(m_PangoFontDesc, DefaultFontFamily);
	pango_font_description_set_style(m_PangoFontDesc, DefaultFontStyle);
	pango_font_description_set_weight(m_PangoFontDesc, DefaultFontWeight);
	pango_font_description_set_variant(m_PangoFontDesc, DefaultFontVariant);
	pango_font_description_set_stretch(m_PangoFontDesc, DefaultFontStretch);
	pango_font_description_set_size(m_PangoFontDesc, DefaultFontSize);
	m_sFontName = pango_font_description_to_string(m_PangoFontDesc);
	m_PangoSmallFontDesc = pango_font_description_new();
	pango_font_description_set_family(m_PangoSmallFontDesc, DefaultFontFamily);
	pango_font_description_set_style(m_PangoSmallFontDesc, DefaultFontStyle);
	pango_font_description_set_weight(m_PangoSmallFontDesc, DefaultFontWeight);
	pango_font_description_set_variant(m_PangoSmallFontDesc, DefaultFontVariant);
	pango_font_description_set_stretch(m_PangoSmallFontDesc, DefaultFontStretch);
	pango_font_description_set_size(m_PangoSmallFontDesc, DefaultFontSize * 2 / 3);
	m_sSmallFontName = pango_font_description_to_string(m_PangoSmallFontDesc);
	m_PangoTinyFontDesc = pango_font_description_new();
	pango_font_description_set_family(m_PangoTinyFontDesc, DefaultFontFamily);
	pango_font_description_set_style(m_PangoTinyFontDesc, DefaultFontStyle);
	pango_font_description_set_weight(m_PangoTinyFontDesc, DefaultFontWeight);
	pango_font_description_set_variant(m_PangoTinyFontDesc, DefaultFontVariant);
	pango_font_description_set_stretch(m_PangoTinyFontDesc, DefaultFontStretch);
	pango_font_description_set_size(m_PangoTinyFontDesc, DefaultFontSize / 2);
	m_sTinyFontName = pango_font_description_to_string(m_PangoTinyFontDesc);
	m_DefaultPadding = DefaultPadding;
	m_DefaultStereoBondWidth = DefaultStereoBondWidth;
	m_DefaultHashWidth = DefaultHashWidth;
	m_DefaultHashDist = DefaultHashDist;
	m_DefaultBondWidth = DefaultBondWidth;
	m_DefaultBondLength = DefaultBondLength;
	m_DefaultBondDist = DefaultBondDist;
	m_DefaultArrowLength = DefaultArrowLength;
	m_DefaultArrowHeadA = DefaultArrowHeadA;
	m_DefaultArrowHeadB = DefaultArrowHeadB;
	m_DefaultArrowHeadC = DefaultArrowHeadC;
	m_DefaultArrowDist = DefaultArrowDist;
	m_width = 400;
	m_height = 300;
	m_ActiveRichText = NULL;
	if (!IsEmbedded()) CreateNewWidget();
}

gcpView::~gcpView()
{
	if (!bCloseAll &&!IsEmbedded())
	{
		void* Item = g_object_get_data(G_OBJECT(m_pWidget), "menu");
		if (Item) gtk_widget_destroy(GTK_WIDGET(Item));
	}
	g_object_unref(G_OBJECT(m_PangoContext));
	if (m_sFontName) g_free(m_sFontName);
	if (m_sSmallFontName) g_free(m_sSmallFontName);
}

static bool bDragging = false;

bool gcpView::OnEvent(GnomeCanvasItem *item, GdkEvent *event, GtkWidget* widget)
{
	if ((!m_pDoc->GetEditable()) || (!pActiveTool)) return true;
	if (item == (GnomeCanvasItem*)m_ActiveRichText)
	{
		GnomeCanvasItemClass* klass = GNOME_CANVAS_ITEM_CLASS(((GTypeInstance*)item)->g_class);
		return klass->event(item, event);
	}
	else if (pActiveTool->OnEvent(event)) return true;
	m_pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(widget), "data");
	m_pWidget = widget;
	double x, y;
	x = event->button.x;
	y = event->button.y;
	gnome_canvas_item_w2i(GNOME_CANVAS_ITEM(m_pData->Group), &x, &y);
	if (event->type == GDK_BUTTON_PRESS)
	{
		if (item == m_pData->Background)
		{
			item = NULL;
			std::map<Object*, GnomeCanvasGroup*>::iterator i = m_pData->Items.begin();
			gcpBond* pBond;
			while (i != m_pData->Items.end())
			{
				if ((*i).first->GetType() == BondType)
				{
					pBond = (gcpBond*)(*i).first;
					if (pBond->GetDist(x / m_pData->ZoomFactor, y / m_pData->ZoomFactor) < (m_DefaultPadding + m_DefaultBondWidth / 2) / m_pData->ZoomFactor)
					{
						item = GNOME_CANVAS_ITEM((*i).second);
						break;
					}
				}
				i++;
			}
		}
	}
	Object* Obj;
	switch (event->type)
	{
	case GDK_BUTTON_PRESS:
		Obj = (item) ? (Object*) g_object_get_data(G_OBJECT(item), "object") : NULL;
		switch (event->button.button)
		{
			case 1:
			{
				if (bDragging) break;
				bool result = pActiveTool->OnClicked(this, Obj, x, y, event->button.state);
				if (item && (item == (GnomeCanvasItem*)m_ActiveRichText))
				{
					GnomeCanvasItemClass* klass = GNOME_CANVAS_ITEM_CLASS(((GTypeInstance*)item)->g_class);
					return klass->event(item, event);
				}
				bDragging = result;
				return true;
			}
			case 2:
			{
				m_lastx = x;
				m_lasty = y;
				GtkClipboard *clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);
				OnPasteSelection (m_pWidget, clipboard);
				return true;
			}
			case 3:
				if (Obj) Obj->ShowContextualMenu(event->button.button, event->button.time);
		}
		break;
	case GDK_MOTION_NOTIFY:
		if (!bDragging) break;
		pActiveTool->OnDrag(x, y, event->button.state);
		return true;
	case GDK_BUTTON_RELEASE:
		switch (event->button.button)
		{
		case 1:
			if (!bDragging) break;
			bDragging = false;
			pActiveTool->OnRelease(x, y, event->button.state);
			clear_status();
			return true;
		}
		break;
	}
	return false;
}

void gcpView::AddObject(Object* pObject)
{
	std::list<GtkWidget*>::iterator i;
	for (i = m_Widgets.begin(); i != m_Widgets.end(); i++)
		pObject->Add(*i);
}

GtkWidget* gcpView::CreateNewWidget()
{
	gtk_widget_push_colormap(gdk_rgb_get_colormap());
	m_pWidget = gnome_canvas_gcp_new();
	gtk_widget_pop_colormap();
	GtkWidget* pWidget = (m_Widgets.size() > 0) ? m_Widgets.front() : NULL;
	if (m_pWidget)
	{
		g_object_set_data(G_OBJECT(m_pWidget), "view", this);
		g_object_set_data(G_OBJECT(m_pWidget), "doc", m_pDoc);
		m_pData = new gcpWidgetData();
		m_pData->Canvas = m_pWidget;
		g_object_set_data(G_OBJECT(m_pWidget), "data", m_pData);
		m_pData->View = this;
		gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(m_pWidget), 1);
		gnome_canvas_set_scroll_region(GNOME_CANVAS(m_pWidget), 0, 0, m_width, m_height);
//		gtk_widget_set_size_request(m_pWidget, m_width, m_height);//FIXME: probably necessary when embedded
		m_pData->ZoomFactor = DefaultZoomFactor;
		m_pData->Padding = m_DefaultPadding * DefaultZoomFactor;
		m_pData->BondWidth = m_DefaultBondWidth * DefaultZoomFactor;
		m_pData->BondLength = m_DefaultBondLength * DefaultZoomFactor;
		m_pData->BondDist = m_DefaultBondDist * DefaultZoomFactor;
		m_pData->ArrowLength = m_DefaultArrowLength * DefaultZoomFactor;
		m_pData->ArrowHeadA = m_DefaultArrowHeadA * DefaultZoomFactor;
		m_pData->ArrowHeadB = m_DefaultArrowHeadB * DefaultZoomFactor;
		m_pData->ArrowHeadC = m_DefaultArrowHeadC * DefaultZoomFactor;
		m_pData->ArrowDist = m_DefaultArrowDist * DefaultZoomFactor;
		m_pData->HashWidth = m_DefaultHashWidth * DefaultZoomFactor;
		m_pData->HashDist = m_DefaultHashDist * DefaultZoomFactor;
		m_pData->StereoBondWidth = m_DefaultStereoBondWidth * DefaultZoomFactor;
		m_pData->Group = GNOME_CANVAS_GROUP (gnome_canvas_item_new (
									gnome_canvas_root(GNOME_CANVAS(m_pWidget)),
									gnome_canvas_group_ext_get_type (),
									NULL));
		m_pData->Background = gnome_canvas_item_new(
									m_pData->Group,
									gnome_canvas_rect_get_type(),
									"x1", 0.0,
									"y1", 0.0,
									"x2", (double) m_width,
									"y2", (double) m_height,
									"fill_color", "white",
									NULL);
		g_signal_connect(G_OBJECT(m_pData->Background), "event", G_CALLBACK(on_event), m_pWidget);
		g_signal_connect(G_OBJECT(m_pWidget), "destroy", G_CALLBACK(on_destroy), this);
		g_signal_connect(G_OBJECT(m_pWidget), "size_allocate", G_CALLBACK(on_size), this);
		gtk_widget_show(m_pWidget);
		m_Widgets.push_back(m_pWidget);
		if (pWidget)
		{
			gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(pWidget), "data");
			std::map<Object*, GnomeCanvasGroup*>::iterator i;
			for (i = pData->Items.begin(); i != pData->Items.end(); i++)
				 (*i).first->Add(m_pWidget);
		}
		else
		{
			m_PangoContext = gtk_widget_create_pango_context(m_pWidget);
			g_object_ref(G_OBJECT(m_PangoContext));
			UpdateFont();
		}
	}
	return m_pWidget;
}

void gcpView::OnDestroy(GtkWidget* widget)
{
	if (IsEmbedded())
	{
		delete (gcpWidgetData*)g_object_get_data(G_OBJECT(widget), "data");
		m_Widgets.remove(widget);
	}
	else delete m_pDoc;
}

GnomeCanvasItem* gcpView::GetCanvasItem(GtkWidget* widget, Object* Object)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(widget), "data");
	if ((!pData) || (pData->View != this)) return NULL;
	return (GnomeCanvasItem*) pData->Items[Object];
}

void gcpView::Print(GnomePrintContext *pc, gdouble width, gdouble height)
{
	g_return_if_fail (G_IS_PRINTABLE (m_pData->Group));
	gnome_print_gsave(pc);
	double matrix [6] = {.75, 0., 0., -.75, 0., height};
	if (!IsEmbedded())	//FIXME: add a print setup dialog and use real margins values
	{
		matrix[4] += 30;
		matrix[5] -= 30;
	}
	m_pData->ShowSelection(false);
	Object* pObj = NULL;
	if (m_ActiveRichText)
	{
		pObj = (Object*) g_object_get_data(G_OBJECT(m_ActiveRichText), "object");
		if (pObj) pObj->SetSelected(m_pWidget, SelStateUnselected);
	}
	gnome_print_concat (pc, matrix);
	GPrintable* printable = G_PRINTABLE (m_pData->Group);
	(* G_PRINTABLE_GET_IFACE (printable)->print) (G_PRINTABLE (printable), pc); 
	gnome_print_grestore(pc);
	m_pData->ShowSelection(true);
	if (pObj) pObj->SetSelected(m_pWidget, SelStateUpdating);
}

void gcpView::Update(Object* pObject)
{
	std::list<GtkWidget*>::iterator i;
	switch (pObject->GetType())
	{
		case AtomType:
		case BondType:
			for (i = m_Widgets.begin(); i != m_Widgets.end(); i++)
			{
				gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(*i), "data");
				GnomeCanvasGroup* group = pData->Items[pObject];
				if (group)
				{
					gtk_object_destroy(GTK_OBJECT(group));
					pObject->Add(*i);//FIXME: replace with Update
				}
				else
				{
					pData->Items.erase(pObject);
					pObject->Update(*i);
				}
			}
			break;
		default:
			if (pObject->HasChildren())
			{
				std::map<std::string, Object*>::iterator i;
				Object* pObj;
				for (pObj = pObject->GetFirstChild(i); pObj; pObj = pObject->GetNextChild(i)) Update(pObj);
			}
			else for (i = m_Widgets.begin(); i != m_Widgets.end(); i++)
			{
				pObject->Update(*i);
			}
			break;

	}
}

GnomeCanvasItem* gcpView::GetBackground()
{
	return m_pData->Background;
}

double gcpView::GetZoomFactor()
{
	return m_pData->ZoomFactor;
}

void gcpView::UpdateFont()
{
	pango_context_set_font_description(m_PangoContext, m_PangoFontDesc);
	PangoLayout* pl = pango_layout_new(m_PangoContext);
	PangoRectangle rect;
	pango_layout_set_text(pl, "lj", 2);
	pango_layout_get_extents(pl, &rect, NULL);
	m_dFontHeight = rect.height / PANGO_SCALE;
}
	
void gcpView::Remove(Object* pObject)
{
	std::list<GtkWidget*>::iterator i;
	for (i = m_Widgets.begin(); i != m_Widgets.end(); i++)
	{
		gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(*i), "data");
		Object* pObj = pObject->GetMolecule();
		if (pObj) pData->SelectedObjects.remove(pObj);
		else pData->SelectedObjects.remove(pObject);
		if (pData->Items[pObject]) gtk_object_destroy(GTK_OBJECT(pData->Items[pObject]));
		pData->Items.erase(pObject);
	}
}

void gcpView::UpdateLabel(const gchar* label)
{
	GtkLabel* pLabel = (GtkLabel*)g_object_get_data(G_OBJECT(m_pWidget), "label");
	if (pLabel) gtk_label_set_text(pLabel, label);
	GtkContainer* pMenu = (GtkContainer*)g_object_get_data(G_OBJECT(m_pWidget), "menu");
	if (!pMenu) return;
	GList* l = gtk_container_get_children(pMenu);
	if (GTK_IS_LABEL(l->data)) gtk_label_set_text((GtkLabel*)(l->data), label);
}

void gcpView::OnDeleteSelection(GtkWidget* w)
{
	m_pWidget = w;
	if (!pActiveTool->DeleteSelection())
	{
		m_pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
		gcpWidgetData* pData;
		std::list<GtkWidget*>::iterator j;
		for (j = m_Widgets.begin(); j != m_Widgets.end(); j++)
		{
			if (*j == m_pWidget) continue;
			pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(*j), "data");
			pData->UnselectAll();
		}
		while (!m_pData->SelectedObjects.empty()) m_pDoc->Remove(m_pData->SelectedObjects.front());
		m_pData->ClearSelection();
	}
	m_pDoc->FinishOperation();
	ActivateMenu(EditMenu, CopyMenu, false);
	ActivateMenu(EditMenu, CutMenu, false);
	ActivateMenu(EditMenu, EraseMenu, false);
}

GtkTargetEntry const targets[] = {
	{(char *) GCHEMPAINT_ATOM_NAME,  0, 0},
	{(char *)"UTF8_STRING", 0, 1},
	{(char *)"STRING", 0, 2}
};

void gcpView::OnCopySelection(GtkWidget* w, GtkClipboard* clipboard)
{
	m_pWidget = w;
	m_pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	if (!pActiveTool->CopySelection(clipboard)) m_pData->Copy(clipboard);
}

void gcpView::OnReceive(GtkClipboard* clipboard, GtkSelectionData* selection_data)
{
	if ((selection_data->length <= 0) || !selection_data->data) return;
	guint *DataType = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? &ClipboardDataType: &ClipboardDataType1;
	g_return_if_fail((selection_data->target == gdk_atom_intern (targets[*DataType].target, FALSE)));
	if (pActiveTool->OnReceive(clipboard, selection_data, *DataType)) return;
	if (ToolsDlg) ToolsDlg->Select(SelectId);
	else if (pActiveTool && pActiveTool->Activate(false))
	{
		pActiveTool = Tools[SelectId];
		pActiveTool->Activate(true);
	}
	if (pActiveTool != Tools[SelectId]) return;
	xmlDocPtr xml;
	m_pData->UnselectAll();
	switch (*DataType)
	{
		case 0:
			xml = xmlParseMemory((const char*)selection_data->data, selection_data->length);
			m_pDoc->AddData(xml->children->children);
			xmlFreeDoc(xml);
			break;
		case 1:
			{
				gcpText* text = new gcpText();
				GtkTextBuffer *buf = (GtkTextBuffer*)text->GetTextBuffer();
				gtk_text_buffer_set_text(buf, (const char*)selection_data->data, selection_data->length);
				GtkTextIter start, end;
				gtk_text_buffer_get_bounds(buf, &start, &end);
				gtk_text_buffer_remove_all_tags(buf, &start, &end);
				gtk_text_buffer_apply_tag_by_name(buf, FontName, &start, &end);
				text->OnChanged(buf);
				m_pDoc->AddObject(text);
				m_pData->SetSelected(text);
			}
			break;
		case 2:
			{
				gcpText* text = new gcpText();
				GtkTextBuffer *buf = (GtkTextBuffer*)text->GetTextBuffer();
				if (!g_utf8_validate( (const char*)selection_data->data, selection_data->length, NULL))
				{
					gsize r, w;
					gchar* newstr = g_locale_to_utf8((const char*)selection_data->data, selection_data->length, &r, &w, NULL);
					gtk_text_buffer_set_text(buf, newstr, w);
					g_free(newstr);
				}
				else
					gtk_text_buffer_set_text(buf, (const char*)selection_data->data, selection_data->length);
				GtkTextIter start, end;
				gtk_text_buffer_get_bounds(buf, &start, &end);
				gtk_text_buffer_remove_all_tags(buf, &start, &end);
				gtk_text_buffer_apply_tag_by_name(buf, FontName, &start, &end);
				text->OnEndUserAction(buf);
				m_pDoc->AddObject(text);
				m_pData->SetSelected(text);
			}
			break;
	}
	ArtDRect rect;
	double dx, dy;
	while(gtk_events_pending()) gtk_main_iteration();
	m_pDoc->AbortOperation();
	m_pData->GetSelectionBounds(rect);
	if (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))
	{
		//center the pasted data at the center of the visible area
		if (IsEmbedded())
		{
			dx = m_pWidget->allocation.width / 2. - (rect.x0 + rect.x1) / 2.;
			dy = m_pWidget->allocation.height / 2. - (rect.y0 + rect.y1) / 2.;
		}
		else
		{
			GtkAdjustment *horiz, *vert;
			GtkWidget* parent = gtk_widget_get_parent(m_pWidget);
			horiz = gtk_viewport_get_hadjustment(GTK_VIEWPORT(parent));
			vert = gtk_viewport_get_vadjustment(GTK_VIEWPORT(parent));
			dx = horiz->value + horiz->page_size / 2.  - (rect.x0 + rect.x1) / 2.;
			dy = vert->value + vert->page_size / 2.  - (rect.y0 + rect.y1) / 2.;
		}
	}
	else //center the pasted data at the mouse click position
	{
		dx = m_lastx - (rect.x0 + rect.x1) / 2.;
		dy = m_lasty - (rect.y0 + rect.y1) / 2.;
	}
	m_pData->MoveSelection(dx, dy);
	SelectionTool.AddSelection(m_pData);
	m_pDoc->PopOperation();
	gcpOperation* pOp = m_pDoc->GetNewOperation(GCP_ADD_OPERATION);
	std::list<Object*>::iterator i;
	for (i = m_pData->SelectedObjects.begin(); i != m_pData->SelectedObjects.end(); i++)
		pOp->AddObject(*i);
	m_pDoc->FinishOperation();
	gnome_canvas_gcp_update_bounds(GNOME_CANVAS_GCP(m_pData->Canvas));
}

void gcpView::OnPasteSelection(GtkWidget* w, GtkClipboard* clipboard)
{
	if (pActiveTool->PasteSelection(clipboard)) return;
	m_pWidget = w;
	m_pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	guint *DataType = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? &ClipboardDataType: &ClipboardDataType1;
	GdkAtom targets_atom  = gdk_atom_intern (targets[*DataType].target, FALSE);
	gtk_clipboard_request_contents(clipboard, targets_atom,  (GtkClipboardReceivedFunc)on_receive, this);
}

void gcpView::OnCutSelection(GtkWidget* w, GtkClipboard* clipboard)
{
	if (!pActiveTool->CutSelection(clipboard))
	{
		OnCopySelection(w, clipboard);
		OnDeleteSelection(w);
	}
	ActivateMenu(EditMenu, CopyMenu, false);
	ActivateMenu(EditMenu, CutMenu, false);
	ActivateMenu(EditMenu, EraseMenu, false);
}

bool gcpView::OnKeyPress(GtkWidget* w, GdkEventKey* event)
{
	if (pActiveTool->OnEvent((GdkEvent*)event)) return true;
	switch(event->keyval)
	{
		case GDK_Delete:
		case GDK_Clear:
		case GDK_BackSpace:
			OnDeleteSelection(w);
			return true;
		case GDK_Shift_L:
		case GDK_Shift_R:
			if (pActiveTool) pActiveTool->OnKeyPressed(GDK_SHIFT_MASK);
			return true;
		case GDK_Control_L:
		case GDK_Control_R:
			if (pActiveTool) pActiveTool->OnKeyPressed(GDK_CONTROL_MASK);
			return true;
		case GDK_Alt_L:
		case GDK_Alt_R:
			if (pActiveTool) pActiveTool->OnKeyPressed(GDK_MOD1_MASK);//FIXME: might be not portable
			return true;
		default:
			break;
	}
	return false;
}

bool gcpView::OnKeyRelease(GtkWidget* w, GdkEventKey* event)
{
	switch(event->keyval)
	{
		case GDK_Shift_L:
		case GDK_Shift_R:
			if (pActiveTool) pActiveTool->OnKeyReleased(GDK_SHIFT_MASK);
			return true;
		case GDK_Control_L:
		case GDK_Control_R:
			if (pActiveTool) pActiveTool->OnKeyReleased(GDK_CONTROL_MASK);
			return true;
		case GDK_Alt_L:
		case GDK_Alt_R:
			if (pActiveTool) pActiveTool->OnKeyReleased(GDK_MOD1_MASK);//FIXME: might be not portable
			return true;
		default:
			break;
	}
	return false;
}
	
bool gcpView::OnSize(int width, int height)
{
	if ((width == m_width) && (height == m_height)) return true;
	std::list<GtkWidget*>::iterator i;
	int x, y;
	m_height = height;
	m_width = width;
	for (i = m_Widgets.begin(); i != m_Widgets.end(); i++)
	{
		gnome_canvas_set_scroll_region(GNOME_CANVAS(*i), 0, 0, width, height);
		gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(*i), "data");
		if (pData->Background) g_object_set(G_OBJECT(pData->Background), "x2", (double) width, "y2", (double) height, NULL);
	}
	return true;
}

void gcpView::UpdateSize(double x1, double y1, double x2, double y2)
{
	if (x1 < 0.0) x2 -= x1;
	if (y1 < 0.0) y2 -= y1;
	list<GtkWidget*>::iterator i;
	if ((x2 > m_width) || (y2 > m_height))
		for (i = m_Widgets.begin(); i != m_Widgets.end(); i++)
			gtk_widget_set_size_request(*i,(int)ceil(x2), (int)ceil(y2));
	if ((x1 < 0.0) || (y1 < 0.0))
	{
		x1 = - x1;
		y1 = - y1;
		gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(m_pWidget), "data");
		m_pDoc->Move(x1 / pData->ZoomFactor, y1 / pData->ZoomFactor);
		Update(m_pDoc);
	}
}

void gcpView::SetGnomeCanvasRichTextActive(GnomeCanvasRichTextExt* item)
{
	m_ActiveRichText = item;
	bDragging = false;
}

bool gcpView::PrepareUnselect()
{
	return (pActiveTool)? pActiveTool->NotifyViewChange(): true;
}

void gcpView::OnSelectAll()
{
	if (ToolsDlg) ToolsDlg->Select(SelectId);
	else
	{
		if (pActiveTool) pActiveTool->Activate(false);
		pActiveTool = Tools[SelectId];
		pActiveTool->Activate(true);
	}
	m_pData->SelectAll();
	SelectionTool.AddSelection(m_pData);
}
