/*  ga_service_provider_t.c */
/*  Copyright 2004-2006 Oswaldo Morizaki Hirakata */

/*  This file is part of ga-nn-ag-2.

    ga-nn-ag is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn-ag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn-ag; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "my_header.h"
#include "aux_prot.h"

/* Global variables and mutexes from service server */
#include "ga_service_server_header.h"

/* 
Provide service to clients 

Also controls the time_wait

command values:
0 = init
1 = agr
2 = test
3 = train
-1 = exit
1000 = no previus command

Transactions:
============
- Send command: command, num_nn (block style)
- Send nn (block style)
- Send pat 
- Send nn_exec_f config
-	Send nn_init_f / nn_agr_f config if required
- Close connection
*/


void 
* ga_service_provider_t(void * arg)
{
int ret;
int retry;
int k,l,m,n;
int num_nn;
int num_layer;
int curr_command;
int cross_flag;
int inm_flag;
int terminate_flag = 0;
int action;	
int num_test = 0;
int num_train = 0;
int num_new = 0;
int num_cross = 0;
int num_inm = 0;
int temp;
int new = 0;

struct io_connection inm_con;

struct ga_service_client_index * local_index = NULL;
struct ga_service_client_index * cross_index = NULL;

struct ga_service_provider_thread_param * param = NULL;

struct ga_pat_index * new_pat_index = NULL;

struct io_block * command_block = NULL;
struct io_block * agr_block = NULL;
struct io_block * exec_block = NULL;
struct io_block * nn_inm_block = NULL; 
struct io_block * base_exec_block = NULL;
struct io_block * base_inm_block = NULL;
struct io_block * input_pat_block = NULL;
struct io_block * output_pat_block = NULL;

struct neural_net * net1 = NULL;
struct neural_net * net2 = NULL;

struct nn_pattern ** pat;

char temp_buffer[BUFFSIZE];
char char_temp[BUFFSIZE];

int ** temp_array;

double * fitness = NULL;

FILE * fileptr;

/* Detach thread */
if (pthread_detach(pthread_self()) != 0)
{
	syslog(LOG_ERR,"Error detaching thread %d: %s",pthread_self(),strerror(errno));
}

if (!(param = (struct ga_service_provider_thread_param *)calloc(
			1,sizeof(struct ga_service_provider_thread_param )) ))
{
	syslog(LOG_CRIT,"Error calloc param in ga_service_provider_t: %s", strerror(errno));
	pthread_exit(NULL);
}
param->conf = ((struct ga_service_provider_thread_param *)arg)->conf;

/* Lock mutex for generations */
if ((ret = pthread_mutex_lock(&generation_mutex) )!= 0)
{
	syslog(LOG_CRIT,"Error locking mutex generation_mutex in ga_service_provider_t(): %d", ret);
	pthread_exit(NULL);
}

param->local_current_generation = current_generation;

/* Unlock mutex for generations */
if ((ret = pthread_mutex_unlock(&generation_mutex) )!= 0)
{
	syslog(LOG_CRIT,"Error unlocking mutex generation_mutex in ga_service_provider_t(): %d", ret);
	pthread_exit(NULL);
}

syslog(LOG_INFO,"ga_service_provider_t thread with tid %d created for generation: %d",pthread_self(), param->local_current_generation);

/******************************************************************************/
/*                                 LOCAL INDEX                                */
/******************************************************************************/
/* Allocate local_index */
if (!(local_index = (struct ga_service_client_index *)ga_calloc_client_index(
																		0, 0, 0, local_index) ))
{
	syslog(LOG_CRIT,"Error calloc local_index in ga_service_provider: %s",strerror(errno) );
	pthread_exit(NULL);
}				

/* Lock client_index while using local_index */
if ((ret = pthread_mutex_lock(&index_mutex) )!= 0)
{
	syslog(LOG_ERR,"Error locking mutex index_mutex in ga_service_provider_t(): %d", ret);
}	

/* Insert in local_index the clients */
for (k=0 ; k< client_index->num_client; k++)
{
	if (client_index->clients[k]->curr_gen == param->local_current_generation)
	{
		/* Insert in case state is REQ or ID == INM_ID */
		if ((client_index->clients[k]->state == 0)  ||
			(client_index->clients[k]->id == INM_ID ))
		{
			syslog(LOG_INFO,"client[%d]: id = %d, curr_gen = %d ",k,client_index->clients[k]->id,
						client_index->clients[k]->curr_gen);
	
			/* Insert data into client_index */
			if (!(local_index = (struct ga_service_client_index *)ga_insert_client_index(
																client_index->clients[k], local_index) ))
			{
				syslog(LOG_ERR,"Error ga_insert_client_index() in ga_service_provider_t(): %d", strerror(errno));
			}
		}
	}
}

/* Unlock client_index */
if ((ret = pthread_mutex_unlock(&index_mutex) )!= 0)
{
	syslog(LOG_ERR,"Error unlocking mutex index_mutex in ga_service_provider_t(): %d", ret);
}	

/* Check if are clients in local_index */
/* time_wait setting logic */
if (local_index->num_client == 0)
{
	if (param->conf->time_wait > 0)
	{
		param->conf->time_wait = (int)(param->conf->time_wait*1.5);
		syslog(LOG_ERR,"No clients ready, increasing time_wait 50 %% to",param->conf->time_wait);
	}
	else
	{
		param->conf->time_wait = 2;
		syslog(LOG_ERR,"No clients ready, setting time_wait to 2 seconds");
	}
	pthread_exit(NULL);
}
else
{
	syslog(LOG_INFO,"Number of clients for generation %d: %d", param->local_current_generation, local_index->num_client);
	if (local_index->num_client >= param->conf->population)
	{
		param->conf->time_wait = (int)(param->conf->time_wait*0.9);
		syslog(LOG_INFO,"num_client >= population, decreasing time_wait by 10 %% to %d",param->conf->time_wait);
	}
}
/******************************************************************************/
/*                              END LOCAL INDEX                               */
/******************************************************************************/

/******************************************************************************/
/*                             PAT_INDEX SETTING                              */
/******************************************************************************/
/* Generate new ga_pat_index */
if (!(new_pat_index = (struct ga_pat_index *)ga_service_ga_pattern(
												local_index, param->conf, pat_index, new_pat_index)))
{
	syslog(LOG_CRIT,"Error ga_service_ga_pattern in ga_service_provider_t");
	pthread_exit(NULL);
}
/* Lock mutex for pat_index */
if ((ret = pthread_mutex_lock(&pat_mutex) )!= 0)
{
	syslog(LOG_CRIT,"Error locking pat_mutex in ga_service_provider_t(): %d", ret);
	pthread_exit(NULL);
}
/* Assign to pat_index new_pat_index */
pat_index = (struct ga_pat_index *)ga_free_pat_index(pat_index);
pat_index = new_pat_index;

/* Unlock mutex for pat_index */
if ((ret = pthread_mutex_unlock(&pat_mutex) )!= 0)
{
	syslog(LOG_CRIT,"Error unlocking pat_mutex in ga_service_provider_t(): %d", ret);
	pthread_exit(NULL);
}

/* Get fitness values for clients */
if (ga_prov_get_fitness(param->conf, local_index) )
{
	syslog(LOG_CRIT,"Error ga_prov_get_fitness() in ga_service_provider_t(): %s", strerror(errno));
	pthread_exit(NULL);
}

/* Clear current pat_index in clients */
for (k=0; k< local_index->num_client; k++)
{
	for (l = local_index->clients[k]->pat_index->num_cat-1; l+1; l--)
	{
		for (m= local_index->clients[k]->pat_index->cat[l]->num_pat-1; m+1; m--)
		{
			local_index->clients[k]->pat_index->cat[l]->pat[m] = (struct ga_pattern *)
							ga_free_pattern(local_index->clients[k]->pat_index->cat[l]->pat[m]);
		}
		free(local_index->clients[k]->pat_index->cat[l]->pat);
		local_index->clients[k]->pat_index->cat[l]->num_pat = new_pat_index->cat[l]->num_stable;
	}
}

/* Set number of patterns according to fitness */
/* Each clients gets at least num_stable patterns */
if (!(fitness = (double *)malloc(local_index->num_client*sizeof(double))))
{
	syslog(LOG_CRIT,"Error fitness in ga_service_provider_t: %s",strerror(errno));
	pthread_exit(NULL);
}
if (!(temp_array = (int **)malloc(new_pat_index->num_cat*sizeof(int *))))
{
	syslog(LOG_CRIT,"Error temp_array in ga_service_provider_t: %s",strerror(errno));
	pthread_exit(NULL);
}
for (k=0; k< new_pat_index->num_cat; k++)
{
	if (!(temp_array[k] = (int *)malloc(new_pat_index->cat[k]->num_pat*sizeof(int))))
	{
		syslog(LOG_CRIT,"Error temp_array[%d] in ga_service_provider_t: %s",k,strerror(errno));
		pthread_exit(NULL);
	}
}

/* Fill fitness vector */
for (k=0; k< local_index->num_client; k++)
{
	fitness[k] = local_index->clients[k]->fitness;
}
/* Get the number of patterns per category */
for (k=new_pat_index->num_cat-1; k+1; k--)
{
	for(l=new_pat_index->cat[k]->num_pat - new_pat_index->cat[k]->num_stable-1; l+1; l--)
	{
		temp = va_roulette(local_index->num_client, fitness);
		temp_array[k][l] = temp;
		local_index->clients[temp]->pat_index->cat[k]->num_pat += 1;
	}
}
free(fitness);

/* Copy patterns to clients */
for (k=0 ; k< local_index->num_client; k++)
{
	for (l=0; l< new_pat_index->num_cat; l++)
	{
		if (!(local_index->clients[k]->pat_index->cat[l]->pat = (struct ga_pattern **)malloc
					(local_index->clients[k]->pat_index->cat[l]->num_pat*sizeof(struct ga_pattern *))))
		{
			syslog(LOG_CRIT,"Error malloc local_index->clients[%d]->pat_index->cat[%d]->pat in ga_service_provider: %s",
							k, l, strerror(errno));
			pthread_exit(NULL);
		}
		
		/* Copy stable patterns */
		for (m=0; m< new_pat_index->cat[l]->num_stable; m++)
		{
			local_index->clients[k]->pat_index->cat[l]->pat[m] = NULL;
			if (!(local_index->clients[k]->pat_index->cat[l]->pat[m] = (struct ga_pattern *)
						ga_pat_copy_pattern(new_pat_index->cat[l]->pat[m], 
						local_index->clients[k]->pat_index->cat[l]->pat[m]) ))
			{
				syslog(LOG_CRIT,"Error ga_pat_copy_pattern in ga_service_provider_t");
				pthread_exit(NULL);
			}
		}
		/* Copy non stable patterns */
		m = new_pat_index->cat[l]->num_stable;
		for (n=new_pat_index->cat[l]->num_stable; n< new_pat_index->cat[l]->num_pat; n++)
		{
			if (temp_array[l][n] == k)
			{
				local_index->clients[k]->pat_index->cat[l]->pat[m] = NULL;
				if (!(local_index->clients[k]->pat_index->cat[l]->pat[m] = (struct ga_pattern *)
						ga_pat_copy_pattern(new_pat_index->cat[l]->pat[n], 
						local_index->clients[k]->pat_index->cat[l]->pat[m]) ))
				{
					syslog(LOG_CRIT,"Error ga_pat_copy_pattern in ga_service_provider_t");
					pthread_exit(NULL);
				}
				m++;
			}
		}
		if (m != local_index->clients[k]->pat_index->cat[l]->num_pat)
		{
			syslog(LOG_CRIT,"Error, numbers doesn't fit m != num_pat in ga_service_provider_t");
			pthread_exit(NULL);
		}
	}
}

/* Lock client_index while using client_index */
if ((ret = pthread_mutex_lock(&index_mutex) )!= 0)
{
	syslog(LOG_ERR,"Error locking mutex index_mutex in ga_service_provider_t(): %d", ret);
}	
/* Insert new pat_index into client_index->clients */
for (k=0 ; k< local_index->num_client; k++)
{
	if ((l = ga_is_in_client_index(local_index->clients[k], client_index))< 10)
	{
		syslog(LOG_ERR,"Error, local_index->clients[%d] not in client_index",k);
		continue;
	}
	else
	{
		l -= 10;
		client_index->clients[l]->pat_index = (struct ga_pat_index *)ga_free_pat_index
							(client_index->clients[l]->pat_index);
		client_index->clients[l]->pat_index = local_index->clients[k]->pat_index;
	}
}
/* Unlock index_mutex */
if ((ret = pthread_mutex_unlock(&index_mutex) )!= 0)
{
	syslog(LOG_ERR,"Error unlocking mutex index_mutex in ga_service_provider_t(): %d", ret);
}	

/* Lock pat_mutex while using pat_index */
if ((ret = pthread_mutex_lock(&pat_mutex) )!= 0)
{
	syslog(LOG_ERR,"Error locking mutex pat_mutex in ga_service_provider_t(): %d", ret);
}	
/* Insert new pat_index into client_index->clients */
pat_index = (struct ga_pat_index *)ga_free_pat_index(pat_index);
pat_index = new_pat_index;

/* Unlock pat_mutex */
if ((ret = pthread_mutex_unlock(&pat_mutex) )!= 0)
{
	syslog(LOG_ERR,"Error unlocking mutex pat_mutex in ga_service_provider_t(): %d", ret);
}	
/******************************************************************************/
/*                             END PAT_INDEX SETTING                          */
/******************************************************************************/


/******************************************************************************/
/*                           CURRENT_GENERATION SETTING                       */
/******************************************************************************/
/* Lock mutex for generations */
if ((ret = pthread_mutex_lock(&generation_mutex) )!= 0)
{
	syslog(LOG_CRIT,"Error locking generation_mutex in ga_service_provider_t(): %d", ret);
	pthread_exit(NULL);
}
/* Check for setting exit flag */
if (current_generation >= generations)
{
	exit_flag = 1;
	syslog(LOG_INFO,"exit_flag set to 1, ready to exit"); 
}
/* Increase current_generation */
else 
{
	current_generation += 1;
	syslog(LOG_INFO,"Increasing current_generation to %d",current_generation);
}
/* Unlock mutex for generations */
if ((ret = pthread_mutex_unlock(&generation_mutex) )!= 0)
{
	syslog(LOG_CRIT,"Error unlocking generation_mutex in ga_service_provider_t(): %d", ret);
	pthread_exit(NULL);
}
/******************************************************************************/
/*                           END CURRENT_GENERATION SETTING                   */
/******************************************************************************/

/******************************************************************************/
/*                                 ALLOC BLOCKS                               */
/******************************************************************************/
if (!(input_pat_block = (struct io_block *)va_calloc_io_block(0,0,0,input_pat_block)))
{
	syslog(LOG_CRIT,"Error va_calloc_io_block(input_pat_block) in ga_service_provider: %s");
	pthread_exit(NULL);
}
if (!(output_pat_block = (struct io_block *)va_calloc_io_block(0,0,0,output_pat_block)))
{
	syslog(LOG_CRIT,"Error va_calloc_io_block(output_pat_block) in ga_service_provider: %s");
	pthread_exit(NULL);
}
if (!(command_block = (struct io_block *)va_calloc_io_block(0,0,0,command_block)))
{
	syslog(LOG_CRIT,"Error va_calloc_io_block(command_block) in ga_service_provider: %s");
	pthread_exit(NULL);
}
if (!(exec_block = (struct io_block *)va_calloc_io_block(0,0,0,exec_block)))
{
	syslog(LOG_CRIT,"Error va_calloc_io_block(exec_block) in ga_service_provider: %s");
	pthread_exit(NULL);
}
if (!(agr_block = (struct io_block *)va_calloc_io_block(0,0,0,agr_block)))
{
	syslog(LOG_CRIT,"Error va_calloc_io_block(agr_block) in ga_service_provider: %s");
	pthread_exit(NULL);
}
if (!(nn_inm_block = (struct io_block *)va_calloc_io_block(0,0,0,nn_inm_block)))
{
	syslog(LOG_CRIT,"Error va_calloc_io_block(nn_inm_block) in ga_service_provider: %s");
	pthread_exit(NULL);
}
if (!(base_inm_block = (struct io_block *)va_calloc_io_block(0,0,0,base_inm_block)))
{
	syslog(LOG_CRIT,"Error va_calloc_io_block(base_inm_block) in ga_service_provider: %s");
	pthread_exit(NULL);
}
/* Memory reserve for exec block */
if (!(base_exec_block = (struct io_block *)va_calloc_io_block(4,BUFFSIZE,0,base_exec_block)))
{
	syslog(LOG_CRIT,"Error va_calloc_io_block(base_exec_block) in ga_service_provider: %s");
	pthread_exit(NULL);
}
/******************************************************************************/
/*                               END ALLOC BLOCKS                             */
/******************************************************************************/

/******************************************************************************/
/*                             SET INMIGRATION BLOCK                          */
/******************************************************************************/
/* Set inm_flag */
if ((inm_flag = ga_prov_inmigrate(param->conf, local_index)) == 1)
{
	/* Set basic parameters for connection to server */
	bzero(&inm_con,sizeof(inm_con)); //set parameters to zero
	sprintf(inm_con.port,"%d\0",param->conf->niche_port);
	inm_con.socket = 1; //connection via socket
	inm_con.number = INM_ID; //client number reserved for inmigration
	
	/**************************************************/
	/* Set base_inm_block, common to all cross clients  */
	/**************************************************/
	
	snprintf(base_inm_block->char_vector[0],BUFFSIZE_1,"ID=%d\0",INM_ID);
	snprintf(base_inm_block->char_vector[1],BUFFSIZE_1,"UPD\0");
	snprintf(base_inm_block->char_vector[2],BUFFSIZE_1,"command=%d\0",2);
	
	for (k=0; k< 3; k++)
	{
		*(base_inm_block->char_vector[k]+BUFFSIZE_1) = '\0';
	}
	
	syslog(LOG_INFO,"Inmigration flag set for generation: %d",param->local_current_generation);
}
/******************************************************************************/
/*                          END SET INMIGRATION BLOCK                         */
/******************************************************************************/

/******************************************************************************/
/*                             SET BASE EXEC BLOCK                            */
/******************************************************************************/
snprintf(base_exec_block->char_vector[0], BUFFSIZE_1,"num_input=%d\0",
					param->conf->nn_train_conf.num_input);
snprintf(base_exec_block->char_vector[1], BUFFSIZE_1,"num_output=%d\0",
					param->conf->nn_train_conf.num_output);
snprintf(base_exec_block->char_vector[2], BUFFSIZE_1,"age_inf=%1.5f\0",
					param->conf->nn_train_conf.age_inf);
snprintf(base_exec_block->char_vector[3], BUFFSIZE_1,"threshold_level=%1.5f\0",
					param->conf->nn_train_conf.threshold_level);
					
for (k=0; k< 4; k++)
{
	*(nn_inm_block->char_vector[k]+BUFFSIZE_1) = '\0';
}
/******************************************************************************/
/*                          END SET BASE EXEC BLOCK                           */
/******************************************************************************/

/******************************************************************************/
/******************************************************************************/
/**                                                                          **/
/**                             PROVIDE SERVICE                              **/
/**                                                                          **/
/******************************************************************************/
/******************************************************************************/
for (k=0; k< local_index->num_client; k++)	
{
	/****************************************************************************/
	/*                                SET BLOCKS                                 */
	/****************************************************************************/
	/* Check generation for command */
	if (param->local_current_generation == 0) 
	{
		/* First generation */
		/***************************************/
		/*           AGGREGATION               */
		/***************************************/
		if ((param->conf->agr != 0) && (param->conf->clean_start == 0))
		{
			/* Setup command_block */
			num_nn = (param->conf->agr < 20) ? 1 : 2;
			if (!(command_block = (struct io_block *)ga_prov_set_command_block(
														command_block, 1, num_nn)))
			{
				syslog(LOG_CRIT,"Error ga_prov_set_command_block() agr");
				pthread_exit(NULL);
			}

			/* Set mode */			
			sprintf(temp_buffer,"mode=%d\0",param->conf->agr);
			if (!(agr_block = (struct io_block *)va_insert_io_block(-1,temp_buffer,agr_block) ))
			{
				syslog(LOG_CRIT,"Error va_insert_io_block(agr_block) inserting: %s",temp_buffer);
				pthread_exit(NULL);
			}			

			switch(param->conf->agr)
			{
				case 10 : //output add
				{
					sprintf(temp_buffer,"output_add=%d\0",param->conf->agr_io_add);
					if (!(agr_block = (struct io_block *)va_insert_io_block(-1,temp_buffer,agr_block) ))
					{
						syslog(LOG_CRIT,"Error va_insert_io_block(agr_block) inserting: %s",temp_buffer);
						pthread_exit(NULL);
					}			
					break;
				}
				case 11: // input add
				{
					sprintf(temp_buffer,"input_add=%d\0",param->conf->agr_io_add);
					if (!(agr_block = (struct io_block *)va_insert_io_block(-1,temp_buffer,agr_block) ))
					{
						syslog(LOG_CRIT,"Error va_insert_io_block(agr_block) inserting: %s",temp_buffer);
						pthread_exit(NULL);
					}			
					break;
				}
				case 20 : // neural_net add
				{
					sprintf(temp_buffer,"old_init1=%s%d\0",param->conf->agr_add_preffix,
											local_index->clients[k]->id);
					if (!(agr_block = (struct io_block *)va_insert_io_block(-1,temp_buffer,agr_block) ))
					{
						syslog(LOG_CRIT,"Error va_insert_io_block(agr_block) inserting: %s",temp_buffer);
						pthread_exit(NULL);
					}			
					break;
				}
			}			
			/* Insert NULL */
			if (!(agr_block = (struct io_block *)va_insert_io_block(-1,NULL,agr_block) ))
			{
				syslog(LOG_CRIT,"Error va_insert_io_block(agr_block) inserting: NULL");
				pthread_exit(NULL);
			}			

			/*****************************/			
			/* Read neural net from file */
			/*****************************/			
			sprintf(temp_buffer,"%s%d\0",param->conf->init_preffix, local_index->clients[k]->id);
			if (!(net1 = (struct neural_net *)nn_fread_neural_net(net1, temp_buffer)))
			{
				syslog(LOG_CRIT,"Error nn_fread_neural_net(net1, %s) in ga_service_provider: %s",
						temp_buffer, strerror(errno));
				pthread_exit(NULL);
			}
			
			/* Read old_init1 if required */
			if (param->conf->agr == 20)
			{
				sprintf(temp_buffer,"%s%d\0",param->conf->agr_add_preffix, local_index->clients[k]->id);
				if (!(net2 = (struct neural_net *)nn_fread_neural_net(net2, temp_buffer)))
				{
					syslog(LOG_CRIT,"Error nn_fread_neural_net(net2, %s) in ga_service_provider: %s",
							temp_buffer, strerror(errno));
					pthread_exit(NULL);
				}
			}
			/*********************************/			
			/* End read neural net from file */
			/*********************************/			
		}	
		/***************************************/
		/*          END AGGREGATION            */
		/***************************************/
		
		/*******************************/
		/*        CLEAN START          */
		/*******************************/
		else if ((param->conf->agr == 0) || (param->conf->clean_start == 1)) 
		{
			new = 1;
			if (!(command_block = (struct io_block *)ga_prov_set_command_block(
														command_block, 0, 0)))
			{
				syslog(LOG_CRIT,"Error ga_prov_set_command_block() init");
				pthread_exit(NULL);
			}			

			if (!(agr_block = (struct io_block *)ga_prov_set_clean_start_block(agr_block, param->conf)))
			{
				syslog(LOG_CRIT,"Error setting agr_block for clean start in ga_service_provider: %s",strerror(errno));
				pthread_exit(NULL);
			}
		}
		/*******************************/
		/*       END CLEAN START       */
		/*******************************/

		/*******************************/
		/*        STOP AND GO          */
		/*******************************/
		else if ((param->conf->agr == -1) || (param->conf->clean_start == -1)) 
		{
			if (!(command_block = (struct io_block *)ga_prov_set_command_block(
														command_block, 3, 1)))
			{
				syslog(LOG_CRIT,"Error ga_prov_set_command_block() stop-and-go");
				pthread_exit(NULL);
			}			

			/* Read neural net from file */
			sprintf(temp_buffer,"%s%d\0",param->conf->init_preffix, local_index->clients[k]->id);
			if (!(net1 = (struct neural_net *)nn_fread_neural_net(net1, temp_buffer)))
			{
				syslog(LOG_CRIT,"Error nn_fread_neural_net(net1, %s) in ga_service_provider: %s",
						temp_buffer, strerror(errno));
				pthread_exit(NULL);
			}
		}
	}
	else // Generations over 0
	{
		/**************************/
		/* START Inmigration CODE */
		/**************************/
		/* Check for inmigration */
		if ((inm_flag) && (va_toss(param->conf->inm_prob, k)))
		{
			num_inm += 1;
			/* Select niche */
			strcpy(inm_con.ip, param->conf->niches[va_dice_toss(k,param->conf->num_niches)]);

			if (!(nn_inm_block = (struct io_block *)va_copy_io_block(base_inm_block, nn_inm_block) ))
			{
				syslog(LOG_CRIT,"Error va_copy_io_block(base_inm_block): %s",strerror(errno));
				pthread_exit(NULL);
			}
				
			/* Set error block */
			if (!(nn_inm_block = (struct io_block *)ga_pack_error(local_index->clients[k]->ret, nn_inm_block)));
			{
				syslog(LOG_CRIT,"Error ga_pack_error %d in ga_service_provider_t",k);
			}

			/* Send UPD as client */
			/* Send UPD */
			if(va_io_connect(&inm_con) < 0)
			{
				syslog(LOG_CRIT,"Error va_io_connect()");
			}
			nn_inm_block->connfd = inm_con.connfd;
			if (va_dwrite_io_block(nn_inm_block) < 0)
			{
				syslog(LOG_INFO,"Error sending UPD");
			}
			if (va_dwrite_io_block(nn_inm_block) < 0)
			{
				syslog(LOG_INFO,"Error sending UPD");
			}
			if (nn_write_neural_net(local_index->clients[k]->ret->net,&inm_con) < 0)
			{
				syslog(LOG_CRIT,"Error nn_write_neural_net(), command %d, type %d");
			}					
					
			/* Close connection */
			va_io_close(&inm_con);
			
			/* Eliminate entries in exec_block */
			nn_inm_block = (struct io_block *)va_delete_io_block(-1,nn_inm_block);			
		}
		/************************/
		/* END Inmigration CODE */
		/************************/

		/* Decide whenever is going to be a mating season or trainning season */
		if (va_toss(1.0 - 1.0/(param->conf->train_ratio + 1), k)) // train
		{
			if (!(command_block = (struct io_block *)ga_prov_set_command_block(
														command_block, 3, 1)))
			{
				syslog(LOG_CRIT,"Error ga_prov_set_command_block()");
				pthread_exit(NULL);
			}			
		}
		else //test
		{
			if (!(command_block = (struct io_block *)ga_prov_set_command_block(
													command_block, 2, 1)))
			{
				syslog(LOG_CRIT,"Error ga_prov_set_command_block()");
				pthread_exit(NULL);
			}			
		}
	}
	
	/***********************/
	/*    SET PAT BLOCK    */
	/***********************/	
	if (!(pat = (struct nn_pattern **)ga_pat_index2nn_pat(local_index->clients[k]->pat_index, pat)))
	{
		syslog(LOG_CRIT,"Error ga_pat_index2nn_pat in ga_service_provider: %s",strerror(errno));
		pthread_exit(NULL);
	}
	
	if (!(input_pat_block = (struct io_block *)ga_pack_patterns(pat[0],input_pat_block)))
	{
		syslog(LOG_CRIT,"Error ga_pack_patterns(input) in ga_service_provider: %s",strerror(errno));
		pthread_exit(NULL);
	}
	if (!(output_pat_block = (struct io_block *)ga_pack_patterns(pat[1],output_pat_block)))
	{
		syslog(LOG_CRIT,"Error ga_pack_patterns in(output) ga_service_provider: %s",strerror(errno));
		pthread_exit(NULL);
	}
	/***************************/
	/*    END SET PAT BLOCK    */
	/***************************/	

	/************************/
	/*    SET EXEC BLOCK    */
	/************************/
	if (!(exec_block = (struct io_block *)va_copy_io_block(base_exec_block, exec_block) ))
	{
		syslog(LOG_CRIT,"Error va_copy_io_block(base_exec_block): %s",strerror(errno));
		pthread_exit(NULL);
	}
	/* Set num_pat */	
	for (l=local_index->clients[k]->pat_index->num_cat-1; l+1; l--)
	{
		temp += local_index->clients[k]->pat_index->cat[l]->num_pat;
	}
	sprintf(temp_buffer,"num_pat=%d\0", temp);
	if (!(exec_block = (struct io_block *)va_insert_io_block(-1,temp_buffer,exec_block) ))
	{
		syslog(LOG_CRIT,"Error va_insert_io_block(exec_block) inserting: %s",temp_buffer);
		pthread_exit(NULL);
	}							
	/* Insert NULL */
	if (!(exec_block = (struct io_block *)va_insert_io_block(-1,NULL,exec_block) ))
	{
		syslog(LOG_CRIT,"Error va_insert_io_block(exec_block) inserting: NULL");
		pthread_exit(NULL);
	}							
	/****************************/
	/*    END SET EXEC BLOCK    */
	/****************************/


	/****************************************************************************/
	/*                                 END SET BLOCKS                           */
	/****************************************************************************/

	/****************************************************************************/
	/*                                SET NEURAL NETS                           */
	/****************************************************************************/

	/************************/
	/* New / Cross networks */
	/************************/
	/* Check previus command */
	if ((local_index->clients[k]->command == 0) || 
			(local_index->clients[k]->command == 1) )
	{
		curr_command = 2; // exec_train
		cross_flag = 0;	// do not cross
	}
	else if (local_index->clients[k]->command == 2)
	{
		curr_command = 3; // exec_test
		cross_flag = 0;	// do not cross
	}
	else if (local_index->clients[k]->command == 3)
	{
		curr_command = 2; // exec_train
		cross_flag = 1;	// cross
	}
	else // exit or exec_terminate
	{
		close(exec_block->connfd); // close connection
		pthread_exit(NULL); //exit
	}

	syslog(LOG_INFO,"curr_command = %d, cross_flag = %d",curr_command, cross_flag);

	/* Check cross_flag */
	if (cross_flag)
	{
		/* Test if new neural net */
		if (va_toss(param->conf->mut_new_prob, k) ) 
		{
			/*********************************/
			/* Set new neural_net for client */
			/*********************************/
			syslog(LOG_INFO,"New neural net for generation");
			num_new += 1;
			new = 1;
			
			/* Eliminate previous (now wrong) entries in command_block */
			command_block = (struct io_block *)va_delete_io_block(-1,command_block);

			if (!(command_block = (struct io_block *)ga_prov_set_command_block(
														command_block, 0, 0)))
			{
				syslog(LOG_CRIT,"Error ga_prov_set_command_block() init");
				pthread_exit(NULL);
			}			

			if (!(agr_block = (struct io_block *)ga_prov_set_clean_start_block(agr_block, param->conf)))
			{
				syslog(LOG_CRIT,"Error setting agr_block for clean start in ga_service_provider: %s",strerror(errno));
				pthread_exit(NULL);
			}
		}		
		/**********************/
		/* END New neural net */
		/**********************/
		
		/*********************/
		/* Cross neural nets */
		/*********************/
		else	
		{
			num_cross += 1;
	
			/* cross_index has a single element */
			if (!(cross_index = (struct ga_service_client_index *)ga_prov_cross(param->conf,
										local_index, cross_index)))
			{
				syslog(LOG_CRIT,"Error ga_prov_cross() in ga_service_provider_t(): %s",strerror(errno));
				pthread_exit(NULL);
			}
				
			/* Mutation */
			if (!(cross_index = (struct ga_service_client_index *)ga_prov_mut_net(param->conf,
												cross_index)))
			{
				syslog(LOG_CRIT,"Error ga_prov_mut_net() in ga_servie_provider_t():%s", strerror(errno));
				pthread_exit(NULL);
			}
		}
		/******************************/
		/* END Cross / New neural net */
		/******************************/
	}
	/****************************************************************************/
	/*                                  SEND BLOCKS                             */
	/****************************************************************************/
	/* Send command block to client */
	command_block->connfd = *(local_index->clients[k]->connptr);
	if (va_dwrite_io_block(command_block))
	{
		syslog(LOG_CRIT,"Error va_dwrite_io_block(command_block) for client %d",k);
		pthread_exit(NULL);
	}
	/* Eliminate entries in command_block */
	command_block = (struct io_block *)va_delete_io_block(-1,command_block);

	/* Send neural_net to ga_client */
	if ((param->local_current_generation == 0) || (new == 1))
	{
		if (nn_dwrite_neural_net(net1, *(local_index->clients[k]->connptr)) < 0)
		{
			syslog(LOG_CRIT,"Error nn_dwrite_neural_net(net1, connptr) %d: %s",k, strerror(errno));
			pthread_exit(NULL);
		}
		net1 = (struct neural_net *)nn_free_neural_net(net1);
		
		if (param->conf->agr == 20)
		{
			if (nn_dwrite_neural_net(net2, *(local_index->clients[k]->connptr)) < 0)
			{
				syslog(LOG_CRIT,"Error nn_dwrite_neural_net(net2, connptr) %d: %s",k, strerror(errno));
				pthread_exit(NULL);
			}
			net2 = (struct neural_net *)nn_free_neural_net(net2);			
		}
	}
	else
	{
		if (nn_dwrite_neural_net(cross_index->clients[0]->ret->net, *(local_index->clients[k]->connptr)) < 0)
		{
			syslog(LOG_CRIT,"Error nn_dwrite_neural_net(cross_index->clients[0]->ret->net, connptr) %d: %s",k, strerror(errno));
			pthread_exit(NULL);
		}
		/* Free neural_net in cross_index */
		cross_index->clients[0]->ret->net = (struct neural_net *)
								nn_free_neural_net(cross_index->clients[0]->ret->net);
	}
	
	/* Send pattern blocks */	
	input_pat_block->connfd = *(local_index->clients[k]->connptr);
	if (va_dwrite_io_block(input_pat_block))
	{
		syslog(LOG_CRIT,"Error va_dwrite_io_block(input_pat_block) for client %d",k);
		pthread_exit(NULL);
	}
	output_pat_block->connfd = *(local_index->clients[k]->connptr);
	if (va_dwrite_io_block(output_pat_block))
	{
		syslog(LOG_CRIT,"Error va_dwrite_io_block(output_pat_block) for client %d",k);
		pthread_exit(NULL);
	}
	/* Eliminate entries in pat blocks */
	input_pat_block = (struct io_block *)va_delete_io_block(-1,input_pat_block);
	output_pat_block = (struct io_block *)va_delete_io_block(-1,output_pat_block);
	
	/* Send exec_block */
	exec_block->connfd = *(local_index->clients[k]->connptr);
	if (va_dwrite_io_block(exec_block))
	{
		syslog(LOG_ERR,"Error sending exec_block %d: %s",k, strerror(errno));
	}
	/* Eliminate entries in exec_block */
	exec_block = (struct io_block *)va_delete_io_block(-1,exec_block);			
	
	if ((new == 1) || (param->local_current_generation == 0))
	{
		agr_block->connfd = *(local_index->clients[k]->connptr);
		if (va_dwrite_io_block(agr_block))
		{
			syslog(LOG_ERR,"Error sending agr_block %d: %s",k, strerror(errno));
		}
		/* Eliminate entries in agr_block */
		agr_block = (struct io_block *)va_delete_io_block(-1,agr_block);			
	}
	new = 0;
	
	/* Free pat */
	nn_free_pattern(pat[0]);
	nn_free_pattern(pat[1]);
	free(pat);
	
	/****************************************************************************/
	/*                                END SEND BLOCKS                           */
	/****************************************************************************/
}
syslog(LOG_INFO,"Exiting provider thread %d", pthread_self());
pthread_exit(0);
}

