/*  ga_service_get_client_info.c */
/* 	Copyright 2004-2006 Oswaldo Morizaki Hirakata */

/* 	This file is part of ga-nn-ag-2.

    ga-nn-ag is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn-ag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn-ag; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "my_header.h"
#include "aux_prot.h"

/*
Reads client info from socket (listen) 
and fill in a ga_service_client_info structure, except for pat_curr_gen, nurse_time (UPD)
and pat_index, or just fill command and state (REQ)

Also sets the current generation

Transactions:
-	Reads REQ / UPD block
-	In case UPD, reads error block
- In case UPD, reads nn
*/
#include "ga_service_server_header.h"

void * ga_service_get_client_info(int * fd, struct ga_service_client_info * info)
{
	int k;
	int temp;
	int counter;
	int allocated_flag;
	int ret;
	int prev_command;
	int prev_state;

	union 
	{
		struct sockaddr addr;
		char data[MAXSOCKADDR];
	} peer;
	
	socklen_t len = MAXSOCKADDR;
	
	struct io_block * block = NULL;
	
	syslog(LOG_INFO,"ga_service_get_client_info thread[%d]",pthread_self());
	
	/* Check if info is already allocated */
	if (info != NULL)
	{
		allocated_flag = 1; //already allocated
	}
	else
	{
		if ( !(info = (struct ga_service_client_info *)calloc(1,sizeof(struct ga_service_client_info)) ))
		{
			syslog(LOG_CRIT,"Error calloc info in ga_service_client_info: %s",strerror(errno));
			return(NULL);
		}
		info->connptr = NULL;
		info->ret = NULL;
		info->pat_index = NULL;
		allocated_flag = 0; //first allocation
	}

	/* Allocate io_block block */
	if (!(block = (struct io_block *)va_calloc_io_block(0, 0, 0,block) ))
	{
		syslog(LOG_CRIT,"Error va_calloc_io_block() in ga_service_get_client_info()");
		return(NULL);
	}
	
	/* Set connfd in io_block block */
	block->connfd = *fd;

	/* Set connfd for client info */
	info->connptr = fd;
		
	/* Get peer inet parameters */
	if (getpeername(*fd, (struct sockaddr *) peer.data, &len) < 0)
	{
		syslog(LOG_CRIT,"Error getpeername() in ga_service_get_client_info: %s",strerror(errno));
		return(NULL);
	}
	memcpy(&(info->addr), &(peer.addr), len);
	
	/* Set current generation */
	if ((ret = pthread_mutex_lock(&generation_mutex) )!= 0)
	{
		syslog(LOG_ERR,"Error locking mutex generation_mutex in ga_service_get_client_info(): %d", ret);
	}	
	info->curr_gen = current_generation;
	if ((ret = pthread_mutex_unlock(&generation_mutex) )!= 0)
	{
		syslog(LOG_ERR,"Error unlocking mutex generation_mutex in ga_service_get_client_info(): %d", ret);
	}	

	/* REQ / UPD block */
	if (!(block = (struct io_block *)va_dread_io_block(block)))
	{
		syslog(LOG_ERR,"Error va_dread_io_block in ga_service_get_client_info");
		block = (struct io_block *)va_free_io_block(block);
		return(NULL);
	}
	
	/* Descramble block */
	/* 2 kinds of blocks: REQ and UPD */
	counter = 0;
	for (k=0; k< block->num ; k++)
	{
		if (!strncmp(block->char_vector[k], "ID", 2) )
		{
			info->id = atoi(block->char_vector[k] + 3);
		}
		else if (!strncmp(block->char_vector[k], "REQ", 3) )
		{
			info->state = 0;
		}
		else if (!strncmp(block->char_vector[k], "UPD", 3))
		{
			info->state = 1;
		}
		else if (!strncmp(block->char_vector[k], "command", 7))
		{
			info->command = atoi(block->char_vector[k] + 8);
		}
	}
		
	if (info->state == 0) // REQ, no further reading
	{
		block = (struct io_block *)va_free_io_block(block);
		return(info);
	}
	
	/********************************/
	/* Continue reading in UPD case */
	/********************************/
	/* ERROR Reading */
	/* Free block */
	if (!(block = (struct io_block *)va_delete_io_block(-1, block)))
	{
		syslog(LOG_ERR,"Error va_delete_io_block in ga_service_get_client_info");
		info->state = 0;
		block = (struct io_block *)va_free_io_block(block);
		return(info);
	}
	
	/* Read error block */
	if (!(block = (struct io_block *)va_dread_io_block(block)))
	{
		syslog(LOG_ERR,"Error va_dread_io_block in ga_service_get_client_info (ERROR)");
		info->state = 0;
		block = (struct io_block *)va_free_io_block(block);
		return(NULL);
	}

	/* Get error */
	if (!(info->ret = (struct nn_return_f *)ga_unpack_error(info->ret, block)))
	{
		syslog(LOG_ERR,"Error ga_unpack_error in ga_service_get_client_info (ERROR)");
		info->state = 0;
		block = (struct io_block *)va_free_io_block(block);
		return(NULL);
	}

	/* NN Reading */
	/* Free block */
	if (!(block = (struct io_block *)va_delete_io_block(-1, block)))
	{
		syslog(LOG_ERR,"Error va_delete_io_block in ga_service_get_client_info");
		info->state = 0;
		block = (struct io_block *)va_free_io_block(block);
		return(info);
	}

	/* Read remote block */
	if (!(block = (struct io_block *)va_dread_io_block(block)))
	{
		syslog(LOG_ERR,"Error va_dread_io_block in ga_service_get_client_info (NN)");
		info->state = 0;
		block = (struct io_block *)va_free_io_block(block);
		return(NULL);
	}
	
	/* Unpack neural net */
	if (info->ret->net != NULL)
	{
		info->ret->net = (struct neural_net *)nn_free_neural_net(info->ret->net);
	}
	if (!(info->ret->net = (struct neural_net *)nn_unpack_neural_net(info->ret->net, block)))
	{
		syslog(LOG_ERR,"Error nn_unpack_neural_net in ga_service_get_client_info (NN)");
		info->state = 0;
		block = (struct io_block *)va_free_io_block(block);
		return(NULL);
	}
	
	/* Free elements */
	block = (struct io_block *)va_free_io_block(block);
	
	return(info);
}
