## dr-theobold.l - a text-based adventure game
## Copyright (C) 2017  Christopher Howard

## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.

(load "tree.l")

(de prinll @ (prog
                (let WW (read-world-var "wrap-width")
                   (apply
                      '((L)
                        (prinl
                           (wrap
                              (if (not WW) 72 WW)
                              (chop L) ) ) )
                      (rest) )
                   (prinl) ) ) )

(de world ARGLST
   (setq World (aa-insert "rooms" NIL NIL))
   (run ARGLST)
   World )

(de save-world (Filename)
   (let Result
      (catch '(NIL)
         (out Filename
            (println World) )
         (prinll "Game progress saved.")
         'Complete)
      (unless (= Result 'Complete)
         (prinll Result) ) ) )

(de load-world (Filename)
   (let Result
      (catch '(NIL)
         (in Filename
            (setq World (read)) )
         (prinll "Saved game loaded.")
         'Complete )
      (if (<> Result 'Complete)
         (prog
            (prinll Result)
            (prinll "Could not find file \"theobold.sav\" in current directory.") )
         (look-at-room (location)) ) ) )

(de location @
   (if (not (arg 1))
      (aa-search-val World "location")
      (setq World (aa-insert "location" (next) World)) ) )

(de rooms ()
   (aa-search-val World "rooms") )

(de room ARGLST
   (if (not (car ARGLST))
      (quit "func `room' missing argument")
      (let (Room ())
         (run (cdr ARGLST))
         (setq World
            (aa-insert
               "rooms"
               (aa-insert (car ARGLST) Room (rooms)) World ) ) ) ) )

(de room-long-description (Description)
   (setq Room (aa-insert "long-description" Description Room)) )

(de room-command @
   (let (ArgLst (rest)
         Cmds (head -1 ArgLst)
         @Fn (car (tail 1 ArgLst)))
      (mapc (fill '((Cmd) (room-command-core Cmd (car (quote @Fn))))) Cmds) ) )

(de room-command-core (Cmd Fn)
   (when (str? Cmd) (setq Cmd (str-to-lst Cmd)))
   (setq Room (aa-insert "room-commands"
                 (aa-insert Cmd Fn
                    (aa-search-val Room "room-commands") )
                 Room ) ) )
   
(de room-exit (Direction Fn)
   (setq Room (aa-insert "room-exits"
                 (aa-insert Direction Fn
                    (aa-search-val Room "room-exits") )
                 Room ) ) )

(de add-room-exit (Direction RoomID Fn)
   (let (OldRooms (aa-search-val World "rooms")
         OldRoom (get-room RoomID)
         OldExits (aa-search-val OldRoom "room-exits")
         NewExits (aa-insert Direction Fn OldExits)
         NewRoom (aa-insert "room-exits" NewExits OldRoom)
         NewRooms (aa-insert RoomID NewRoom OldRooms)
         NewWorld (aa-insert "rooms" NewRooms World) )
      (setq World NewWorld) ) )

# SEGFAULTS
(de delete-room-exit (RoomID Direction)
   (let (OldRooms (aa-search-val World "rooms")
         OldRoom (get-room RoomID)
         OldExits (aa-search-val OldRoom "room-exits")
         NewExits (aa-delete Direction OldExits)
         NewRoom (aa-insert "room-exits" NewExits OldRoom)
         NewRooms (aa-insert RoomID NewRoom OldRooms)
         NewWorld (aa-insert "rooms" NewRooms World) )
      (setq World NewWorld) ) )

(de get-room (RoomID)
   (aa-search-val (rooms) RoomID) )

(de print-exits (RoomID)
   (let (Exit-Str
         (glue ", "
            (aa-keys-to-list
               (aa-search-val
                  (get-room RoomID) "room-exits" ) ) ) )
      (when Exit-Str
         (prinll (glue "" (list "Exits: " Exit-Str))) ) ) )

(de look-at-room (RoomID)
   (let D (aa-search-val (get-room RoomID) "long-description")
      (if (fun? D)
         (D)
         (prinll D) ) )
   (print-exits RoomID) )

(de enter-room (RoomID)
   (location RoomID)
   (look-at-room RoomID) )

(de enter-room-quietly (RoomID)
   (location RoomID) )

(de c-loop X
   (let (continue '(() (throw 'CONTINUE NIL))
         break '(() (throw 'BREAK NIL)))
      (catch 'BREAK
         (loop
            (catch 'CONTINUE
               (run X) ) ) ) ) )

(de print-help ()
   (prinll "These are the commands you will need most often:")
   (prinll "help, quit, save, load, width, go <exit>, look, look at <item>, search \
            <item>, use <item>, put <item> in <item>, speak <word>.")
   (prinll "These are not all of the commands available. You may need to guess other \
            commands depending on the type of object involved. Keep commands simple \
            and avoid using articles such as 'a' and 'the'.")
   (prinll "For convenience, the commands 'n', 's', 'w', and 'e' translate to 'go \
            north', 'go south', etc. Also, 'i', 'o', 'u', and 'd', translate \
            to 'go in', 'go out', 'go up', and 'go down'.")
   (prinll "It is not necessary to explicitly use an 'open door' command before \
            going through one. However, some doors are locked and it is necessary to \
            find some way to unlock them.")
   (prinll "The 'width' command is very useful if you need to adjust the text \
            wrapping width for a narrow screen size."))

(de print-warranty ()
   (prinll
      "This program is distributed in the hope that it will be useful, but \
       WITHOUT ANY WARRANTY; without even the implied warranty of \
       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU \
       General Public License for more details." ) )

(de print-license ()
   (prinll
      "This program is free software: you can redistribute it and/or modify it \
       under the terms of the GNU General Public License as published by the \
       Free Software Foundation, either version 3 of the License, or (at your \
       option) any later version. You should have received a copy of the GNU \
       General Public License along with this program. If not, see \
       <http://www.gnu.org/licenses/>." ) )

(de print-inventory ()
   (let (inventory-lst
         (sort
            (mapcar 'car
               (aa-to-list
                  (aa-search-val World "inventory") ) ) ) )
      (if (not inventory-lst)
         (prinll "You are not carrying anything.")
         (prog
            (prinll "Your are carrying the following items:")
            (prinll (glue "\n" inventory-lst)) ) ) ) )

(de set-wrap-width ()
   (prinll "To change the wrap width of the text, input a value between 30 and 100:")
   (let INP (read)
      (if (and (num? INP) (>= INP 30) (<= INP 100))
         (prog
            (set-world-var "wrap-width" INP) )
         (prinll "Invalid value.") ) ) )

(de handle-unhandled-cmd (Cmd)
   (cond
      ((= (head 2 Cmd) '("look" "at"))
       (prinll "You notice nothing of interest.") )
      ((= Cmd '("search"))
       (prinll "You must search some thing, like 'search cabinet'.") )
      ((= (car Cmd) "search")
       (prinll "You find nothing of interest.") )
      ((= (car Cmd) "put")
       (prinll "You cannot do that.") )
      ((or (= (car Cmd) "use")
          (= (car Cmd) "break")
          (= (car Cmd) "take") )
       (prinll "You do not see any value in doing that.") )
      (T
         (prinll "Nothing happens.") ) ) )

(de str-to-lst (Str)
   (mapcar '((X) (lowc (pack X))) (split (chop Str) " ")) )

(de read-to-lst ()
   (str-to-lst (line T)) )

(de read-loop ()
   (c-loop
      (prin "> ")
      (flush)
      (let (Cmd (read-to-lst))
         (prinl "")

         (cond
            ((= Cmd '("n")) (setq Cmd '("go" "north")))
            ((= Cmd '("w")) (setq Cmd '("go" "west")))
            ((= Cmd '("s")) (setq Cmd '("go" "south")))
            ((= Cmd '("e")) (setq Cmd '("go" "east")))
            ((= Cmd '("i")) (setq Cmd '("go" "in")))
            ((= Cmd '("o")) (setq Cmd '("go" "out")))
            ((= Cmd '("u")) (setq Cmd '("go" "up")))
            ((= Cmd '("d")) (setq Cmd '("go" "down"))) )
         
         (when (= Cmd '("help"))
            (prog (print-help) (continue)))

         (when (= Cmd '("warranty"))
            (prog (print-warranty) (continue)))
         
         (when (= Cmd '("license"))
            (prog (print-license) (continue)) )

         (when (= Cmd '("width"))
            (prog (set-wrap-width) (continue)) )
         
         (when (= Cmd '("look"))
            (prog (look-at-room (location)) (continue)) )

         (when (or (= Cmd '("quit")) (= Cmd '("bye")))
            (prog (prinll "Goodbye!") (bye)) )

         (when (or (= Cmd '("inventory")) (= Cmd '("show" "inventory")))
            (prog (print-inventory) (continue)) )

         (when (or (= Cmd '("save")) (= Cmd '("save" "game")))
            (save-world "theobold.sav")
            (continue) )
         
         (when (or (= Cmd '("load")) (= Cmd '("load" "game")))
            (load-world "theobold.sav")
            (continue) )
         
         (let (RoomCmd
               (aa-search-val
                  (aa-search-val
                     (get-room (location))
                     "room-commands" ) Cmd ) )
            (when RoomCmd
               (prog (RoomCmd) (continue)) ) )
         
         (if (= (car Cmd) "go")
            (let (ExitCmd
                  (aa-search-val
                     (aa-search-val
                        (get-room (location))
                        "room-exits" ) (cadr Cmd) ) )
               (if ExitCmd
                  (prog (ExitCmd) (continue))
                  (prog (prinll "You cannot go that direction.") (continue)) ) ) )
         
         (handle-unhandled-cmd Cmd) ) ) )

(de set-world-var (Name Val)
   (setq World (aa-insert "variables"
                  (aa-insert Name Val
                     (aa-search-val World "variables") ) World ) ) )

(de read-world-var (Name)
   (aa-search-val (aa-search-val World "variables") Name) )

(de add-to-inventory (ItemName ItemData)
   (setq World (aa-insert "inventory"
                  (aa-insert ItemName ItemData
                     (aa-search-val World "inventory") ) World ) ) )

(de remove-from-inventory (ItemName)
   (setq World (aa-insert "inventory"
                  (aa-delete ItemName
                     (aa-search-val World "inventory") ) World ) ) )

(de inventory-contains? (ItemName)
   (car (aa-search (aa-search-val World "inventory") ItemName)) )

(de get-inventory-data (ItemName)
   (aa-search-val (aa-search-val World "inventory") ItemName) )

(de use-threshold-keypad ()
   (prin "Enter a five digit code: ") (flush)
   (let (Input (format (car (read-to-lst))))
      (if (or (not (num? Input))
             (< Input 0) (> Input 99999) )
         (prinll "Five digits, please.")
         (if (= Input 31415)
            (prog (prinll "\nYou hear a clicking sound.")
               (set-world-var "threshold-door-unlocked" T))
            (prinll "\nYou input the code but nothing happens.") ) ) ) )

# Thanks Mike Pechkin!
(de az-AZ (N)
   (or
      (<= `(char "a") N `(char "z"))
      (<= `(char "A") N `(char "Z")) ) )

(de alpha? (S)
   (let S (chop S)
      (and
         (not (cdr S))
         (az-AZ (char (car S))) ) ) )

(de cryptogram-encrypt (Message SubsTree)
   (let (SubFn
         '((C) (if (not (alpha? C))
                  C
                  (let (UC (uppc C))
                     (if (aa-search-val SubsTree UC)
                        @
                        "_" ) ) ) ) )
      (glue "" (mapcar 'SubFn (chop Message))) ) )

(de space-out (Str)
   (glue " " (chop Str)) )

(de use-crypto-tablet ()
   (prinll "The tablet displays a cryptogram:")
   (prinll "Enter a letter, a space, and then the letter you wish to substitute for \
            it. If you wish to take a break, enter \"done\".")
   (let (EncMessage
         "GPB J YZVPNB OXANUT YE RNDEUB IJABEO, GPB OGBPNAA, GPB TEUUX:\n\
          TEV IDGY LGP YDN OGP BE YDGY LEONA GTYNV YDN HJPF?\n\
          NMNP YDGY IDJLD DGA RNNP GUVNGBX BEPN."
         DecMessage
         "AND I TURNED MYSELF TO BEHOLD WISDOM, AND MADNESS, AND FOLLY:\n\
          FOR WHAT CAN THE MAN DO THAT COMES AFTER THE KING?\n\
          EVEN THAT WHICH HAS BEEN ALREADY DONE." )
      (c-loop
         (prin " ")
         (prinll (space-out EncMessage))
         (prin " ")
         (let (RenderedMessage (cryptogram-encrypt
                                  EncMessage
                                  (read-world-var
                                     "crypto-tablet-substitutions" ) ) )
            (prinll
               (space-out
                  (cryptogram-encrypt
                     EncMessage
                     (read-world-var "crypto-tablet-substitutions") ) ) )
            (when (= RenderedMessage DecMessage)
               (prinll "Puzzle complete. Thank you for playing.")
               (prinll "To relax, be sure to try out the radio in the next room.")
               (break) ) )
         (prin "> ")
         (flush)
         (let (Cmd (read-to-lst))
            (prinl)
            (when (= Cmd '("done"))
               (prinll "You set the tablet back on the table.")
               (break) )
            (when (and
                   (alpha? (car Cmd))
                   (alpha? (cadr Cmd)) )
               (let (L1 (uppc (car Cmd))
                     L2 (uppc (cadr Cmd)) )
                  (when (rassoc
                           L2
                           (aa-to-list
                              (read-world-var "crypto-tablet-substitutions") ) )
                     (set-world-var
                        "crypto-tablet-substitutions"
                        (aa-delete
                           (car @)
                           (read-world-var "crypto-tablet-substitutions") ) ) )
                  (set-world-var
                     "crypto-tablet-substitutions"
                     (aa-insert L1 L2
                        (read-world-var "crypto-tablet-substitutions") ) ) ) ) ) ) ) )

(de use-old-radio ()
   (prinll "You turn the tuning knob a little. You hear some static and \
            crackling as you settle on a station.")
   (case (rand 1 11)
      (1 (prinll "\"...Who knows what evil lurks in the hearts of men? The \
                    Shadow knows...\"") )
      (2 (prinll "\"...Ladies and gentlemen. The story you are about to \
                    hear is true. The names have been changed to \
                    protect the innocent...\"") )
      (3 (prinll "\"...Who's afraid of the big bad wolf? Tra la la la la...\""))
      (4 (prinll "\"...Hi-yo, Silver, away!...\""))
      (5 (prinll "\"...And the gold of her hair, crowns the blue of her eyes...\""))
      (6 (prinll "\"...We didn't exactly believe your story, Miss \
                       Wonderly. We believed your 200 dollars...\""))
      (7 (prinll "You hear the gentle notes of the Blue Danube."))
      (8 (prinll "You hear Sinatra's velvety voice singing \"Close To You\"."))
      (9 (prinll "You hear the fiery oration of a fundamentalist preacher."))
      (10 (prinll "\"...speak 'friend' and enter...\""))
      (11 (prinll "You hear a jazzy tune you don't recognize."))))

(de process-submarine-lock-mechanism (Action)
   (set-world-var "sub-unlock-sequence"
      (head 4
         (cons Action (read-world-var "sub-unlock-sequence")) ) )
   (when (= (read-world-var "sub-unlock-sequence")
            '("pull" "rotate" "push" "rotate") )
      (set-world-var "submarine-unlocked" T)
      (prinll "The hatch of the submarine suddenly becomes loose. You release that you \
               understand how the locking mechanism works and can enter and exit the \
               submarine at will.") ) )

(de process-stuck-sub-move (Action)
   (set-world-var "sub-unstuck-sequence"
      (head 6
         (cons Action (read-world-var "sub-unstuck-sequence")) ) )
   (when (let Sequence (read-world-var "sub-unstuck-sequence")
            (or (= Sequence '(up west east up west east))
               (= Sequence '(up east west up east west)) ) )
      (set-world-var "sub-unstuck" T)
      (prinll "With an effort, the submarine pulls itself free of the mud.") ) )

(de set-sphere-dial ()
   (prinll "Input a value between 0 and 100:")
   (let INP (read)
      (if (and (num? INP) (>= INP 0) (<= INP 100))
         (prog
            (set-world-var "sphere-frequency" INP)
            (prinll (glue "" (list "You adjust the dial to a value of " INP "."))) )
         (prinll "It is not possible to set that value on the dial.") ) ) )

(de use-laser ()
   (let (SB (read-world-var "sphere-broken")
         FC (= (read-world-var "sphere-frequency") 72))
      (cond
         (SB
            (prinll "The device fires a powerful but focused blast of energy, scorching the \
                     wall." ) )
         ((not (or SB FC))
          (prinll "The device fires a powerful but focused blast of energy, which passes \
                   through the glass sphere and scorches the wall."))
         (T (prog
               (prinll "The device fires a powerful but focused blast of energy, which strikes \
                        the ball of metal hovering in the center of the glass sphere. A \
                        brilliant light engulfs the room, and then the sphere explodes. After \
                        the smoke clears, you find a glowing, green rock amidst the broken \
                        remains of the sphere apparatus.")
               (set-world-var "sphere-broken" T)
               (add-to-inventory "glowing rock")
               (prinll "A glowing rock has been added to your inventory.") ) ) ) ) )

(de desc-transporter ()
   (cond
      ((and
          (read-world-var "transporter-energized")
          (not (read-world-var "transporter-destination")) )
       (prinll "A faint energy shimmers inside the circle.") )
      ((and
          (read-world-var "transporter-energized")
          (read-world-var "transporter-destination") )
       (prinll "To your amazement, through the formerly empty circle you can see into \
                another room in some distant place. Unlike a camera or video display, \
                you see depth and it seems as though you could simply step inside. \
                Nevertheless, it is difficult to make out what is on the other side, as \
                the lighting there seems to be pale and dim."))
      (T (prinll "The symmetry and complexity of this device is awe-inspiring. The \
                  electronics, coils, and rods are pointed towards the inside of \
                  the circle. You notice an empty chamber near the base of the \
                  structure." ) ) ) )

(de desc-transporter-energy-chamber ()
   (if (read-world-var "transporter-energized")
      (prinll "A brilliant green light pulsates within the chamber.")
      (prinll "Inside, the chamber looks like a pouch, made of a mesh of coils and strips of metal.") ) )

(de put-rock-in-chamber ()
   (if (inventory-contains? "glowing rock")
      (prog
         (remove-from-inventory "glowing rock")
         (set-world-var "transporter-energized" T)
         (prinll "As you push the rock inside, the chamber begins to throb with light and energy.") )
      (prinll "The chamber is already filled.") ) )

(de load-transporter-parameters ()
   (if (not (inventory-contains? "equations"))
      (prinll "You do not have equations with you.")
      (if (< (read-world-var "physics-understanding") 3)
         (prinll "You try to enter parameters into the control panel, but you do not \
                  understand enough physics to know what you are doing.")
         (prog
            (prinll "You enter a set of parameters into the control panel, representing \
                     coordinates and frequencies and motions. You hear, emanating from \
                     the main apparatus, the sounds of crystal-like vibrations and electrical \
                     discharges.")
            (set-world-var "transporter-destination" "mars") ) ) ) )

(de desc-weapon (weapon)
   (if (read-world-var "studied-weapons")
      (prinll
         (cdr
            (assoc weapon
               '(("longsword" . "This longsword is about five feet long from bottom of the handle \
                                 to tip of the double-edged blade. Its long handle is to be \
                                 held double-handed. This allows it to be used for powerful, vicious \
                                 slicing, which could amputate in a single stroke.")
                 ("pollaxe" . "This pollaxe is part warhammer and part axe blade, with a \
                               spike at the top. This combination makes it a versatile weapon, \
                               useful for impact, cutting, and stabbing damage.")
                 ("longbow" . "This large bow, over six feet in length, is cut from a yew branch. \
                               Longbows were famous for their power and distance. With the proper \
                               arrow, they can pierce even plate armor, and that up to a range of \
                               several hundred feet.")
                 ("crossbow" . "This simpler crossbow design is essentially a piece of wood attached \
                                to a metal bow and bowstring, with a latch and trigger mechanism for \
                                holding and releasing the bowstring. The bowstring is pulled back with \
                                the help of a separate lever also carried by the crossbowman. Unlike a \
                                longbow (which requires years to master) the crossbow was easy to use \
                                and only required about a week of training, and therefore could quickly \
                                turn any peasant into an effective ranged soldier. Unfortunately \
                                crossbows were much slower to reload than longbows, owing to the \
                                difficulty of 'cocking', or pulling back, the bowstring.")
                 ("mace" . "This mace is essentially a wooden rod with a heavy weight at the top. \
                            The weight is flanged, or ridged, so as to more effectively concentrate \
                            the swinging force. Maces were quite popular weapons as they were cheap \
                            to make, easy to use, and effective against heavily armoured foes. \
                            Basically, you bludgeon your opponent, and the impact force travels \
                            well even through mail or plate armour." ) ) ) ) )
      (prinll
         (cdr
            (assoc weapon
               '(("longsword" . "This is a really long sword, nearly as tall as a person. It is hard to \
                                 imagine anyone holding it.")
                 ("pollaxe" . "This weapon appears to be a combination of axe, hammer, and spear.")
                 ("longbow" . "A very tall bow, made of some kind of flexible wood.")
                 ("crossbow" . "It looks like a metal bow and a trigger mechanism attached to a block \
                                of wood.")
                 ("mace" . "It looks like a stick with a heavy metal weight at the top. You'd hate \
                            to get clobbered with one of these!" ) ) ) ) ) ) )

(de use-drink-machine ()
   (prinll "The machine is apparently programmed to mix you a random fruit juice drink.")
   (prinll (glue "" (list "This time it dispenses "
      (case (rand 1 5)
         (1 "an apple juice base with a touch of raspberry and cranberry. You could \
             see yourself drinking this regularly.")
         (2 "mango juice with a little lime and orange juice added. You wish you were \
             on a tropical beach!")
         (3 "a mix of blueberry and blackberry. Not bad, though you find the \
             blueberry taste a bit overpowering.")
         (4 "lemonade, but with a minty taste to it. An interesting idea.")
         (5 "some kind of strawberry drink with some other fruits. Very tasty, though \
             you can't quite tell what the other flavors are.") ) ) )))

(de read-physics-book ()
   (case (read-world-var "physics-understanding")
      (NIL (prog
              (set-world-var "physics-understanding" 1)
              (prinll "You skim through the book, but the concepts are very technical.") ) )
      (1 (prog
            (set-world-var "physics-understanding" 2)
            (prinll "You spend a few hours carefully studying the book. You begin to \
                     see some core concepts, but you understanding of them is rather \
                     fuzzy.")))
      (2 (prog
            (set-world-var "physics-understanding" 3)
            (prinll "After studying the book again, you start to truly grasp the material.") ) )
      (3 (prinll "You contemplate the material and are struck by many fascinating ideas.")) ) )

(de drive-car ()
   (let fuel (read-world-var "car-fuel")
      (cond
         ((not fuel) (prinll "The engine will not start."))
         ((= fuel 'jet) (prog
                           (prinll "You feel a deep vibration and hear a loud humming noise. As soon as you \
                                    put the car in gear, it lurches forward and goes rocketing down the \
                                    track.")
                           (prinll "As you apply acceleration you approach 200mph but must begin braking to \
                                    avoid hitting the wall at the end of the road. Your heart pumping with \
                                    the thrill, you turn the vehicle around and race back to the beginning \
                                    of the track." ) ))
         ((= fuel 'theo) (prog
                            (prinll "You feel a deep vibration and hear a loud humming noise. As soon as you \
                                     put the car in gear, it lurches forward and goes rocketing down the \
                                     track.")
                            (prinll "As you apply acceleration, you quickly reach 200mph, and then 250mph. In \
                                     your excitement, you notice the end of the track much too late, and madly \
                                     slam the brakes.")
                            (prinll "Before you are able to come to a complete stop, you ram into the wall, \
                                     and an airbag deploys. After a few minutes of recovery, you push your \
                                     door out and exit the vehicle.")
                            (set-world-var "track-car-destroyed" T)
                            (delete-room-exit "track" "in")
                            (add-room-exit "east" "track"
                               '(() (prinll "You take a long walk down the track.")
                                 (enter-room "hidden-track-room") ) )
                            (enter-room "hidden-track-room") ) ) ) ) )

(de read-bible ()
   (prinll "You turn to a random page and find an underlined verse.")
   (case (rand 1 11)
      (1 (prinll "\"It is the glory of God to conceal a thing, but the glory of kings is to \
                    search out a matter.\""))
      (2 (prinll "\"In him was life, and the life was the light of men.\""))
      (3 (prinll "\"All we like sheep have gone astray. Everyone has turned to his own way; \
                    and Yahweh has laid on him the iniquity of us all.\""))
      (4 (prinll "\"Beloved, let us love one another, for love is of God; and everyone who \
                    loves has been born of God, and knows God.\""))
      (5 (prinll "\"In the beginning, God created the heavens and the earth.\""))
      (6 (prinll "\"Why do you spend money for that which is not bread? and your labor for \
                    that which does not satisfy?\""))
      (7 (prinll "\"Remember also your Creator in the days of your youth, before the evil \
                    days come, and the years draw near, when you will say, \"I have no \
                    pleasure in them\".\"" ))
      (8 (prinll "\"I have seen all the works that are done under the sun; and behold, all \
                    is vanity and a chasing after wind.\""))
      (9 (prinll "\"I heard a loud voice out of heaven saying, 'Behold, God’s dwelling is \
                    with people, and he will dwell with them, and they will be his people, \
                    and God himself will be with them as their God.'\""))
      (10 (prinll "\"The Word became flesh, and lived among us. We saw his glory, such glory \
                     as of the one and only Son of the Father, full of grace and truth.\""))
      (11 (prinll "\"The wicked are like the troubled sea; for it can’t rest, and its waters \
                     cast up mire and dirt. 'There is no peace', says my God, 'for the \
                     wicked.'\""))))

(de enter-tower ()
   # TODO: pauses between paragraphs
   (prinll "You step into a room that appears to be a minature observatory. You see \
            a telescope and a glass ceiling, the starry night glittering above. Star \
            charts line the walls.")
   (prinll "In the center of the room is a large desk, and between it and you, a big \
            chair, turned away from you. You hear a deep but wispy voice from behind \
            the chair.")
   (prinll "'Please, come in. My apologies it took you so long to get here, and if \
            it was a rather confusing route. I supposes the locks and secret rooms \
            and such was a bunch of nonsense. I just wanted to give you a bit of a \
            challenge. I wasn't quite sure if the world was ready for ALL of my \
            inventions. So, that meant you were the only one I could really trust.'")
   (prinll "You hear a quiet chuckle.")
   (prinll "You feel a strong urge to turn the chair around, and to see the speaker. \
            But at the same time, you feel a certain hesitation, as though you are \
            afraid of learning something you already know.")
   (enter-room-quietly "tower") )

(de turn-chair ()
   (prinll "You turn the chair slowly. A handsome but age-worn face stares back at \
            you. You have seen that face many times...")
   (prinll "'Yes,' he says, 'I am Dr. Theobold. And I believe we have met.' He \
            smiles.")
   (prinll "Your head begins to hurt. You close your eyes and hold your head, \
            waiting for the pain to subside. You become dizzy, and you feel the \
            world around you vanishing like a mist.")
   (prinll "Suddenly you awake, as if from a long dream, and find yourself in your \
            familiar office at the university.")
   (enter-room "university-office") )

(de start-game ()
   (prinll "")
   (prinll "Dr. Theobold: A Text Adventure Game")
   (prinll "Copyright (C) 2017 Christopher Howard")
   (prinll "This program comes with ABSOLUTELY NO WARRANTY; for details type 'warranty'.")
   (prinll "This is free software, and you are welcome to redistribute it under \
            certain conditions; type 'license' for details.")
   (prinll "Use the 'help' command to get more information about available game \
            commands.")
   (init-world)
   (enter-room "threshold")
   (read-loop) )

(de init-world ()
   (world
      (location "threshold")
      (room "threshold"
         (room-long-description "You are standing at the entrance to a building, reputed to be the home \
                                 of one Dr. Theobold. The entrance is barred by a solid oak door. A \
                                 rubber mat lines the threshold.")
         (room-command "search mat" "search under mat" "look under mat"
            '(() (prinll "You found a small note. It is a picture of the greek letter Pi, a famous \
                          mathematical symbol." ) ) )
         (room-command "knock on door"
            '(() (prinll "No one answers.")) )
         (room-command "break door"
            '(() (prinll "You would just hurt yourself trying to break that door down.")) )
         (room-command "look at keypad"
            '(() (prinll "It is a keypad lock that takes a five digit code. Those are a lot of \
                          combinations to try.")) )
         (room-command "look at building"
            '(() (prinll "There is nothing particularly interesting about the outside of the \
                          building, except there is some kind of tower structure jutting a floor \
                          or two above the rest of the building.")) )
         (room-command "look at threshold"
            '(() (prinll "You see a rubber mat.")) )
         (room-command
            "look at door" "look at oak door" "search door" 
            '(() (prinll "You hesitate... But if half the rumors are true, it is worth the risk. \
                          You'll never find out any other way.")
              (prinll "Instead of a traditional lock, the door has a five digit numeric keypad.") ) )
         (room-command "look at mat"
            '(() (prinll "It is an uninteresting rubber mat.")))
         (room-command "use keypad" 'use-threshold-keypad)
         (room-exit "north" '(() (if (read-world-var "threshold-door-unlocked")
                                    (enter-room "atrium")
                                    (prinll "The door is locked.") ) ) ) )
      (room "atrium"
         (room-long-description "You are in a large atrium. The walls are covered in ornate carvings. The \
                                 room is mostly open and empty except a few chairs and couches in the \
                                 middle, centered around a small table. A small computer tablet is laying \
                                 on the table. There is a door heading back out to the front entrance, \
                                 and another door into a living room.")
         (room-exit "south" '(() (enter-room "threshold")))
         (room-exit "east" '(() (enter-room "living-room")))
         (room-command "look at tablet" "look at computer tablet"
            '(() (prinll "It is a small computer tablet with a puzzle game loaded.")) )
         (room-command "use tablet" "use computer tablet" '(() (use-crypto-tablet)))
         (room-command "look at table" '(() (prinll "There is a computer tablet on the table.")))
         (room-command "look at walls" "look at carvings"
            '(() (prinll "They are neat and beautiful geometric patterns.")) )
         (room-command "look at chairs" "look at couches" "look at chair" "look at couch"
            '(() (prinll "Comfortable but otherwise uninteresting furniture.")) )
         (room-command "sit on chair" "sit on couch"
            '(() (prinll "You relax for a few minutes.")) )
         (room-command '("use" "computer" "tablet") '(() (use-crypto-tablet))) )
      (room "living-room"
         (room-long-description "You are in a cozy living room with a fuzzy carpet and plush furniture. \
                                 An old-fashioned radio leans against the wall. You also see an small \
                                 elevator. You see exits to the west and the east, and as well as a door \
                                 on the north wall.")
         (room-command "look at elevator"
            '(() (prinll "It looks like one of the small, classy hotel elevators from an old \
                          movie, but with buttons instead of an operator.")) )
         (room-command "sit in recliner" "sit on couch" "lay on couch"
            '(() (prinll "You feel so relaxed you almost fall to sleep.")) )
         (room-command "look at carpet"
            '(() (prinll "The carpet is well worn, but thick and soft.")) )
         (room-command "look at furniture"
            '(() (prinll "There is a very comfortable looking couch, and several recliners.")) )
         (room-command "search couch"
            '(() (prinll "You find some pocket change, a pen, and a few pieces of lint. None of \
                          which you particularly need at the moment.")))
         (room-exit "west" '(() (enter-room "atrium")))
         (room-exit "north"
            '(() (if (not (read-world-var "bedroom-door-unlocked"))
                    (prinll "The door seems to be locked.")
                    (prog
                       (set-world-var "bedroom-door-unlocked" NIL)
                       (enter-room "bedroom") ) ) ) )
         (room-command "use card on reader" "use card on card reader"
            "use card in reader" "use card in card reader"
            "insert card in reader" "insert card in card reader"
            "put card in reader" "put card in card reader"
            "insert card into reader" "insert card into card reader"
            "put card into reader" "put card into card reader"
            '(() (if (not (inventory-contains? "card"))
                    (prinll "You do not have a card.")
                    (prog
                       (set-world-var "bedroom-door-unlocked" T)
                       (prinll "You hear a loud click.") ) ) ) )
         (room-command "look at door" "search door"
            '(() (prinll "Just to the side of the door is a small card reader.")) )
         (room-exit "east" '(() (enter-room "kitchen")))
         (room-exit "down" '(() (if (read-world-var "elevator-unlocked")
                                   (enter-room "programming-room")
                                   (prinll "The elevator does not appear to be functional.") ) ) )
         (room-command '("use" "radio") '(() (use-old-radio)))
         (room-command "look at radio" "look at old radio" "look at old-fashioned radio"
            '(() (prinll "It is big and bulky, but obviously quality material, and in good \
                          condition.")) )
         (room-command '("use" "old" "radio") '(() (use-old-radio)))
         (let (elevator-release '(() (prog
                                        (prinll "You hear a buzzing noise \
                                                 coming from the elevator.")
                                        (set-world-var "elevator-unlocked" T) ) ) )
            (room-command '("speak" "friend") elevator-release)
            (room-command '("speak" "\"friend\"") elevator-release)
            (room-command '("say" "friend") elevator-release)
            (room-command '("say" "\"friend\"") elevator-release) )
         (room-command '("use" "old-fashioned" "radio") '(() (use-old-radio))) )
      (room "bedroom"
         (room-long-description
            "This appears to be the doctor's bedroom. The bed is unmade, and some \
             books and items of clothing are scattered here and there.")
         (room-command "look at bed"
            '(() (prinll "An uninteresting full size bed.")) )
         (room-command "look at books"
            '(() (prinll "A few interesting tomes scattered here and there.")) )
         (room-command "search books"
            '(() (prinll "You find several volumes of Tolkien's Lord of the Rings series. They are \
                          well worn.")))
         (room-command "read book" "read books" "read lord of the rings" "read tolkien"
            '(() (prinll "You spent a few minutes catching up on the adventures of Frodo Baggins \
                          and Gandalf the Grey.")))
         (room-command "search bed"
            '(() (prinll "Mysteriously, you find a sock hidden under the pillow. You decide to \
                          leave it be.")))
         (room-exit "west"
            '(() (enter-room "chapel")) )
         (room-exit "south"
            '(() (enter-room "living-room")) ) )
      (room "chapel"
         (room-long-description "You are in what looks to be a small chapel. It is peaceful here. There \
                                 is only a single bench, and you see a Bible on a reading stand. Hues of \
                                 blue and green and red and purple glow from stained glass in the east \
                                 wall. An organ occupies one corner.")
         (room-command "look at glass" "look at stained glass"
            '(() (prinll "The mixture of glowing colors is pleasant.")) )
         (room-command "look at bible"
            '(() (prinll "It is open for reading.")) )
         (room-command "read bible" 'read-bible)
         (room-command "use cable on organ" "use cable on electric organ"
            '(() (if (not (inventory-contains? "cable"))
                    (prinll "What cable?")
                    (prog
                       (prinll "You use the cable to connect the organ to a power port in the wall.")
                       (set-world-var "organ-powered" T)
                       (remove-from-inventory "cable") ) ) ) )
         (room-command "look at organ"
            '(() (prinll "It is an electric organ.") ))
         (room-command "turn on organ" "switch on organ" "power on organ"
            '(() (if (read-world-var "organ-powered")
                    (prinll "It is already powered on.")
                    (prinll "You flip switches, but there is no response.") ) ) )
         (room-command "play organ"
            '(() (if (not (read-world-var "organ-powered"))
                    (prinll "You press the keys but there is no sound.")
                    (if (read-world-var "organ-played")
                       (prinll "You play a few more pieces on the organ.")
                       (prog
                          (prinll "Having, fortunately, some background in the musical arts, you are \
                                   familiar with the organ and begin to play the composition already set on \
                                   it: Bach's 'Jesu, Joy of Man's Desiring'.")
                          (prinll "After the first verse, you begin to hear sounds reverberating from \
                                   somewhere in the walls — an accompaniment to your playing! You hear \
                                   other instruments, and even voices! Some electronic trick of Doctor \
                                   Theobold...?")
                          (prinll "Whether real or imagined, you cannot tell, but at the climax of your \
                                   playing, the rooms grows brighter and more colorful. In that moment, you \
                                   notice one of the wall panels seem to be slightly out of position.")
                          (set-world-var "chapel-panel-unlocked" T)
                          (set-world-var "organ-played" T) ) ) ) ) )
         (room-command "look at panel" "search panel"
            '(() (if (not (read-world-var "chapel-panel-unlocked"))
                    (prinll "What panel?")
                    (prinll "It looks as though the panel could slide open.") ) ) )
         (room-command "slide panel" "slide open panel" "slide wall panel" "slide open wall panel"
            '(() (if (not (read-world-var "chapel-panel-unlocked"))
                    (prinll "What panel?")
                    (prog
                       (prinll "The panel slides open, allowing you enter a stairway leading up.")
                       (enter-room "tower-door") ) ) ) )
         (room-exit "east"
            '(() (enter-room "bedroom")) ) )
      (room "tower-door"
         (room-long-description "You are at the top of the stairway, in front of a closed door.")
         (room-exit "in"
            '(() (if (read-world-var "tower-door-unlocked")
                    (enter-tower)
                    (prinll "The door seems to be locked.") ) ) )
         (room-command "break door"
            '(() (prinll "You contemplate a flying karate kick, but it seems more likely that you \
                          would break before the door did.")))
         (room-command "look at microphone"
            '(() (prinll "Voice activated entry, perhaps?")) )
         (room-command "look at door" "search door"
            '(() (prinll "There seems to be a small microphone built into the wall next to the \
                          door handle.")) )
         (room-command "speak elaine" "say elaine" "speak \"elaine\"" "say \"elaine\""
            '(() (prinll "You hear a loud click.")
              (set-world-var "tower-door-unlocked" T) ) )
         (room-exit "down"
            '(() (enter-room "chapel")) ) )
      (room "tower"
         (room-long-description
            "You are in a minature observatory with a telescope and a glass window. \
             Star charts line the walls. Someone is sitting at a large table, in a \
             chair facing away from you.")
         (room-exit "out"
            '(() (prinll "You are unable to leave — surely this place must be the climax of your \
                          adventure.")))
         (room-command "look above" "look at sky" "look at glass" "look through glass"
            '(() (prinll "You see a beautiful night sky, with crisp, brilliant stars, and the glow \
                          of the milky way galaxy.")))
         (room-command "use telescope" "look in telescope" "look through telescope"
            '(() (prinll "It is pointed at the moon.")) )
         (room-command "look at telescope"
            '(() (prinll "It is somewhat bulky, and clearly a precision instrument.")) )
         (room-command "look at table"
            '(() (prinll "The table is covered in papers and calculators and tablets.")) )
         (room-command
            "look at charts" "look at wall" "look at walls"
            "look at wall charts" "look at star charts"
            '(() (prinll "You see various posters, including diagrams of star formations, tables \
                          of numbers, maps of planetary movement, and other astronomical data." ) ) )
         (room-command "look at chair"
            '(() (prinll "The chair is turned away from you, so that you cannot see who was \
                          speaking to you.")) )
         (room-command "look at theobold" "look at dr. theobold" "look at doctor"
            '(() (prinll "You know it must be him behind the chair.")) )
         (room-command "turn chair" "rotate chair" "turn chair around"
            turn-chair ) )
      (room "university-office"
         (room-long-description "You are sitting a your office desk. The walls are covered in \
                                 bookshelves.")
         (room-exit "out"
            '(() (prinll "You don't want to leave yet. There is still something you are trying to \
                          remember, something about your dream.")))
         (room-command "look at books" "look at bookshelves" "look at shelves"
            '(() (prinll "You see books on many subjects, such as mathematics, astronomy, physics, \
                          history, and theology.")))
         (room-command "read book" "read books"
            '(() (prinll "You do not feel like reading right at the moment. Something is bothering \
                          you.")))
         (room-command "look at papers"
            '(() (prinll "An assortment of academic papers and homework assignments.")) )
         (room-command "look at desk"
            '(() (prinll "You see many papers piled on the desk, as well as a name plate.")) )
         (room-command "look at name plate" "look at plate" "read name plate" "read plate"
            '(() (prinll "The name plate reads:")
              (prinll "DR. JOHN THEOBOLD")
              (prinll "Game complete. Thank you for playing.")
              (wait 10000)
              (prinll "Credits:")
              (wait 5000)
              (prinll "Game developed by Christopher Howard of Fairbanks, Alaska in 2017.")
              (wait 5000)
              (prinll "Special thanks to Weston Howard <alaskalinuxuser>, who completed the \
                       first beta testing pass.")
              (wait 5000)
              (prinll "And also to the PicoLisp mailing list for assistance to a newbie \
                       PicoLisp programmer.")
              (wait 5000)
              (prinll "And also my wife Emily <3")
              (wait 5000)
              (prinll "And especially my Creator who gave me the ability to learn and to study \
                       and to be creative.")
              (wait 10000)
              (prinll "Press enter or return to quit the game.")
              (line T)
              (bye))))
      (room "kitchen"
         (room-long-description
            "You find a kitchen that is sophisticated but slightly \
             messy. Remnants of frozen dinner packaging decorate the \
             counter tops. You see a refrigerator, and also a bulky \
             microwave with several tubes connected to it." )
         (room-command "look at packaging" "look at dinner packaging"
            "look at frozen dinner packaging"
            '(() (prinll "You find a variety of frozen dinner brands and flavors.")) )
         (room-command "look at refrigerator"
            '(() (prinll "The refrigerator is plain and unadorned.")) )
         (room-command '("look" "at" "microwave")
            '(() (prog
                    (prinll "You see a label:")
                    (prinll "WARNING: POWERED BY FUSION. TEST CONTAINMENT COILS REGULARLY. COOKING \
                             TEMPERATURES MAY NOT MATCH CONVENTIONAL MICROWAVES." ) ) ) )
         (room-command '("search" "refrigerator")
            '(() (prog
                    (prinll "You found some eggs.")
                    (if (not (read-world-var "egg-counter"))
                       (set-world-var "egg-counter" 1)
                       (set-world-var "egg-counter" (inc (read-world-var "egg-counter"))) )
                    (unless (inventory-contains? "egg")
                       (add-to-inventory "egg"
                          (= (read-world-var "egg-counter") 4) )
                       (prinll "An egg has been added to your inventory.") ) ) ) )
         (room-command '("put" "egg" "in" "microwave")
            '(() (if (not (inventory-contains? "egg"))
                    (prinll "You do not have an egg.")
                    (if (read-world-var "microwave-contents")
                       (prinll "The microwave already has something in it.")
                       (prog
                          (if (not (get-inventory-data "egg"))
                             (set-world-var "microwave-contents" "egg")
                             (set-world-var "microwave-contents" "the egg") )
                          (remove-from-inventory "egg")
                          (prinll "You put an egg in the microwave.") ) ) ) ) )
         (room-command '("break" "egg")
            '(() (if (not (inventory-contains? "egg"))
                    (prinll "You do not have an egg.")
                    (if (get-inventory-data "egg")
                       (prinll "Surprisingly, the egg is solid and does not break.")
                       (prog
                          (remove-from-inventory "egg")
                          (prinll "The egg cracks and leaves a mess on the floor.") ) ) ) ) )
         (room-command '("use" "microwave")
            '(() (if (not (read-world-var "microwave-contents"))
                    (prinll "There is nothing in the microwave to cook.")
                    (if (= (read-world-var "microwave-contents") "egg")
                       (prog 
                          (set-world-var "microwave-contents" NIL)
                          (case (rand 1 4)
                             (1 (prinll "The egg explodes, covering the inside of the microwave with pieces of \
                                         shell and yolk."))
                             (2 (prinll "The egg bursts into a ball of plasma, then vaporizes into nothing."))
                             (3 (prinll "The egg implodes dramatically, bending space and time. For a brief \
                                         moment, you see a glimpse into another part of the universe, before the \
                                         tunnel collapses in on itself."))
                             (4 (prinll "The atomic structure of the egg breaks down and becomes a vortex of \
                                         spinning particles. For several seconds, the particles arrange \
                                         themselves into a beautiful crystalline structure, then dissipate, \
                                         settling and leaving a thin film on the microwave plate." ) ) ) )
                       # "the egg"
                       (prog
                          (prinll "The bulk of the egg-like object in the microwave melts away, leaving \
                                   behind a mysterious, glittering crystal.")
                          (add-to-inventory "crystal")
                          (set-world-var "microwave-contents" NIL)
                          (prinll "A crystal has been added to your inventory.") ) ) ) ) )
         (room-exit "west" '(() (enter-room "living-room"))) )
      (room "programming-room"
         (room-long-description "In the center of this room is a rotating chair, nested in an assortment \
                                 of keyboards and control panels, and surrounded above by a half dozen \
                                 huge flatscreen computer monitors.")
         (room-command '("sit" "in" "chair")
            '(() (prinll "You feel very cool in the middle of a setup like this.")) )
         (room-command
            "use keyboard"
            "use keyboards"
            "use control panel"
            "use control panels"
            "use monitor"
            "use monitors"
            "touch keyboard"
            "touch keyboards"
            "touch control panel"
            "touch control panels"
            "touch monitor"
            "touch monitors"
            "type on keyboard"
            "type on keyboards"
            '(() (prinll "You think it best not to touch anything, as you are not sure what you \
                          are doing." ) ) )
         (room-command "look at monitor" "look at monitors"
            '(() (prinll "The monitors are filled with mysterious computer codes embedded in \
                          nested parentheses.")))
         (room-command "look at chair" "look at rotating chair"
            '(() (prinll "This snazzy chair provides great lumbar support and rotates easily in \
                          any direction. Includes a drink holder!" )))
         (room-command "look at keyboard" "look at keyboards"
            '(() (prinll "These keyboards are similar to the usual ones, but the 'CTRL' character \
                          is in the wrong place, and there are a bunch of weird keys added, like \
                          the 'META' key.")))
         (room-command "look at control panel" "look at control panels"
            "look at panel" "look at panels"
            '(() (prinll "You see an assortment of statistical and graphical displays.")) )
         (room-exit "up" '(() (enter-room "living-room")))
         (room-exit "east" '(() (enter-room "laser-room")))
         (room-exit "west" '(() (enter-room "water-entry")))
         (room-exit "south" '(() (enter-room "weapons-room")))
         (room-exit "north" '(() (enter-room "transporter-room"))) )
      (room "weapons-room"
         (room-long-description "This area is sparsely furnished, with only a chair and a small table. \
                                 Nevertheless, someone has decorated the walls with a rich assortment \
                                 of ancient medieval weaponry.")
         (run
            (let Fgen
               '((@weapon)
                 (macro '(room-command '("look" "at" @weapon) '(() (desc-weapon @weapon)))) )
               (mapcar 'Fgen '("mace" "longsword" "pollaxe" "longbow" "crossbow")) ) )
         (run
            (let Fgen
               '((@weapon)
                 (macro '(room-command '("take" @weapon) '(() (prinll "It is firmly attached to the wall.")))) )
               (mapcar 'Fgen '("longsword" "pollaxe" "longbow" "crossbow")) ) )
         (room-command "take mace" "pull mace"
            '(() (prinll "The mace will not come off the wall, but you feel it shift a little in \
                          place.")) )
         (let turn-mace
            '(() (prinll "The mace rotates several degrees. Immediately, a panel in the wall opens \
                          just long enough for you to jump in.")
              (enter-room "lounge") )
            (room-command '("turn" "mace") turn-mace)
            (room-command '("rotate" "mace") turn-mace) )
         (let desc-table '(() (prinll "There is a book on the table."))
            (room-command '("look" "at" "table") desc-table)
            (room-command '("search" "table") desc-table) )
         (room-command "look at chair"
            '(() (prinll "A simple and unadorned chair.")) )
         (room-command '("look" "at" "book")
            '(() (prinll "It is short book about medieval weaponry.")) )
         (room-command '("read" "book")
            '(()
              (set-world-var "studied-weapons" T)
              (prinll "It is a very interesting book about weaponry used in the Middle Ages. \
                       After perusing the material for a few minutes, you feel more \
                       knowledgeable on the subject.") ) )
         (let desc-weapons '(() (prinll "Some of the weapons mounted include: a longsword, a mace, a pollaxe, \
                                         a longbow, and a crossbow.") )
            (room-command '("look" "at" "wall") desc-weapons)
            (room-command '("look" "at" "walls") desc-weapons)
            (room-command '("look" "at" "weaponry") desc-weapons)
            (room-command '("look" "at" "weapons") desc-weapons) )
         (room-exit "north" '(() (enter-room "programming-room")))
         (room-exit "south" '(() (enter-room "mirror-hall-1"))) )
      (room "lounge"
         (room-long-description "You are in a private lounge. There are several comfortable couches and \
                                 chairs, books scattered about, and a drink machine of some kind.")
         (let desc-drink-machine '(() (prinll "The machine dispenses fruit juice!"))
            (room-command '("look" "at" "drink" "machine") desc-drink-machine)
            (room-command '("look" "at" "machine") desc-drink-machine) )
         (room-command
            "look at chair" "look at chairs"
            "look at couch" "look at couches"
            '(() (prinll "Obviously meant for relaxation.")) )
         (room-command "sit in chair" "sit in couch"
            '(() (prinll "Very comfortable. You could spend a lot of time here.")) )
         (room-command '("use" "drink" "machine") use-drink-machine)
         (room-command '("use" "machine") use-drink-machine)
         (room-command '("drink" "juice") use-drink-machine)
         (room-command '("drink" "fruit" "juice") use-drink-machine)
         (room-command '("look" "at" "books")
            '(() (prinll "There is quite an assortment of books scattered about. Most are on \
                          technical subjects.")))
         (room-command '("look" "at" "book") '(() (prinll "Which book?")))
         (room-command '("look" "at" "physics" "book")
            '(() (prinll "It is entitled \"Discussions in Classical and Quantum Physics\"")) )
         (room-command '("search" "books")
            '(() (prinll "You find an interesting title. It appears to be a physics book.")) )
         (room-command '("read" "book") '(() (prinll "Which book?")))
         (room-command '("read" "physics" "book") read-physics-book)
         (room-command '("read" "discussions in classical and quantum physics") read-physics-book)
         (room-command '("read" "discussions" "in" "classical" "and" "quantum" "physics") read-physics-book)
         (room-exit "out" '(() (enter-room "weapons-room"))) )
      (let (look-at-mirror 
            '(() (prinll "It is a tall, skinny mirror, about right for your size."))
            break-mirror
            '(() (prinll "Why does everybody always want to break the mirror!?")) )
         (room "mirror-hall-1"
            (room-long-description "This room has nothing in it except a mirror on one wall. There is an exit \
                                 to the north.")
            (room-exit "north" '(() (enter-room "weapons-room")))
            (room-command "look at mirror" look-at-mirror)
            (room-command "break mirror" break-mirror)
            (room-command "look in mirror" "look through mirror"
               '(() (prinll "A handsome fellow looks back at you.")
                 (enter-room-quietly "mirror-hall-2") ) ) )
         (room "mirror-hall-2"
            (room-long-description "This room has nothing in it except a mirror on one wall. There is an exit \
                                 to the south.")
            (room-exit "south" '(() (enter-room "mirror-hall-3")))
            (room-command "look at mirror" look-at-mirror)
            (room-command "break mirror" break-mirror)
            (room-command "look in mirror" "look through mirror"
               '(() (prinll "A handsome fellow looks back at you.")
                 (enter-room-quietly "mirror-hall-1") ) ) )
         (room "mirror-hall-3"
            (room-long-description "This room has nothing in it except a mirror on one wall. There is an exit \
                                 to the north.")
            (room-exit "north" '(() (enter-room "mirror-hall-2")))
            (room-command "look at mirror" look-at-mirror)
            (room-command "break mirror" break-mirror)
            (room-command "look in mirror"
               '(() (prinll "A handsome fellow looks back at you.")
                 (enter-room-quietly "mirror-hall-4") ) ) )
         (room "mirror-hall-4"
            (room-long-description "This room has nothing in it except a mirror on one wall. There is an exit \
                                 to the south.")
            (room-exit "south" '(() (enter-room "stairs-1")))
            (room-command "look at mirror" look-at-mirror)
            (room-command "break mirror" break-mirror)
            (room-command "look in mirror"
               '(() (prinll "A handsome fellow looks back at you.")
                 (enter-room-quietly "mirror-hall-3") ) ) ) )
      (room "stairs-1"
         (room-long-description "This room is empty, but there are several exits, including some stairs \
                                 leading downwards.")
         (room-exit "east" '(() (enter-room "hologram-generator")))
         (room-exit "west" '(() (enter-room "aviary")))
         (room-exit "down" '(() (enter-room "stairs-2")))
         (room-exit "north" '(() (enter-room "mirror-hall-4"))) )
      (room "stairs-2"
         (room-long-description "This room is empty, but there is an exit to the east.")
         (room-exit "up" '(() (enter-room "stairs-1")))
         (room-exit "east" '(() (enter-room "track"))) )
      (room "track"
         (room-long-description
            '(() (prinll "You find yourself in an enormous underground cavern, which is somewhat \
                          narrow in width, but seems to stretch for at least a mile to the east. \
                          The cavern appears to be mostly natural, but there are a few places \
                          where rock appears to have been blasted away. The ground has been \
                          leveled and paved into a smooth road, and you see shafts for \
                          ventilation, and many bright lamps for lighting.")
              (unless (read-world-var "track-car-destroyed")
                     (prinll "At the beginning of the track you see a large vehicle. There is also a \
                              large fuel tank, as well as racks of tires, and other mechanical \
                              equipment."))))
         (room-command "look at air intakes" "look at intakes"
            "look at air intake" "look at intakes"
            '(() (if (read-world-var "track-car-destroyed")
                    (prinll "The car is no longer here.")
                    (prinll "The engine must require a great deal of oxygen.") ) ) )
         (room-command "look at road"
            '(() (prinll "This looks like high quality work. But how was it constructed?")) )
         (room-command "look at shafts" "look at shafts for ventilation"
            '(() (prinll "You see shafts in the roof and wall, presumably leading to the surface \
                          and a source of fresh air.")) )
         (room-command "look at lamps" "look at light"
            '(() (prinll "They must use a great deal of electricity.")) )
         (room-command "look at tires" "look at rack" "look at rack of tires"
            '(() (prinll "These are high-grade racing tires.")) )
         (room-command "look at equipment" "look at mechanical equipment"
            '(() (prinll "You find air compressers, welding apparatus, jacks, and an \
                          assortment of other common tools and automobile parts.")))
         (room-command "put tires on car" "put new tires on car" "take tires" "take tire"
            '(() (prinll "The current tires seem to be in good shape.")) )
         (room-command "look at hose"
            '(() (prinll "It looks like the hose you normally see at a filling station.")))
         (room-command "look at fuel tank" "look at tank"
            '(() (prinll "It is large and has a hose attached. You also see a large switch.")) )
         (room-command "look at switch" "look at large switch"
            '(() (prinll "You can move the switch left or right. The left side is labeled 'Jet A' \
                          while the right side is labeled 'Theobold 7'.")
              (let fuel (read-world-var "fuel-setting")
              (cond
                 ((not fuel) (prinll "The switch is in the middle."))
                 ((= fuel 'jet) (prinll "The switch is set to the left."))
                 ((= fuel 'theo) (prinll "The switch is set to the right.")) ) ) ) )
         (room-command "move switch" "turn switch" "push switch" "press switch"
            '(() (prinll "What direction?")) )
         (room-command
            "move switch left" "turn switch left"
            "push switch left" "press switch left"
            "move switch to the left" "turn switch to the left"
            "move switch to left" "turn switch to left"
            '(() (set-world-var "fuel-setting" 'jet)
              (prinll "You move the switch on the fuel tank to the left position.") ) )
         (room-command
            "move switch right" "turn switch right"
            "push switch right" "press switch right"
            "move switch to the right" "turn switch to the right"
            "move switch to right" "turn switch to right"
            '(() (set-world-var "fuel-setting" 'theo)
              (prinll "You move the switch on the fuel tank to the right position.") ) )
         (room-command
            "use fuel"
            "put fuel in vehicle"
            "put fuel in car"
            "use fuel pump"
            "use hose"
            "use hose on car"
            "use fuel hose"
            "use fuel hose on car"
            "use hose on vehicle"
            "use fueld hose on vehicle"
            "fuel car"
            "fuel up car"
            "fuel vehicle"
            "fuel up vehicle"
            "top off vehicle"
            "top off car"
            "refuel car"
            "refuel vehicle"
            "pump fuel"
            "pump fuel into car"
            "pump fuel into vehicle"
            '(() (if (read-world-var "track-car-destroyed")
                    (prinll "The car is no longer here.")
                    (let fuel (read-world-var "fuel-setting")
                       (cond
                       ((= fuel 'jet)
                        (set-world-var "car-fuel" 'jet)
                        (prinll "You pump some fuel into the car.") )
                       ((= fuel 'theo)
                        (set-world-var "car-fuel" 'theo)
                        (prinll "You pump some fuel into the car.") )
                       (T (prinll "Nothing comes out of the hose when you squeeze the trigger.")) ) ) ) ) )
         (room-command "look at turbine" "look at turbines" "look at turbine components"
            '(() (if (read-world-var "track-car-destroyed")
                    (prinll "The car is no longer here.")
                    (prinll "The turbine is the third stage of a gas turbine engine. The rest of the \
                             engine must be hidden in the vehicle." ) ) ))
         (room-exit "in" '(() (prinll "You climb in the vehicle.")
                           (enter-room "car") ) )
         (room-exit "west" '(() (enter-room "stairs-2")))
         (room-command "look at car" "look at vehicle"
            '(() (if (read-world-var "track-car-destroyed")
                    (prinll "The car is no longer here.")
                    (prinll "This sleek machine is yellow and has a similar shape and form to a lamborghini, \
                             except it is larger. Also, you see air intakes and turbine components at \
                             the sides and rear of the vehicle." ) ) ) ) )
      (room "hidden-track-room"
         (room-long-description "You are in a vehicle bay filled with machinery. What you thought earlier \
                                 to be a wall, was actually a sliding door of some kind, though now it is \
                                 torn apart and wrecked. You see the remains of the car, salvageable \
                                 perhaps, but clearly not in driving condition. An assortment of other \
                                 machinery fills up the space in the bay.")
         (room-command "look at bay" "search bay"
            '(() (prinll "Aside from a lot of large and small machinery, you find a large lift or \
                          elevator system, evidently designed to bring materials and equipment in \
                          from the surface.")))
         (room-command "look at lift"
            '(() (prinll "It is an impressive design.")) )
         (room-command "use lift"
            '(() (prinll "You do not wish to escape to the surface at this time.")) )
         (room-command "look at car" "look at vehicle"
            '(() (prinll "The bulk of the car seems to be intact, but there is some nasty damage \
                          along the front end. One of the front wheels is bent out at an awkward \
                          angle.")))
         (room-command "look at door" "look at sliding door"
            '(() (prinll "It is a torn and twisted mess of metal.")) )
         (room-exit "west"
            '(() (prinll "You take a long walk across the track.")
              (enter-room "track") ) )
         (room-command "look at machinery" "look at equipment"
            '(() (prinll "You see various forms of excavating and construction equipment, and some \
                          other fancy equipment you do not recognize.")))
         (room-command "use machinery" "drive machinery" "use equipment" "drive equipment"
            '(() (prinll "You would rather not, as you don't have any clear purpose in mind.")) )
         (room-command "search machinery" "search equipment"
            '(() (if (inventory-contains? "cable")
                    (prinll "The machinery is interesting, but you find nothing else you can take \
                             with you.")
                    (prog
                       (prinll "You find a power cable.")
                       (add-to-inventory "cable")
                       (prinll "A cable has been added to your inventory.") ) ) ) ) )
      (room "car"
         (room-long-description "The inside of the vehicle is comfortable, but some missing interior \
                                 panels reveal a lot of wiring and circuitry.")
         (room-command "look at wiring" "look at circuitry"
            '(() (prinll "It looks as though someone was in the process of installing some \
                          additional sensor and control consoles.")))
         (room-exit "out" '(() (enter-room "track")))
         (room-command "drive" "drive car" "drive vehicle" "start engine" "start vehicle" "start car"
            drive-car ) )
      (room "aviary"
         (room-long-description "You are inside a large chamber. Cylinder shaped walls, covered in a \
                                 steel lattice, stretch upwards toward a high ceiling. This space appears \
                                 to be part workshop and part aviary. On one side is a workbench with \
                                 tools, shelves, and drawers. On the other side is an elaborate bird \
                                 cage.")
         (room-command
            "put gear in bird" "put gear in robot"
            "put gear in housing" "put gear in empty spot"
            '(() (if (not (inventory-contains? "gear"))
                    (prinll "You do not have a gear")
                    (prog
                       (set-world-var "bird-gear-installed" T)
                       (remove-from-inventory "gear")
                       (prinll "With a few adjustments, you are able to install the gear in the housing \
                                of the robot." ) ) ) ) )
         (room-command
            "attach wing" "attach wing to robot" "attach wing to bird"
            "put wing on bird" "put wing on robot" "fix bird" "fix robot"
            "reattach wing"
            '(() (if (read-world-var "bird-complete")
                    (prinll "You did that earlier!")
                    (if (not (read-world-var "bird-gear-installed"))
                       (prinll "It looks as though there is something else that needs to be installed \
                                first.")
                       (if (not (inventory-contains? "screws"))
                          (prinll "It is missing some screws of a particular size.")
                          (prog
                             (remove-from-inventory "screws")
                             (set-world-var "bird-complete" T)
                             (prinll "Suddenly, the robot comes to life and springs out of your hands. It \
                                      whirls up through the air for a while, moving in larger and larger \
                                      circles, and finally perches on a small ledge near the ceiling." ) ) ) ) ) ) )
         (room-command "search drawers" "search drawers for screws"
            '(() (if (or (read-world-var "bird-complete")
                        (not (read-world-var "bird-gear-installed")) )
                    (prinll "You find many small parts, but you are not sure what it is you are \
                             looking for.")
                    (prog
                       (prinll "You find some screws of the correct size.")
                       (add-to-inventory "screws") ) ) ) )
         (room-command "search tools" "search shelves"
            '(() (prinll "You find a number of tools and parts, but nothing of interest.")) )
         (room-command "look at tools" "look at shelves"
            '(() (prinll "An assortment of tools and parts.")) )
         (room-command "look at drawers"
            '(() (prinll "The drawers are filled with many small parts, both electronic and mechanical.")) )
         (room-command "look at workbench"
            '(() (if (read-world-var "bird-complete")
                    (prinll "The workbench is empty except for a few scattered tools.")
                    (prinll "On the workbench you find the hardware for a bird-like robot.") ) ) )
         (room-command "look at robot bird" "look at robot" "look at hardware"
            '(() (if (read-world-var "bird-complete")
                    (prinll "The robot bird is perched on a ledge near the ceiling.")
                    (if (read-world-var "bird-gear-installed")
                       (prinll "The robot appears to be mostly complete except for a missing wing, \
                                which you find detached on the workbench.")
                       (prinll "The robot appears to be mostly complete except for a missing wing, which \
                                you find detached on the workbench. There is an empty spot in the \
                                robot's housing.")))))
         (room-command "look at bird cage" "look at cage"
            '(() (prinll "A sophisticated bird cage houses two pidgeons. Various mechanical devices \
                          provide food and liquids, while removing waste automatically.")))
         (room-command "look at bird" "look at birds"
            '(() (prinll "They look to be a common variety of pidgeon — mostly gray in color. They \
                         do not seem to be frightened by you." ) ) )
         (room-command "take bird" "pet bird" "take birds" "pet birds" "open cage" "release birds"
            '(() (prinll "You open the cage and play with the birds a little, but then put them \
                          back in the cage. It seems like the safest place for them.")))
         (room-command "look at wall" "look at walls" "look at lattice" "look at steel lattice"
            '(() (prinll "The lattice along the wall is occasionally dotted by cameras and other \
                          sensory equipment, as well as some perches and stands. It looks as \
                          though you could climb the lattice, though it would not be very easy.")))
         (room-command "look at tools" "look at shelves" "look at drawers"
            '(() (prinll "You see common items you would expect to find in a workshop.")) )
         (room-exit "up"
            '(() (if (not (read-world-var "bird-complete"))
                    (prinll "Climbing up the lattice looks somewhat challenging, and you can't \
                             think of a good reason to go up there.")
                    (prog
                       (prinll "You carefully climb up the lattice toward the robot.")
                       (enter-room "bird-ledge") ) ) ) )
         (room-exit "east" '(() (enter-room "stairs-1"))) )
      (room "bird-ledge"
         (room-long-description
            "You are hanging onto the lattice near the ceiling, next to the ledge \
             where the bird robot is resting.")
         (room-command "touch bird" "pet bird" "take bird" "grab bird" "hit bird"
            "touch robot" "pet robot" "take robot" "grab robot" "hit robot"
            "touch robot bird" "pet robot bird" "take robot bird" "grab robot bird" "hit robot bird"
            '(() (prinll "As you try to touch the robot, it simply hops away from your hand.")) )
         (room-command "search ledge"
            '(() (if (inventory-contains? "card")
                    (prinll "You find nothing of interest.")
                    (prog
                       (prinll "You find a small card-shaped piece of plastic. It appears to have \
                                circuitry embedded in it.")
                       (add-to-inventory "card")
                       (prinll "A card has been added to your inventory.") ) ) ) )
         (room-command "look at bird" "look at robot" "look at robot bird"
            '(() (prinll "The robot is perched calmly on the ledge. It does not seem to have any \
                          intention of going anywhere, and does not seem to be interested in you. \
                          Its beady mechanical eyes reveal little.")))
         (room-command "look at ledge"
            '(() (if (inventory-contains? "card")
                    (prinll "The robot bird is perched on the ledge.")
                    (prinll "There seems to be something left on the ledge, near the robot.") ) ) )
         (room-exit "down"
            '(() (prinll "You climb carefully back down the lattice.")
              (enter-room "aviary") ) ) )
      (room "hologram-generator"
         (room-long-description
            '(() (if (read-world-var "hologram-active")
                    (prinll "You are in a large city park, with benches, and large trees, and \
                             footpaths. The light is dim as dusk approaches, and a gentle wind pushes \
                             autumn leaves across the grass. But you see no animals or people, except \
                             one young lady, who sits contemplatively on a bench only a few feet away. \
                             Yet, behind you, an exit is visible, and a small crystal sparkles on a \
                             pad.")
                    (prinll "The walls and ceiling of this room are covered in curious \
                             semi-transparent plates and small, black pyramids set at varying \
                             angles. Near the entrance, a small pad juts out of the wall, laying flat \
                             and round like a drink coaster. The pad glows orange."))))
         (room-command "look at plates" "look at pyramids"
            "look at plate" "look at pyramid"
            "look at semi-transparent plates"
            "look at black pyramids"
            '(() (prinll "They are quite interesting, but you really don't know what it is they \
                          do." ) ) )
         (room-command "look at pad" "look at small pad"
            '(() (if (read-world-var "hologram-active")
                    (prinll "A sparkling crystal sits on the pad.")
                    (prinll "The pad glows orange.") ) ) )
         (room-command "look at trees" "look at tree"
            "look at large trees" "look at large tree"
            '(() (prinll "The trees sway a little with the wind, but you see no squirrels or \
                          birds.")) )
         (room-command "look at footpath" "look at footpaths"
            '(() (prinll "Dirt paths wind there way around trees and bushes and benches")) )
         (room-command "look at bench" "look at benches"
            '(() (prinll "A young woman is sitting on one of the benches.")) )
         (let (put-crystal
               '(() (if (not (inventory-contains? "crystal"))
                       (prinll "You do not have a crystal.")
                       (prog
                          (remove-from-inventory "crystal")
                          (set-world-var "hologram-active" T)
                          (prinll "The pad flickers, and the crystal begins to sparkle energetically. \
                                   Streams of light and color pulsate throughout the room, until they \
                                   coalesce and the room vanishes, and you find yourself in the middle \
                                   of a city park." ) ) ) ) )
            (room-command "use crystal on pad" put-crystal)
            (room-command "put crystal on pad" put-crystal)
            (room-command "place crystal on pad" put-crystal) )
         (let (look-at-girl
               '(() (prinll "She is young — 19 or 20 years old, you would guess — and small in \
                             figure. She is beautiful with dark brown hair. Her eyes are hazel with \
                             flecks of green. You notice she is holding a small book.")))
            (room-command "look at girl" look-at-girl)
            (room-command "look at woman" look-at-girl)
            (room-command "look at lady" look-at-girl) )
         (room-command "look at book" "look at small book" "look at journal"
            '(() (prinll "It looks like a journal. Her fingers partially cover a name inscribed in \
                          gold lettering. The first word is 'Elaine'.")))
         (room-command "read book" "read small book" "read journal"
            "take book" "take small book" "take journal"
            '(() (prinll "The image of the book flickers a little as your fingers pass through it.")) )
         (let (talk-to-girl
               '(() (if (read-world-var "talked-to-girl")
                       (prinll "You try to ask the girl some questions. She looks at you but does not \
                                seem to be listening.")
                       (prog
                          (prinll "You begin to speak, but she interrupts you. \"How long will you cling to \
                                   this memory, John?\"")
                          (set-world-var "talked-to-girl" T) ) ) ) )
            (room-command
               "talk to girl" "talk to lady" "talk to woman"
               "speak to girl" "speak to lady" "speak to woman"
               talk-to-girl ) )
         (room-command "look at eyes" "look at her eyes"
            '(() (prinll "They are beautiful.")) )
         (room-command "look at hair" "look at her hair"
            '(() (prinll "Her dark hair is one of her more distracting features.")) )
         (room-command "look at leaf" "look at leaves"
            '(() (prinll "A leaf occasionally stirs, or tumbles a few inches across the ground. \
                          However, you do not feel any wind.")) )
         (let (touch-girl
               '(() (prinll "You attempt to touch her, but your hand moves through her, as through a \
                             ghost.")))
            (room-command "touch girl" touch-girl)
            (room-command "kiss girl" touch-girl)
            (room-command "touch woman" touch-girl)
            (room-command "kiss woman" touch-girl)
            (room-command "touch lady" touch-girl)
            (room-command "kiss lady" touch-girl)
            )
         (room-command "look at park"
            '(() (prinll "There is something sad about this place. It is nothing obvious, only a feeling.")) )
         (room-exit "west" '(() (enter-room "stairs-1"))) )
      (room "transporter-room"
         (room-long-description "The bulk of this room is taken up by a large, upright circle, crafted \
                                 of an incredible array of mechanical and electrical components. In \
                                 front of this apparatus is a sleek, desk-shaped control panel.")
         (let desc-panel '(() (prinll "A set of nobs and keys allows you to enter several number parameters \
                                       into the machine.") )
            (room-command '("look" "at" "panel") desc-panel)
            (room-command '("look" "at" "control" "panel") desc-panel) )
         (room-command '("look" "at" "circle") '(() (desc-transporter)))
         (room-command '("look" "at" "apparatus") '(() (desc-transporter)))
         (room-command '("look" "at" "chamber") '(() (desc-transporter-energy-chamber)))
         (room-command '("put" "glowing" "rock" "in" "chamber") '(() (put-rock-in-chamber)))
         (room-command '("put" "rock" "in" "chamber") '(() (put-rock-in-chamber)))
         (room-command '("put" "glowing" "rock" "into" "chamber") '(() (put-rock-in-chamber)))
         (room-command '("put" "rock" "into" "chamber") '(() (put-rock-in-chamber)))
         (room-command '("use" "equations") '(() (load-transporter-parameters)))
         (room-command '("use" "equations" "on" "panel") '(() (load-transporter-parameters)))
         (room-command '("use" "equations" "on" "control" "panel") '(() (load-transporter-parameters)))
         (room-exit "in"
            '(() (if (nand
                        (read-world-var "transporter-energized")
                        (read-world-var "transporter-destination") )
                    (prinll "You step through the circle, but nothing happens.")
                    (prog
                       (prinll "As you step through the portal, your sight becomes dim and the air \
                                becomes stale and thin. You feel some blood rush to your head, and a \
                                lightness in your body, and become disoriented. You trip and flail \
                                around for a while before regaining your composure and taking in your \
                                surroundings.")
                       (enter-room "moon-living-room") ) ) ) )
         (room-exit "south" '(() (enter-room "programming-room"))) )
      (room "moon-living-room"
         (room-long-description "You see you are in a small, stuffy room. The lighting is dim and \
                                 flickers continually. There is very little free space as there are \
                                 several austere bunks attached to the walls, and various other items \
                                 scattered about. You notice curious symbols on the wall and ceiling \
                                 panels. A ladder ascends into a chamber above. One wall has a hatch. A \
                                 disc-shaped portal hovers in the air on one side of the room, shimmering \
                                 occasionally.")
         (room-command "look at bunks" "look at austere bunks"
            '(() (prinll "Sleeping beds, apparently.")) )
         (room-command "look at floor"
            '(() (prinll "It is very cluttered.")) )
         (room-command "look at bin"
            '(() (prinll "It contains an assortment of mechanical parts.")) )
         (room-command "look at hatch"
            '(() (prinll "It just looks like an access hatch.")) )
         (room-command
            "look at portal" "look at disc-shaped portal"
            "look through portal" "look through disc-shaped portal"
            '(() (prinll "Through it, you can see a brightly lit control panel, with a throbbing \
                          green reflection.")))
         (room-command "look at ladder"
            '(() (prinll "It looks like a rather thin ladder, but you are feeling rather light.")) )
         (room-command '("search" "floor")
            '(() (prinll "You find some exercise machines, pieces of clothing, and packages of \
                          freeze-dried food. There is also a bin containing mechanical parts.")))
         (room-command "look at light" "look at lighting"
            '(() (prinll "The lighting flickers as though on a weak source of power.")) )
         (room-command "look at wall" "look at ceiling" "look at symbols"
            '(() (prinll "Labeling on the panels includes Soviet symbols and Russian characters.")) )
         (let search-bin
            '(() (prinll "Most of the parts are common and of little interest... bolts, latches, \
                          wire, and such like. One item does catch your attention: a curious \
                          composite gear assembly. As you play with it, it seems to rotate easily \
                          in any direction, without feeling rough or weak.")
              (add-to-inventory "gear")
              (prinll "A gear has been added to your inventory.") )
            (room-command '("search" "bin") search-bin)
            (room-command '("search" "parts") search-bin)
            (room-command '("search" "mechanical" "parts") search-bin)
            (room-command '("search" "bin" "containing" "mechanical" "parts") search-bin) )
         (room-exit "up" '(() (prog
                                 (prinll "You climb the ladder quite easily, almost floating up.")
                                 (enter-room "moon-viewing-bubble") ) ) )
         (room-exit "south" '(() (prinll "You attempt to open the hatch. The handle turns a little, but the hatch \
                                          will not budge. Whether locked or otherwise blocked, you cannot tell.")))
         (room-exit "out" '(() (enter-room "transporter-room"))) )
      (room "moon-viewing-bubble"
         (room-long-description "This small chamber is encased in a thick glass or glass-like material, \
                                 allowing for a breathtaking view of the surrounding landscape. A ladder \
                                 leads back down to the lower level.")
         (room-exit "down" '(() (enter-room "moon-living-room")))
         (room-command "look at glass"
            '(() (prinll "The glass, if that is what it is, seems very strong.")) )
         (room-command "look at earth" "look at globe"
            '(() (prinll "It is the home of mankind.")) )
         (room-command "look at stars" "look at sky"
            '(() (prinll "Without an atmosphere, the view is breathtaking. The heavenly bodies \
                          seem innumerable.")) )
         (room-command "look at construction materials" "look at materials"
            '(() (prinl "There are a few crates and metal beams and other industrial components.")) )
         (room-command "look at vehicle" "look at small vehicle"
            '(() (prinl "You see a small buggy, and an unattached trailer. There are wheel tracks \
                         stretching and curving in many directions.")))
         (let look-at-landscape
            '(() (prog
                    (prinll "For a great distance, you can see a gray, sandy landscape. Its surface \
                             is pocked with small holes and larger craters, as well as the occasional \
                             rolling hill. The sky above gives a crisp view of an awesome starry sky \
                             which includes, to your amazement, a globe that you recognize as the planet \
                             earth.")
                    (prinll "Nearby, you see some equipment, such as a small vehicle and some \
                             construction materials. A section of dirt bears scorch marks like that \
                             of a landing or takeoff pad. Nevertheless, all is still and lifeless.")))
            (room-command '("look" "at" "landscape") look-at-landscape)
            (room-command '("look" "at" "view") look-at-landscape)
            (room-command '("look" "at" "surrounding" "landscape") look-at-landscape)
            (room-command '("look" "through" "glass") look-at-landscape)
            (room-command '("look" "out" "glass") look-at-landscape)
            (room-command '("view" "landscape") look-at-landscape)
            (room-command '("look" "outside") look-at-landscape) ) )
      (room "laser-room"
         (room-long-description
            '(() (if (read-world-var "sphere-broken")
                    (prinll "The bulk of the room is filled with a metal gun-shaped device, covered \
                             in complex wiring and electronics. The gun is pointed at the remnants \
                             of a transparent glass sphere.")
                    (prinll "The bulk of the room is filled with a metal gun-shaped device, covered \
                             in complex wiring and electronics. The gun is pointed at a transparent \
                             glass sphere."))))
         (let describe-laser '(() (prinll "The device is covered is with several placards relating the dangers of \
                                           high-powered laser energy. The activation mechanism seems fairly simply \
                                           to operate."))
            (room-command '("look" "at" "gun") describe-laser)
            (room-command '("look" "at" "laser") describe-laser) )
         (room-command "look at dial"
            '(() (prinll "The dial has settings between 0 and 100.")))
         (let describe-sphere
            '(()
              (let (SF (read-world-var "sphere-frequency")
                    D "It is a glass sphere partially filled with some kind of strange \
                       metal dust. Attached to the outside of the sphere are several electronic \
                       devices connected to cabling and wire coils. You see a dial underneath.")
                 (cond
                    ((read-world-var "sphere-broken")
                     (prinll "Bits of the sphere and associated electronics are strewn about the floor.") )
                    ((or (not SF) (= SF 0))
                     (prinll D)
                     (prinll "The dust rests motionless on the bottom of the sphere.") )
                    ((or (< SF 40) (>= SF 90))
                     (prinll D)
                     (prinll "The dust in the sphere is vibrating.") )
                    ((or (< SF 60) (>= SF 80))
                     (prinll D)
                     (prinll "The dust in the sphere is flying about chaotically, filling the entire \
                              sphere."))
                    ((or (<> SF 72))
                     (prinll D)
                     (prinll "The dust swirls about fluidly, and has contracted to its own smaller \
                              spherical shape in the center of the glass.") )
                    (T (prinll D)
                       (prinll "The dust has concentrated itself into a single point which hovers in the \
                                center of the glass." ) ) ) ))
            (room-command '("use" "dial") set-sphere-dial)
            (room-command '("set" "dial") set-sphere-dial)
            (room-command '("use" "gun") use-laser)
            (room-command '("use" "laser") use-laser)
            (room-command '("look" "at" "sphere") describe-sphere)
            (room-command '("look" "at" "glass" "sphere") describe-sphere)
            (room-command '("look" "at" "glass") describe-sphere) )
         (room-command "look at wall"
            '(() (prinll "You see some scorch marks.")) )
         (room-exit "west" '(() (enter-room "programming-room"))) )
      (room "water-entry"
         (room-long-description "You are in a cavern with a tile floor and a small but deep pool of water \
                                 at one end. Metal pipes cover the walls. A minature submarine, the right \
                                 size for one person, is docked near the edge of the water.")
         (room-command "look at pipes" "look at metal pipes"
            '(() (prinll "They look like water pipes. Perhaps for controlling the water level \
                          somehow?")))
         (room-command '("look" "at" "submarine")
            '(() (prinll "On the frame of the submarine, you see a lever, which can be either \
                          pushed or pulled, and a handle which rotates.") ))
         (room-command '("push" "lever")
            '(()
              (if (read-world-var "submarine-unlocked")
                 (prinll "You already understand how to use the locking mechanism.")
                 (prog
                    (prinll "You push the lever forward.")
                    (process-submarine-lock-mechanism "push") ) ) ) )
         (room-command '("pull" "lever")
            '(()
              (if (read-world-var "submarine-unlocked")
                 (prinll "You already understand how to use the locking mechanism.")
                 (prog
                    (prinll "You pull the lever back.")
                    (process-submarine-lock-mechanism "pull") ) ) ) )
         (let turn-cmd
            '(()
              (if (read-world-var "submarine-unlocked")
                 (prinll "You already understand how to use the locking mechanism.")
                 (prog
                    (prinll "You rotate the handle.")
                    (process-submarine-lock-mechanism "rotate") ) ) )
            (room-command '("rotate" "handle") turn-cmd)
            (room-command '("turn" "handle") turn-cmd) )
         (room-exit "east" '(() (enter-room "programming-room")))
         (room-exit "down"
            '(() (if (not (read-world-var "submarine-unlocked"))
                    (prinll "You try to enter the submarine but cannot find an opening.")
                    (prog
                       (prinll "You enter the submarine. After figuring out the engine and directional \
                                controls, you dive the vessel down into the pool.")
                       (enter-room "water-passage-1") ) ) ) ) )
      (let prefix-sub-description
         '((Str)
           (pack ""
              (list "You are inside the minature submarine. Lights on the vessel brighten the \
                     surrounding water. " Str ) ) )
         (room "water-passage-1"
            (room-long-description
               (prefix-sub-description
                  "Outside the viewports you see you are in a water filled cavern that continues \
                   upward and downward." ) )
            (room-exit "up" '(() (enter-room "water-entry")))
            (room-exit "down" '(() (enter-room "water-passage-2"))) )
         (room "water-passage-2"
            (room-long-description
               (prefix-sub-description
                  "Outside the viewports you see you are in a water filled cavern that continues \
                   upwards and downwards." ) )
            (room-exit "up" '(() (enter-room "water-passage-1")))
            (room-exit "down"
               '(() (unless (read-world-var "sub-unstuck")
                       (prinll "You feel a jarring thud as the submarine collides with the bottom of the \
                                tunnel and embeddes itself deeply into some gooey mud."))
                 (enter-room "mud-trap") ) ) )
         (room "mud-trap"
            (room-long-description
               (prefix-sub-description
                  "The cavern bottoms out here into a pit of thick mud. Through the \
                   viewports, you can see an opening above, as well off toward the west \
                   and toward the east." ) )
            (room-command "look at mud" "look at gooey mud"
               '(() (if (read-world-var "sub-unstuck")
                 (prinll "It looks grayish-brown in the dim light.")
                 (prinll "You are definitely in the mud, but not very deep. If you could rock \
                             the submarine just right, you might be able to pull free." ) ) ))
            (room-exit "up"
               '(() (if (read-world-var "sub-unstuck")
                       (enter-room "water-passage-2")
                       (prog
                          (prinll "You attempt to raise the vessel, pulling it against the mud.")
                          (process-stuck-sub-move 'up) ) ) ) )
            (room-exit "west"
               '(() (if (read-world-var "sub-unstuck")
                       (enter-room "water-passage-3")
                       (prog
                          (prinll "You throttle forwards, nudging the submarine a little through the mud. \
                                   Fortunately, the propellers are above the top of the mud.")
                          (process-stuck-sub-move 'west) ) ) ) )
            (room-exit "east"
               '(() (if (read-world-var "sub-unstuck")
                       (enter-room "water-passage-4")
                       (prog
                          (prinll "You throttle backwards, nudging the submarine a little through the mud. \
                                   Fortunately, the propellers are above the top of the mud.")
                          (process-stuck-sub-move 'east) ) ) ) ) )
         (room "water-passage-3"
            (room-long-description
               (prefix-sub-description
                  "Outside the viewports you see you are in a water filled cavern that continues \
                   to the west and to the east." ))
            (room-exit "west" '(() (enter-room "water-passage-5")))
            (room-exit "east" '(() (enter-room "mud-trap"))) )
         (room "water-passage-4"
            (room-long-description
               (prefix-sub-description
                  "Outside the viewports you see you are in a water filled cavern which ends at \
                   a rock wall." ) )
            (room-command "look at wall" "look at rock wall"
               '(() (prinll "Quite solid.")))
            (room-exit "west" '(() (enter-room "mud-trap"))) )
         (room "water-passage-5"
            (room-long-description
               (prefix-sub-description
                  "Outside the viewports you see you are in a water filled cavern that continues \
                   to the north and to the east. The tunnel to the north looks narrow and low but \
                   should be large enough for the submarine." ))
            (room-command "look at tunnel"
               '(() (prinll "The submarine should be able to fit through.")))
            (room-exit "north" '(() (enter-room "ladder-cavern")))
            (room-exit "east" '(() (enter-room "water-passage-3"))) )
         (room "ladder-cavern"
            (room-long-description
               (prefix-sub-description
                  "Your submarine is in a bottle shaped cave mostly filled with water. The top \
                   narrows some more and you see a bubble of air, large enough that you can open \
                   the hatch into it. A sturdy ladder hangs down out of a hole in the ceiling." ))
            (room-command "look at ladder" "look at sturdy ladder"
               '(() (prinll "It looks strong enough to hold your weight.")) )
            (room-command "look at hole" "look at hold in the ceiling"
               '(() (prinll "It is certainly large enough for you to fit through.")) )
            (room-exit "south" '(() (enter-room "water-passage-5")))
            (room-exit "up" '(() (enter-room "garden-sanctuary"))) ) )
      (room "garden-sanctuary"
         (room-long-description
            "This small, but tall, domed room, contains a little garden. The floor is \
             soft dirt, very comfortable. The circle of the wall is lined with \
             flowers and other colorful plants. In the center grows a beautiful \
             pomegranate tree. Somehow, natural light shines through some shafts in \
             the ceiling. The air is very moist and warm, though it is not obvious to \
             you how this is being contrived. You feel as though you could rest here \
             for many hours. A sturdy ladder stretches down a hole in the floor.")
         (let FloorCommand
            '(() (if (inventory-contains? "equations")
                    (prinll "You find nothing of interest.")
                    (prog
                       (prinll "You find a sheet of paper filled with complex equations, which simplify \
                                to several numbers at the bottom of the page.")   
                       (add-to-inventory "equations")
                       (prinll "You have added a sheet of equations to your inventory.") ) ) )
            (room-command '("search" "floor") FloorCommand)
            (room-command '("search" "ground") FloorCommand)
            (room-command '("search" "dirt") FloorCommand) )
         (room-command '("search" "flowers")
            '(() (prinll "You find a tracked robot hidden behind some flowers.")) )
         (room-command '("look" "at" "robot")
            '(() (prinll "The robot appears to be in some kind of sleep mode.")) )
         (room-command '("use" "robot")
            '(() (prinll "You touch the robot and it suddenly awakes from its dormant state. It \
                          drives over to a well-hidden water spigot, fills up a watering can, \
                          drives back, and proceeds to pour water down your legs. After you are \
                          thoroughly soaked, it returns to sleep mode.") ) )
         (room-command "look at garden"
            '(() (prinll "It is colorful and fresh.")) )
         (room-command
            "look at shaft" "look at shafts"
            "look at light" "look at natural light"
            '(() (prinll "The light looks and feels like real sunlight.")) )
         (let TreeCommand
            '(() (prinll "The tree is filled with pomegranate fruit."))
            (room-command '("look" "at" "tree") TreeCommand)
            (room-command '("look" "at" "pomegranate" "tree") TreeCommand) )
         (let FruitCommand
            '(() (prinll "You enjoy a tasty, juicy fruit snack."))
            (room-command '("eat" "fruit") FruitCommand)
            (room-command '("eat" "pomegranate") FruitCommand) )
         (room-command "look at ladder" "look at sturdy ladder"
            '(() (prinll "It looks strong enough to hold your weight.")) )
         (room-command "look at hole" "look at hold in the ceiling"
            '(() (prinll "It is certainly large enough for you to fit through.")) )
         (room-command "rest" "sleep"
            '(() (prinll "You take a short nap, feeling refreshed.")) )
         (room-command "search garden"
            '(() (prinll "You look through through the flowers and plants but do not find anything \
                          else of interest.")))
         (room-command "look at ground" "look at floor"
            '(() (prinll "The dirt looks soft and fertile.")) )
         (room-exit "down" '(() (enter-room "ladder-cavern"))) )
 ) )
