#
# Muxer.pm - Multiplex I/O, non-blocking
#
# Copyright (C) 2008 Oskar Liljeblad
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#

package IO::Muxer;

use strict;
use POSIX qw(:errno_h BUFSIZ);

use constant MUX_RUNNING => 0;
use constant MUX_END => 1;
use constant MUX_FD_REGULAR => 0;
use constant MUX_FD_UNCONNECTED => 1;
use constant MUX_FD_LISTEN => 2;
use constant MUX_FD_CLOSE_WHEN_FLUSHED => 3;
use constant MUX_FD_CLOSING => 4;

# FD_SET($this, $fdset, $fh, $enabled)
# Change a file descriptor set.
#
sub FD_SET ($$$$) {
  my ($this, $fdset, $fh, $enabled) = @_;
  vec ($_[1], $this->{'fhs'}{$fh}{'fd'}, 1) = $enabled if exists $this->{'fhs'}{$fh};
}

# FD_ISSET($this, $fdset, $fh)
# Check if a handle is in the specified file descriptor set.
#
sub FD_ISSET ($$$) {
  my ($this, $fdset, $fh) = @_;
  return exists $this->{'fhs'}{$fh} ? vec($fdset, $this->{'fhs'}{$fh}{'fd'}, 1) : 0;
}

# IO::Muxer->new($callback)
# Create a new instance of IO::Muxer.
#
sub new ($$) {
  my ($class, $callback) = @_;
  my $this = {
    'state' => MUX_RUNNING,	# current state, MUX_*
    'fhs' => {},		# hash of file handles
    'readfds' => '',		# readable file descriptor set
    'writefds' => '',		# writable file descriptor set
    'timers' => [],		# timer list, sorted
    'cbobj' => $callback,	# callback object
  };
  bless($this, $class);
  return $this;
}

# _internal_error
# Prevent mux_error from being called recursively.
#
sub _internal_error ($$$) {
  my ($this, $fh, $syscall) = @_;
  if (!defined $fh) {
    $this->{'cbobj'}->mux_error($this, undef, $syscall);
  } elsif (!exists $this->{'fhs'}{$fh}{'error'}) {
    $this->{'fhs'}{$fh}{'error'} = 1;
    $this->{'cbobj'}->mux_error($this, $fh, $syscall);
    delete $this->{'fhs'}{$fh}{'error'} if exists $this->{'fhs'}{$fh};
  }
}

# _internal_add
# Add a file handle.
#
sub _internal_add ($$$$) {
  my ($this, $fh, $fdset, $state) = @_;
  $this->{'fhs'}{$fh} = {
    'inbuf' => '',		# input buffer
    'outbuf' => '',		# output buffer
    'state' => $state,		# current state, MUX_FD_*
    # 'timeout'			# time when timeout triggers
    # 'error'			# set when inside mux_error
    'fh' => $fh,		# reference to the file handle
    'fd' => fileno($fh),	# file descriptor
  };
  FD_SET($this, $this->{$fdset}, $fh, 1);
  $fh->autoflush(1);
  $this->_internal_error($fh, 'fcntl') if !defined $fh->blocking(0);
}

# $mux->add($fh)
# Add a file handle to the muxer. The handle will be made non-blocking and
# all I/O operations must be done through the muxer object.
#
sub add ($$) {
  my ($this, $fh) = @_;
  $this->_internal_add($fh, 'readfds', MUX_FD_REGULAR);
}

# $mux->add_unconnected($fh)
# Add an unconnected socket handle to the muxer. The handle will be
# made non-blocking and all I/O operations must be done through the muxer
# object. When connected, mux_connected will be called.
#
sub add_unconnected ($$) {
  my ($this, $fh) = @_;
  $this->_internal_add($fh, 'writefds', MUX_FD_UNCONNECTED);
}

# $mux->listen($fh)
# Add a listening socket handle to the muxer. The handle will be made
# non-blocking so that the accept system call will not block. All I/O
# operations must on the handle must be done through the muxer. When a
# connection is made to the socket, mux_connection will be called.
#
sub listen ($$) {
  my ($this, $fh) = @_;
  $this->_internal_add($fh, 'readfds', MUX_FD_LISTEN);
}

# $mux->set_timeout($fh, $time)
# Set a timeout on a file handle. When the specified time has elapsed,
# mux_timeout will be called.
# 
sub set_timeout ($$$) {
  my ($this, $fh, $time) = @_;
  @{$this->{'timers'}} = grep { $_->[0] != $fh } @{$this->{'timers'}};
  if (defined $time) {
    $time += time;
    @{$this->{'timers'}} = sort { $a->[1] <=> $b->[1] } (@{$this->{'timers'}}), [ $fh, $time ];
    $this->{'fhs'}{$fh}{'timeout'} = $time;
  } else {
    delete $this->{'fhs'}{$fh}{'timeout'};
  }
}

# $mux->loop()
# Start the loop that uses the select system call to wait for I/O events.
#
sub loop ($) {
  my ($this) = @_;

  while ($this->{'state'} == MUX_RUNNING && keys %{$this->{'fhs'}} != 0) {
    my $timeout = undef;
    if (@{$this->{'timers'}} > 0) {
      $timeout = $this->{'timers'}->[0]->[1] - time;
      if ($timeout <= 0) {
        my ($fh) = @{shift @{$this->{'timers'}}};
        delete $this->{'fhs'}{$fh}{'timeout'};
        $this->{'cbobj'}->mux_timeout($this, $fh);
        next;
      }
    }

    my $readfds = $this->{'readfds'};
    my $writefds = $this->{'writefds'};
    my $res = select($readfds, $writefds, undef, $timeout);
    if (!defined $res || $res < 0) {
      # XXX: I don't understand why select returns -1 and sets $! to 0,
      # but it happens with Perl 5.8.8 on Linux 2.6.23.12.
      $this->_internal_error(undef, 'select') if $! != EINTR && $! != 0;
      next;
    }
    next if $res == 0;

    foreach my $strfh (keys %{$this->{'fhs'}}) {
      next if !exists $this->{'fhs'}{$strfh};
      my $fh = $this->{'fhs'}{$strfh}{'fh'};

      if (FD_ISSET($this, $readfds, $fh)) {
        if ($this->{'fhs'}{$fh}{'state'} == MUX_FD_LISTEN) {
          my $newfh = $fh->accept();
          if (defined $newfh) {
            $this->{'cbobj'}->mux_connection($this, $fh, $newfh);
          } else {
            $this->_internal_error($fh, 'accept');
          }
        } else {
          my $totread = 0;
          for (;;) {
            my $data;
            $res = sysread($fh, $data, BUFSIZ);
            last if !defined $res || $res <= 0;
            $totread += $res;
            $this->{'fhs'}{$fh}{'inbuf'} .= $data;
          }
          if ($totread != 0) {
            my $olderr = $!;
            $this->{'cbobj'}->mux_input($this, $fh, \$this->{'fhs'}{$fh}{'inbuf'});
            next if !exists $this->{'fhs'}{$fh};
            last if $this->{'state'} != MUX_RUNNING || keys %{$this->{'fhs'}} == 0;
            $! = $olderr;
          }
          if (!defined $res || $res < 0) {
            $this->_internal_error($fh, 'read') if $! != EAGAIN;
          } elsif ($res == 0) {
            $this->{'cbobj'}->mux_eof($this, $fh, \$this->{'fhs'}{$fh}{'inbuf'});
          }
        }
        next if !exists $this->{'fhs'}{$fh};
        last if $this->{'state'} != MUX_RUNNING || keys %{$this->{'fhs'}} == 0;
      }

      if (FD_ISSET($this, $writefds, $fh)) {
        if ($this->{'fhs'}{$fh}{'state'} == MUX_FD_UNCONNECTED) {
          $! = $fh->getsockopt(Socket::SOL_SOCKET(), Socket::SO_ERROR());
          if ($! != 0) {
            $this->_internal_error($fh, 'connect');
          } else {
            $this->{'fhs'}{$fh}{'state'} = MUX_FD_REGULAR;
            FD_SET($this, $this->{'writefds'}, $fh, 0) if $this->{'fhs'}{$fh}{'outbuf'} eq '';
            FD_SET($this, $this->{'readfds'}, $fh, 1);
            $this->{'cbobj'}->mux_connected($this, $fh);
          }
        } else {
          while ($this->{'fhs'}{$fh}{'outbuf'} ne '') {
            $res = syswrite($fh, $this->{'fhs'}{$fh}{'outbuf'});
            last if !defined $res || $res < 0;
            substr($this->{'fhs'}{$fh}{'outbuf'}, 0, $res) = '';
          }
          if ((!defined $res || $res < 0) && $! != EAGAIN) {
            $this->_internal_error($fh, 'write') if $! != EAGAIN;
          } elsif ($this->{'fhs'}{$fh}{'outbuf'} eq '') {
            FD_SET($this, $this->{'writefds'}, $fh, 0);
            $this->close($fh) if $this->{'fhs'}{$fh}{'state'} == MUX_FD_CLOSE_WHEN_FLUSHED;
          }
        }
        last if $this->{'state'} != MUX_RUNNING || keys %{$this->{'fhs'}} == 0;
      }
    }
  }
}

# $mux->end()
# Terminate the loop as soon as possible.
#
sub end ($) {
  my ($this) = @_;
  $this->{'state'} = MUX_END;
}

# $mux->force_flush($fh)
# Make the specified file handle blocking and write all of its pending
# output. Once that is done, make the file handle non-blocking and return.
#
sub force_flush ($$) {
  my ($this, $fh) = @_;
  $this->_internal_error($fh, 'fcntl') if !defined $fh->blocking(1);
  while ($this->{'state'} == MUX_RUNNING
      && exists $this->{'fh'}{$fh}
      && $this->{'fhs'}{$fh}{'outbuf'} ne '') {
    my $len = syswrite($fh, $this->{'fhs'}{$fh}{'outbuf'});
    if ($len < 0) {
      $this->_internal_error($fh, 'write');
    } else {
      substr($this->{'fhs'}{$fh}{'outbuf'}, 0, $len) = '';
    }
  }
  $this->_internal_error($fh, 'fcntl') if !defined $fh->blocking(0);
}

# $mux->write($fh, $data)
# Queue up the data to the specified file handle.
#
sub write ($$$) {
  my ($this, $fh, $data) = @_;
  if ($data ne '') {
    $this->{'fhs'}{$fh}{'outbuf'} .= $data;
    FD_SET($this, $this->{'writefds'}, $fh, 1);
  }
}

# $mux->remove($fh)
# Remove the file handle from the muxer.
#
sub remove ($$) {
  my ($this, $fh) = @_;
  return if !exists $this->{'fhs'}{$fh};
  FD_SET($this, $this->{'readfds'}, $fh, 0);
  FD_SET($this, $this->{'writefds'}, $fh, 0);
  $this->set_timeout($fh, undef);
  delete ($this->{'fhs'}{$fh});
}

# $mux->close($fh)
# Close the specified handle and remove it from the muxer.
#
sub close ($$) {
  my ($this, $fh) = @_;
  return if !exists $this->{'fhs'}{$fh} || $this->{'fhs'}{$fh}{'state'} == MUX_FD_CLOSING;
  $this->{'fhs'}{$fh}{'state'} = MUX_FD_CLOSING;
  close($fh) || $this->_internal_error($fh, 'close');
  $this->remove($fh) if exists $this->{'fhs'}{$fh};
}

# $mux->close_when_flushed ($$)
# Close the specified handle and remove it from the muxer as soon as its
# pending output has been written. No input will be read from the handle and
# a possible timer will be removed.
#
sub close_when_flushed ($$) {
  my ($this, $fh) = @_;
  return if !exists $this->{'fhs'}{$fh} || $this->{'fhs'}{$fh}{'state'} == MUX_FD_CLOSING;
  if ($this->{'fhs'}{$fh}{'outbuf'} eq '') {
    $this->close($fh);
  } else {
    $this->set_timeout($fh, undef);
    FD_SET($this, $this->{'readfds'}, $fh, 0);
    $this->{'fhs'}{$fh}{'state'} = MUX_FD_CLOSE_WHEN_FLUSHED;
  }
}

1;
