/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include "tbox.h"
#include "sexpr.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "where.h"
#include "_where.h"


/*
** This function creates IN operator syntax tree expressions.
*/

int _where_in_syntax( in_cond_t *in_cond, sexpr_t *sexpr, dfile_t *dfile )
{
	char	*variable_name;
	dfile_bind_t	bind, *key, **bind_entry;
	sexpr_t	*list;
	unsigned long	list_cnt, hash_table_slot_cnt;
	void	*hash_table, *hash_slot;
	const unsigned long	max_collisions_allowed = 10;
	char	*value;
	dherrno_t	dherrno;
	int	ret;

	assert( in_cond != (in_cond_t *)0 );
	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	assert( SEXPR_CAR_TYPE( sexpr ) == string_sexpr );
	variable_name = SEXPR_CAR_STRING( sexpr );

	if ( *variable_name != '$' ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Expected variable but found [", stderr );
		(void) fputs( variable_name, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	++variable_name;
	bind.field_name = variable_name;
	key = &bind;

	bind_entry = (dfile_bind_t **)bsearch( (void *)&key, dfile->sorted_bind, dfile->bind_cnt, sizeof( dfile_bind_t * ), dfile_bind_field_name_cmp );

	if ( bind_entry == (dfile_bind_t **)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Variable [", stderr );
		(void) fputs( variable_name, stderr );
		(void) fputs( "] was not defined.\n", stderr );
		RETURN_INT( WHERE_UNKNVAR );
	}

	in_cond->dfile_bind = *bind_entry;

	if ( SEXPR_CDR_TYPE( sexpr ) != list_sexpr ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Expected list of literals to follow variable [", stderr );
		(void) fputs( variable_name, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	sexpr = SEXPR_CDR_LIST( sexpr );

	if ( sexpr == (sexpr_t *)0 || SEXPR_CAR_TYPE( sexpr ) != list_sexpr ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Expected list of literals to follow variable [", stderr );
		(void) fputs( variable_name, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	list = SEXPR_CAR_LIST( sexpr );
	list_cnt = 0;

	/*
	** Count elements in list.
	*/
	while ( list != (sexpr_t *)0 ) {
		if ( SEXPR_CAR_TYPE( list ) != string_sexpr ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Nested list encountered.\n", stderr );
			RETURN_INT( WHERE_SYNTAX );
		}
		++list_cnt;
		list = SEXPR_CDR_LIST( list );
	}

	if ( list_cnt == 0UL ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "List must contain at least on element.\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	hash_table_slot_cnt = list_cnt * 2UL;
	hash_table = dhcreate( hash_table_slot_cnt, max_collisions_allowed );
	if ( hash_table == (void *)0 ) {
		UNIX_ERROR( "dhcreate() failed" );
		RETURN_INT( WHERE_UNIXERR );
	}

	list = SEXPR_CAR_LIST( sexpr );

	/*
	** Fill hash table.
	*/
	while ( list != (sexpr_t *)0 ) {
		value = SEXPR_CAR_STRING( list );

		if ( *value == '$' ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Cannot use variable in list.\n", stderr );
			RETURN_INT( WHERE_SYNTAX );
		}

		if ( *value == '\'' ) {
			/*
			** Skip leading tick mark.
			*/
			++value;
		}

		if ( Debug ) {
			(void) fprintf( stderr, "Adding [%s] to hash table.\n", value );
		}

		hash_slot = dhesearch( (void *)value, &hash_table, (unsigned long (*)(const void *))strhkey, (int (*)(const void *, const void *))strcmp );

		if ( hash_slot == (void *)0 ) {
			dherrno = dherror();

			switch ( dherrno ) {
			case Dh_alloc:
				UNIX_ERROR( "dhesearch() failed" );
				ret = WHERE_UNIXERR;
				break;
			case Dh_collision:
				ret = WHERE_HASHFAIL;
				break;
			default:
				FPUT_SRC_CODE( stderr );
				(void) fputs( "programming error--unknown error (", stderr );
				(void) fput_int( dherrno, stderr );
				(void) fputs( ") occurred in dhesearch().\n", stderr );
				abort();
			}

			RETURN_INT( ret );
		}

		list = SEXPR_CDR_LIST( list );
	}

	in_cond->hash_table = hash_table;

	if ( SEXPR_CDR_LIST( sexpr ) != (sexpr_t *)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Too many arguments for IN operator.\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	RETURN_INT( WHERE_NOERR );
}
