/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include "tbox.h"


/*
** This function trims trailing spaces from end of string.
** The new string length is returned.
*/

size_t trim_spaces( char *str )
{
	char	*end;
	size_t	len;

	assert( str != (char *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( [%s] )\n", __func__, str );
	}

	DEBUG_FUNC_START;

	len = strlen( str );

	if ( len == (size_t)0 ) {
		if ( Debug ) {
			(void) fputs( "String is zero length.\n", stderr );
		}
		RETURN_INT( (size_t)0 );
	}

	end = &str[ len - (size_t)1 ];

	while ( end >= str ) {
		if ( !isspace( *end ) ) {
			break;
		}
		*end = (char)0;
		--end;
	}

	len = (size_t)( end - str + (ptrdiff_t)1 );

	if ( Debug ) {
		(void) fputs( "Trimmed string [", stderr );
		(void) fputs( str, stderr );
		(void) fputs( "].\n", stderr );
	}

	RETURN_INT( len );
}

#ifdef MT_trim_spaces

#include <stdlib.h>
/*
** This function is used to regression test trim_spaces().
** The following command is used to compile:
**   x=trim_spaces; make "MT_CC=-DMT_$x" $x
*/
int main( void )

{
	static const char	complete_msg[] =  ">>> Module test on function %s() is complete.\n";
	static const char	test_func[] = "trim_spaces";
	static const char	successful[] = ">>>\n>>> %s() was successful.\n";
	static const char	unsuccessful[] = ">>>\n>>> %s() was unsuccessful.\n";
	static const char	blank_line[] = ">>>\n";
	static const char	example[] = "Hello world!";
	char	str[1024];

	Debug = 1;

	(void) fprintf( stderr, ">>> Start module test on function %s().\n", test_func );
	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #1\n", stderr );
	(void) fputs( ">>> Trim string with no trailing spaces.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( str, example );
	if ( trim_spaces( str ) == strlen( example ) ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #2\n", stderr );
	(void) fputs( ">>> Trim zero length string.\n", stderr );
	(void) fputs( blank_line, stderr );

	str[ 0 ] = (char)0;
	if ( trim_spaces( str ) == (size_t)0 ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #3\n", stderr );
	(void) fputs( ">>> Trim string with one trailing space.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( str, example );
	(void) strcat( str, " " );
	if ( trim_spaces( str ) == strlen( example ) ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #4\n", stderr );
	(void) fputs( ">>> Trim string with three trailing spaces.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( str, example );
	(void) strcat( str, "   " );
	if ( trim_spaces( str ) == strlen( example ) ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #5\n", stderr );
	(void) fputs( ">>> Trim string that only contains spaces.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( str, "   " );
	if ( trim_spaces( str ) == (size_t)0 ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fprintf( stderr, complete_msg, test_func );
	exit( 0 );
}
#endif
