/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>

#include "tbox.h"

/*
** This function does an ASCII to double conversion.
*/
int atod( double *result, const char *str )
{
	char	*ptr, msg[100];
	size_t	len;

	assert( result != (double *) 0 );
	assert( str != (const char *) 0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( result %p, str [%s] )\n", __func__, result, str );
	}

	DEBUG_FUNC_START;

	len = strlen( str );

	if ( len == (size_t)0 ) {
		/*
		** Zero length string.
		*/
		*result = 0.0;

		if ( Debug ) {
			(void) fprintf( stderr, "result = %g\n", *result );
		}

		RETURN_INT( 0 );
	}

	errno = 0;
	*result = strtod( str, &ptr );

	if ( errno != 0 ) {
		(void) strcpy( msg, "strtod() failed to convert [" );
		(void) strncat( msg, str, (size_t)24 );
		(void) strcat( msg, "]" );
		UNIX_ERROR( msg );

		RETURN_INT( -1 );
	}

	if ( ptr != &str[ len ] ) {
		/*
		** Entire string was not converted.
		*/
		(void) fputs( __FILE__, stderr );
		(void) fprintf( stderr, "(%d)", __LINE__ );
		(void) fputs( ": strtod() failed to convert entire string [", stderr );
		(void) fputs( str, stderr );
		(void) fputs( "].\n", stderr );

		RETURN_INT( -1 );
	}

	if ( Debug ) {
		(void) fprintf( stderr, "result = %g\n", *result );
	}

	RETURN_INT( 0 );
}

#ifdef MT_atod
/*
** This function is used to regression test atod().
** The following command is used to compile:
** x=atod; make "MT_CC=-DMT_$x" "MT_PRE=DEFINE=MT_$x" $x
*/
int main( void )

{
	static const char	complete_msg[] =  ">>> Module test on function %s() is complete.\n";
	static const char	test_func[] = "atod";
	static const char	incorrectly_successful[] = ">>>\n>>> %s() was incorrectly successful.\n";
	static const char	correctly_unsuccessful[] = ">>>\n>>> %s() was correctly unsuccessful.\n";
	static const char	correctly_successful[] = ">>>\n>>> %s() was correctly successful.\n";
	static const char	incorrectly_unsuccessful[] = ">>>\n>>> %s() was incorrectly unsuccessful.\n";
	static const char	blank_line[] = ">>>\n";
	double	value;

	Debug = 1;

	(void) fprintf( stderr, ">>> Start module test on function %s().\n", test_func );
	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #1\n", stderr );
	(void) fputs( ">>> Check failure for non-numeric format.\n", stderr );
	(void) fputs( blank_line, stderr );

	if ( atod( &value, "abc" ) == 0 ) {
		(void) fprintf( stderr, incorrectly_successful, test_func );
	} else {
		(void) fprintf( stderr, correctly_unsuccessful, test_func );
	}

	if ( atod( &value, "123abc456" ) == 0 ) {
		(void) fprintf( stderr, incorrectly_successful, test_func );
	} else {
		(void) fprintf( stderr, correctly_unsuccessful, test_func );
	}

	if ( atod( &value, "1-2" ) == 0 ) {
		(void) fprintf( stderr, incorrectly_successful, test_func );
	} else {
		(void) fprintf( stderr, correctly_unsuccessful, test_func );
	}

	if ( atod( &value, "1-" ) == 0 ) {
		(void) fprintf( stderr, incorrectly_successful, test_func );
	} else {
		(void) fprintf( stderr, correctly_unsuccessful, test_func );
	}

	if ( atod( &value, "-" ) == 0 ) {
		(void) fprintf( stderr, incorrectly_successful, test_func );
	} else {
		(void) fprintf( stderr, correctly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #2\n", stderr );
	(void) fputs( ">>> Check for valid signed integer.\n", stderr );

	if ( atod( &value, "-1" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	if ( atod( &value, "+12345678" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #3\n", stderr );
	(void) fputs( ">>> Check for valid unsigned integer.\n", stderr );

	if ( atod( &value, "5" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	if ( atod( &value, "567890" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #4\n", stderr );
	(void) fputs( ">>> Check for valid signed real.\n", stderr );

	if ( atod( &value, "-1.5" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	if ( atod( &value, "+.5" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	if ( atod( &value, "-.5098765" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	if ( atod( &value, "+598765.5098765" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #5\n", stderr );
	(void) fputs( ">>> Check for valid unsigned real.\n", stderr );

	if ( atod( &value, "1.5" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	if ( atod( &value, ".5" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	if ( atod( &value, ".5098765" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	if ( atod( &value, "598765.5098765" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #6\n", stderr );
	(void) fputs( ">>> Check for valid scientific notation.\n", stderr );

	if ( atod( &value, "1.567e-07" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	if ( atod( &value, "1.567E+003" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	if ( atod( &value, "-1.567E+003" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fprintf( stderr, complete_msg, test_func );
	exit( 0 );
}
#endif
