/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"


#define	MAGIC_OFFSET	0
#define	METHOD_OFFSET	2
#define	FLAGS_OFFSET	3
#define	TIME_OFFSET	4
#define	XFLAGS_OFFSET	8
#define	OS_OFFSET	9
#define	XLEN_OFFSET	10

/* gzip flag byte */
#define ASCII_FLAG	0x01 /* bit 0 set: file probably ascii text */
#define HEAD_CRC	0x02 /* bit 1 set: header CRC present */
#define EXTRA_FIELD	0x04 /* bit 2 set: extra field present */
#define ORIG_NAME	0x08 /* bit 3 set: original file name present */
#define COMMENT		0x10 /* bit 4 set: file comment present */
#define RESERVED	0xE0 /* bits 5..7: reserved */

#define	DEFLATE_METHOD	8

#define	XLEN_SIZE	2

static int skip_str( dfile_t * );

/*
** This function reads and verifies the header of a GZIP compressed file.
** Header is assumed to be at the beginning of the file, and not larger
** than the I/O buffer.
*/

int _dfile_read_gz_header( dfile_t *dfile )
{
	static const unsigned char	gz_magic[2] = {0x1f, 0x8b};
	int	flags;
	size_t	xlen_len;
	unsigned char	header[OS_OFFSET+1], xlen[XLEN_SIZE];

	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	/*
	** Attempt to get first byte of header.
	*/
	if ( _dfile_copy_raw_gz_data( header, (size_t)1, dfile ) == -1 ) {
		RETURN_INT( -1 );
	}

	if ( _dfile_copy_raw_gz_data( &header[ 1 ], sizeof( header ) - (size_t)1, dfile ) == -1 ) {
		if ( dfile->error == Dfile_end_of_file ) {
			dfile->error = Dfile_invalid_gzip_format;
		}
		RETURN_INT( -1 );
	}

	if ( memcmp( (void *)&header[ MAGIC_OFFSET ], (void *)gz_magic, sizeof( gz_magic ) ) != 0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "GZIP file has invalid magic value.\n", stderr );
		dfile->error = Dfile_invalid_gzip_format;
		RETURN_INT( -1 );
	}

	if ( header[ METHOD_OFFSET ] != (unsigned char)DEFLATE_METHOD ) {
		dfile->error = Dfile_invalid_gzip_format;
		FPUT_SRC_CODE( stderr );
		(void) fputs( "File was not zipped using deflate compression (gzip's default compression).\n", stderr );
		RETURN_INT( -1 );
	}

	flags = (int)header[ FLAGS_OFFSET ];

	if ( ( flags & RESERVED ) != 0 ) {
		dfile->error = Dfile_invalid_gzip_format;
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Header contains invalid flags.\n", stderr );
		RETURN_INT( -1 );
	}

	/*
	** Ignore time and OS type.
	*/

	if ( ( flags & EXTRA_FIELD ) == 0 ) {
		xlen_len = (size_t)0;
	} else {
		/*
		** Skip extra fields.
		*/
		if ( _dfile_copy_raw_gz_data( xlen, sizeof( xlen ), dfile ) == -1 ) {
			if ( dfile->error == Dfile_end_of_file ) {
				dfile->error = Dfile_invalid_gzip_format;
			}
			RETURN_INT( -1 );
		}
		xlen_len = (size_t)xlen[ 0 ];
		xlen_len += (size_t)xlen[ 1 ] << 8;
	}

	if ( xlen_len > (size_t)0 ) {
		if ( _dfile_copy_raw_gz_data( (unsigned char *)0, xlen_len, dfile ) == -1 ) {
			if ( dfile->error == Dfile_end_of_file ) {
				dfile->error = Dfile_invalid_gzip_format;
			}
			RETURN_INT( -1 );
		}
	}

	if ( ( flags & ORIG_NAME ) != 0 ) {
		/*
		** Find end of original file name.
		*/
		if ( skip_str( dfile ) == -1 ) {
			if ( dfile->error == Dfile_end_of_file ) {
				dfile->error = Dfile_invalid_gzip_format;
			}
			RETURN_INT( -1 );
		}
	}

	if ( ( flags & COMMENT ) != 0 ) {
		/*
		** Find end of .gz file comment.
		*/
		if ( skip_str( dfile ) == -1 ) {
			if ( dfile->error == Dfile_end_of_file ) {
				dfile->error = Dfile_invalid_gzip_format;
			}
			RETURN_INT( -1 );
		}
	}

	if ( ( flags & HEAD_CRC ) != 0 ) {
		/*
		** Skip header CRC.
		*/
		if ( _dfile_copy_raw_gz_data( (unsigned char *)0, (size_t)2, dfile ) == -1 ) {
			if ( dfile->error == Dfile_end_of_file ) {
				dfile->error = Dfile_invalid_gzip_format;
			}
			RETURN_INT( -1 );
		}
	}

	RETURN_INT( 0 );
}

static int skip_str( dfile_t *dfile )
{
	unsigned char	ch;

	do {
		if ( _dfile_copy_raw_gz_data( &ch, (size_t)1, dfile ) == -1 ) {
			return -1;
		}
	} while ( ch != (unsigned char)0 );

	return 0;
}
