/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <zlib.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"


/*
** This function GZIPs an ASCII buffer to a compressed buffer. When
** the compressed buffer is full, it is written to disk.
*/

int _dfile_flush_zipped_buffer( dfile_t *dfile )
{
	ssize_t	write_cnt;
	size_t	byte_cnt;
	dfile_buffer_t	*buffer;
	int	zerr, flush_flag;
	z_streamp	zstream;

	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	zstream = (z_streamp)dfile->zstream;
	assert( zstream != (z_stream *)0 );

	buffer = dfile->io_buffer;
	assert( buffer != (dfile_buffer_t *)0 );
	assert( buffer->buf_ptr >= buffer->buffer_area );
	zstream->avail_in = (unsigned int)( buffer->buf_ptr - buffer->buffer_area );
	zstream->next_in = (unsigned char *)buffer->buffer_area;
	dfile->crc = crc32( dfile->crc, zstream->next_in, zstream->avail_in );
	dfile->file_char_cnt += zstream->avail_in;

	flush_flag = ( buffer->processing_complete_flag == 'Y' ) ? Z_FINISH : Z_NO_FLUSH;
	zerr = Z_OK;

	while ( zstream->avail_in > 0U || ( flush_flag == Z_FINISH && zerr != Z_STREAM_END ) ) {
		if ( Debug ) {
			(void) fprintf( stderr, "deflate() started at %s\n", get_ctime() );
		}
		zerr = deflate( zstream, flush_flag );
		if ( Debug ) {
			(void) fprintf( stderr, End_func_msg_int, "deflate", zerr, get_ctime() );
		}
		if ( zerr != Z_OK && zerr != Z_STREAM_END ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "deflate() failed (", stderr );
			(void) fput_int( zerr, stderr );
			(void) fputs( "): ", stderr );
			if ( zstream->msg == (char *)0 ) {
				(void) fputs( "no description available of error", stderr );
			} else {
				(void) fputs( zstream->msg, stderr );
			}
			(void) fputs( ".\n", stderr );
			dfile->error = Dfile_invalid_gzip_format;
			RETURN_INT( -1 );
		}

		if ( zstream->avail_out == 0U || zerr == Z_STREAM_END ) {
			byte_cnt = dfile->buffer_length - zstream->avail_out;
			if ( Debug ) {
				(void) fprintf( stderr, "Writing %u bytes.\n", byte_cnt );
			}
			write_cnt = write( dfile->open_descriptor, (void *)dfile->zipped_buffer, byte_cnt );
			if ( write_cnt < (ssize_t)0 ) {
				UNIX_ERROR( "write() failed" );
				dfile->error = Dfile_io_failed;
				RETURN_INT( -1 );
			}
			if ( write_cnt != byte_cnt ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "write() did not output expected number bytes.\n", stderr );
				dfile->error = Dfile_io_failed;
				RETURN_INT( -1 );
			}
			zstream->avail_out = (unsigned int)dfile->buffer_length;
			zstream->next_out = dfile->zipped_buffer;
		}
	}

	buffer->buf_ptr = buffer->buffer_area;
	buffer->buffer_rec_cnt = 0UL;

	if ( buffer->processing_complete_flag == 'Y' ) {
		if ( _dfile_write_zipped_crc( dfile ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	RETURN_INT( 0 );
}
