/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <limits.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_utility.h"
#include "sexpr.h"
#include "dfile_sort.h"


/*
** This function parses one key from command line key argument.
*/
int parse_one_key( order_by_t *key, const char *field_beg, const char *field_end )
{
	const char 	*beg, *delimiter;
	size_t	parse_len, field_name_len;
	char	ascend_descend, value_type;
	const char	default_ascend_descend = 'A';
	const char	default_value_type = 'A';
	const char	expected_delimiter = '.';

	assert( key != (order_by_t *)0 );
	assert( field_beg != (const char *)0 );
	assert( field_end != (const char *)0 );

	DEBUG_FUNC_START;

	/*
	** Initialize order_by_t structure.
	*/
	key->field_name = (char *)0;
	key->ascend_descend = default_ascend_descend;
	key->value_type = default_value_type;

	beg = field_beg;
	assert( field_end >= field_beg );
	parse_len = (size_t) ( field_end - field_beg );

	delimiter = memchr( beg, expected_delimiter, parse_len );

	if ( delimiter == (const char *)0 ) {
		field_name_len = parse_len;
	} else {
		assert( delimiter > beg );
		field_name_len = (size_t)( delimiter - beg );
	}

	key->field_name = (char *)malloc( field_name_len + (size_t)1 );
	if ( key->field_name == (char *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	(void) memcpy( (void *)key->field_name, (void *)beg, field_name_len );
	key->field_name[ field_name_len ] = (char)0;

	assert( parse_len >= field_name_len );
	parse_len -= field_name_len;

	if ( parse_len <= (size_t)1 ) {
		/*
		** Only field name and possibly its delimiter was given.
		*/
		RETURN_INT( 0 );
	}

	beg = &delimiter[ 1 ];

	ascend_descend = toupper( *beg );

	if ( ascend_descend != 'A' && ascend_descend != 'D' ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Encountered [", stderr );
		(void) fputc( ascend_descend, stderr );
		(void) fputs( "] when parsing (A)scend/(D)escend.\n", stderr );
		RETURN_INT( -1 );
	}

	key->ascend_descend = ascend_descend;

	parse_len -= (size_t)2;

	if ( parse_len >= (size_t)1 ) {
		delimiter = &beg[ 1 ];
		if ( *delimiter != expected_delimiter ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Expected delimiter character (,) but found [", stderr );
			(void) fputc( *delimiter, stderr );
			(void) fputs( "].\n", stderr );
			RETURN_INT( -1 );
		}
	}

	if ( parse_len <= (size_t)1 ) {
		/*
		** value_type was omitted.
		*/
		RETURN_INT( 0 );
	}

	parse_len -= (size_t)2;

	if ( parse_len != (size_t)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Expected value type field [", stderr );
		(void) fwrite( (void *)beg, sizeof( char ), parse_len, stderr );
		(void) fputs( "] to be a single character.", stderr );
		RETURN_INT( -1 );
	}

	beg = &delimiter[ 1 ];
	value_type = toupper( *beg );
	if ( value_type != 'A' && value_type != 'N' && value_type != 'H' ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Encountered [", stderr );
		(void) fputc( value_type, stderr );
		(void) fputs( "] when parsing (A)SCII/(N)umeric/(H)igh value null.\n", stderr );
		RETURN_INT( -1 );
	}

	key->value_type = value_type;

	RETURN_INT( 0 );
}
