/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <limits.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_utility.h"
#include "sexpr.h"
#include "where.h"
#include "dfile_sort.h"


/*
** This function creates a merge queue.
*/
int create_merge_queue( merge_queue_t **merge_queue, dfile_t **merge_dfile_tbl, unsigned short merge_dfile_tbl_cnt, unsigned short **merge_io_field_map, order_by_t *key_tbl, unsigned short **merge_key_ndx_tbl, unsigned short key_tbl_cnt, sm_dfile_t *sm_dfile )
{
	dfile_t	*dfile;
	merge_queue_t	*link;
	int	read_ret;
	where_result_t	where_result;
	void	*where;
	char	err_msg[ 256 ];

	assert( merge_queue != (merge_queue_t **)0 );
	assert( merge_dfile_tbl != (dfile_t **)0 );
	assert( merge_io_field_map != (unsigned short **)0 );
	assert( merge_key_ndx_tbl != (unsigned short **)0 );
	assert( key_tbl != (order_by_t *)0 );
	assert( sm_dfile != (sm_dfile_t *)0 );

	DEBUG_FUNC_START;

	*merge_queue = (merge_queue_t *)0;

	for ( ; merge_dfile_tbl_cnt > (unsigned short)0; --merge_dfile_tbl_cnt ) {
		dfile = *merge_dfile_tbl;

		if ( sm_dfile->where == (sexpr_t *)0 ) {
			where = (void *)0;
		} else {
			if ( where_compile_sexpr( &where, err_msg, sizeof( err_msg ), (void *)sm_dfile->where, dfile ) != WHERE_NOERR ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "where_compile_sexpr() failed [", stderr );
				(void) fputs( err_msg, stderr );
				(void) fputs( "].\n", stderr );

				RETURN_INT( -1 );
			}

			assert( where != (void *)0 );
		}

		/*
		** Loop until an unfiltered record is found.
		*/
		while ( ( read_ret = dfile_read( dfile ) ) == 0 ) {
			if ( where == (void *)0 ) {
				/*
				** No filter defined.
				*/
				break;
			}

			if ( where_condition( &where_result, err_msg, sizeof( err_msg ), where ) != WHERE_NOERR ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "where_condition() failed [", stderr );
				(void) fputs( err_msg, stderr );
				(void) fputs( "].\n", stderr );

				RETURN_INT( -1 );
			}

			if ( where_result == Where_result_true ) {
				/*
				** Unfiltered record found.
				*/
				break;
			}
		}

		if ( read_ret == -1 ) {
			if ( dfile->error == Dfile_all_data_processed ) {
				/*
				** File is empty.
				*/
				++merge_dfile_tbl;
				++sm_dfile;
				++merge_io_field_map;
				++merge_key_ndx_tbl;
				continue;
			} else {
				RETURN_INT( -1 );
			}
		}

		link = (merge_queue_t *)malloc( sizeof( merge_queue_t ) );
		if ( link == (merge_queue_t *)0 ) {
			UNIX_ERROR( "malloc() failed" );
			RETURN_INT( -1 );
		}

		link->dfile = dfile;
		link->where = where;
		link->io_field_map = *merge_io_field_map;
		link->key_ndx_tbl = *merge_key_ndx_tbl;
		link->next = (merge_queue_t *)0;

		if ( add_to_merge_queue( merge_queue, link, key_tbl, key_tbl_cnt ) == -1 ) {
			RETURN_INT( -1 );
		}

		++merge_dfile_tbl;
		++sm_dfile;
		++merge_io_field_map;
		++merge_key_ndx_tbl;
	}

	RETURN_INT( 0 );
}
