/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <search.h>
#include <assert.h>
#include "tbox.h"
#include "rbtree.h"
#include "sexpr.h"
#include "dfile_exec.h"


/*
** This function starts as many available application procs (slices) as
** possible without exceeding maximum concurrent procs.
*/
int start_procs( unsigned short *procs_running_cnt, unsigned short *allocated_proc_cnt, proc_t *proc_list, void **proc_tree, char **divvy_text, unsigned short divvy_text_cnt, unsigned short max_concurrent_procs, int semid, unsigned short procs_remaining_cnt, step_t *step, int log_fd, const char *job_name )
{
	char	*stdin_fname, *stdout_fname, *stderr_fname;
	pid_t	pid;
	int	stdout_fd, stderr_fd;
	proc_t	**ptr;
	unsigned short	proc_cnt;
	int	resource_flag;
	char	*slice_text;

	assert( procs_running_cnt != (unsigned short *)0 );
	assert( proc_list != (proc_t *)0 );
	assert( proc_tree != (void **)0 );
	assert( step != (step_t *)0 );
	assert( log_fd >= 0 );

	DEBUG_FUNC_START;

	proc_cnt = ( divvy_text_cnt == (unsigned short)0 ) ? (unsigned short)1 : divvy_text_cnt;

	/*
	** Cycle through proc table looking for procs to start.
	*/
	for ( ; proc_list != (proc_t *)0; proc_list = proc_list->next ) {
		if ( *procs_running_cnt >= max_concurrent_procs ) {
			/*
			** Don't start any more procs.
			*/
			break;
		}

		if ( Debug ) {
			fprintf( stderr, "*allocated_proc_cnt = %hu, max_concurrent_procs = %hu, procs_remaining_cnt = %hu, *procs_running_cnt = %hu\n", *allocated_proc_cnt, max_concurrent_procs, procs_remaining_cnt, *procs_running_cnt );
		}

		if ( semid >= 0 && max_concurrent_procs > *allocated_proc_cnt && procs_remaining_cnt > *allocated_proc_cnt && *procs_running_cnt >= *allocated_proc_cnt ) {
			if ( request_proc_resource( &resource_flag, *allocated_proc_cnt, semid ) == -1 ) {
				RETURN_INT( -1 );
			}
			if ( resource_flag == 0 ) {
				/*
				** Could not get another resource.
				*/
				break;
			}
			++*allocated_proc_cnt;
		}

		if ( proc_list->pid != (pid_t)0 ) {
			/*
			** Job previously started.
			*/
			if ( proc_list->status <= 0 ) {
				/*
				** proc currently running, or
				** proc completed successfully.
				*/
				continue;
			}
		}

		/*
		** Job ran and failed, retry failed,
		** or start proc for first attempt.
		*/

		if ( divvy_text_cnt == (unsigned short)0 ) {
			slice_text = "%s";
#if 0
			proc_list->partition = (const char *)0;
#endif
		} else {
			assert( divvy_text_cnt > proc_list->partition_ndx );
			slice_text = divvy_text[ proc_list->partition_ndx ];
#if 0
			proc_list->partition = slice_text;
#endif
		}

		if ( assign_exec_args( &proc_list->exec_args, step->exec_arg, step->exec_arg_cnt, slice_text, proc_list->partition_ndx ) == -1 ) {
			RETURN_INT( -1 );
		}

		/*
		** Assign file names to capture stdout and stderr of
		** application about to be started.
		*/
		if ( assign_fname( &stdin_fname, step->stdin_fname, proc_list->partition_ndx, divvy_text_cnt, slice_text ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( assign_fname( &stdout_fname, step->stdout_fname, proc_list->partition_ndx, divvy_text_cnt, slice_text ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( assign_fname( &stderr_fname, step->stderr_fname, proc_list->partition_ndx, divvy_text_cnt, slice_text ) == -1 ) {
			RETURN_INT( -1 );
		}

		/*
		** Start new instance of application (proc) to process a slice.
		*/
		pid = exec_proc( &stdout_fd, &stderr_fd, proc_list->exec_args, stdin_fname, stdout_fname, stderr_fname );
		if ( pid == (pid_t)-1 ) {
			RETURN_INT( -1 );
		}

		/*
		** Save proc information.
		*/
		proc_list->stdout_fd = stdout_fd;
		proc_list->stderr_fd = stderr_fd;
		proc_list->pid = pid;
		proc_list->status = -1;

		if ( write_log_record( log_fd, job_name, step->step_name, proc_list->partition, pid, 'S', (int *)0, (int *)0 ) == -1 ) {
			RETURN_INT( -1 );
		}

		/*
		** Record proc in tree for Unix process ID lookups.
		*/
		ptr = (proc_t **)rbtsearch( (void *)proc_list, proc_tree, proc_cmp );
		if ( ptr == (proc_t **)0 ) {
			/*
			** Out of memory.
			*/
			RETURN_INT( -1 );
		}

		if ( *ptr != proc_list ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Duplicate process id ", stderr );
			(void) fprintf( stderr, "%d", pid );
			(void) fputs( " encountered.\n", stderr );
			RETURN_INT( -1 );
		}

		/*
		** Job successfully started and recorded.
		*/
		++*procs_running_cnt;

#if 0
		(void) fputs( get_ctime(), stderr );
		(void) fputs( " Started process ", stderr );
		(void) fprintf( stderr, "%d", pid );
		(void) fputs( " for slice ", stderr );
		(void) fprintf( stderr, "%hu", proc_list->partition_ndx );
		if ( slice_text != (char *)0 ) {
			(void) fputs( " [", stderr );
			(void) fputs( slice_text, stderr );
			(void) fputs( "]", stderr );
		}
		(void) fputs( ".\n", stderr );
#endif
	}

	RETURN_INT( 0 );
}
