/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <tbox.h>
#include "sexpr.h"
#include "dfile_exec.h"

/*
** This function processes a job step.
*/
int process_step( step_t *step, completed_t *completed, unsigned long completed_cnt, int cpu_semid, int log_fd, const char *job_name )
{
	unsigned short	procs_remaining_cnt, procs_running_cnt;
	unsigned short	divvy_text_cnt, proc_list_cnt;
	unsigned short	allocated_proc_cnt, defined_proc_cnt;
	unsigned short	unneeded_resource_cnt;
	char	**divvy_text;
	int	exit_code, ignore_exit_code;
	proc_t	*proc_tbl, *proc_list;
	void 	*proc_tree;

	assert( step != (step_t *)0 );
	assert( log_fd >= 0 );

	DEBUG_FUNC_START;

	if ( step->partition == (char *)0 ) {
		defined_proc_cnt = (unsigned short)1;
		divvy_text_cnt = (unsigned short)0;
		divvy_text = (char **)0;
	} else {
		if ( get_divvy_text( &divvy_text, &divvy_text_cnt, step->partition ) == -1 ) {
			RETURN_INT( -1 );
		}

		defined_proc_cnt = divvy_text_cnt;
	}

	/*
	** This function allocates proc table array. Its size corresponds to
	** number of slices to be processed.
	*/
	proc_tbl = alloc_proc_tbl( defined_proc_cnt );
	if ( proc_tbl == (proc_t *)0 ) {
		RETURN_INT( -1 );
	}

	if ( set_proc_list( &proc_list, &proc_list_cnt, proc_tbl, completed, completed_cnt, divvy_text, divvy_text_cnt, step->step_name, step->successful_exit ) == -1 ) {
		RETURN_INT( -1 );
	}

	procs_running_cnt = (unsigned short)0;
	procs_remaining_cnt = proc_list_cnt;
	allocated_proc_cnt = (unsigned short)0;
	proc_tree = (void *)0;
	exit_code = 0;

	/*
	** Loop until all slices have been processes.
	*/
	while ( procs_remaining_cnt > (unsigned short)0 ) {
		/*
		** Start as many available application procs (slices) as
		** possible without exceeding maximum concurrent procs.
		*/
		if ( start_procs( &procs_running_cnt, &allocated_proc_cnt, proc_list, &proc_tree, divvy_text, divvy_text_cnt, step->max_processes, cpu_semid, procs_remaining_cnt, step, log_fd, job_name ) == -1 ) {
			RETURN_INT( -1 );
		}

		/*
		** Wait for a proc to complete, interpret proc's exit code
		** and perform post proc cleanup.
		*/
		if ( wait_proc( &procs_remaining_cnt, &proc_tree, &exit_code, &proc_list, log_fd, job_name, step->step_name ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( Debug ) {
			fprintf( stderr, "allocated_proc_cnt = %hu, procs_remaining_cnt = %hu\n", allocated_proc_cnt, procs_remaining_cnt );
		}

		if ( allocated_proc_cnt > procs_remaining_cnt ) {
			unneeded_resource_cnt = allocated_proc_cnt - procs_remaining_cnt;
			(void) release_allocated_procs( unneeded_resource_cnt, cpu_semid );
			allocated_proc_cnt -= unneeded_resource_cnt;
		}

		if ( Debug ) {
			(void) fprintf( stderr, "procs_remaining_cnt = %hu\n", procs_remaining_cnt );
		}

		--procs_running_cnt;

		if ( !step->successful_exit[ exit_code ] ) {
			break;
		}

	}

	while ( procs_running_cnt > (unsigned short)0 ) {
		if ( wait_proc( &procs_remaining_cnt, &proc_tree, &ignore_exit_code, &proc_list, log_fd, job_name, step->step_name ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( Debug ) {
			fprintf( stderr, "allocated_proc_cnt = %hu, procs_remaining_cnt = %hu\n", allocated_proc_cnt, procs_remaining_cnt );
		}

		if ( allocated_proc_cnt > (unsigned short)0 ) {
			(void) release_allocated_procs( (unsigned short)1, cpu_semid );
			--allocated_proc_cnt;
		}
		--procs_running_cnt;
	}

	if ( !step->successful_exit[ exit_code ] ) {
		RETURN_INT( -1 );
	}

	RETURN_INT( 0 );
}
